/***************************************************************************
 *            camel-kolab-imapx-metadata.c
 *
 *  Tue Oct 19 18:58:03 2010
 *  Copyright  2010  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
/*----------------------------------------------------------------------------*/

#include <string.h>

#include <camel/camel.h>

#include <camel/providers/imapx/camel-imapx-store.h>
#include <camel/providers/imapx/camel-imapx-server.h>
#include <camel/providers/imapx/camel-imapx-utils.h>

/* Kolab error reporting */
#include <libekolabutil/kolab-util-error.h>

#include "camel-kolab-imapx-metadata.h"

/*----------------------------------------------------------------------------*/

static gchar *kolab_folder_type_inv_map[] = {
	"---INVALID---",	/* KOLAB_FOLDER_TYPE_INVAL */
	"---UNKNOWN---",	/* KOLAB_FOLDER_TYPE_UNKNOWN */

	"mail",			/* KOLAB_FOLDER_TYPE_EMAIL */
	"mail.inbox",		/* KOLAB_FOLDER_TYPE_EMAIL_INBOX */
	"mail.drafts",		/* KOLAB_FOLDER_TYPE_EMAIL_DRAFTS */
	"mail.sentitems",	/* KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS */
	"mail.junkemail",	/* KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL */

	"event",		/* KOLAB_FOLDER_TYPE_EVENT */
	"event.default",	/* KOLAB_FOLDER_TYPE_EVENT_DEFAULT */
	"journal",		/* KOLAB_FOLDER_TYPE_JOURNAL */
	"journal.default",	/* KOLAB_FOLDER_TYPE_JOURNAL_DEFAULT */
	"task",			/* KOLAB_FOLDER_TYPE_TASK */
	"task.default",		/* KOLAB_FOLDER_TYPE_TASK_DEFAULT */
	"note",			/* KOLAB_FOLDER_TYPE_NOTE */
	"note.default",		/* KOLAB_FOLDER_TYPE_NOTE_DEFAULT */

	"contact",		/* KOLAB_FOLDER_TYPE_CONTACT */
	"contact.default"	/* KOLAB_FOLDER_TYPE_CONTACT_DEFAULT */
};

static KolabFolderTypeID kolab_folder_type_nums[KOLAB_FOLDER_LAST_TYPE];
static GHashTable *kolab_folder_type_map = NULL;

/*----------------------------------------------------------------------------*/

static void
kolab_folder_destroy_type_map (void)
{
	if (kolab_folder_type_map != NULL) {
		g_hash_table_destroy (kolab_folder_type_map);
		kolab_folder_type_map = NULL;
	}
}

static void
kolab_folder_build_type_map (void)
{
	gint ii;
	kolab_folder_destroy_type_map ();
	kolab_folder_type_map = g_hash_table_new (g_str_hash, g_str_equal);
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++) {
		kolab_folder_type_nums[ii] = ii;
		g_hash_table_insert (kolab_folder_type_map,
		                     kolab_folder_type_inv_map[ii],
		                     &(kolab_folder_type_nums[ii]));
	}
}

static KolabFolderTypeID
kolab_folder_map_get_type_id (const gchar *typestring, GError **err)
{
	/* when looking up a value from kolab_folder_type_map, store
	 * it in gpointer, check for NULL, then dereference and cast
	 * to KolabFolderTypeID
	 */
	gpointer map_entry;
	KolabFolderTypeID id;

	g_assert (kolab_folder_type_map != NULL);
	g_assert (typestring != NULL);

	map_entry = g_hash_table_lookup (kolab_folder_type_map,
	                                 typestring);

	if (map_entry == NULL) {
		/* this would be a Kolab format error from server */
		g_set_error (err,
		             KOLAB_CAMEL_KOLAB_ERROR,
		             KOLAB_CAMEL_KOLAB_ERROR_FORMAT,
		             "%s: invalid folder type string",
		             __func__);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	
	id = *((KolabFolderTypeID*)map_entry);
	return id;
}

/*----------------------------------------------------------------------------*/

CamelKolabFolderMetaData*
kolab_folder_meta_data_new (void)
{
	CamelKolabFolderMetaData *kfmd;
	kfmd = g_new0 (CamelKolabFolderMetaData, 1);
	kfmd->folder_type = KOLAB_FOLDER_TYPE_INVAL;
	return kfmd;
}

gboolean
kolab_folder_meta_data_free (CamelKolabFolderMetaData *kfmd)
{
	if (kfmd == NULL)
		return TRUE;
	g_free (kfmd);
	return TRUE;
}

static void
kolab_folder_meta_data_gdestroy (gpointer data)
{
	CamelKolabFolderMetaData *kfmd = (CamelKolabFolderMetaData *)data;
	(void)kolab_folder_meta_data_free (kfmd);
}

static CamelKolabFolderMetaData*
kolab_folder_meta_data_new_from_imapx_annotation (CamelIMAPXMetaAnnotation *man,
                                                  CamelIMAPXServer *server,
                                                  GError **err)
{
	CamelKolabFolderMetaData *kfmd;
	CamelIMAPXMetaAttrib *ma;
	CamelIMAPXMetaDataSpec *spec;
	GError *tmp_err = NULL;
	gchar *typestring;

	g_assert (server != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, NULL);

	if (man == NULL)
		return NULL;
	g_assert (man->entries != NULL);

	/* search for folder type info */
	spec = __KOLAB_imapx_meta_data_spec_new (server->metadata, /* to get protocol type */
	                                         NULL, /* search starts at annotation */
	                                         "/vendor/kolab/folder-type",
	                                         NULL, /* "value" will be substituted */
	                                         &tmp_err);
	if (spec == NULL) {
		g_propagate_error (err, tmp_err);
		return NULL;
	}

	kfmd = kolab_folder_meta_data_new ();
	
	ma = __KOLAB_imapx_meta_data_get_attrib_from_annotation (man, spec);
	__KOLAB_imapx_meta_data_spec_free (spec);

	if (ma == NULL) {
		/* no folder type info available */
		kfmd->folder_type = KOLAB_FOLDER_TYPE_UNKNOWN;
		return kfmd;
	}

	g_assert (ma->data != NULL);
	g_assert (ma->type != NULL);

	if (ma->type[CAMEL_IMAPX_METADATA_ACCESS_SHARED] != CAMEL_IMAPX_META_ATTRIB_TYPE_UTF8) {
		/* protocol violation - kolab folder type
		 * string must be an UTF-8 shared value
		 */
		kolab_folder_meta_data_free (kfmd);
		g_set_error (err,
		             KOLAB_CAMEL_KOLAB_ERROR,
		             KOLAB_CAMEL_KOLAB_ERROR_FORMAT,
		             "%s: invalid folder type string encoding",
		             __func__);
		return NULL;
	}

	typestring = (gchar*) ma->data[CAMEL_IMAPX_METADATA_ACCESS_SHARED]->data;
	kfmd->folder_type = kolab_folder_map_get_type_id (typestring, &tmp_err);
	if (kfmd->folder_type == KOLAB_FOLDER_TYPE_INVAL) {
		/* protocol violation - invalid kolab folder type string */
		kolab_folder_meta_data_free (kfmd);
		g_propagate_error (err, tmp_err);
		return NULL;
	}
	
	return kfmd;
}

/*----------------------------------------------------------------------------*/

CamelKolabIMAPXMetaData*
kolab_imapx_meta_data_new (void)
{
	CamelKolabIMAPXMetaData *kmd;
	kmd = g_new0 (CamelKolabIMAPXMetaData, 1);
	kmd->mdb = NULL;
	kmd->kolab_metadata = g_hash_table_new_full (g_str_hash,
	                                             g_str_equal,
	                                             g_free,
	                                             kolab_folder_meta_data_gdestroy);

	return kmd;
}

gboolean
kolab_imapx_meta_data_free (CamelKolabIMAPXMetaData *kmd)
{
	if (kmd == NULL)
		return TRUE;

	if (kmd->kolab_metadata)
		/* need hash table with key:val destroy function set */
		g_hash_table_destroy (kmd->kolab_metadata);

	g_free (kmd);
	return TRUE;
}

gboolean
kolab_imapx_meta_data_init (CamelKolabIMAPXMetaData *kmd,
                            CamelService* service,
                            CamelSession* session,
                            GError** err)
{
	GError *tmp_err = NULL;
	gboolean db_ok;

	g_assert (kmd != NULL);
	g_assert (service != NULL);
	g_assert (session != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	if (kmd->mdb == NULL)
		kmd->mdb = kolab_imapx_meta_data_db_new ();

	db_ok = kolab_imapx_meta_data_db_open (kmd->mdb, service, session, &tmp_err);
	if (!db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	db_ok = kolab_imapx_meta_data_db_init (kmd->mdb, &tmp_err);
	if (!db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	kolab_folder_build_type_map ();

	/* TODO call to __KOLAB_imapx_meta_data_init? */

	return TRUE;
}

gboolean
kolab_imapx_meta_data_uninit (CamelKolabIMAPXMetaData *kmd,
                              GError** err)
{
	GError *tmp_err = NULL;
	gboolean db_ok;

	g_assert (kmd != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	db_ok = kolab_imapx_meta_data_db_close (kmd->mdb, &tmp_err);
	if (!db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	kolab_folder_destroy_type_map ();

	/* TODO call to __KOLAB_imapx_meta_data_uninit? */

	return TRUE;
}

static void
kolab_meta_data_addreplace_from_imapx (CamelKolabIMAPXMetaData *kmd,
                                       CamelIMAPXMetaData *md,
                                       CamelIMAPXServer *server)
{
	/* TODO error reporting */

	GHashTableIter mbox_iter;
	gpointer mbox_key;
	gpointer mbox_value;
	GError *tmp_err = NULL;

	gchar *folder_name;
	CamelIMAPXMetaAnnotation *man;
	CamelKolabFolderMetaData *kfmd;
	
	g_assert (kmd != NULL);
	g_assert (md != NULL);
	g_assert (server != NULL);
	
	g_hash_table_iter_init (&mbox_iter, md->mboxes);
	while (g_hash_table_iter_next (&mbox_iter, &mbox_key, &mbox_value)) {
		folder_name = (gchar*) mbox_key;
		man = (CamelIMAPXMetaAnnotation*) mbox_value;

		kfmd = kolab_folder_meta_data_new_from_imapx_annotation (man,
		                                                         server,
		                                                         &tmp_err);
		if (kfmd == NULL) {
			g_warning ("%s: kolab annotation error for [%s]: %s",
			           __func__, folder_name, tmp_err->message);
			g_clear_error (&tmp_err);
			continue;
		}
		/* need hash table with key:value destroy functions set */
		g_hash_table_replace (kmd->kolab_metadata,
		                      g_strdup (folder_name),
		                      kfmd);
	}
}

/* FIXME CAUTION -- TESTING purposes only! The server response may
 *                  become too long!
 */
gboolean
kolab_imapx_meta_data_query_all_folder_types (CamelService *service,
                                              GError **err)
{		
	CamelIMAPXServer *server;
	CamelIMAPXMetaDataSpec *spec;

	GError *tmp_err = NULL;
	CamelException *tmp_ex;

	g_assert (service != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	tmp_ex = camel_exception_new ();
	server = __KOLAB_camel_imapx_store_get_server ((CamelIMAPXStore *)service, tmp_ex);
	if (server == NULL) {
		kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
		return FALSE;
	}

	spec = __KOLAB_imapx_meta_data_spec_new (server->metadata,
	                                         "*",
	                                         "/vendor/kolab/folder-type",
	                                         "value",
	                                         &tmp_err);
	if (spec == NULL) {
		camel_exception_free (tmp_ex);
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	(void)__KOLAB_imapx_server_get_metadata (server, spec, FALSE, tmp_ex);
	if (camel_exception_is_set (tmp_ex)) {
		__KOLAB_imapx_meta_data_spec_free (spec);
		kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
		return FALSE;
	}
	__KOLAB_imapx_meta_data_spec_free (spec);
	camel_exception_free (tmp_ex);

	return TRUE;
}

KolabFolderTypeID
kolab_imapx_meta_data_get_foldertype (CamelKolabIMAPXMetaData *kmd,
                                      CamelService *service,
                                      const gchar *folder_name,
                                      gboolean do_updatedb,
                                      GError **err)
{
	/* TODO better error reporting */
	
	CamelIMAPXServer *server;
	CamelStore *store;
	CamelIMAPXMetaData *md;
	CamelKolabFolderMetaData *kfmd;
	CamelIMAPXMetaDataSpec *spec;
	GError *tmp_err = NULL;
	CamelException *tmp_ex;
	gboolean db_ok;

	g_assert (kmd != NULL);
	g_assert (kmd->kolab_metadata != NULL);
	g_assert (kmd->mdb != NULL);
	g_assert (service != NULL);
	g_assert (folder_name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, KOLAB_FOLDER_TYPE_INVAL);

	/* hash table lookup */
	kfmd = g_hash_table_lookup (kmd->kolab_metadata, folder_name);
	if (kfmd != NULL)
		return kfmd->folder_type;

	/* if not in hash table: sqlite db lookup */
	kfmd = kolab_imapx_meta_data_db_lookup (kmd->mdb,
	                                        folder_name,
	                                        &tmp_err);
	if (tmp_err != NULL) {
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	if (kfmd != NULL) {
		g_hash_table_insert (kmd->kolab_metadata,
		                     g_strdup (folder_name),
		                     kfmd);
		return kfmd->folder_type;
	}

	/* check whether we're online */
	store = (CamelStore *) service;
	if (CAMEL_OFFLINE_STORE (store)->state == CAMEL_OFFLINE_STORE_NETWORK_UNAVAIL)
		return KOLAB_FOLDER_TYPE_UNKNOWN;

	/* TODO check whether we're authenticated */

	/* if not in sqlite db: issue IMAP query */
	tmp_ex = camel_exception_new ();
	server = __KOLAB_camel_imapx_store_get_server ((CamelIMAPXStore *)service, tmp_ex);
	if (server == NULL) {
		kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
		return KOLAB_FOLDER_TYPE_INVAL;
	}
	spec = __KOLAB_imapx_meta_data_spec_new (server->metadata,
	                                         folder_name,
	                                         "/vendor/kolab/folder-type",
	                                         "value",
	                                         &tmp_err);
	if (spec == NULL) {
		camel_object_unref (server);
		g_propagate_error (err, tmp_err);
		return KOLAB_FOLDER_TYPE_INVAL;
	}

	md = __KOLAB_imapx_server_get_metadata (server, spec, TRUE, tmp_ex);

	if (md == NULL) {
		if (camel_exception_is_set (tmp_ex)) {
			camel_object_unref (server);
			__KOLAB_imapx_meta_data_spec_free (spec);
			kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
			return KOLAB_FOLDER_TYPE_INVAL;
		}
	}
	__KOLAB_imapx_meta_data_spec_free (spec);
	camel_exception_free (tmp_ex);
	
	/* create kolab "flat" data structure */
	kolab_meta_data_addreplace_from_imapx (kmd, md, server);
	__KOLAB_imapx_meta_data_free (md);

	/* unref server */
	camel_object_unref (server);

	/* stuff folder types into metadata_db */
	if (do_updatedb) {
		db_ok = kolab_imapx_meta_data_db_update (kmd->mdb,
			                                 kmd->kolab_metadata,
			                                 &tmp_err);
		if (!db_ok) {
			g_propagate_error (err, tmp_err);
			return KOLAB_FOLDER_TYPE_INVAL;
		}
	}
	
	/* final hash table lookup */
	kfmd = g_hash_table_lookup (kmd->kolab_metadata, folder_name);
	if (kfmd != NULL)
		return kfmd->folder_type;
	
	return KOLAB_FOLDER_TYPE_UNKNOWN;
}

gboolean
kolab_imapx_meta_data_set_foldertype (CamelKolabIMAPXMetaData *kmd,
                                      CamelService *service,
                                      const gchar *folder_name,
                                      KolabFolderTypeID folder_type,
                                      GError **err)
{
	CamelIMAPXServer *server;
	CamelStore *store;
	CamelIMAPXMetaData *md;
	CamelIMAPXMetaAnnotation *man;
	CamelIMAPXMetaEntry *me;
	CamelIMAPXMetaAttrib *ma;
	gchar *type_str;
	imapx_meta_data_access_t acc;
	CamelException *tmp_ex;
	GError *tmp_err = NULL;
	gboolean metadata_ok;
	gboolean db_ok;

	g_assert (kmd != NULL);
	g_assert (kmd->kolab_metadata != NULL);
	g_assert (kmd->mdb != NULL);
	g_assert (service != NULL);
	g_assert (folder_name != NULL);
	g_assert ((folder_type > KOLAB_FOLDER_TYPE_UNKNOWN) && (folder_type < KOLAB_FOLDER_LAST_TYPE));
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* check whether we're online */
	store = (CamelStore *) service;
	if (CAMEL_OFFLINE_STORE (store)->state == CAMEL_OFFLINE_STORE_NETWORK_UNAVAIL) {
		g_set_error (err,
		             KOLAB_CAMEL_KOLAB_ERROR,
		             KOLAB_CAMEL_KOLAB_ERROR_SERVER,
		             "%s: must be online to complete this operation",
		             __func__);
		return FALSE;
	}

	/* TODO check whether we're authenticated */

	/* get the server object */
	tmp_ex = camel_exception_new ();
	server = __KOLAB_camel_imapx_store_get_server ((CamelIMAPXStore *)service, tmp_ex);
	if (server == NULL) {
		kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
		return FALSE;
	}
	
	/* create local CamelIMAPXMetaData for setting type */
	acc = CAMEL_IMAPX_METADATA_ACCESS_SHARED;
	ma = __KOLAB_imapx_meta_attrib_new ();
	ma->type[acc] = CAMEL_IMAPX_META_ATTRIB_TYPE_UTF8;
	type_str = kolab_folder_type_inv_map[folder_type];
	ma->data[acc] = g_byte_array_new ();
	g_byte_array_append (ma->data[acc],
	                     (guchar *) type_str,
	                     strlen (type_str));
	me = __KOLAB_imapx_meta_entry_new ();
	g_hash_table_insert (me->attributes,
	                     g_strdup ("value"),
	                     ma);
	man = __KOLAB_imapx_meta_annotation_new ();
	g_hash_table_insert (man->entries,
	                     g_strdup ("/vendor/kolab/folder-type"),
	                     me);
	md = __KOLAB_imapx_meta_data_new (__KOLAB_imapx_meta_data_get_proto (server->metadata),
	                                  FALSE);
	g_hash_table_insert (md->mboxes,
	                     g_strdup (folder_name),
	                     man);

	/* set folder type on the server */
	metadata_ok = __KOLAB_imapx_server_set_metadata (server, md, tmp_ex);
	if (!metadata_ok) {
		__KOLAB_imapx_meta_data_free (md);
		camel_object_unref (server);
		kolab_gerror_propagate_from_camelexception (err, tmp_ex, KOLAB_CAMEL_KOLAB_ERROR);
		return FALSE;
	}

	/* create kolab "flat" data structure */
	kolab_meta_data_addreplace_from_imapx (kmd, md, server);
	__KOLAB_imapx_meta_data_free (md);

	/* unref server */
	camel_object_unref (server);

	/* stuff folder types into metadata_db */
	db_ok = kolab_imapx_meta_data_db_update (kmd->mdb,
		                                 kmd->kolab_metadata,
		                                 &tmp_err);
	if (!db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}
	
	return TRUE;
}

gboolean
kolab_imapx_meta_data_remove (CamelKolabIMAPXMetaData *kmd,
                              const gchar *folder_name,
                              GError **err)
{
	GError *tmp_err = NULL;
	gboolean db_ok;
	gboolean mem_ok;
	
	/* remove from local cache only, let the imap daemon
	 * take care of the server side.
	 * Handle removal of non-existent folder names gracefully
	 * (i.e. report I/O errors only, return okay if folder name
	 * does not exist in sqlite/in-mem db)
	 */

	g_assert (kmd != NULL);
	g_assert (kmd->mdb != NULL);
	g_assert (kmd->kolab_metadata != NULL);
	g_assert (folder_name != NULL);
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* remove from metadata db */
	db_ok = kolab_imapx_meta_data_db_remove_folder (kmd->mdb,
	                                                folder_name,
	                                                &tmp_err);
	if (!db_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	/* remove from in-mem */
	/* need hash table with key:value destroy functions set */
	mem_ok = g_hash_table_remove (kmd->kolab_metadata,
	                              folder_name);
	if (!mem_ok)
		g_warning ("%s: [%s] not in mem cache",
		           __func__, folder_name);
	
	return TRUE;
}
