/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            camel-imapx-extd-server.c
 *
 *  2011-11-28, 20:16:38
 *  Copyright 2011, Christian Hilberg
 *  <hilberg@unix-ag.org>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n.h>

#include "camel-imapx-extd-store.h"
#include "camel-imapx-extd-store-friend.h"
#include "camel-imapx-extd-server.h"

/*----------------------------------------------------------------------------*/

static guint32 imapx_capa_flag_annotatemore = 0;

static gboolean
camel_imapx_extd_server_untagged_annotation (CamelIMAPXServer *is,
                                             GCancellable *cancellable,
                                             GError **err);

static const CamelIMAPXUntaggedRespHandlerDesc desc = {
	camel_imapx_extd_server_untagged_annotation, /* handler               */
	NULL,                                        /* next_response         */
	TRUE                                         /* skip_stream_when_done */
};

/*----------------------------------------------------------------------------*/

static gboolean
camel_imapx_extd_server_untagged_annotation (CamelIMAPXServer *is,
                                             GCancellable *cancellable,
                                             GError **err)
{
	CamelIMAPXExtdStore *estore = NULL;
	CamelImapxMetadata *md = NULL;
	guint32 capa = 0;
	GError *tmp_err = NULL;
	gboolean parse_and_add_ok = FALSE;

	g_assert (CAMEL_IS_IMAPX_SERVER (is));
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	estore = CAMEL_IMAPX_EXTD_STORE (is->store);

	/* capability check */
	capa = is->cinfo->capa & imapx_capa_flag_annotatemore;
	if (! capa) {
		g_set_error (err,
		             CAMEL_IMAPX_ERROR,
		             1, /* FIXME define and add a sensible code here */
		             _("Got ANNOTATION response but server did not advertise ANNOTATEMORE capability"));
		return FALSE;
	}

	md = camel_imapx_extd_store_get_md_table (estore);
	parse_and_add_ok =
		camel_imapx_metadata_add_from_server_response (md,
		                                               is->stream,
		                                               cancellable,
		                                               &tmp_err);
	g_object_unref (md);

	if (! parse_and_add_ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	return TRUE;
}

/*----------------------------------------------------------------------------*/

gboolean
camel_imapx_extd_server_init (CamelIMAPXServer *is,
                              GCancellable *cancellable,
                              GError **err)
{
	const CamelIMAPXUntaggedRespHandlerDesc *prev = NULL;
	static gboolean is_inited = FALSE;

	g_assert (CAMEL_IS_IMAPX_SERVER (is));
	(void)cancellable; /* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* TODO change this for a g_once() */
	if (is_inited)
		return TRUE;

	/* ANNOTATEMORE server capability flag */
	imapx_capa_flag_annotatemore =
		imapx_register_capability (IMAPX_IMAP_TOKEN_ANNOTATEMORE);
	g_assert (imapx_capa_flag_annotatemore > 0);

	prev = camel_imapx_server_register_untagged_handler (is,
	                                                     IMAPX_IMAP_TOKEN_ANNOTATION,
	                                                     &desc);
	/* make sure we do not have been registered before */
	g_assert (prev == NULL);

	is_inited = TRUE;

	return TRUE;
}

gboolean
camel_imapx_extd_server_get_metadata (CamelIMAPXServer *is,
                                      CamelImapxMetadataSpec *spec,
                                      GCancellable *cancellable,
                                      GError **err)
{
	CamelIMAPXCommand *ic = NULL;
	CamelIMAPXJob *job = NULL;
	gchar *cmd = NULL;
	GError *tmp_err = NULL;
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_IMAPX_SERVER (is));
	g_assert (spec != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	/* TODO move GETANNOTATION string to -metadata.[hc]
	 *      (add a function to return proper string depending
	 *      on protocol type)
	 */
	cmd = g_strdup_printf ("%s \"%s\" \"%s\" \"%s\"",
	                       IMAPX_IMAP_TOKEN_GETANNOTATION,
	                       spec->mailbox_name,
	                       spec->entry_name,
	                       spec->attrib_name);
	ic = camel_imapx_command_new (is,
	                              IMAPX_IMAP_TOKEN_GETANNOTATION,
	                              NULL,
	                              cmd);
	job = camel_imapx_job_new (cancellable);
	camel_imapx_command_set_job (ic, job);
	/* TODO set more job details like priority? */

	/* run GETANNOTATION command */
	ok = camel_imapx_server_command_run (is,
	                                     ic,
	                                     cancellable,
	                                     &tmp_err);
	g_free (cmd);

	/* wait for the command to complete
	 * (will improve our chances that the server will
	 * already have sent an untagged response when we
	 * want to read the metadata information)
	 */
	if (ok)
		camel_imapx_command_wait (ic);

	camel_imapx_command_done (ic);
	camel_imapx_command_unref (ic);

	if (! ok) {
		g_propagate_error (err, tmp_err);
		return FALSE;
	}

	return TRUE;
}

gboolean
camel_imapx_extd_server_set_metadata (CamelIMAPXServer *is,
                                      CamelImapxMetadata *md,
                                      GCancellable *cancellable,
                                      GError **err)
{
	CamelIMAPXCommand *ic = NULL;
	CamelIMAPXJob *job = NULL;
	GSList *commands = NULL;
	GSList *cur_cmd = NULL;
	GError *tmp_err = NULL;
	gboolean ok = FALSE;

	g_assert (CAMEL_IS_IMAPX_SERVER (is));
	g_assert (md != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (err == NULL || *err == NULL, FALSE);

	commands = camel_imapx_metadata_new_commandlist (md);

	if (commands == NULL)
		return TRUE;

	cur_cmd = commands;
	while (cur_cmd != NULL) {
		/* TODO move SETANNOTATION string to -metadata.[hc]
		 *      (add a function to return proper string depending
		 *      on protocol type)
		 */
		ic = camel_imapx_command_new (is,
		                              IMAPX_IMAP_TOKEN_SETANNOTATION,
		                              NULL,
		                              (gchar *) cur_cmd->data);
		job = camel_imapx_job_new (cancellable);
		camel_imapx_command_set_job (ic, job);
		/* TODO set more job details like priority? */

		/* run SETANNOTATION command */
		ok = camel_imapx_server_command_run (is,
		                                     ic,
		                                     cancellable,
		                                     &tmp_err);

		camel_imapx_command_done (ic);
		camel_imapx_command_unref (ic);

		if (! ok) {
			g_slist_free (commands);
			g_propagate_error (err, tmp_err);
			return FALSE;
		}

		cur_cmd = g_slist_next (cur_cmd);
	}

	g_slist_free (commands);

	return TRUE;
}

/*----------------------------------------------------------------------------*/
