/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * folder-browser-factory.c: A Bonobo Control factory for Folder Browsers
 *
 * Author:
 *   Miguel de Icaza (miguel@helixcode.com)
 *
 * (C) 2000 Helix Code, Inc.
 */

#include <config.h>

#include <gnome.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-object.h>
#include <bonobo/bonobo-generic-factory.h>
#include <bonobo/bonobo-control.h> 
#include <bonobo/bonobo-ui-component.h>

#include <gal/util/e-util.h>
#include <gal/widgets/e-gui-utils.h>

#include "folder-browser-factory.h"

#include "folder-browser.h"
#include "mail.h"
#include "shell/Evolution.h"
#include "mail-config.h"
#include "mail-ops.h"

/* The FolderBrowser BonoboControls we have.  */
static GList *control_list = NULL;

static void
register_ondemand (RuleContext *f, FilterRule *rule, FolderBrowser *fb, BonoboUIComponent *uic)
{
	BonoboUIHandler *uih;
	gchar *text;
	struct fb_ondemand_closure *oc;
	
	oc = g_new (struct fb_ondemand_closure, 1);
	oc->rule = rule;
	oc->fb = fb;
	oc->path = g_strdup_printf ("/*Component Placeholder*/Folder/Filter-%s", rule->name);

#warning FIXME: this should not use the bonobo_ui_handler API.

	uih = bonobo_ui_handler_new_from_component (uic);

	if (fb->filter_menu_paths == NULL)
		bonobo_ui_handler_menu_new_separator (uih, "/*Component Placeholder*/Folder/separator1", -1);
	
	text = g_strdup_printf (_("Run filter \"%s\""), rule->name);
	fb->filter_menu_paths = g_slist_prepend (fb->filter_menu_paths, oc);
	
	bonobo_ui_handler_menu_new_item (uih, oc->path, text,
					 NULL, -1,
					 BONOBO_UI_HANDLER_PIXMAP_NONE,
					 0,
					 0, 0, run_filter_ondemand, oc);
	g_free (text);
}

static void
create_ondemand_hooks (FolderBrowser *fb, BonoboUIComponent *uic)
{
	gchar *system, *user;
	FilterRule *rule = NULL;

	user = g_strdup_printf ("%s/filters.xml", evolution_dir);
	system = EVOLUTION_DATADIR "/evolution/filtertypes.xml";
	fb->filter_context = filter_context_new();
	rule_context_load ((RuleContext *) fb->filter_context, system, user);
	while ( (rule = rule_context_next_rule((RuleContext *)fb->filter_context, rule, FILTER_SOURCE_DEMAND)) != NULL ) {
		register_ondemand((RuleContext *)fb->filter_context, rule, fb, uic);
	}
	g_free (user);
}

/*
 * Add with 'folder_browser'
 */
BonoboUIVerb verbs [] = {
	BONOBO_UI_UNSAFE_VERB ("PrintMessage", print_msg),
	BONOBO_UI_UNSAFE_VERB ("PrintPreviewMessage", print_preview_msg),
	
	/* Edit Menu */
	BONOBO_UI_UNSAFE_VERB ("EditSelectAll", select_all),
	BONOBO_UI_UNSAFE_VERB ("EditInvertSelection", invert_selection),
	
	/* Settings Menu */
	BONOBO_UI_UNSAFE_VERB ("SetMailFilter", filter_edit),
	BONOBO_UI_UNSAFE_VERB ("SetVFolder", vfolder_edit_vfolders),
	BONOBO_UI_UNSAFE_VERB ("SetMailConfig", providers_config),
	BONOBO_UI_UNSAFE_VERB ("SetSubscribe", manage_subscriptions),
	BONOBO_UI_UNSAFE_VERB ("SetForgetPwd", forget_passwords),
	
	/* Message Menu */
	BONOBO_UI_UNSAFE_VERB ("MessageOpenNewWnd", view_message),
	BONOBO_UI_UNSAFE_VERB ("MessageEdit", edit_message),
	BONOBO_UI_UNSAFE_VERB ("MessagePrint", print_msg),
	BONOBO_UI_UNSAFE_VERB ("MessageReplySndr", reply_to_sender),
	BONOBO_UI_UNSAFE_VERB ("MessageReplyAll", reply_to_all),
	BONOBO_UI_UNSAFE_VERB ("MessageForward", forward_msg),
	
	BONOBO_UI_UNSAFE_VERB ("MessageMarkAsRead", mark_as_seen),
	BONOBO_UI_UNSAFE_VERB ("MessageMarkAsUnRead", mark_as_unseen),
	BONOBO_UI_UNSAFE_VERB ("MessageDelete", delete_msg),
	BONOBO_UI_UNSAFE_VERB ("MessageMove", move_msg),
	BONOBO_UI_UNSAFE_VERB ("MessageCopy", copy_msg),
	
	BONOBO_UI_UNSAFE_VERB ("MessageVFolderSubj", vfolder_subject),
	BONOBO_UI_UNSAFE_VERB ("MessageVFolderSndr", vfolder_sender),
	BONOBO_UI_UNSAFE_VERB ("MessageVFolderRecip", vfolder_recipient),
	
	BONOBO_UI_UNSAFE_VERB ("MessageFilterSubj", filter_subject),
	BONOBO_UI_UNSAFE_VERB ("MessageFilderSndr", filter_sender),
	BONOBO_UI_UNSAFE_VERB ("MessageFilderRecip", filter_recipient),
	
	/* Folder Menu */
	BONOBO_UI_UNSAFE_VERB ("FolderExpunge", expunge_folder),
	BONOBO_UI_UNSAFE_VERB ("FolderConfig", configure_folder),
	
	/* Toolbar specific */
	BONOBO_UI_UNSAFE_VERB ("MailGet", send_receieve_mail),
	BONOBO_UI_UNSAFE_VERB ("MailCompose", compose_msg),
	
	BONOBO_UI_VERB_END
};

static void
set_pixmap (BonoboUIComponent *uic,
	    const char        *xml_path,
	    const char        *icon)
{
	char *path;
	GdkPixbuf *pixbuf;

	path = g_concat_dir_and_file (EVOLUTION_DATADIR "/images/evolution/buttons", icon);

	pixbuf = gdk_pixbuf_new_from_file (path);
	g_return_if_fail (pixbuf != NULL);

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}

static void
update_pixmaps (BonoboUIComponent *uic)
{
	set_pixmap (uic, "/Toolbar/MailGet", "fetch-mail.png");
	set_pixmap (uic, "/Toolbar/MailCompose", "compose-message.png");
	set_pixmap (uic, "/Toolbar/Reply", "reply.png");
	set_pixmap (uic, "/Toolbar/ReplyAll", "reply-to-all.png");
	set_pixmap (uic, "/Toolbar/Forward", "forward.png");
	set_pixmap (uic, "/Toolbar/Move", "move-message.png");
	set_pixmap (uic, "/Toolbar/Copy", "copy-message.png");
}

static void
control_activate (BonoboControl     *control,
		  BonoboUIComponent *uic,
		  FolderBrowser     *fb)
{
	GtkWidget         *folder_browser;
	Bonobo_UIContainer container;

	container = bonobo_control_get_remote_ui_container (control);
	bonobo_ui_component_set_container (uic, container);
	bonobo_object_release_unref (container, NULL);

	g_assert (container == bonobo_ui_component_get_container (uic));
	g_return_if_fail (container != CORBA_OBJECT_NIL);
		
	folder_browser = bonobo_control_get_widget (control);

	bonobo_ui_component_add_verb_list_with_data (
		uic, verbs, folder_browser);

	bonobo_ui_component_freeze (uic, NULL);

	bonobo_ui_util_set_ui (
		uic, EVOLUTION_DATADIR,
		"evolution-mail.xml", "evolution-mail");

	if (mail_config_thread_list ())
		bonobo_ui_component_set_prop (
			uic, "/commands/ViewThreaded", "state", "1", NULL);
	else
		bonobo_ui_component_set_prop (
			uic, "/commands/ViewThreaded", "state", "0", NULL);

	bonobo_ui_component_add_listener (
		uic, "ViewThreaded",
		message_list_toggle_threads,
		FOLDER_BROWSER (folder_browser)->message_list);

	create_ondemand_hooks (fb, uic);

	update_pixmaps (uic);

	bonobo_ui_component_thaw (uic, NULL);
}

static void
control_deactivate (BonoboControl     *control,
		    BonoboUIComponent *uic,
		    FolderBrowser     *fb)
{
	bonobo_ui_component_rm (uic, "/", NULL);
 	bonobo_ui_component_unset_container (uic);

	if (fb->folder)
		mail_do_sync_folder (fb->folder);
}

static void
control_activate_cb (BonoboControl *control, 
		     gboolean activate, 
		     gpointer user_data)
{
	BonoboUIComponent *uic;

	uic = bonobo_control_get_ui_component (control);
	g_assert (uic != NULL);

	if (activate)
		control_activate (control, uic, user_data);
	else
		control_deactivate (control, uic, user_data);
}

static void
control_destroy_cb (BonoboControl *control,
		    gpointer       user_data)
{
	GtkWidget *folder_browser = user_data;

	control_list = g_list_remove (control_list, control);

	gtk_object_destroy (GTK_OBJECT (folder_browser));
}

BonoboControl *
folder_browser_factory_new_control (const char *uri,
				    const Evolution_Shell shell)
{
	BonoboControl *control;
	GtkWidget *folder_browser;

	folder_browser = folder_browser_new (shell);
	if (folder_browser == NULL)
		return NULL;

	if (!folder_browser_set_uri (FOLDER_BROWSER (folder_browser), uri)) {
		gtk_object_sink (GTK_OBJECT (folder_browser));
		return NULL;
	}

	gtk_widget_show (folder_browser);
	
	control = bonobo_control_new (folder_browser);
	
	if (control == NULL) {
		gtk_object_destroy (GTK_OBJECT (folder_browser));
		return NULL;
	}
	
	gtk_signal_connect (GTK_OBJECT (control), "activate",
			    control_activate_cb, folder_browser);

	gtk_signal_connect (GTK_OBJECT (control), "destroy",
			    control_destroy_cb, folder_browser);

	control_list = g_list_prepend (control_list, control);

	return control;
}

GList *
folder_browser_factory_get_control_list (void)
{
	return control_list;
}
