from desktop.GlassWindow import GlassWindow
from TargetGroup import TargetGroup
from DisplayTarget import DisplayTarget
from utils.Observable import Observable
from DisplayConfigurator import DisplayConfigurator
from main import ICON

import gtk
import os


#
# Class for display windows.
#
class Display(GlassWindow, Observable):

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1

    # mapping: str -> window type
    __WINDOW_FLAGS = {"below": GlassWindow.TYPE_KEEP_BELOW,
                      "above": GlassWindow.TYPE_KEEP_ABOVE,
                      "sticky": GlassWindow.TYPE_STICKY,
                      "managed": GlassWindow.TYPE_MANAGED}
    

    def __init__(self, id):

        # the path of the .display file
        self.__path = os.getcwd()

        # the unique ID of this display
        self.__id = id

        # the sensors of this window
        self.__sensors = {}

        # the last selected targets (used for detecting events)
        self.__last_targets = []
        self.__last_target_path = []

        # timeout flag for mouse buttons
        self.__button_timeout = 1

        # timeout flag for menu detection
        self.__menu_timeout = 0

        # window position for detecting moves
        self.__window_pos = (-1, -1)

        # window size for detecting resizing
        self.__window_size = (0, 0)

        # mapping between sensors and targets; which targets watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # temporary data used for dragging windows
        self.__is_dragging = 0
        self.__drag_offset = (0, 0)


        GlassWindow.__init__(self, gtk.WINDOW_TOPLEVEL)
        self.set_size_request(-1, -1)

        # set the icon
        self.set_icon(gtk.gdk.pixbuf_new_from_file(ICON))

        # set up event handlers
        #self.connect("configure-event", self.__on_configure)
        self.connect("button-press-event", self.__on_button, 0)
        self.connect("button-release-event", self.__on_button, 1)
        self.connect("motion-notify-event", self.__on_motion, 0)
        self.connect("leave-notify-event", self.__on_motion, 1)
        self.connect("delete-event", self.__on_close)
        self.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                        gtk.gdk.BUTTON_RELEASE_MASK |
                        gtk.gdk.LEAVE_NOTIFY_MASK |
                        gtk.gdk.POINTER_MOTION_MASK)

        # sawfish needs this
        self.realize()
        self.set_property("skip-taskbar-hint", 1)
        self.set_property("skip-pager-hint", 1)
        # don't show up in the taskbar or pager
        #gtk.idle_add(self.set_property, "skip-taskbar-hint", 1)
        #gtk.idle_add(self.set_property, "skip-pager-hint", 1)



    def __del__(self):

        print "DEL", self


    #
    # Returns the path of the .display file.
    #
    def get_path(self):  return self.__path
    


    def add_children(self, childrendata):

        # create the root TargetGroup
        self.__group = TargetGroup(None, self)
        self.__group.set_position(-1, -1)
        self.__group.add_children(childrendata)
        self.add(self.__group)
        self.__group.add_observer(self.__on_observe_group)
        


    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):

        configurators = []
        sensors = self.__sensors.values()
        for s in sensors:
            configurators.append(s.get_configurator())

        dconf = DisplayConfigurator(configurators)
        dconf.set_transient_for(self)


    #
    # Removes this display. When the restart flag is set, the configuration
    # entries will not be purged.
    #
    def __remove_display(self, restart = 0):

        for s in self.__sensors.values():
            s.stop(restart)
            
        del self.__sensors
        del self.__mapping
        self.__group._forget_parent() #destroy()
        self.__group.destroy()
        #del self.__group
        self.destroy()



    #
    # Reacts on configuring the window.
    #
    def __on_configure(self, src, event):

        width = event.width
        height = event.height

        if ((width, height) != self.__window_size):
            self.__window_size = (width, height)
            self.set_config("width", width)
            self.set_config("height", height)
        #end if

        


    #
    # Reacts on closing the window.
    #
    def __on_close(self, src, event):

        self.__remove_display()
        self.update_observer(self.OBS_CLOSE, self.__id)
        


    #
    # Reacts on moving the mouse.
    #
    def __on_move(self):

        if (self.__is_dragging):
            offx, offy = self.__drag_offset
            winx, winy = self.get_position()
            x, y = self.get_pointer()
            x += winx; y += winy
            self.move(x - offx, y - offy)

            return gtk.TRUE

        else:
            return gtk.FALSE



    #
    # Reacts on button events.
    #
    def __on_button(self, src, event, is_release = 0):

        # local timeout function
        def do_timeout(self): self.__button_timeout = 1

        # function for detecting holding down the mouse long enough for menu
        def do_menu(self, targets, path):
            if (not self.__menu_timeout): return
            targets.reverse()
            for t in targets:
                if (t.has_action(t.ACTION_MENU)):
                    call = t.get_action_call(t.ACTION_MENU)
                    index = targets.index(t)
                    targetpath = path[:index + 1]
                    self.__call_sensor(call, targetpath, button)
                    break
            #end for
        #end def


        self.__menu_timeout = 0
        px, py = self.get_pointer()
        lx, ly = self.__pointer_pos
        button = event.button
        targets, path = self.__get_target_at(px, py)

        # determine action
        if (not is_release):
            self.__pointer_pos = (px, py)
            if (button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
                action = DisplayTarget.ACTION_DOUBLECLICK
            elif (button == 1):
                action = DisplayTarget.ACTION_PRESS
            elif (button == 2):
                #action = DisplayTarget.ACTION_PRESS
                x, y = self.get_pointer()
                self.__is_dragging = 1
                self.__drag_offset = (x, y)
                self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.FLEUR))
                gtk.timeout_add(20, self.__on_move)
                return
            elif (button == 3):
                return
            else:
                return

        else:
            if (button == 1):
                if (abs(lx - px) < 10 and abs(ly - py) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                #action = DisplayTarget.ACTION_RELEASE
                self.__is_dragging = 0
                self.window.set_cursor(None)
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return

        #end if

                        
        # find target and invoke action handler
        called = 0
        for t in targets:
            if (t.has_action(action)):
                call = t.get_action_call(action)
                index = targets.index(t)
                targetpath = path[:index + 1]
                self.__call_sensor(call, targetpath, button)
                called = 1
                break
        #end for

        # make sure that there is always a popup menu
        if (action == DisplayTarget.ACTION_MENU and not called):
            self.__call_sensor(["_default_:menu"], [], 3)
        


    #
    # Reacts on moving the mouse.
    #
    def __on_motion(self, src, event, is_leave):

        px, py = self.get_pointer()
        targets, path = self.__get_target_at(px, py)


        # how to detect enter and leave:
        # get covered targets and compare with previously covered targets;
        # what's new in covered targets is entered;
        # what's only in previously covered targets is left

        if (is_leave): targets = []

        # TODO: make this more efficient; don't check for existence in lists
        for t in targets + self.__last_targets:
            # enter
            if (not t in self.__last_targets and t.has_action(t.ACTION_ENTER)):
                index = targets.index(t)
                targetpath = path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_ENTER),
                                   targetpath)

            # leave
            elif (not t in targets and t.has_action(t.ACTION_LEAVE)):
                index = self.__last_targets.index(t)
                targetpath = self.__last_target_path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_LEAVE),
                                   targetpath)

            # motion
            elif (t in targets and t.has_action(t.ACTION_MOTION)):
                tx, ty = t.get_pointer()
                index = targets.index(t)
                targetpath = path[:index + 1]
                self.__call_sensor(t.get_action_call(t.ACTION_MOTION),
                                   targetpath, tx, ty)

        #end for
            
        self.__last_targets = targets
        self.__last_target_path = path


        # save the window position if the window has moved
        x, y = self.get_position()
        if ((x, y) != self.__window_pos):
            nil, nil, w, h = self.__group.get_geometry()
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)
            dx, dy = x - ax, y - ay
            self.__call_sensor(["_default_:move"], [""], x + dx, y + dy)

            self.__window_pos = (x, y)
            #print "save window @", x, y
            self.__group.set_position(x + dx, y + dy, update = 0)


            

    #
    # Observer for sensors.
    #
    def __on_observe_sensor(self, src, cmd, data):

        # propagate the incoming sensor output
        if (cmd == src.OBS_OUTPUT):
            #if (self.window): self.window.freeze_updates()
            for key, value in data.get_entries():
                if ("[" in key):
                    indexpart = key[key.find("[") + 1:-1]
                    indexes = indexpart.split("][")
                    key = key[:key.find("[")]
                else:
                    indexes = []

                entries = self.__mapping.get((src, key), [])
                for target, property in entries:
                    if (indexes):
                        sensor = self.__get_sensor_id(src)
                        target.distribute_sensor_output(sensor, indexes[:],
                                                        key, value)
                    else:
                        target.set_config(property, value)
                #end for
            #end for
            #if (self.window): self.window.thaw_updates()
            self.queue_draw()

        elif (cmd == src.OBS_CMD_CONFIGURE):
            self.__open_configurator()

        elif (cmd == src.OBS_CMD_REMOVE):
            self.__remove_display(restart = 0)
            self.update_observer(self.OBS_CLOSE, self.__id)

        elif (cmd == src.OBS_CMD_RESTART):
            self.__remove_display(restart = 1)
            self.update_observer(self.OBS_RESTART, self.__id)
            



    #
    # Observer for the root group.
    #
    def __on_observe_group(self, src, cmd, *args):

        if (cmd == src.OBS_MOVE):
            x, y, w, h = args
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)

            ax = min(max(0, ax), gtk.gdk.screen_width())
            ay = min(max(0, ay), gtk.gdk.screen_height())

            if ((x != -1 and y != -1) and (ax, ay) != self.__window_pos):
                self.move(ax, ay)
                self.__window_pos = (ax, ay)
                
            if (w != 0 and h != 0):
                self.resize(w, h)
                # storing the size is useless, but it's by added by request;
                # it makes life easy for desklets pagers
                self.__call_sensor(["_default_:size"], [""], w, h)

            # don't show this window before its position has been set
            if (x != -1 and y != -1):
                self.show()
                gtk.idle_add(self.move, ax, ay)



    #
    # Calls a function of a Sensor.
    #
    def __call_sensor(self, call, path, *args):
        assert(call)

        for c in call:
            index = c.index(":")
            id, callname = c[:index], c[index + 1:]
            sensor = self.__get_sensor(id)

            # the sensor is an external module, so we make sure it cannot crash
            # the application
            try:
                sensor.send_action(callname, path, args)
            
            except StandardError, e:
                print "The sensor produced an error:", e
        #end for



    #
    # Returns the target and its path at the given position.
    #
    def __get_target_at(self, px, py):

        ret = self.__group.get_target_at(px, py, 1)
        if (ret): targets, path = ret
        else: targets, path = [], []

        return (targets, path)



    #
    # Sets the configuration.
    #
    def set_config(self, key, value):

        if (key == "window-flags"):
            flags = 0
            value = value.split(",")
            for p in value: flags |= self.__WINDOW_FLAGS[p.strip()]
            self.set_window_type(flags)

        else:
            self.__group.set_config(key, value)



    #
    # Adds a sensor to this display.
    #
    def add_sensor(self, id, sensor):

        self.__sensors[id] = sensor
        sensor.add_observer(self.__on_observe_sensor)



    #
    # Returns the sensor with the given ID.
    #
    def __get_sensor(self, id): return self.__sensors[id]



    #
    # Returns the ID of the given sensor.
    #
    def __get_sensor_id(self, sensor):

        for k, v in self.__sensors.items():
            if (v == sensor): return k

        return ""



    #
    # Maps a sensor output to a target.
    #
    def add_mapping(self, sensorplug, target, property):

        id, port = sensorplug.split(":")
        sensor = self.__get_sensor(id)
        if (not self.__mapping.has_key((sensor, port))):
            self.__mapping[(sensor, port)] = []

        if (not (target, property) in self.__mapping[(sensor, port)]):
            self.__mapping[(sensor, port)].append((target, property))
