/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-tb-button.h"
#include "gul-gui.h"
#include "gul-string.h"
#include "galeon-go-button.h"
#include <gtk/gtkstock.h>
#include <string.h>
#include "galeon-shell.h"
#include "toolbar.h"

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GaleonGoButtonPrivate 
{
	GulTbButton *widget;
	gboolean sensitive;
};

/**
 * Private functions, only availble from this file
 */
static void		galeon_go_button_class_init			(GaleonGoButtonClass *klass);
static void		galeon_go_button_init				(GaleonGoButton *tb);
static void		galeon_go_button_finalize_impl		(GObject *o);
static GtkWidget *	galeon_go_button_get_widget_impl		(GulTbItem *i);
static GdkPixbuf *	galeon_go_button_get_icon_impl		(GulTbItem *i);
static gchar *		galeon_go_button_get_name_human_impl		(GulTbItem *i);
static gchar *		galeon_go_button_to_string_impl		(GulTbItem *i);
static GulTbItem *	galeon_go_button_clone_impl			(GulTbItem *i);
static void		galeon_go_button_parse_properties_impl	(GulTbItem *i, const gchar *props);
static void		galeon_go_button_clicked_cb			(GtkWidget *w, GaleonGoButton *b);


static gpointer gul_tb_item_class;

/**
 * TbiZoom object
 */

MAKE_GET_TYPE (galeon_go_button, "GaleonGoButton", GaleonGoButton,
	       galeon_go_button_class_init, 
	       galeon_go_button_init, GALEON_TYPE_TBI);

static void
galeon_go_button_class_init (GaleonGoButtonClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_go_button_finalize_impl;
	
	GUL_TB_ITEM_CLASS (klass)->get_widget = galeon_go_button_get_widget_impl;
	GUL_TB_ITEM_CLASS (klass)->get_icon = galeon_go_button_get_icon_impl;
	GUL_TB_ITEM_CLASS (klass)->get_name_human = galeon_go_button_get_name_human_impl;
	GUL_TB_ITEM_CLASS (klass)->to_string = galeon_go_button_to_string_impl;
	GUL_TB_ITEM_CLASS (klass)->clone = galeon_go_button_clone_impl;
	GUL_TB_ITEM_CLASS (klass)->parse_properties = galeon_go_button_parse_properties_impl;

	gul_tb_item_class = g_type_class_peek_parent (klass);
}

static void 
galeon_go_button_init (GaleonGoButton *tbi)
{
	GaleonGoButtonPrivate *p = g_new0 (GaleonGoButtonPrivate, 1);
	tbi->priv = p;

	p->sensitive = TRUE;
}

GaleonGoButton *
galeon_go_button_new (void)
{
	GaleonGoButton *ret = g_object_new (GALEON_TYPE_GO_BUTTON, NULL);
	return ret;
}

static void
galeon_go_button_finalize_impl (GObject *o)
{
	GaleonGoButton *it = GALEON_GO_BUTTON (o);
	GaleonGoButtonPrivate *p = it->priv;

	if (p->widget)
	{
		g_object_unref (p->widget);
	}

	g_free (p);
	
	DEBUG_MSG (("GaleonGoButton finalized\n"));
	
	G_OBJECT_CLASS (gul_tb_item_class)->finalize (o);
}

static void
galeon_go_button_setup_widget (GaleonGoButton *b)
{
	GaleonGoButtonPrivate *p = b->priv;

	if (!p->widget)
	{
		galeon_go_button_get_widget_impl (GUL_TB_ITEM (b));
	}
	g_assert (GUL_IS_TB_BUTTON (p->widget));

	gul_tb_button_set_label (p->widget, GTK_STOCK_JUMP_TO);
	gul_tb_button_set_tooltip_text (p->widget, _("Go to the specified location"));
	gul_tb_button_set_priority (p->widget, FALSE);
	gul_tb_button_set_show_arrow (p->widget, FALSE);
	gul_tb_button_set_sensitivity (p->widget, p->sensitive);
}

static GaleonEmbed *
galeon_go_button_get_embed (GaleonGoButton *b)
{
	GaleonWindow *window;
	GaleonEmbed *embed;

	window = galeon_tbi_get_window (GALEON_TBI (b));
	g_return_val_if_fail (window != NULL, NULL);

	embed = galeon_window_get_active_embed (window);
	g_return_val_if_fail (embed != NULL, NULL);

	return embed;
}

static gchar *
galeon_go_button_get_location (GaleonGoButton *b)
{
	GaleonWindow *window;
	Toolbar *tb;
	char *ret;
	
	window = galeon_tbi_get_window (GALEON_TBI (b));
	g_return_val_if_fail (window != NULL, NULL);
	
	tb = galeon_window_get_toolbar (window);
	g_return_val_if_fail (tb != NULL, NULL);

	ret = toolbar_get_location (tb);

	return ret;
}

static gboolean
galeon_go_button_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					GaleonGoButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));

	if (event->button == 2 && IS_GALEON_WINDOW (window))
	{
		char *location;

		location = galeon_go_button_get_location (b);
		g_return_val_if_fail (location != NULL, FALSE);
	
		if (location)
		{
			GaleonTab *tab = galeon_window_get_active_tab (window);
			galeon_shell_new_tab (galeon_shell, window, tab, location, 
					      GALEON_NEW_TAB_JUMP);
			g_free (location);
		}
		return TRUE;
	}
	
	return FALSE;
}

static void
galeon_go_button_go_activate_cb (GtkMenuItem *mi, GaleonGoButton *b)
{
	GaleonEmbed *embed = galeon_go_button_get_embed (b);
	char *location;
	
	g_return_if_fail (embed != NULL);

	location = galeon_go_button_get_location (b);
	g_return_if_fail (location != NULL);
	
	galeon_embed_load_url (embed, location);

	g_free (location);
}

static void 
galeon_go_button_go_new_tab_activate_cb (GtkMenuItem *mi, GaleonGoButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));
	char *location;

	location = galeon_go_button_get_location (b);
	g_return_if_fail (location != NULL);
	
	if (location)
	{
		GaleonTab *tab = galeon_window_get_active_tab (window);
		galeon_shell_new_tab (galeon_shell, window, tab, location, 
				      GALEON_NEW_TAB_JUMP | GALEON_NEW_TAB_IN_EXISTING_WINDOW);
		g_free (location);
	}
}

static void 
galeon_go_button_go_new_window_activate_cb (GtkMenuItem *mi, GaleonGoButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));
	char *location;

	location = galeon_go_button_get_location (b);
	g_return_if_fail (location != NULL);
	
	if (location)
	{
		GaleonTab *tab = galeon_window_get_active_tab (window);
		galeon_shell_new_tab (galeon_shell, window, tab, location, 
				      GALEON_NEW_TAB_JUMP | GALEON_NEW_TAB_IN_NEW_WINDOW);
		g_free (location);
	}
}

static GtkWidget *
galeon_go_button_get_widget_impl (GulTbItem *i)
{
	GaleonGoButton *iz = GALEON_GO_BUTTON (i);
	GaleonGoButtonPrivate *p = iz->priv;

	if (!p->widget)
	{
		GtkMenuShell *ms;
		
		p->widget = gul_tb_button_new ();
		g_object_ref (p->widget);
		gtk_object_sink (GTK_OBJECT (p->widget));

		gul_tb_button_set_use_stock (p->widget, TRUE);
		gul_tb_button_set_enable_menu (p->widget, TRUE);
		
		galeon_go_button_setup_widget (iz);

		gtk_widget_show (GTK_WIDGET (p->widget));
		
		g_signal_connect (gul_tb_button_get_button (p->widget), "clicked",
				  G_CALLBACK (galeon_go_button_clicked_cb), i);
		g_signal_connect (gul_tb_button_get_button (p->widget), "button-press-event",
				  G_CALLBACK (galeon_go_button_button_press_event_cb), i);

		ms = gul_tb_button_get_menu (p->widget);

		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified _Location"), 
					     G_CALLBACK (galeon_go_button_go_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified Location in Another _Tab"), 
					     G_CALLBACK (galeon_go_button_go_new_tab_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("Go To The Specified Location in Another _Window"), 
					     G_CALLBACK (galeon_go_button_go_new_window_activate_cb), i);
	}

	return GTK_WIDGET (p->widget);
}

static GdkPixbuf *
galeon_go_button_get_icon_impl (GulTbItem *i)
{
	static GdkPixbuf *pb_go = NULL;

	if (!pb_go)
	{
		/* what's the easier way? */
		GtkWidget *b = gtk_button_new ();
		pb_go = gtk_widget_render_icon (b,
						GTK_STOCK_JUMP_TO,
						GTK_ICON_SIZE_SMALL_TOOLBAR,
						NULL);
		gtk_widget_destroy (b);
	}

	return g_object_ref (pb_go);
}

static gchar *
galeon_go_button_get_name_human_impl (GulTbItem *i)
{
	return g_strdup (_("Go"));
}

static gchar *
galeon_go_button_to_string_impl (GulTbItem *i)
{
	/* if it had any properties, the string should include them */

	return g_strdup_printf ("%s=go_button", i->id);
}

static GulTbItem *
galeon_go_button_clone_impl (GulTbItem *i)
{
	GulTbItem *ret = GUL_TB_ITEM (galeon_go_button_new ());

	gul_tb_item_set_id (ret, i->id);

	return ret;
}

static void
galeon_go_button_parse_properties_impl (GulTbItem *it, const gchar *props)
{
	/* we have no properties */
}

void
galeon_go_button_set_sensitive (GaleonGoButton *b, gboolean s)
{
	GaleonGoButtonPrivate *p = b->priv;
	p->sensitive = s;
	if (p->widget)
	{
		gul_tb_button_set_sensitivity (p->widget, s);
	}
	else
	{
		galeon_go_button_get_widget_impl (GUL_TB_ITEM (b));
	}
}

static void
galeon_go_button_clicked_cb (GtkWidget *w, GaleonGoButton *b)
{
	GaleonEmbed *embed = galeon_go_button_get_embed (b);
	char *location;
	
	g_return_if_fail (embed != NULL);

	location = galeon_go_button_get_location (b);
	g_return_if_fail (location != NULL);
	
	galeon_embed_load_url (embed, location);

	g_free (location);
}
