
/*
 * GNOME Basic Array support
 *
 * Authors:
 *    Michael Meeks (mmeeks@gnu.org)
 *
 * Copyright 2000, Helix Code Inc.
 */
#include <math.h>
#include <setjmp.h>

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-array.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-statement.h>

#undef ARRAY_DEBUG

#define PRIV(o) ((GBRunArray *)gb_object_get_priv ((o), gbrun_array_get_class ()))

typedef struct {
	GBLong min, max;
} GBRunARange;

static GBValue *
array_deref (GBRunEvalContext *ec,
	     GBRunArray       *a,
	     const GBObjRef   *ref,
	     const GBValue    *assign,
	     gboolean          try_only)
{
	gpointer   *data;
	GBValue   **pos = NULL;
	GSList     *i, *offset;

	if (g_slist_length (a->indicees) !=
	    g_slist_length (ref->parms))
		return gbrun_exception_firev (ec, "Too many / few array indicees");

	i      = a->indicees;
	data   = a->data;
	offset = ref->parms;

#ifdef ARRAY_DEBUG
	fprintf (stderr, "Array index : %s (", (ref->name)?(ref->name):"[Unknown]");
#endif

	while (i && offset) {
		GBRunARange *r = i->data;
		GBValue *v;
		int      into;
		
		v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), offset->data);
		if (!v)
			return NULL;

		into = gb_value_get_as_int (v);
		gb_value_destroy (v);
		
		if (into < r->min ||
		    into > r->max)
			return gbrun_exception_firev (ec, "Out of bounds array index "
						      "%d !<= %d !<= %d", r->min, into, r->max);

		pos    = (GBValue **)&(data [into - r->min]);
		data   = data [into - r->min];
#ifdef ARRAY_DEBUG
		fprintf (stderr, "%d%s", into, i->next?", ":"");
#endif
		offset = offset->next;
		i      = i->next;
	}

#ifdef ARRAY_DEBUG
	if (assign)
		fprintf (stderr, ") := '%s'\n",
			 gb_value_get_as_string (assign)->str);
	else
		fprintf (stderr, ") == '%s'\n",
			 gb_value_get_as_string ((GBValue *)data)->str);
#endif

	if (assign) {
		if (!pos)
			return gbrun_exception_firev (ec, "Wierd, nowhere  to assign");
		if (*pos)
			gb_value_destroy (*pos);

		*pos = gb_value_copy (GB_EVAL_CONTEXT (ec), assign);

		return gb_value_new_empty ();
	}

	if (!data || (!GB_IS_AN_OBJECT (((GBValue *)data)->gtk_type) &&
		      !GB_IS_A_FUNDAMENTAL (((GBValue *)data)->gtk_type)))
		return gbrun_exception_firev (ec, "Serious internal error in array dereference");

	return gb_value_copy (GB_EVAL_CONTEXT (ec), (GBValue *)data);
}

static GBValue *
gbrun_array_deref (GBEvalContext  *ec,
		   GBObject       *object,
		   const GBObjRef *ref,
		   gboolean        try_deref)
{
	GBValue *ans;

	g_return_val_if_fail (GBRUN_IS_ARRAY (object), NULL);

	ans = array_deref (GBRUN_EVAL_CONTEXT (ec), GBRUN_ARRAY (object),
			   ref, NULL, try_deref);

	return ans;
}

static gboolean
gbrun_array_assign (GBEvalContext  *ec,
		    GBObject       *object,
		    const GBObjRef *ref,
		    GBValue        *value,
		    gboolean        try_assign)
{
	GBValue    *ans;

	g_return_val_if_fail (GBRUN_IS_ARRAY (object), FALSE);

	ans = array_deref (GBRUN_EVAL_CONTEXT (ec), GBRUN_ARRAY (object),
			   ref, value, try_assign);

	if (try_assign) /* FIXME: slow */
		gb_eval_context_reset (ec);

	if (!ans)
		return FALSE;

	gb_value_destroy (ans);

	return TRUE;
}

static gboolean
get_as_long (GBRunEvalContext *ec,
	     const GBExpr     *expr,
	     GBLong           *ans)
{
	gboolean ret;
	GBValue *v, *i;

	v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), expr);
	if (!v) {
		gb_value_destroy (v);
		return FALSE;
	}
	
	i = gbrun_value_promote (ec, v, GB_VALUE_LONG);
	if (!i || i->gtk_type != gb_gtk_type_from_value (GB_VALUE_LONG))
		ret = FALSE;
	else
		ret = TRUE;

	*ans = i->v.l;

	gb_value_destroy (v);
	gb_value_destroy (i);

	return ret;
}

static GBRunARange *
range_create (GBRunEvalContext *ec,
	      GBIndex          *idx)
{
	GBRunARange *r = g_new (GBRunARange, 1);

	if (!get_as_long (ec, idx->min, &r->min) ||
	    !get_as_long (ec, idx->max, &r->max)) {
		g_free (r);
		return NULL;
	}

	if (r->min > r->max) {
		GBLong t = r->min;
		g_warning ("Testme: swapping indicees");
		r->min = r->max;
		r->max = t;
	}

	return r;
}

static gpointer
alloc_array (GBRunEvalContext *ec,
	     GSList           *l,
	     GBRunArray       *a)
{
	GBRunARange *r;
	GBLong       size, i;
	gpointer    *data;

	if (!l) {
		if (gtk_type_is_a (a->content_type, GBRUN_TYPE_ARRAY))
			return gb_value_new_empty ();
		else
			return gb_value_new_default (
				GB_EVAL_CONTEXT (ec), a->content_type);
	}
	r = l->data;

	size = r->max - r->min + 1;
	if (size < 0)
		size = -size;

	data = g_new (gpointer, size);

	for (i = 0; i < size; i++)
		data [i] = alloc_array (ec, l->next, a);

	return data;
}

static GBRunArray *
array_new_for_type (GBRunEvalContext *ec, const char *type)
{
	GBRunArray *a;

	a = gtk_type_new (GBRUN_TYPE_ARRAY);

	a->content_type = gb_gtk_type_from_name (type);

	if (a->content_type == GB_VALUE_EMPTY) {
		gbrun_exception_firev (ec, "Invalid type '%s'", type);
		gtk_object_destroy (GTK_OBJECT (a));

		return NULL;
	}

	return a;
}

GBObject *
gbrun_array_new (GBRunEvalContext *ec,
		 const GBVar      *var)
{
	GBRunArray *a;
	GSList     *l;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (var != NULL, NULL);
	g_return_val_if_fail (GB_IS_EVAL_CONTEXT (ec), NULL);

	a = array_new_for_type (ec, var->type);
	
	/* Evaluate indicees */
	a->indicees = NULL;
	for (l = var->indicees; l; l = l->next) {
		GBRunARange *r = range_create (ec, l->data);

		if (!r) /* FIXME: leak */
			return NULL;

		a->indicees = g_slist_append (a->indicees, r);
	}

	a->data = alloc_array (ec, a->indicees, a);

	return GB_OBJECT (a);
}

GBObject *
gbrun_array_new_vals (GBRunEvalContext *ec,
		      GSList           *values)
{
	GBRunArray  *a;
	GBRunARange *r;
	GBValue     *v;
	GBValue    **data;
	int          i;

	g_return_val_if_fail (ec != NULL, NULL);
	g_return_val_if_fail (values != NULL, NULL);
	g_return_val_if_fail (values->data != NULL, NULL);

	a = gtk_type_new (GBRUN_TYPE_ARRAY);

	r = g_new0 (GBRunARange, 1);
	r->min = 0;
	r->max = g_slist_length (values) - 1;

	a->indicees = g_slist_append (NULL, r);

	v = values->data;
	a->content_type = v->gtk_type;

	data = g_new (GBValue *, r->max + 1);
	a->data = (gpointer)data;

	for (i = 0; i < r->max + 1; i++) {
		data [i] = gb_value_promote (GB_EVAL_CONTEXT (ec),
					     a->content_type, values->data);
		values = values->next;
	}

	return GB_OBJECT (a);
}

static void
gbrun_array_copy (GBEvalContext  *ec,
		  GBObject       *src,
		  GBObject       *dest)
{
	g_warning ("Array copy unimplemented");
}

static void
gbrun_array_class_init (GBObjectClass *klass)
{
	klass->copy   = gbrun_array_copy;
	klass->assign = gbrun_array_assign;
	klass->deref  = gbrun_array_deref;
}

GtkType
gbrun_array_get_type (void)
{
	static GtkType array_type = 0;

	if (!array_type) {
		static const GtkTypeInfo array_info = {
			"GBRunArray",
			sizeof (GBRunArray),
			sizeof (GBRunArrayClass),
			(GtkClassInitFunc)  gbrun_array_class_init,
			(GtkObjectInitFunc) NULL,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		array_type = gtk_type_unique (GB_TYPE_OBJECT, &array_info);
		gtk_type_class (array_type);
	}

	return array_type;	
}
