/*
 * GNOME Basic Statement evaluator
 *
 * Author:
 *    Michael Meeks <michael@imaginator.com>
 *
 * Copyright 2000, Helix Code, Inc
 */

#include <gbrun/gbrun.h>
#include <gbrun/gbrun-eval.h>
#include <gbrun/gbrun-value.h>
#include <gbrun/gbrun-stack.h>
#include <gbrun/gbrun-statement.h>
#include <gbrun/gbrun-object.h>
/* for gettimeofday */
#include <sys/time.h> 
#include <unistd.h>
/* for file I/O */
#include <gbrun/gbrun-file.h>

#undef STMT_DEBUG

#ifdef STMT_DEBUG
static void
gbrun_frame_dump (GBRunFrame *rf, const char *type)
{
	GBRunSubFrame *sf = rf->cur;

	fprintf (stderr, "%s dump <%p>: | ", type, rf);

	while (sf) {
		GSList *sl = sf->stmts;

		if (sl && sl->data)
			fprintf (stderr, "%s ", gb_stmt_type (sl->data));
		else
			fprintf (stderr, "NullStack ");
		
		sf = sf->parent;
	}
	
	fprintf (stderr, "|\n");
}
#endif

static void
gbrun_frame_stmts_push (GBRunEvalContext *ec, GSList *stmts)
{
	GBRunFrame    *rf = gbrun_stack_frame (ec->stack);
	GBRunSubFrame *sf = g_new0 (GBRunSubFrame, 1);

	g_return_if_fail (stmts != NULL);
	g_return_if_fail (stmts->data != NULL);

#ifdef STMT_DEBUG
	fprintf (stderr, "Pushing '%s'\n", gb_stmt_type (stmts->data));
#endif

	sf->parent = rf->cur;
	rf->cur    = sf;

	sf->stmts  = stmts;
	sf->init   = TRUE;
	sf->pushed = TRUE;

#ifdef STMT_DEBUG
	gbrun_frame_dump (rf, "after push");
#endif
}

static void
gbrun_frame_crop_to (GBRunEvalContext *ec, guint depth)
{
	GBRunFrame    *rf = gbrun_stack_frame (ec->stack);
	GBRunSubFrame *sf;
	int            i;

	g_return_if_fail (rf != NULL);

	sf = rf->cur;
	for (i = 0; sf; i++)
		sf = sf->parent;

	while ((sf = rf->cur) && (i >= depth)) {
		rf->cur = sf->parent;
		g_free (sf);
		i--;
	}
}

static GBStatement *
gbrun_frame_stmt_next (GBRunEvalContext *ec, gboolean *init)
{
	GBRunFrame    *rf = gbrun_stack_frame (ec->stack);
	GBRunSubFrame *sf;

	if (!rf || !rf->cur) {
#ifdef STMT_DEBUG
		fprintf (stderr, "Hit end of stmt list <%p>\n", rf);
#endif
		return NULL;
	}

#ifdef STMT_DEBUG
	gbrun_frame_dump (rf, "before next");
#endif
	sf = rf->cur;

	*init    = sf->init;
	sf->init = TRUE;
	if (sf->pushed) {
		sf->pushed = FALSE;
		g_assert (sf->stmts->data != NULL);
		return sf->stmts->data;
	}

	sf->stmts = g_slist_next (sf->stmts);

	if (!sf->stmts) {
		rf->cur = sf->parent;
		g_free (sf);
		
		if (rf->cur) {
			rf->cur->init   = FALSE;
			rf->cur->pushed = TRUE;
#ifdef STMT_DEBUG
			fprintf (stderr, "Pop to %s\n", gb_stmt_type (
				rf->cur->stmts->data));
#endif
		}
		
		return gbrun_frame_stmt_next (ec, init);
	}
	g_assert (sf->stmts->data != NULL);

#ifdef STMT_DEBUG
	gbrun_frame_dump (rf, "after next");
#endif
	return sf->stmts->data;
}

static GSList *
seek_label (GSList *stmts, const char *label, int *depth)
{
	GSList  *l, *ret = NULL;

	(*depth)++;

	for (l = stmts; l && !ret; l = l->next) {
		const GBStatement *stmt = l->data;

		switch (stmt->type) {
		case GBS_LABEL:
			if (!g_strcasecmp (stmt->parm.label, label))
				ret = l;
			break;

		case GBS_FOR: {
			ret = seek_label (stmt->parm.forloop.body, label, depth);
			break;

		case GBS_WHILE:
			ret = seek_label (stmt->parm.do_while.body, label, depth);
			break;
			
		case GBS_IF:
			ret = seek_label (stmt->parm.if_stmt.body, label, depth);
			if (ret)
				break;
			ret = seek_label (stmt->parm.if_stmt.else_body, label, depth);
			break;
			
		case GBS_FOREACH:
		case GBS_DO:
			g_warning ("Goto can't cope with flow control stmt %d",
				   stmt->type);
		default:
			break;
		};
	}
	}

	if (!ret)
		(*depth)--;
	return ret;
}

static gboolean
handle_stmt_goto (GBRunEvalContext *ec, const char *label)
{
	GBRunFrame    *rf = gbrun_stack_frame (ec->stack);
	GBRunSubFrame *sf;
	GSList        *label_stmt;
	int            depth, i;

	if (!rf || !rf->func_root)
		goto stmt_goto_err;

	depth = 0;
	if (!(label_stmt = seek_label (rf->func_root, label, &depth)))
		goto stmt_goto_err;

	sf = rf->cur;
	for (i = 0; sf; i++)
		sf = sf->parent;

	/*
	 * For now we will only roll back the frame, forwards
	 * is more interesting.
	 */
	if (depth > i) {
		gbrun_exception_firev (ec, "Can't goto label '%s' at depth %d, "
				       "when we are at depth %d", label, depth, i);
		return FALSE;
	}

	gbrun_frame_crop_to (ec, depth - 1);
	gbrun_frame_stmts_push (ec, label_stmt);
	
	return TRUE;
	
 stmt_goto_err:
	gbrun_exception_firev (ec, "Can't find label '%s'", label);
	return FALSE;
}

gboolean
gbrun_stmt_assign (GBRunEvalContext *ec,
		   const GBExpr *lexpr, const GBExpr *rexpr)
{
	GBValue *rval;
	gboolean result;

/*	printf ("Assign\n");
	gb_expr_print (stdout, lexpr);
	printf (" = ");
	gb_expr_print (stdout, rexpr);*/

	if (lexpr->type != GB_EXPR_OBJREF) {
		gbrun_exception_fire (ec, "Duff lvalue");
		return FALSE;
	}

	rval = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), rexpr);
	if (!rval)
		return FALSE;

	result = gbrun_eval_assign (ec, lexpr->parm.objref, rval);
	gb_value_destroy (rval);

	return result;
}

gboolean
gbrun_stmt_set (GBRunEvalContext *ec, const GBStatement *stmt)
{
	GBValue *tmp;
	gboolean result;
		
	g_return_val_if_fail (stmt->parm.set.var != NULL, FALSE);
	g_return_val_if_fail (stmt->parm.set.objref != NULL, FALSE);

	if (stmt->parm.set.objref->type != GB_EXPR_OBJREF) {
		gbrun_exception_fire (ec, "Reference must be to an object in Set");
		return FALSE;
	}

	tmp = gbrun_eval_objref (ec, stmt->parm.set.objref);
	if (!tmp)
		return FALSE;
	
	result = gbrun_eval_assign (ec, stmt->parm.set.var->parm.objref, tmp);
	gb_value_destroy (tmp);
	
	return result;
}

static gboolean
handle_stmt_case (GBRunEvalContext *ec, GBValue *val, GBSelectCase *c,
		  gboolean *err)
{
	gboolean ret = FALSE;

	g_return_val_if_fail (c != NULL, FALSE);
	g_return_val_if_fail (val != NULL, FALSE);

	switch (c->type) {
	case GB_CASE_ELSE:
		ret = TRUE;
		gbrun_frame_stmts_push (ec, c->statements);
		*err = FALSE;
		break;

	case GB_CASE_EXPR:
	{
		GBValue *v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
						   c->u.expr);
		if (!v)
			ret = *err = TRUE;
		else {
			GBBoolean ok;

			if (!gb_eval_compare (GB_EVAL_CONTEXT (ec), v,
					      GB_EXPR_EQ, val, &ok))
				ret = *err = TRUE;
			if (ok) {
				*err = FALSE;
				gbrun_frame_stmts_push (ec, c->statements);
				ret = TRUE;
			}
			gb_value_destroy (v);
		}
		break;
	}
	case GB_CASE_EXPR_TO_EXPR:
	{
		GBValue   *from, *to;
		GBBoolean  ok;

		from = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
					     c->u.expr_to_expr.from);
		if (!from) {
			*err = TRUE;
			return TRUE;
		}

		to   = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
					     c->u.expr_to_expr.to);
		if (!to) {
			gb_value_destroy (from);
			*err = TRUE;
			return TRUE;
		}

		if (!gb_eval_compare (GB_EVAL_CONTEXT (ec), val,
				      GB_EXPR_GE, from, &ok)) {
			ret = *err = TRUE;
			break;
		}
		
		if (ok) {
			if (!gb_eval_compare (GB_EVAL_CONTEXT (ec), val,
					      GB_EXPR_LE, to, &ok)) {
				ret = *err = TRUE;
				break;
			}
			
			if (ok) {
				*err = FALSE;
				gbrun_frame_stmts_push (ec, c->statements);
				ret = TRUE;
			}
		}

		gb_value_destroy (from);
		gb_value_destroy (to);
		break;
	}
	case GB_CASE_COMPARISON:
	{
		GBValue  *to = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
						    c->u.comparison.to);
		GBBoolean ok;

		if (!to || !gb_eval_compare (GB_EVAL_CONTEXT (ec), val,
					     c->u.comparison.op, to, &ok))
			ret = *err = TRUE;

		else if (ok) {
			*err = FALSE;
			gbrun_frame_stmts_push (ec, c->statements);
			ret = TRUE;
			gb_value_destroy (to);
		}
		break;
	}
	
	case GB_CASE_CSV:
        { 
		const GBExprList *tmp;
		
		for (tmp = c->u.exprs; tmp ; tmp = tmp->next) {
		
			GBValue *v = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
							   tmp->data);
			if (!v)
				ret = *err = TRUE;
			else {
				GBBoolean ok;
				
				if (!gb_eval_compare (GB_EVAL_CONTEXT (ec), v,
						      GB_EXPR_EQ, val, &ok))
					ret = *err = TRUE;
				if (ok) {
					*err = FALSE;
					gbrun_frame_stmts_push (ec, c->statements);
					ret = TRUE;
				}
				gb_value_destroy (v);
			}
		}

		break;
	}
	default:
		g_warning ("Unimplemented select syntax");
		break;
	}

	return ret;
}

gboolean
gbrun_stmt_for (GBRunEvalContext *ec, const GBStatement *stmt, gboolean init)
{
	GBObjRef i;
	GBValue  *to;
	GBBoolean ok;
	gboolean ret=TRUE;
	
	i.name   = stmt->parm.forloop.var;
	i.method = FALSE;
	i.parms  = NULL;
	
	if (init) {
		GBValue *from;

		from = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), 
					     stmt->parm.forloop.from);
		if (!from)
			return FALSE;
		
		gbrun_stack_set (ec, i.name, from);

		gb_value_destroy (from);
	} else {
		GBValue *step;
		GBValue *tmp;

		if (stmt->parm.forloop.step)
			step = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
						     stmt->parm.forloop.step);
		else
			step = gb_value_new_int (1);
		
		if (!step)
			return FALSE;

		tmp = gb_eval_add (gbrun_objref_deref (ec, NULL, &i, TRUE), step);
		gbrun_stack_set (ec, i.name, tmp);

		gb_value_destroy (tmp);
		gb_value_destroy (step);
	}

	to = gb_eval_context_eval (GB_EVAL_CONTEXT (ec), 
				   stmt->parm.forloop.to);
	if (!to ||
	    !gb_eval_compare (GB_EVAL_CONTEXT (ec),
			      gbrun_objref_deref (ec, NULL, &i, TRUE),
			      GB_EXPR_LE, to, &ok)) {
		gb_value_destroy (to);
		return FALSE;
	}
	gb_value_destroy (to);

	if (ok)
		gbrun_frame_stmts_push (ec, stmt->parm.forloop.body);

	return ret;
}


static gboolean
gbrun_stmt_while (GBRunEvalContext *ec, const GBStatement *stmt, gboolean init)
{
	GBValue      *val;
	GBBoolean     going = TRUE;

	val = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
				    stmt->parm.do_while.expr);

	if (!gb_eval_compare (GB_EVAL_CONTEXT (ec),
			      val, GB_EXPR_EQ, GBTrue_value, &going))
		return FALSE;
			
	gb_value_destroy (val);
			
	if (going)
		gbrun_frame_stmts_push (ec, stmt->parm.do_while.body);
	       
	return TRUE;
}

/* 
 * Returns error flag.
 */
static gboolean
gbrun_stmt_evaluate (GBRunEvalContext *ec, const GBStatement *stmt, gboolean init)
{
        g_return_val_if_fail (ec != NULL, FALSE);
	g_return_val_if_fail (stmt != NULL, FALSE);

	gb_eval_context_set_line (GB_EVAL_CONTEXT (ec), stmt->line);

	switch (stmt->type) {

	case GBS_ASSIGN:
		return gbrun_stmt_assign (ec, stmt->parm.assignment.dest,
					  stmt->parm.assignment.val);

	case GBS_FOR:
		return gbrun_stmt_for (ec, stmt, init);

	case GBS_WHILE:
		return gbrun_stmt_while (ec, stmt, init);

	case GBS_CALL:
		/* FIXME: named parameters are not handled at all seemingly */

		if (stmt->parm.func.call->type != GB_EXPR_OBJREF) {
			g_warning ("Duff function expression");
		} else {
			GBValue *discard;

			discard = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
							stmt->parm.func.call);

			if (discard)
				gb_value_destroy (discard);
			else
				return FALSE;
		}
		break;

	case GBS_SELECT:
		if (init) {
			GBValue *val;
			GSList  *l;
			gboolean err = FALSE;

			val = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
						    stmt->parm.select.test_expr);
			if (!val)
				return FALSE;

			for (l = stmt->parm.select.cases; l; l = l->next) {
				if (handle_stmt_case (ec, val, l->data, &err))
					break;
			}
		
			gb_value_destroy (val);
			if (err)
				return FALSE;
		}
		break;

	case GBS_IF:
		if (init) {
			GBValue  *cond;
			GBBoolean bool;

			cond = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
						     stmt->parm.if_stmt.condition);
			if (!cond)
				return FALSE;

			bool = gb_value_get_as_boolean (cond);
			if (bool)
				gbrun_frame_stmts_push (ec, stmt->parm.if_stmt.body);
			else 
				if (stmt->parm.if_stmt.else_body)
					gbrun_frame_stmts_push (ec, stmt->parm.if_stmt.else_body);
			gb_value_destroy (cond);
		}
		break;

	case GBS_RANDOMIZE:
	{
	        GBValue * tmpseed;
		GBInt seed;
		struct timeval tv;

	        if ((stmt->parm.randomize != NULL) &&
		    (ec->random.reseed    == TRUE)) {
		        /* we actually have the seed */
		        ec->random.reseed = FALSE;
			tmpseed = gb_eval_context_eval (GB_EVAL_CONTEXT (ec),
							stmt->parm.randomize);
			/* FIXME: check if value is number and transform to int */
			seed = gb_value_get_as_int (tmpseed);
			gb_value_destroy (tmpseed);

			ec->random.randseed = seed;
		} else {
		        /* we don't have a seed */
		        gettimeofday (&tv,NULL);
			ec->random.randseed = tv.tv_usec;
		}
		break;
	}
	
	case GBS_LOAD:
		gtk_main ();
		g_warning ("Load stubbed %d", gtk_main_level ());
		break;

	case GBS_UNLOAD:
		gtk_main_quit ();
		g_warning ("UnLoad stubbed %d", gtk_main_level ());
		break;

	case GBS_OPEN:
		return gbrun_stmt_open (ec, stmt);

	case GBS_INPUT:
		return gbrun_stmt_input (ec, stmt);

	case GBS_LINE_INPUT:
		return gbrun_stmt_line_input (ec, stmt);
	
	case GBS_CLOSE:
		return gbrun_stmt_close (ec, stmt);

	case GBS_ON_ERROR:
		ec->on_error = stmt->parm.on_error;
		break;
		
	case GBS_GOTO:
		return handle_stmt_goto (ec, stmt->parm.label);

	case GBS_LABEL:
		return TRUE;
		
        case GBS_GET:
		return gbrun_stmt_get (ec, stmt);
		
	case GBS_PUT:
		return gbrun_stmt_put (ec, stmt);

	case GBS_SEEK:
		return gbrun_stmt_seek (ec, stmt);

	case GBS_SET:
		return gbrun_stmt_set (ec, stmt); 

	case GBS_WITH:
	case GBS_FOREACH:
	case GBS_DO:

	default:
		gbrun_exception_fire (ec, "Unhandled statement");
		return FALSE;
	}

	return TRUE;
}

gboolean
gbrun_stmts_evaluate (GBRunEvalContext *ec, GSList *stmts)
{
	GBRunFrame  *rf = gbrun_stack_frame (ec->stack);
	GBStatement *stmt;
	gboolean     init;

	g_return_val_if_fail (rf != NULL, FALSE);

	if (!stmts)
		return TRUE;

	gbrun_frame_stmts_push (ec, stmts);
	rf->func_root = stmts;

	while ((stmt = gbrun_frame_stmt_next (ec, &init))) {
		gboolean err;
		
		err = !gbrun_stmt_evaluate (ec, stmt, init);

	stmt_eval_recheck:
		
		if (err || gbrun_eval_context_exception (ec)) {
			
			if      (ec->on_error.type == GB_ON_ERROR_FLAG)
				break;
			else if (ec->on_error.type == GB_ON_ERROR_GOTO) {
				handle_stmt_goto (ec, ec->on_error.label);
				goto stmt_eval_recheck;
			} else {
				g_assert (ec->on_error.type == GB_ON_ERROR_NEXT);
				gb_eval_context_reset (GB_EVAL_CONTEXT (ec));
			}
		}
	}

	if (stmt)
		gbrun_frame_crop_to (ec, 0);
	
	return stmt == NULL;
}
