
#include <gconfmm/client.h>
#include <gconfmm/private/client_p.h>

// -*- C++ -*-

/* client.ccg
 * 
 * Copyright (C) 2000-2002 GConfmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gconfmm/list_conversion.h>
#include <glibmm/private/object_p.h>
#include <glib/gerror.h>

namespace Gnome
{

namespace Conf
{


Glib::RefPtr<Client> Client::get_default_client()
{
  return Glib::wrap( gconf_client_get_default() );
}

Glib::RefPtr<Client> Client::get_client_for_engine(GConfEngine* engine)
{
  return Glib::wrap( gconf_client_get_for_engine(engine) );
}


guint Client::notify_add(const Glib::ustring& namespace_section,
			 Callback callback)
{
  CallbackHolder* pHolder = new CallbackHolder(callback); //destroyed by callback_destroy.

  GError* pError = 0;
  guint iResult = gconf_client_notify_add(gobj(),
                  namespace_section.c_str(),
                  &CallbackHolder::call,
                  (gpointer)pHolder, //used by GConfClient as an arg to callback_call and callback_destroy.
                  &CallbackHolder::destroy,
                  &pError);
  handle_error(pError);

  //pHolder->add(this);

  return iResult;
}

void Client::notify_remove(guint cnxn)
{
  //Also call's the destroy callback,
  //so the Callback_Holder that was created in notify_add()
  //will be destroyed.

  handle_error(0); //clear error.

  gconf_client_notify_remove(gobj(), cnxn);
}

void Client::add_dir(const Glib::ustring& dir, ClientPreloadType preload)
{
  GError* pError = 0;
  gconf_client_add_dir(gobj(), dir.c_str(), (GConfClientPreloadType)preload, &pError);
  handle_error(pError);
}

void Client::remove_dir(const Glib::ustring& dir)
{
  GError* pError = 0;
  gconf_client_remove_dir(gobj(), dir.c_str(), &pError);
  handle_error(pError);
}

void Client::set_error_handling(ClientErrorHandlingMode mode)
{
  handle_error(0); //clear error.

  gconf_client_set_error_handling(gobj(), (GConfClientErrorHandlingMode)mode);
}

void Client::clear_cache()
{
  handle_error(0); //clear error.

  gconf_client_clear_cache(gobj());
}


bool Client::unset(const Glib::ustring& key)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_unset(gobj(), key.c_str(), &pError);
  handle_error(pError);

  return bResult;
}

Client::type_list_entries Client::all_entries(const Glib::ustring& dir)
{
  GError* pError = 0;
  GSList* pList = gconf_client_all_entries(gobj(), dir.c_str(), &pError);
  handle_error(pError);

  typedef ListHelpers::container_list_converter< type_list_entries, GConfEntry* > type_converter;
  return type_converter::container_from_gslist(
    pList,
    &gconf_entry_free );
}

Glib::SArray Client::all_dirs(const Glib::ustring& dir)
{
  GError* pError = 0;
  GSList* pList = gconf_client_all_dirs(gobj(), dir.c_str(), &pError);
  handle_error(pError);

  typedef  ListHelpers::container_list_converter< std::list<Glib::ustring>, gchar* > type_converter;
  return type_converter::container_from_gslist(
    pList,
    (type_converter::citem_free_func)(&g_free) );
}

void Client::suggest_sync()
{
  GError* pError = 0;
  gconf_client_suggest_sync(gobj(), &pError);
  handle_error(pError);
}

bool Client::dir_exists(const Glib::ustring& dir)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_dir_exists(gobj(), dir.c_str(), &pError);
  handle_error(pError);

  return bResult;
}

bool Client::key_is_writable(const Glib::ustring& key)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_key_is_writable(gobj(), key.c_str(), &pError);
  handle_error(pError);

  return bResult;
}

gdouble Client::get_float(const Glib::ustring& key)
{
  GError* pError = 0;
  gdouble result = gconf_client_get_float(gobj(), key.c_str(), &pError);
  handle_error(pError);

  return result;
}

gint Client::get_int(const Glib::ustring& key)
{
  GError* pError = 0;
  gint result = gconf_client_get_int(gobj(), key.c_str(), &pError);
  handle_error(pError);

  return result;
}

Glib::ustring Client::get_string(const Glib::ustring& key)
{
  std::string strResult;
  GError* pError = 0;
  gchar* pchResult = gconf_client_get_string(gobj(), key.c_str(), &pError);
  handle_error(pError);

  if(pchResult)
  {
    strResult = pchResult;
    g_free(pchResult);
    pchResult = 0;
  }

  return strResult;
}

bool Client::get_bool(const Glib::ustring& key)
{
  GError* pError = 0;
  bool result = (bool)gconf_client_get_bool(gobj(), key.c_str(), &pError);
  handle_error(pError);

  return result;
}

Schema Client::get_schema(const Glib::ustring& key)
{
  Schema schema;

  GError* pError = 0;
  GConfSchema* pSchema = gconf_client_get_schema(gobj(), key.c_str(), &pError);
  if(pSchema)
  {
    schema = Schema(pSchema);
    gconf_schema_free(pSchema);
    pSchema = 0;
  }

  handle_error(pError);

  return schema;
}

Client::type_list_float Client::get_list_of_floats(const Glib::ustring& key)
{
  GSList* pList = get_list(key, GCONF_VALUE_FLOAT);
  return ListHelpers::container_from_gslist_of_floats(pList);
}

Client::type_list_int Client::get_list_of_ints(const Glib::ustring& key)
{
  typedef ListHelpers::container_list_converter< type_list_int, gint > type_converter;
  return type_converter::container_from_gslist(
    get_list(key, GCONF_VALUE_INT) );
}


Client::type_list_bool Client::get_list_of_bools(const Glib::ustring& key)
{
  typedef  ListHelpers::container_list_converter< type_list_bool, bool > type_converter;
  return type_converter::container_from_gslist(
    get_list(key, GCONF_VALUE_BOOL) );
}

Client::type_list_schema Client::get_list_of_schemas(const Glib::ustring& key)
{
  typedef ListHelpers::container_list_converter< type_list_schema, GConfSchema* > type_converter;
  return type_converter::container_from_gslist(
    get_list(key, GCONF_VALUE_SCHEMA),
    &gconf_schema_free );
}


Glib::SArray Client::get_list_of_strings(const Glib::ustring& key)
{
  typedef ListHelpers::container_list_converter< std::list<Glib::ustring>, gchar* > type_converter;
  return type_converter::container_from_gslist(
    get_list(key, GCONF_VALUE_STRING),
    (type_converter::citem_free_func)(&g_free) );
}

GSList* Client::get_list(const Glib::ustring& key, GConfValueType list_type)
{
  GError* pError = 0;
  GSList* pList = gconf_client_get_list(gobj(), key.c_str(), list_type, &pError);
  handle_error(pError);

  return pList;
}


void Client::preload(const Glib::ustring& dirname, ClientPreloadType type)
{
  GError* pError = 0;
  gconf_client_preload(gobj(), dirname.c_str(), (GConfClientPreloadType)type, &pError);
  handle_error(pError);
}


Value Client::get(const Glib::ustring& key)
{
  Value value;

  GError* pError = 0;
  GConfValue* pValue = gconf_client_get(gobj(), key.c_str(), &pError);
  if(pValue)
  {
    value = Value(pValue);
    gconf_value_free(pValue);
    pValue = 0;
  }

  handle_error(pError);

  return value;
}

Value Client::get_without_default(const Glib::ustring& key)
{
 Value value;

  GError* pError = 0;
  GConfValue* pValue = gconf_client_get_without_default(gobj(), key.c_str(), &pError);
  if(pValue)
  {
    value = Value(pValue);
    gconf_value_free(pValue);
    pValue = 0;
  }

  handle_error(pError);

  return value;
}

Entry Client::get_entry(const Glib::ustring& key,
                const Glib::ustring& locale,
                bool use_schema_default /* = true */)
{
  Entry entry;

  GError* pError = 0;
  GConfEntry* pEntry = gconf_client_get_entry(gobj(), key.c_str(), locale.c_str(), use_schema_default, &pError);
  if(pEntry)
  {
    entry = Entry(pEntry);
    gconf_entry_free(pEntry);
    pEntry = 0;
  }

  handle_error(pError);

  return entry;
}


bool Client::set(const Glib::ustring& key, gdouble val)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_float(gobj(), key.c_str(), val, &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, gint val)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_int(gobj(), key.c_str(), val, &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const Glib::ustring& val)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_string(gobj(), key.c_str(), val.c_str(), &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, bool val)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_bool(gobj(), key.c_str(), val, &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const Value& val)
{
  GError* pError = 0;
  gconf_client_set(gobj(), key.c_str(), const_cast<GConfValue*>(val.gobj()), &pError);
  handle_error(pError);

  return true;
  //gconf_client_set doesn't return a success gboolean, like the other _set_ methods.
  //But we can't have on set override which has a different return type.
}

bool Client::set(const Glib::ustring& key, const Schema& schema)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_schema(gobj(), key.c_str(), const_cast<GConfSchema*>(schema.gobj()), &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const ValuePair& pair)
{
  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_pair(gobj(),
					     key.c_str(),
					     pair.first.get_type(), pair.second.get_type(),
					     pair.first.gobj(), pair.second.gobj(),
					     &pError);
  handle_error(pError);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const Glib::SArray& list)
{
  GSList* pList = ListHelpers::gslist_from_container(list);

  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_list(gobj(), key.c_str(), GCONF_VALUE_STRING, pList, &pError);
  handle_error(pError);

  typedef ListHelpers::container_list_converter< Glib::SArray, gchar* > type_converter;
  type_converter::gslist_of_pointers_destroy(pList, (type_converter::citem_free_func)&g_free);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const type_list_int& list)
{
  typedef ListHelpers::container_list_converter< type_list_int, gint > type_converter;
  GSList* pList = type_converter::gslist_from_container(list);

  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_list(gobj(), key.c_str(), GCONF_VALUE_INT, pList, &pError);
  handle_error(pError);

  g_slist_free(pList);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const type_list_bool& list)
{
  typedef ListHelpers::container_list_converter< type_list_bool, gboolean > type_converter;
  GSList* pList = type_converter::gslist_from_container(list);

  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_list(gobj(), key.c_str(), GCONF_VALUE_BOOL, pList, &pError);
  handle_error(pError);

  g_slist_free(pList);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const type_list_float& list)
{
  GSList* pList = ListHelpers::gslist_from_container(list);

  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_list(gobj(), key.c_str(), GCONF_VALUE_FLOAT, pList, &pError);
  handle_error(pError);

  typedef ListHelpers::container_list_converter< type_list_float, gdouble* > type_converter;
  type_converter::gslist_of_pointers_destroy(pList, (type_converter::citem_free_func)&g_free);

  return bResult;
}

bool Client::set(const Glib::ustring& key, const type_list_schema& list)
{
  GSList* pList = ListHelpers::gslist_from_container(list);

  GError* pError = 0;
  bool bResult = (bool)gconf_client_set_list(gobj(), key.c_str(), GCONF_VALUE_SCHEMA, pList, &pError);
  handle_error(pError);

  typedef ListHelpers::container_list_converter< type_list_schema, GConfSchema* > type_converter;
  type_converter::gslist_of_pointers_destroy(pList, &gconf_schema_free);

  return bResult;
}

ValuePair Client::get_pair(const Glib::ustring& key, ValueTypePair types)
{
  ValuePair pair;

  GError* pError = 0;
  bool bResult = (bool)gconf_client_get_pair(gobj(),
					     key.c_str(),
					     types.first, types.second,
					     pair.first.gobj(), pair.second.gobj(),
					     &pError);
  handle_error(pError);

  return pair;
}


void Client::handle_error(GError* pError)
{
  if(pError)
    Glib::Error::throw_exception(pError);
}

} /* namespace Conf */
} /* namespace Gnome */


namespace
{


void Client_signal_value_changed_callback(GConfClient*, const gchar* p0,GConfValue* p1,void* data)
{
  using namespace Gnome::Conf;
  typedef SigC::Slot2<void,const Glib::ustring&,const Value&> SlotType;

  try
  {
    SigC::SlotNode *const slot = Glib::SignalProxyNormal::data_to_slot(data);
    (*(SlotType::Proxy)(slot->proxy_))
        (Glib::convert_const_gchar_ptr_to_ustring(p0)
,Value(p1)
, slot);
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

const Glib::SignalProxyInfo Client_signal_value_changed_info =
{
  "value_changed",
  (GCallback) &Client_signal_value_changed_callback,
  (GCallback) &Client_signal_value_changed_callback
};


void Client_signal_unreturned_error_callback(GConfClient*, GError* p0,void* data)
{
  using namespace Gnome::Conf;
  typedef SigC::Slot1<void,const Glib::Error&> SlotType;

  try
  {
    SigC::SlotNode *const slot = Glib::SignalProxyNormal::data_to_slot(data);
    (*(SlotType::Proxy)(slot->proxy_))
        (Glib::Error(p0, true)
, slot);
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

const Glib::SignalProxyInfo Client_signal_unreturned_error_info =
{
  "unreturned_error",
  (GCallback) &Client_signal_unreturned_error_callback,
  (GCallback) &Client_signal_unreturned_error_callback
};


void Client_signal_error_callback(GConfClient*, GError* p0,void* data)
{
  using namespace Gnome::Conf;
  typedef SigC::Slot1<void,const Glib::Error&> SlotType;

  try
  {
    SigC::SlotNode *const slot = Glib::SignalProxyNormal::data_to_slot(data);
    (*(SlotType::Proxy)(slot->proxy_))
        (Glib::Error(p0, true)
, slot);
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }
}

const Glib::SignalProxyInfo Client_signal_error_info =
{
  "error",
  (GCallback) &Client_signal_error_callback,
  (GCallback) &Client_signal_error_callback
};

} // anonymous namespace

// static
GType Glib::Value<Gnome::Conf::ClientErrorHandlingMode>::value_type()
{
  return gconf_client_error_handling_mode_get_type();
}

// static
GType Glib::Value<Gnome::Conf::ClientPreloadType>::value_type()
{
  return gconf_client_preload_type_get_type();
}


Gnome::Conf::Error::Error(GError* gobject)
:
  Glib::Error (gobject)
{}

Gnome::Conf::Error::Code Gnome::Conf::Error::code() const
{
  return static_cast<Code>(Glib::Error::code());
}

void Gnome::Conf::Error::throw_func(GError* gobject)
{
  throw Gnome::Conf::Error(gobject);
}

// static
GType Glib::Value<Gnome::Conf::Error::Code>::value_type()
{
  return gconf_error_get_type();
}


namespace Glib
{

Glib::RefPtr<Gnome::Conf::Client> wrap(GConfClient* object, bool take_copy /* = false */)
{
  return Glib::RefPtr<Gnome::Conf::Client>( dynamic_cast<Gnome::Conf::Client*> (Glib::wrap_auto ((GObject*)(object), take_copy)) );
  //We use dynamic_cast<> in case of multiple inheritance.
}

} /* namespace Glib */


namespace Gnome
{

namespace Conf
{


/* The *_Class implementation: */

GType Client_Class::get_type()
{
  if(!gtype_) // create the GType if necessary
  {
    // TODO: This is currently just optimized away, apparently with no harm.
    // Is it actually needed?
    // Make sure that the parent type has been created.
    CppClassParent::CppObjectType::get_type();

    // Create the wrapper type, with the same class/instance size as the base type.
    register_derived_type(gconf_client_get_type(), (GClassInitFunc) &class_init_function);

    // Add derived versions of interfaces, if the C type implements any interfaces:
  }

  return gtype_;
}

void Client_Class::class_init_function(BaseClassType* klass)
{
  CppClassParent::class_init_function((CppClassParent::BaseClassType*) klass);
  klass->value_changed = &value_changed_callback;
  klass->unreturned_error = &unreturned_error_callback;
  klass->error = &error_callback;
}


void Client_Class::value_changed_callback(GConfClient* self, const gchar* p0, GConfValue* p1)
{
  CppObjectType *const obj = dynamic_cast<CppObjectType*>(
      Glib::ObjectBase::_get_current_wrapper((GObject*)self));

  if(obj)
  {
    try
    {
      obj->on_value_changed(Glib::convert_const_gchar_ptr_to_ustring(p0)
, Value(p1)
);
    }
    catch(...)
    {
      Glib::exception_handlers_invoke();
    }
  }
  else
  {
    BaseClassType *const base = static_cast<BaseClassType*>(
        g_type_class_peek_parent(G_OBJECT_GET_CLASS(self)) // Get the parent class of the object class.
);
    g_assert(base != 0);

    if(base->value_changed)
      (*base->value_changed)(self, p0, p1);
  }
}

void Client_Class::unreturned_error_callback(GConfClient* self, GError* p0)
{
  CppObjectType *const obj = dynamic_cast<CppObjectType*>(
      Glib::ObjectBase::_get_current_wrapper((GObject*)self));

  if(obj)
  {
    try
    {
      obj->on_unreturned_error(Glib::Error(p0, true)
);
    }
    catch(...)
    {
      Glib::exception_handlers_invoke();
    }
  }
  else
  {
    BaseClassType *const base = static_cast<BaseClassType*>(
        g_type_class_peek_parent(G_OBJECT_GET_CLASS(self)) // Get the parent class of the object class.
);
    g_assert(base != 0);

    if(base->unreturned_error)
      (*base->unreturned_error)(self, p0);
  }
}

void Client_Class::error_callback(GConfClient* self, GError* p0)
{
  CppObjectType *const obj = dynamic_cast<CppObjectType*>(
      Glib::ObjectBase::_get_current_wrapper((GObject*)self));

  if(obj)
  {
    try
    {
      obj->on_error(Glib::Error(p0, true)
);
    }
    catch(...)
    {
      Glib::exception_handlers_invoke();
    }
  }
  else
  {
    BaseClassType *const base = static_cast<BaseClassType*>(
        g_type_class_peek_parent(G_OBJECT_GET_CLASS(self)) // Get the parent class of the object class.
);
    g_assert(base != 0);

    if(base->error)
      (*base->error)(self, p0);
  }
}


Glib::ObjectBase* Client_Class::wrap_new(GObject* o)
{
  return new Client((GConfClient*)(o));
}


/* The implementation: */

GConfClient* Client::gobj_copy()
{
  reference();
  return gobj();
}

Glib::RefPtr<Client> Client::wrap_specific_type(GConfClient* gobject, bool take_copy /* = false */) //static
{
  Glib::RefPtr<Client> refPtr;

  if(gobject)
  {
    //Check for an existing wrapper:
    Client* pCppObject = dynamic_cast<Client*>(Glib::ObjectBase::_get_current_wrapper(G_OBJECT(gobject)));
    if(pCppObject)
    {
      //Return the existing wrapper:
      refPtr = Glib::RefPtr<Client>(pCppObject);
    }
    else
    {
      //Create a new wrapper:
      refPtr = Glib::RefPtr<Client>( new Client(gobject) );
    }

    if(take_copy && refPtr)
      refPtr->reference();
  }

  return refPtr;
}

Client::Client(GConfClient* castitem)
: Glib::Object((GObject*)(castitem))
{}

Client::~Client()
{
}

Client::CppClassType Client::client_class_; //Initialize static member.

GType Client::get_type()
{
  return client_class_.get_type();
}

GType Client::get_base_type()
{
  return gconf_client_get_type();
}

void Client::value_changed(const Glib::ustring& key, const Value& value)
{
  gconf_client_value_changed(gobj(), key.c_str(), const_cast<GConfValue*>(value.gobj()));
}

void Client::unreturned_error(const Glib::Error& error)
{
  gconf_client_unreturned_error(gobj(), const_cast<GError*>(error.gobj()));
}

void Client::error(const Glib::Error& error)
{
  gconf_client_error(gobj(), const_cast<GError*>(error.gobj()));
}


Glib::SignalProxy2<void,const Glib::ustring&,const Value&> Client::signal_value_changed()
{
  return Glib::SignalProxy2<void,const Glib::ustring&,const Value&>(this, &Client_signal_value_changed_info);
}

Glib::SignalProxy1<void,const Glib::Error&> Client::signal_unreturned_error()
{
  return Glib::SignalProxy1<void,const Glib::Error&>(this, &Client_signal_unreturned_error_info);
}

Glib::SignalProxy1<void,const Glib::Error&> Client::signal_error()
{
  return Glib::SignalProxy1<void,const Glib::Error&>(this, &Client_signal_error_info);
}


void Gnome::Conf::Client::on_value_changed(const Glib::ustring& key, const Value& value)
{
  BaseClassType *const base = static_cast<BaseClassType*>(
      g_type_class_peek_parent(G_OBJECT_GET_CLASS(gobject_)) // Get the parent class of the object class.
);
  g_assert(base != 0);

  if(base->value_changed)
    (*base->value_changed)(gobj(),key.c_str(),const_cast<GConfValue*>(value.gobj()));
}

void Gnome::Conf::Client::on_unreturned_error(const Glib::Error& error)
{
  BaseClassType *const base = static_cast<BaseClassType*>(
      g_type_class_peek_parent(G_OBJECT_GET_CLASS(gobject_)) // Get the parent class of the object class.
);
  g_assert(base != 0);

  if(base->unreturned_error)
    (*base->unreturned_error)(gobj(),const_cast<GError*>(error.gobj()));
}

void Gnome::Conf::Client::on_error(const Glib::Error& error)
{
  BaseClassType *const base = static_cast<BaseClassType*>(
      g_type_class_peek_parent(G_OBJECT_GET_CLASS(gobject_)) // Get the parent class of the object class.
);
  g_assert(base != 0);

  if(base->error)
    (*base->error)(gobj(),const_cast<GError*>(error.gobj()));
}


} // namespace Conf

} // namespace Gnome

