/* GDM - The Gnome Display Manager
 * Copyright (C) 1998, 1999 Martin Kasper Petersen <mkp@mkp.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* Code for cookie handling. This really needs to be modularized to
 * support other XAuth types and possibly DECnet...
 */

#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <X11/Xauth.h>

#include "gdm.h"

static const gchar RCSid[]="$Id: auth.c,v 1.1.1.1 1999/04/18 22:08:08 mkp Exp $";

extern gchar *GdmServAuthDir;
extern gchar *GdmUserAuthDir;
extern gchar *GdmUserAuthFile;
extern gchar *GdmUserAuthFB;
extern gint  GdmUserMaxFile;
extern gint  GdmRelaxPerms;

extern void gdm_cookie_generate (GdmDisplay *d);
extern void gdm_debug (const gchar *, ...);
extern void gdm_error (const gchar *, ...);
extern gboolean gdm_file_check (gchar *caller, uid_t user, gchar *dir, gchar *file, 
				gboolean absentok, gint maxsize, gint perms);

gboolean gdm_auth_secure_display (GdmDisplay *d);
gboolean gdm_auth_user_add (GdmDisplay *d, uid_t user, gchar *homedir);
void gdm_auth_user_remove (GdmDisplay *d, uid_t user);


gboolean
gdm_auth_secure_display (GdmDisplay *d)
{
    FILE *af;
    struct hostent *hentry;
    struct in_addr *ia;
    gchar *addr;
    Xauth *xa;
    guint i;

    if (!d)
	return (FALSE);

    gdm_debug ("gdm_auth_secure_display: Setting up access for %s", d->name);

    if (!d->authfile)
	d->authfile = g_strconcat (GdmServAuthDir, "/", d->name, ".Xauth", NULL);
    
    unlink (d->authfile);

    af = fopen (d->authfile, "w");

    if (!af)
	return (FALSE);

    gdm_cookie_generate (d);

    /* FQDN or IP of display host */
    hentry = gethostbyname (d->hostname);

    if (!hentry) {
	gdm_error ("gdm_auth_secure_display: Error getting hentry for %s", d->hostname);
	return (FALSE);
    }

    /* Local access */
    if (d->type == DISPLAY_LOCAL) {
	gdm_debug ("gdm_auth_secure_display: Setting up socket access");

	xa = g_new0 (Xauth, 1);
	
	if (!xa)
	    return (FALSE);

	xa->family = FamilyLocal;
	xa->address = d->hostname;
	xa->address_length = strlen (d->hostname);
	xa->number = g_strdup_printf ("%d", d->dispnum);
	xa->number_length = 1;
	xa->name = "MIT-MAGIC-COOKIE-1";
	xa->name_length = 18;
	xa->data = d->bcookie;
	xa->data_length = strlen (d->bcookie);
	XauWriteAuth (af, xa);
	d->auths = g_slist_append (d->auths, xa);
    }

    gdm_debug ("gdm_auth_secure_display: Setting up network access");
    
    /* Network access */
    for (i=0 ; i < hentry->h_length ; i++) {
	xa = g_new0 (Xauth, 1);

	if (!xa)
	    return (FALSE);

	xa->family = FamilyInternet;

	addr = g_new0 (gchar, 4);

	if (!addr)
	    return (FALSE);

	ia = (struct in_addr *) hentry->h_addr_list[i];

	if (!ia)
	    break;

	memcpy (addr, &ia->s_addr, 4);
	xa->address = addr; 
	xa->address_length = 4;
	xa->number = g_strdup_printf ("%d", d->dispnum);
	xa->number_length = 1;
	xa->name = "MIT-MAGIC-COOKIE-1";
	xa->name_length = 18;
	xa->data = d->bcookie;
	xa->data_length = strlen (d->bcookie);

	XauWriteAuth (af, xa);

	d->auths = g_slist_append (d->auths, xa);
    }

    fclose (af);
    setenv ("XAUTHORITY", d->authfile, TRUE);

    gdm_debug ("gdm_auth_secure_display: Setting up access for %s ... done", d->name);

    return (TRUE);
}


gboolean
gdm_auth_user_add (GdmDisplay *d, uid_t user, gchar *homedir)
{
    gchar *authdir;
    gint authfd;
    FILE *af;
    GSList *auths = NULL;

    if (!d || !user)
	return (FALSE);

    gdm_debug ("gdm_auth_user_add: Adding cookie for %d", user);

    /* Determine whether UserAuthDir is specified. Otherwise ~user is used */
    if (strlen (GdmUserAuthDir))
	authdir = GdmUserAuthDir;
    else
	authdir = homedir;

    umask (077);

    /* Find out if the Xauthority file passes the paranoia check */
    if (! gdm_file_check ("gdm_auth_user_add", user, authdir, GdmUserAuthFile, 
			  TRUE, GdmUserMaxFile, GdmRelaxPerms)) {

	/* No go. Let's create a fallback file in GdmUserAuthFB (/tmp) */
	d->authfb = TRUE;
	d->userauth = g_strconcat (GdmUserAuthFB, "/.gdmXXXXXX", NULL);
	authfd = mkstemp (d->userauth);

	if (authfd == -1) {
	    g_free (d->userauth);
	    d->userauth = NULL;
	    return (FALSE);
	}

	af = fdopen (authfd, "w");
    }
    else { /* User's Xauthority file is ok */
	d->authfb = FALSE;
	d->userauth = g_strconcat (authdir, "/", GdmUserAuthFile, NULL);

	/* FIXME: Better implement my own locking. The libXau one is not kosher */
	if (XauLockAuth (d->userauth, 3, 3, 0) != LOCK_SUCCESS) {
	    g_free (d->userauth);
	    d->userauth = NULL;
	    return (FALSE);
	}

	af = fopen (d->userauth, "a+");	
    }

    if (!af) {
	/* Really no need to clean up here - this process is a goner anyway */
	XauUnlockAuth (d->userauth);
	g_free (d->userauth);
	d->userauth = NULL;
	return (FALSE); 
    }

    gdm_debug ("gdm_auth_user_add: Using %s for cookies", d->userauth);

    /* Write the authlist for the display to the cookie file */
    auths = d->auths;

    while (auths) {
	XauWriteAuth (af, auths->data);
	auths = auths->next;
    }

    fclose (af);
    XauUnlockAuth (d->userauth);
    setenv ("XAUTHORITY", d->userauth, TRUE);

    gdm_debug ("gdm_auth_user_add: Done");

    return (TRUE);
}


void 
gdm_auth_user_remove (GdmDisplay *d, uid_t user)
{
    FILE *af;
    Xauth *xa;
    GSList *keep = NULL;
    gchar *authfile, *authdir;

    if (!d || !d->userauth)
	return;

    gdm_debug ("gdm_auth_user_remove: Removing cookie from %s (%d)", d->userauth, d->authfb);

    /* If we are using the fallback cookie location, simply nuke the
     * cookie file */
    if (d->authfb) {
	unlink (d->userauth);
	g_free (d->userauth);
	d->userauth = NULL;
	return;
    }

    authfile = g_basename (d->userauth);
    authdir = g_dirname (d->userauth);

    /* Now, the cookie file could be owned by a malicious user who
     * decided to concatenate something like /dev/kcore or his entire
     * MP3 collection to it. So we better play it safe... */

    if (! gdm_file_check ("gdm_auth_user_remove", user, authdir, authfile, 
			  FALSE, GdmUserMaxFile, GdmRelaxPerms)) {
	gdm_error (_("gdm_auth_user_remove: Ignoring suspicious looking cookie file %s"), d->userauth);
	return; 
    }

    g_free (authfile);
    g_free (authdir);

    if (XauLockAuth (d->userauth, 3, 3, 0) != LOCK_SUCCESS)
	return;

    af = fopen (d->userauth, "r");

    if (!af) {
	XauUnlockAuth (d->userauth);
	return;
    }

    /* Read the user's entire Xauth file into memory to avoid temporary file
     * issues */
    while ( (xa = XauReadAuth (af)) ) 
	if (memcmp (d->bcookie, xa->data, xa->data_length)) 
	    keep = g_slist_append (keep, xa);
	else
	    XauDisposeAuth (xa);

    fclose (af);

    /* Truncate and write out all cookies not belonging to this display */
    af = fopen (d->userauth, "w");

    if (!af) {
	XauUnlockAuth (d->userauth);
	return;
    }

    while (keep) {
	XauWriteAuth (af, keep->data);
	XauDisposeAuth (keep->data);
	keep = keep->next;
    }

    g_slist_free (keep);

    fclose (af);
    XauUnlockAuth (d->userauth);

    g_free (d->userauth);
    d->userauth = NULL;

    return;
}


/* EOF */
