/* gIDE
 * Copyright (C) 1999-2000 Jody Goldberg <jgoldberg@home.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
/*
 * Handlers to undo & redo commands
 */

#include <config.h>
#include "gide-type-util.h"
#include "gI_document.h"
#include "gI_commands.h"

#define GIDE_COMMAND_TYPE        (gide_command_get_type ())
#define GIDE_COMMAND(o)          (GTK_CHECK_CAST ((o), GIDE_COMMAND_TYPE, GideCommand))
#define GIDE_COMMAND_CLASS(k)    (GTK_CHECK_CLASS_CAST((k), GIDE_COMMAND_TYPE, GideCommandClass))
#define IS_GIDE_COMMAND(o)       (GTK_CHECK_TYPE ((o), GIDE_COMMAND_TYPE))
#define IS_GIDE_COMMAND_CLASS(k) (GTK_CHECK_CLASS_TYPE ((k), GIDE_COMMAND_TYPE))

typedef struct
{
	GtkObject parent;
	char const *cmd_descriptor;	/* A string to put in the menu */
} GideCommand;

typedef gboolean (* UndoCmd)(GideCommand *this, CommandContext *context);
typedef gboolean (* RedoCmd)(GideCommand *this, CommandContext *context);

typedef struct {
	GtkObjectClass parent_class;

	UndoCmd		undo_cmd;
	RedoCmd		redo_cmd;
} GideCommandClass;

static GIDE_MAKE_TYPE(gide_command, "GideCommand",
			  GideCommand, NULL, NULL,
			  gtk_object_get_type());

/* Store the real GtkObject dtor pointer */
static void (* gtk_object_dtor) (GtkObject *object) = NULL;

static void
gide_command_destroy (GtkObject *obj)
{
	GideCommand *cmd = GIDE_COMMAND(obj);

	g_return_if_fail (cmd != NULL);

	/* The const was to avoid accidental changes elsewhere */
	g_free ((gchar *)cmd->cmd_descriptor);

	/* Call the base class dtor */
	g_return_if_fail (gtk_object_dtor);
	(*gtk_object_dtor) (obj);
}

#define GIDE_MAKE_COMMAND(type, func) \
static gboolean \
func ## _undo (GideCommand *me, CommandContext *context); \
static gboolean \
func ## _redo (GideCommand *me, CommandContext *context); \
static void \
func ## _destroy (GtkObject *object); \
static void \
func ## _class_init (GideCommandClass * const parent) \
{	\
	parent->undo_cmd = (UndoCmd)& func ## _undo;		\
	parent->redo_cmd = (RedoCmd)& func ## _redo;		\
	if (gtk_object_dtor == NULL)				\
		gtk_object_dtor = parent->parent_class.destroy;	\
	parent->parent_class.destroy = & func ## _destroy;	\
} \
static GIDE_MAKE_TYPE_WITH_PARENT(func, #type, type, GideCommandClass, func ## _class_init, NULL, \
				      gide_command_get_type())

/******************************************************************/

/**
 * get_menu_label : Utility routine to get the descriptor associated
 *     with a list of commands.
 *
 * @cmd_list : The command list to check.
 *
 * Returns : A static reference to a descriptor.  DO NOT free this.
 */
static gchar const *
get_menu_label (GSList *cmd_list)
{
	if (cmd_list != NULL) {
		GideCommand *cmd = GIDE_COMMAND (cmd_list->data);
		return cmd->cmd_descriptor;
	}

	return NULL;
}

/**
 * undo_redo_menu_labels : Another utility to set the menus correctly.
 *
 * doc : The document whose undo/redo queues we are modifying
 */
void
command_undo_redo_menu_labels (GideDocument *doc)
{
	gI_document_set_undo_redo_state( doc,
					 get_menu_label (doc->undo_commands),
					 get_menu_label (doc->redo_commands));
}

/*
 * command_undo : Undo the last command executed.
 *
 * @context : The command context which issued the request.
 *            Any user level errors generated by undoing will be reported
 *            here.
 *
 * @doc : The document whose commands to undo.
 */
void
command_undo (CommandContext *context, GideDocument *doc)
{
	GideCommand *cmd;

	g_return_if_fail (doc != NULL);
	g_return_if_fail (doc->undo_commands != NULL);

	cmd = GIDE_COMMAND(doc->undo_commands->data);
	g_return_if_fail (cmd != NULL);

	doc->undo_commands = g_slist_remove (doc->undo_commands,
					     doc->undo_commands->data);
	GIDE_COMMAND_CLASS(cmd->parent.klass)->undo_cmd (cmd, context);
	doc->redo_commands = g_slist_prepend (doc->redo_commands, cmd);
	command_undo_redo_menu_labels (doc);

	/* TODO : Should we mark the workbook as clean or pristine too */
}

/*
 * command_redo : Redo the last command that was undone.
 *
 * @context : The command context which issued the request.
 *            Any user level errors generated by redoing will be reported
 *            here.
 *
 * @doc : The document whose commands to redo.
 */
void
command_redo (CommandContext *context, GideDocument *doc)
{
	GideCommand *cmd;

	g_return_if_fail (doc);
	g_return_if_fail (doc->redo_commands);

	cmd = GIDE_COMMAND(doc->redo_commands->data);
	g_return_if_fail (cmd != NULL);

	/* Remove the command from the undo list */
	doc->redo_commands = g_slist_remove (doc->redo_commands,
					     doc->redo_commands->data);
	GIDE_COMMAND_CLASS(cmd->parent.klass)->redo_cmd (cmd, context);
	doc->undo_commands = g_slist_prepend (doc->undo_commands, cmd);
	command_undo_redo_menu_labels (doc);
}

/*
 * command_list_pop_top : utility routine to free the top command on
 *        the undo list, and to regenerate the menus if needed.
 *
 * @cmd_list : The set of commands to free from.
 */
void
command_list_pop_top_undo (GideDocument *doc)
{
	GtkObject *cmd;
	
	g_return_if_fail (doc->undo_commands != NULL);

	cmd = GTK_OBJECT (doc->undo_commands->data);
	g_return_if_fail (cmd != NULL);

	gtk_object_unref (cmd);
	doc->undo_commands = g_slist_remove (doc->undo_commands,
					     doc->undo_commands->data);
	command_undo_redo_menu_labels (doc);
}

/*
 * command_list_release : utility routine to free the resources associated
 *    with a list of commands.
 *
 * @cmd_list : The set of commands to free.
 */
void
command_list_release (GSList *cmd_list)
{
	while (cmd_list != NULL) {
		GtkObject *cmd = GTK_OBJECT (cmd_list->data);

		g_return_if_fail (cmd != NULL);

		gtk_object_unref (cmd);
		cmd_list = g_slist_remove (cmd_list, cmd_list->data);
	}
	cmd_list = NULL;
}

/**
 * command_push_undo : An internal utility to tack a new command
 *    onto the undo list.
 *
 * @wb : The workbook the command operated on.
 * @cmd : The new command to add.
 * @trouble : A flag indicating whether there was a problem with the
 *            command.
 *
 * returns : TRUE if there was an error.
 */
static gboolean
command_push_undo (GideDocument *doc, GtkObject *cmd, gboolean const trouble)
{
	/* TODO : trouble should be a variable not an argument.
	 * We should call redo on the command object and use
	 * the result as the value for trouble.
	 */
	if  (!trouble) {
		g_return_val_if_fail (doc != NULL, TRUE);
		g_return_val_if_fail (cmd != NULL, TRUE);

		command_list_release (doc->redo_commands);
		doc->redo_commands = NULL;

		doc->undo_commands = g_slist_prepend (doc->undo_commands, cmd);

		command_undo_redo_menu_labels (doc);
	} else
		gtk_object_unref (cmd);

	return trouble;
}

/******************************************************************/

#define CMD_INSERT_TEXT_TYPE        (cmd_insert_text_get_type ())
#define CMD_INSERT_TEXT(o)          (GTK_CHECK_CAST ((o), CMD_INSERT_TEXT_TYPE, CmdInsertText))

typedef struct
{
	GideCommand parent;

	GideDocument *doc;
	gchar *text;
	gint from;
	gint to;
} CmdInsertText;

GIDE_MAKE_COMMAND (CmdInsertText, cmd_insert_text);

static gboolean
cmd_insert_text_undo (GideCommand *cmd, CommandContext *context)
{
	CmdInsertText *me = CMD_INSERT_TEXT( cmd );

	gI_document_disconnect_text_signals( me->doc );
	
	gI_text_set_point( me->doc, me->from );
	gtk_editable_select_region( GTK_EDITABLE( me->doc ),
				    me->from, me->to );
	gI_document_delete_selection( me->doc );
	gtk_editable_set_position( GTK_EDITABLE( me->doc ),
				   me->to );

	gI_document_connect_text_signals( me->doc );
	
	return FALSE;
}

static gboolean
cmd_insert_text_redo (GideCommand *cmd, CommandContext *context)
{
	CmdInsertText *me = CMD_INSERT_TEXT(cmd);

	gI_document_disconnect_text_signals( me->doc );
		
	gI_text_set_point( me->doc, me->to );
	gI_document_insert( me->doc, NULL, NULL, NULL,
			    me->text, (me->to-me->from) );
	gtk_editable_set_position( GTK_EDITABLE( me->doc ), me->from );

	gI_document_connect_text_signals( me->doc );
	
	return FALSE;
}

static void
cmd_insert_text_destroy (GtkObject *cmd)
{
	CmdInsertText *me = CMD_INSERT_TEXT(cmd);
	if (me->text != NULL) {
		g_free (me->text);
		me->text = NULL;
	}
	gide_command_destroy (cmd);
}

gboolean
cmd_insert_text (GideDocument *doc, const gchar *text, gint to, gint from)
{
	GtkObject *obj;
	CmdInsertText *me;

	g_return_val_if_fail (doc != NULL, TRUE);
	g_return_val_if_fail (doc != NULL, TRUE);

	obj = gtk_type_new (CMD_INSERT_TEXT_TYPE);
	me = CMD_INSERT_TEXT (obj);

	/* Store from/to for undo */
	me->doc = doc;
	me->text = g_strdup(text);
	me->from = from;
	me->to = to;

	me->parent.cmd_descriptor = g_strdup_printf (_("Typing \"%.10s\""),
						     me->text);

	/* Register the command object */
	return command_push_undo (doc, obj, FALSE);
}

/******************************************************************/

#define CMD_DELETE_TEXT_TYPE        (cmd_delete_text_get_type ())
#define CMD_DELETE_TEXT(o)          (GTK_CHECK_CAST ((o), CMD_DELETE_TEXT_TYPE, CmdDeleteText))

typedef struct
{
	GideCommand parent;

	GideDocument *doc;
	gchar *text;
	gint from;
	gint to;
} CmdDeleteText;

GIDE_MAKE_COMMAND (CmdDeleteText, cmd_delete_text);

static gboolean
cmd_delete_text_undo (GideCommand *cmd, CommandContext *context)
{
	CmdDeleteText *me = CMD_DELETE_TEXT(cmd);

	gI_document_disconnect_text_signals( me->doc );
	
	gI_text_set_point( me->doc, me->to );
	gI_document_insert( me->doc, NULL, NULL, NULL,
			    me->text, (me->to-me->from) );
	gtk_editable_set_position( GTK_EDITABLE( me->doc ), me->from );

	gI_document_connect_text_signals( me->doc );
		
	return FALSE;
}

static gboolean
cmd_delete_text_redo (GideCommand *cmd, CommandContext *context)
{
	CmdDeleteText *me = CMD_DELETE_TEXT( cmd );

	gI_document_disconnect_text_signals( me->doc );	

	gI_text_set_point( me->doc, me->from );
	gtk_editable_select_region( GTK_EDITABLE( me->doc ),
				    me->from, me->to );
	gI_document_delete_selection( me->doc );
	gtk_editable_set_position( GTK_EDITABLE( me->doc ),
				   me->to );

	gI_document_connect_text_signals( me->doc );	

	return FALSE;
}

static void
cmd_delete_text_destroy (GtkObject *cmd)
{
	CmdDeleteText *me = CMD_DELETE_TEXT(cmd);
	if (me->text != NULL) {
		g_free (me->text);
		me->text = NULL;
	}
	gide_command_destroy (cmd);
}

gboolean
cmd_delete_text (GideDocument *doc, const gchar *text, gint to, gint from)
{
	GtkObject *obj;
	CmdDeleteText *me;
	
	g_return_val_if_fail (doc != NULL, TRUE);
	g_return_val_if_fail (doc != NULL, TRUE);

	obj = gtk_type_new (CMD_DELETE_TEXT_TYPE);
	me = CMD_DELETE_TEXT (obj);

	/* Store from/to for undo */
	me->doc = doc;
	me->text = g_strdup(text);
	me->from = from;
	me->to = to;

	me->parent.cmd_descriptor = g_strdup_printf (_("Deleting \"%.10s\""),
						     me->text);

	/* Register the command object */
	return command_push_undo (doc, obj, FALSE);
}













