/* $Header: /cvs/gnome/gIDE/src/gI_debug_backtrace.c,v 1.2 2000/04/22 17:23:01 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include <ctype.h>
#include "gI_debug_backtrace.h"
#include "gI_debug_p.h"
#include "gI_compile.h"
#include "gI_common.h"
#include "gI_file.h"

/* local prototypes */
static gint backtrace_list_select(GtkWidget* widget, GdkEventButton* event);
static void backtrace_install_popup(gI_bt_window* bt);
static void backtrace_goto_frame(GtkWidget* widget, gI_debug_window* dw);
static void backtrace_view_source(GtkWidget* widget, gI_debug_window* dw);
static void backtrace_add_frame(gI_debug_window* dw, gchar* frame);

/* local functions */
/*
 * View the source of a backtrace
 */
static void
backtrace_view_source(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	GList*								selection;
	gchar*								file;
	gchar*								line;

	/* since we decided that there can be only one debug window */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	selection = GTK_CLIST(dw->bt_window->list)->selection;
	if(!selection)
	{
		return;
	}

	gtk_clist_get_text(GTK_CLIST(dw->bt_window->list), (gint)selection->data,
		2, &file);
	gtk_clist_get_text(GTK_CLIST(dw->bt_window->list), (gint)selection->data,
		3, &line);

	if(!file || !line)
	{
		return;
	}

	if(check_file_open(file))
	{
		goto_file(file);
	}
	else
	{
		if(!file_exist(file))
		{
			return;
		}

		file_open_by_name(main_window, file);
	}

	goto_line(atoi(line));
}


/*
 * Go to a backtrace frame
 */
static void
backtrace_goto_frame(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	GList*								selection;
	gchar*								frameno;
	gchar								cmdstr[100];

	/* since we decided that there can be only one debug window */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	selection = GTK_CLIST(dw->bt_window->list)->selection;
	if(!selection)
	{
		g_warning("Can't execute because there's no selection\n");
		return;
	}

	gtk_clist_get_text(GTK_CLIST(dw->bt_window->list), (gint)selection->data,
		0, &frameno);

	g_snprintf(cmdstr, sizeof(cmdstr), "frame %s", frameno);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
}


/*
 * Install the backtrace popups
 */
static void
backtrace_install_popup(
	gI_bt_window*						bt
)
{
	GtkAccelGroup*						group;
	GtkItemFactoryEntry					bt_popup_items[] =
	{
		{ "/Goto frame", NULL, backtrace_goto_frame, 0 },
		{ "/View source", NULL, backtrace_view_source, 0 }
	};

	group = gtk_accel_group_new();

	bt->popup_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Backtrace>", group);
	gtk_item_factory_create_items(bt->popup_menu,
		sizeof(bt_popup_items) / sizeof(bt_popup_items[0]), bt_popup_items,
		NULL);

	gtk_accel_group_attach(group, GTK_OBJECT(bt->window));
}


/*
 * Callback for the backtrace list
 */
static gint
backtrace_list_select(
	GtkWidget*							widget,
	GdkEventButton*						event
)
{
	gI_bt_window*						bt;

	bt = main_window->debug_window->bt_window;
	if(!bt)
	{
		return FALSE;
	}

	if(event->button == 3)
	{
		gtk_item_factory_popup(GTK_ITEM_FACTORY(bt->popup_menu),
			(guint)event->x_root, (guint)event->y_root, event->button,
			event->time);
		return TRUE;
	}

	return FALSE;
}


/*
 * Add a new backtrace frame
 */
static void
backtrace_add_frame(
	gI_debug_window*					dw,
	gchar*								frame
)
{
	glong								count;
	gchar								frameno[10];
	gchar								funcaddr[STRLEN];
	gchar								_funcaddr[STRLEN];
	gchar*								row[4];
	gchar*								ptr;
	gchar*								fn[3];
	gchar*								file = NULL;
	gchar*								line = NULL;

	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}
	if(!dw->bt_window)
	{
		gI_error_dialog(_("No backtrace window available"));
		return;
	}

	count = sscanf(frame, "%s %s", frameno, funcaddr);
	if(count == 2)
	{
		if(sscanf(frame, "%*s %*s in %s", _funcaddr) == 1)
		{
			strcpy(funcaddr, _funcaddr);
		}

		ptr = &frame[strlen(frame) - 1];
		if(isspace(*ptr))
		{
			while(isspace(*ptr))
			{
				ptr--;
			}
		}

		while(!isspace(*ptr))
		{
			ptr--;
		}

		if(strchr(ptr, ':'))
		{
			SK_GetFields(ptr, fn, ':');

			file = fn[1];
			line = fn[2];

			while(isspace(*file))
			{
				file++;
			}
			while(isspace(*line))
			{
				line++;
			}
		}
		else
		{
			fn[1] = NULL;
			fn[2] = NULL;
		}

		if(!dw->bt_window)
		{
			return;
		}

		row[0] = frameno;
		row[1] = funcaddr;
		row[2] = file;
		row[3] = line;
		gtk_clist_append(GTK_CLIST(dw->bt_window->list), row);

		if(fn[0] != NULL)
		{
			free(fn[0]);
		}
		if(fn[1] != NULL)
		{
			free(fn[1]);
		}
		if(fn[2] != NULL)
		{
			free(fn[2]);
		}
	}
	else
	{
		return;
	}
}


/* public functions */
/*
 * Create a new backtrace window
 */
gI_bt_window*
gI_backtrace_window_new(
	void
)
{
	gI_bt_window*						bt;
	gchar* list_titles[] = { "No.", "Function/Address", "File", "Line" };

	bt = (gI_bt_window*)g_malloc0(sizeof(gI_bt_window));

	bt->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(bt->window), "Backtrace");
	gtk_widget_set_usize(bt->window, 550, 225);
	gtk_signal_connect(GTK_OBJECT(bt->window), "destroy",
		GTK_SIGNAL_FUNC(gI_backtrace_window_destroy), (gpointer)bt);
	backtrace_install_popup(bt);

	bt->list = gtk_clist_new_with_titles(4, list_titles);
	gtk_clist_set_column_width(GTK_CLIST(bt->list), 0, 30);
	gtk_clist_set_column_width(GTK_CLIST(bt->list), 1, 200);
	gtk_clist_set_column_width(GTK_CLIST(bt->list), 2, 200);
	gtk_clist_set_column_width(GTK_CLIST(bt->list), 3, 50);
	gtk_clist_column_titles_passive(GTK_CLIST(bt->list));
	gtk_container_add(GTK_CONTAINER(bt->window), bt->list);
	gtk_signal_connect(GTK_OBJECT(bt->list), "button_press_event",
		GTK_SIGNAL_FUNC(backtrace_list_select), (gpointer)bt);
	gtk_widget_show(bt->list);

	gtk_widget_show(bt->window);

	return bt;
}


void
gI_backtrace_window_destroy(
	GtkWidget*							widget,
	gI_bt_window*						bt
)
{
	g_free(bt);
	bt = NULL;
	main_window->debug_window->bt_window = NULL;
}


void
gI_backtrace_process_output(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	gchar*								ptr;
	gchar*								retptr;
	glong								end = 0;
	gchar								frame[MAXLEN];

	if(dw->bt_window)
	{
		gtk_clist_clear(GTK_CLIST(dw->bt_window->list));
	}

	if(!strncmp(msg, "No stack", 8))
	{
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	if(!dw->bt_window)
	{
		dw->bt_window = gI_backtrace_window_new();
	}

	ptr = &msg[0];

	while((ptr = strchr(msg, '\n')))
	{
		*ptr = ' ';
	}

	ptr = &msg[0];

	ptr = strchr(ptr, '#');
	ptr++;

	end = 0;

	while(strncmp(ptr, GDB_PROMPT, strlen(GDB_PROMPT)))
	{
		/* this may get an endless loop */
		retptr = strchr(ptr, '#');
		if(!retptr && !end)
		{
			end = 1;
			retptr = strchr(ptr, '\0');
			if(retptr)
			{
				--retptr;
			}
		}

		if(!retptr)
		{
			/* end of backtrace reached */
			dw->wait_for = N_WAIT_NOTHING;
			return;
		}

		strncpy(frame, ptr, retptr - ptr);
		frame[retptr - ptr] = '\0';

		backtrace_add_frame(dw, frame);

		ptr = retptr + 1;
	}

	dw->wait_for = N_WAIT_NOTHING;
}
