/* $Header: /cvs/gnome/gIDE/src/gI_debug_breakpoint.c,v 1.2 2000/04/22 17:23:01 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include <ctype.h>
#include "gI_debug_breakpoint.h"
#include "gI_debug_p.h"
#include "gI_common.h"
#include "gI_compile.h"
#include "gI_file.h"


/* local prototypes */
static void breakpoint_install_popup(gI_brkpnt_window* bpw);
static void breakpoint_clear(GtkWidget* widget, gpointer data);
static void breakpoint_view_source(GtkWidget* widget, gpointer data);
static void breakpoint_list_select(GtkWidget* widget, gint row, gint column,
	GdkEventButton* bevent, gI_brkpnt_window* bpw);
static gint breakpoint_event_handler(GtkWidget* widget, GdkEventButton* event);
static void breakpoint_add(gI_debug_window* dw, gchar* brkpnt);
static gI_brkpnt* breakpoint_new(void);
static void breakpoint_clear_ok(GtkWidget* widget, GtkWidget* entry);
static gboolean breakpoint_exists(GideDocument* current, gchar* file,
	glong line);


/* local functions */
/*
 * Create a new breakpoint structure
 */
gI_brkpnt*
breakpoint_new(
	void
)
{
	gI_brkpnt*							brkpnt;

	brkpnt = g_malloc0(sizeof(gI_brkpnt));
	brkpnt->file = NULL;
	brkpnt->line = -1;
	brkpnt->no = -1;

	return brkpnt;
}


/*
 * Add a breakpoint
 */
static void
breakpoint_add(
	gI_debug_window*					dw,
	gchar*								brkpnt
)
{
	gchar*								row[4];
	gchar								brkno[10];
	gchar								function[256];
	gchar								fileln[STRLEN];
	gchar								file[STRLEN];
	gchar								line[10];
	gchar*								ptr;
	glong								count;
	gI_brkpnt*							_brkpnt;

	/* since we decided that there can be only one debug window */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No breakpoint window available"));
		return;
	}

	/* this should not happen, abort */
	if(isspace(brkpnt[0]))
	{
		g_warning("Corrupt breakpoint\n");
		return;
	}

	/* only breakpoints, no watchpoints */
	if(strstr(brkpnt, "watchpoint"))
	{
		gI_error_dialog(_("Can't add a watchpoint as a breakpoint"));
		return;
	}

	count = sscanf(brkpnt, "%s %*s %*s %*s %*s in %s at %s", brkno, function,
		fileln);
	if(count == 3)
	{
		/* get file and line no */
		ptr = strchr(fileln, ':');
		ptr++;
		strcpy(line, ptr);
		ptr--;
		*ptr = '\0';
		strcpy(file, fileln);

		/* add breakpoint to list */
		row[0] = brkno;
		row[1] = function;
		row[2] = file;
		row[3] = line;
		gtk_clist_append(GTK_CLIST(dw->brkpnt_window->list), row);

		_brkpnt = breakpoint_new();
		_brkpnt->file = g_strdup(file);
		_brkpnt->line = atoi(line);
		_brkpnt->no = atoi(brkno);

/***
		if(!dw->breakpoints)
		{
			dw->breakpoints = g_list_alloc();
		}
***/

		dw->breakpoints = g_list_prepend(dw->breakpoints, (gpointer)_brkpnt);
	}
}


/*
 * Popup function to view the source
 */
static void
breakpoint_view_source(
	GtkWidget*							widget,
	gpointer							data
)
{
	GList*								selection;
	gI_brkpnt_window* bpw = main_window->debug_window->brkpnt_window;
	gchar*								file;
	gchar*								line;

	if(!bpw)
	{
		gI_error_dialog(_("No breakpoint window available"));
		return;
	}

	selection = GTK_CLIST(bpw->list)->selection;
	if(!selection)
	{
		return;
	}

	gtk_clist_get_text(GTK_CLIST(bpw->list), (gint)selection->data, 2, &file);
	gtk_clist_get_text(GTK_CLIST(bpw->list), (gint)selection->data, 3, &line);

	if(!file_exist(file))
	{
		return;
	}

	if(check_file_open(file))
	{
		goto_file(file);
	}
	else
	{
		file_open_by_name(main_window, file);
	}

	goto_line(atoi(line));
}


/*
 * Popup function to clear a breakpoint
 */
static void
breakpoint_clear(
	GtkWidget*							widget,
	gpointer							data
)
{
	GList*								selection;
	gI_brkpnt_window* bpw = main_window->debug_window->brkpnt_window;
	gI_debug_window* dw = main_window->debug_window;
	gchar*								brkno;
	gchar								cmdstr[50];

	if(!bpw)
	{
		gI_error_dialog(_("No breakpoint window available"));
		return;
	}

	selection = GTK_CLIST(bpw->list)->selection;
	if(!selection)
	{
		return;
	}

	gtk_clist_get_text(GTK_CLIST(bpw->list), (gint)selection->data, 0, &brkno);
	g_snprintf(cmdstr, sizeof(cmdstr), "del %s", brkno);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);

	/* update breakpoint window */
	gtk_clist_clear(GTK_CLIST(bpw->list));
	debug_send_dbg_cmd(main_window->debug_window, "info breakpoints", TRUE);
	main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
}


/*
 * Install the popup for the breakpoint window
 */
static void
breakpoint_install_popup(
	gI_brkpnt_window*					bpw
)
{
	GtkAccelGroup*						group;
	GtkItemFactoryEntry					bpw_popup_items[] =
	{
		{ "/Clear breakpoint", "Delete", breakpoint_clear, 0 },
		{ "/View source", NULL, breakpoint_view_source, 0 }
	};

	group = gtk_accel_group_new();

	bpw->popup_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Breakpoint>",
		group);
	gtk_item_factory_create_items(bpw->popup_menu,
		sizeof(bpw_popup_items) / sizeof(bpw_popup_items[0]), bpw_popup_items,
		NULL);

	gtk_accel_group_attach(group, GTK_OBJECT(bpw->window));
}


/*
 * Callback for the breakpoint list
 */
static void
breakpoint_list_select(
	GtkWidget*							widget,
	gint								row,
	gint								column,
	GdkEventButton*						bevent,
	gI_brkpnt_window*					bpw
)
{
	gchar*								file;
	gchar*								line;

	if(!bevent)
	{
		gI_error_dialog(_("Expecting an event"));
		return;
	}

	if(bevent->type == GDK_2BUTTON_PRESS)
	{
		/* get file & line from list */
		gtk_clist_get_text(GTK_CLIST(bpw->list), row, 2, &file);
		gtk_clist_get_text(GTK_CLIST(bpw->list), row, 3, &line);

		/* go to file & line */
		goto_file(file);
		goto_line(atoi(line));
	}
}


/*
 * Callback for breakpoint window buttons
 */
static gint
breakpoint_event_handler(
	GtkWidget*							widget,
	GdkEventButton*						event
)
{
	if(!event)
	{
		return FALSE;
	}

	if(event->button == 3)
	{
		gI_brkpnt_window* bpw = main_window->debug_window->brkpnt_window;

		if(!bpw)
		{
			return FALSE;
		}

		gtk_item_factory_popup(GTK_ITEM_FACTORY(bpw->popup_menu),
			(guint)event->x_root, (guint)event->y_root, event->button,
			event->time);
		return TRUE;
	}

	return FALSE;
}


/*
 * Callback function used for clearing a breakpoint
 */
static void
breakpoint_clear_ok(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								ptr;
	gI_debug_window*					dw;
	gchar								cmdstr[STRLEN];

	dw = main_window->debug_window;
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(entry));
	if(!ptr || isempty(ptr))
	{
		return;
	}

	g_snprintf(cmdstr, sizeof(cmdstr), "clear %s", ptr);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
}


/*
 * Check whether there is a breakpoint at the specified position
 */
static gboolean
breakpoint_exists(
	GideDocument*						current,
	gchar*								file,
	glong								line
)
{
	GList*								tmp;
	gI_brkpnt*							brkpnt;

	tmp = main_window->debug_window->breakpoints;

	tmp = g_list_first(tmp);

	while(tmp != NULL)
	{
		brkpnt = (gI_brkpnt*)tmp->data;
		if(brkpnt)
		{
			if(!strcmp(brkpnt->file, file) && brkpnt->line == line)
			{
				return TRUE;
			}
		}
		tmp = g_list_next(tmp);
	}

	return FALSE;
}


/* public functions */
/*
 * Create a new breakpoint window
 */
gI_brkpnt_window*
gI_brkpnt_window_new(
	void
)
{
	gI_brkpnt_window*					bpw;
	gchar* list_titles[] = { "No.", "Function", "File", "Line" };

	bpw = (gI_brkpnt_window*)g_malloc0(sizeof(gI_brkpnt_window));

	bpw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(bpw->window), _("Breakpoints"));
	gtk_widget_set_usize(bpw->window, 550, 225);
	gtk_signal_connect(GTK_OBJECT(bpw->window), "destroy",
		GTK_SIGNAL_FUNC(gI_brkpnt_window_destroy), (gpointer)bpw);

	breakpoint_install_popup(bpw);

	bpw->list = gtk_clist_new_with_titles(4, list_titles);
	gtk_clist_set_column_width(GTK_CLIST(bpw->list), 0, 30);
	gtk_clist_set_column_width(GTK_CLIST(bpw->list), 1, 200);
	gtk_clist_set_column_width(GTK_CLIST(bpw->list), 2, 200);
	gtk_clist_set_column_width(GTK_CLIST(bpw->list), 3, 50);
	gtk_clist_column_titles_passive(GTK_CLIST(bpw->list));
	gtk_container_add(GTK_CONTAINER(bpw->window), bpw->list);
	gtk_signal_connect(GTK_OBJECT(bpw->list), "select_row",
		GTK_SIGNAL_FUNC(breakpoint_list_select), (gpointer)bpw);
	gtk_signal_connect(GTK_OBJECT(bpw->list), "button_press_event",
		GTK_SIGNAL_FUNC(breakpoint_event_handler), (gpointer)bpw);
	gtk_widget_show(bpw->list);

	gtk_widget_show(bpw->window);

	return bpw;
}


/*
 * Destroy the breakpoint window
 */
void
gI_brkpnt_window_destroy(
	GtkWidget*							widget,
	gI_brkpnt_window*					bpw
)
{
	g_free(bpw);
	bpw = NULL;

	main_window->debug_window->brkpnt_window = NULL;
}


/*
 * Parsing routine for debugger output
 */
void
gI_breakpoint_process_output(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	gchar*								ptr;
	gchar*								retptr;
	gchar								brkpnt[MAXLEN];

	if(dw->breakpoints)
	{
		g_list_free(dw->breakpoints);
	}

	dw->breakpoints = NULL;

	if(dw->brkpnt_window)
	{
		gtk_clist_clear(GTK_CLIST(dw->brkpnt_window->list));
	}

	if(!strncmp(msg, "No breakpoints or watchpoints", 29))
	{
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	if(!strncmp(msg, GDB_PROMPT, strlen(GDB_PROMPT)))
	{
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	if(dw->brkpnt_window == NULL)
	{
		dw->brkpnt_window = gI_brkpnt_window_new();
	}

	/* skip first line */
	ptr = strchr(msg, '\n');
	if(!ptr)
	{
		g_warning("FATAL PARSING ERROR: (data: :%s:)\nFile: %s, Line: %d\n"
			"Please report to sk@pn.org\n", msg, __FILE__, __LINE__);
		return;
	}
	ptr++;

	while(strncmp(ptr, GDB_PROMPT, strlen(GDB_PROMPT)))
	{
		/* this may get an endless loop */
		retptr = strchr(ptr, '\n');
		if(!retptr)
		{
			return;
		}

		if(isspace(*++retptr))
		{
			/* line break */
			--retptr;
			*retptr = ' ';
			retptr = strchr(ptr, '\n');
		}
		--retptr;

		/*
		 * maybe it works, if we correct the pointer here, i don't know
		 * why it isnt correct sometimes, but it works with this work-
		 * around!
		 */
		while(*ptr != '\n')
		{
			--ptr;
		}
		ptr++;

		strncpy(brkpnt, ptr, retptr - (ptr - 1));
		brkpnt[retptr - (ptr - 1)] = '\0';

		breakpoint_add(dw, brkpnt);

		ptr = retptr + 2;
	}
	dw->wait_for = N_WAIT_NOTHING;
}

/*
 * Show the breakpoints
 */
void
gI_breakpoint_view(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	/* since we decided that there can be only one debug window */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		return;
	}

	debug_send_dbg_cmd(dw, "info breakpoints", TRUE);
	dw->wait_for = N_WAIT_BREAKPOINTS;
}


/*
 * Clear a breakpoint
 */
void
gI_breakpoint_clear(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	entry_dialog(_("Enter breakpoint: "), _("Debug - Clear breakpoint"),
		breakpoint_clear_ok);
}


/*
 * Toggle a breakpoint
 */
void
gI_breakpoint_toggle(
	GtkWidget*							widget,
	gpointer							data
)
{
	gchar								cmdstr[100];
	gchar*								file;
	glong								line;
	GideDocument*						current;
	gI_debug_window*					dw;

	current = gI_window_get_current_doc(main_window);
	if(!current)
	{
		gI_error_dialog(_("No document available"));
		return;
	}

	/* abort, if no debug window */
	if(!main_window->debug_window)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	/* abort, if no filename is set */
	if(!current->filename)
	{
		gI_error_dialog(_("No filename set"));
		return;
	}

	dw = main_window->debug_window;

	/* get filename and line number */
	file = current->filename;
	line = get_line_from_point(gI_text_get_point(current));

	if(breakpoint_exists(current, file, line))
	{
		/* remove breakpoint */
		sprintf(cmdstr, "clear %s:%ld", file, line);
	}
	else
	{
		/* set breakpoint */
		sprintf(cmdstr, "break %s:%ld", file, line);
	}

	/* send command */
	debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);

	if(dw->brkpnt_window != NULL &&
		GTK_WIDGET_VISIBLE(dw->brkpnt_window->window))
	{
		/* update breakpoint window */
		gtk_clist_clear(GTK_CLIST(dw->brkpnt_window->list));
		debug_send_dbg_cmd(main_window->debug_window, "info breakpoints", TRUE);
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
	else
	{
		/* we need this anyway .. to update the breakpoints GList */
		debug_send_dbg_cmd(main_window->debug_window, "info breakpoints", TRUE);
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
}


