/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gbf_project-tree.c
 *
 * Copyright (C) 2000  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gal/e-table/e-table.h>
#include <gal/e-table/e-tree-scrolled.h>
#include <gal/e-table/e-tree-memory-callbacks.h>
#include <gal/e-table/e-tree-table-adapter.h>

#include "images.h"
#include "gbf-project-tree.h"

enum {
	ARG_PROJECT = 1,
};

static GtkVBoxClass *parent_class = NULL;

struct _GbfProjectTreePrivate
{	
	GbfProjectClient *prj;

	GtkWidget *etree_sc;
	GtkWidget *etree;

	ETreeModel *etree_model;
	ETreePath etree_root;

	GBF_SourceList *sources;
	GHashTable *dirs;

	Bonobo_EventSource_ListenerId listener_id;

	BonoboEventSource *event_source;
};

typedef enum {
	GBF_PROJECT_TREE_FOLDER,
	GBF_PROJECT_TREE_FILE,
} GbfProjectTreeNodeType;

typedef struct 
{
	GbfProjectTreeNodeType type;
	gpointer data;
} GbfProjectTreeNodeData;

#define PROJECT_TREE_SPEC  "<ETableSpecification cursor-mode=\"line\" selection-mode=\"browse\"> \
   <ETableColumn model_col=\"0\" _title=\"Files\" expansion=\"1.0\" minimum_width=\"20\" resizable=\"true\" cell=\"tree-string\" compare=\"string\"/> \
	 <ETableState> \
		 <column source=\"0\"/> \
		 <grouping></grouping>                                         \
	 </ETableState>
 </ETableSpecification>"

static void class_init (GbfProjectTreeClass *klass);
static void init (GbfProjectTree *tree);
static void get_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void set_arg (GtkObject *object, GtkArg *arg, guint arg_id);

static GdkPixbuf *gbf_pt_icon_at (ETreeModel *model, ETreePath path, void *data);
static gint gbf_pt_col_count (ETreeModel *model, void *data);
static void *gbf_pt_dup_val (ETreeModel *model, int col, const void *val, void *data);
static void gbf_pt_free_val (ETreeModel *model, int col, void *val, void *data);
static void *gbf_pt_init_val (ETreeModel *model, int col, void *data);
static gboolean gbf_pt_val_is_empty (ETreeModel *model, int col, const void *val, void *data);
static char *gbf_pt_val_to_string (ETreeModel *model, int col, const void *val, void *data);
static void *gbf_pt_val_at (ETreeModel *model, ETreePath path, int col, void *data);
static void gbf_pt_set_val_at (ETreeModel *model, ETreePath path, int col, const void *val, void *data);
static gboolean gbf_pt_is_editable (ETreeModel *model, ETreePath path, int col, void *data);

static void gbf_pt_load_be_data (GbfProjectTree *tree);

static void event_cb (BonoboListener *listener, const char *event,
		      CORBA_any *any, CORBA_Environment *ev,
		      gpointer user_data);
static void double_click_cb (ETree *et, int row, ETreePath path, int col, GdkEvent *event, gpointer data);

GtkType
gbf_project_tree_get_type (void)
{
   static GtkType type = 0;

   if (type == 0)
     {
       static const GtkTypeInfo info =
       {
	 "GbfProjectTree",
	 sizeof (GbfProjectTree),
	 sizeof (GbfProjectTreeClass),
	 (GtkClassInitFunc) class_init,
	 (GtkObjectInitFunc) init,
	 /* reserved_1 */ NULL,
	 /* reserved_2 */ NULL,
	 (GtkClassInitFunc) NULL,
       };

       type = gtk_type_unique (gtk_vbox_get_type (), &info);
     }

   return type;
}

static void
class_init (GbfProjectTreeClass *klass)
{
	 GtkObjectClass *object_class;

	 object_class = GTK_OBJECT_CLASS (klass);

	 gtk_object_add_arg_type ("GbfProjectTree::project",
				  GTK_TYPE_OBJECT,
				  GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
				  ARG_PROJECT);

	 parent_class = gtk_type_class (gtk_vbox_get_type ());

	 object_class->get_arg = get_arg;
	 object_class->set_arg = set_arg;

	 object_class->destroy = gbf_project_tree_destroy;
}

static void
init (GbfProjectTree *tree)
{
	GbfProjectTreePrivate *priv;

	priv = g_new0 (GbfProjectTreePrivate, 1);

	tree->priv = priv;

	priv->etree_model = e_tree_memory_callbacks_new (gbf_pt_icon_at,
							 gbf_pt_col_count,
							 NULL,
							 NULL,
							 NULL,
							 NULL,
							 gbf_pt_val_at,
							 gbf_pt_set_val_at,
							 gbf_pt_is_editable,
							 gbf_pt_dup_val,
							 gbf_pt_free_val,
							 gbf_pt_init_val,
							 gbf_pt_val_is_empty,
							 gbf_pt_val_to_string,
							 tree);

	e_tree_memory_set_expanded_default (E_TREE_MEMORY (priv->etree_model), FALSE);

	priv->etree_sc = e_tree_scrolled_new (priv->etree_model, NULL, PROJECT_TREE_SPEC, NULL);
	priv->etree = GTK_WIDGET (e_tree_scrolled_get_tree (E_TREE_SCROLLED (priv->etree_sc)));

	gtk_signal_connect (GTK_OBJECT (priv->etree), "double_click",
			    GTK_SIGNAL_FUNC (double_click_cb), tree);

	priv->dirs = g_hash_table_new (g_str_hash, g_str_equal);

	gtk_box_pack_start (GTK_BOX (tree), priv->etree_sc, TRUE, TRUE, 5);
	gtk_widget_show (priv->etree_sc);

	priv->event_source = bonobo_event_source_new ();
	priv->etree_root = NULL;
}

void
get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
     GbfProjectTree *tree = GBF_PROJECT_TREE (object);

     switch (arg_id) {
     case ARG_PROJECT:
	     GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (gbf_project_tree_get_project (tree));
	     break;
     default :
	 arg->type = GTK_TYPE_INVALID;
     }
}

void
set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GbfProjectTree *tree = GBF_PROJECT_TREE (object);

	switch (arg_id) {
	case ARG_PROJECT:
		gbf_project_tree_set_project (tree, GTK_VALUE_OBJECT (*arg) ? GBF_PROJECT_CLIENT (GTK_VALUE_OBJECT (*arg)) : NULL);
		break;
	default :
		break;
	}   
}




GtkWidget *
gbf_project_tree_new      (void)
{
	return GTK_WIDGET (gtk_type_new (GBF_TYPE_PROJECT_TREE));
}

static GbfProjectTreeNodeData *
gbf_pt_node_data_new (GbfProjectTreeNodeType type, gpointer data) 
{
	GbfProjectTreeNodeData *node_data = g_new0 (GbfProjectTreeNodeData, 1);

	node_data->type = type;
	node_data->data = data;

	return node_data;
}

static ETreePath
gbf_pt_find_parent (GbfProjectTree *tree, const char *path) 
{
	GbfProjectTreePrivate *priv;
	GbfProjectTreeNodeData *node_data;
	ETreePath etp = NULL;
	ETreePath last_etp = NULL;
	char **dirs;
	char *dir, *search_path = NULL, *tmp;
	int i;
	
	priv = tree->priv;

	etp = g_hash_table_lookup (priv->dirs, path);
	if (etp)
		return etp;

	last_etp = priv->etree_root;
	dirs = g_strsplit (path, "/", 0);
	for (i = 0, dir = dirs[0]; dir != NULL; i++, dir = dirs[i]) {
		if (search_path) {
			tmp = g_strconcat (search_path, "/", dir, NULL);
			g_free (search_path);
			search_path = tmp;
		} else {
			search_path = g_strdup (dir);
		}
		
		etp = g_hash_table_lookup (priv->dirs, search_path);
		
		if (!etp) {
			node_data = gbf_pt_node_data_new (GBF_PROJECT_TREE_FOLDER, g_strdup (dir));
			etp = e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), last_etp, -1, node_data);
			g_hash_table_insert (priv->dirs, g_strdup (search_path), etp);
		}
		
		last_etp = etp;
	}

	return etp;
}

static gboolean
gbf_pt_destroy_node (gpointer key, gpointer value, gpointer user_data) 
{
	g_free (key);
	
	return TRUE;
}

static void
gbf_pt_load_be_data (GbfProjectTree *tree)
{
	GbfProjectTreePrivate *priv;
	GbfProjectClientResult res;
	GbfProjectTreeNodeData *node_data;
	CORBA_char *tmp, *tmp2;
	int i;
	
	priv = tree->priv;

	if (priv->etree_root) {
		e_tree_memory_node_remove (E_TREE_MEMORY (priv->etree_model), priv->etree_root);
		priv->etree_root = NULL;
		g_hash_table_foreach_remove (priv->dirs, gbf_pt_destroy_node, NULL);
	}

	if (!priv->prj)
		return;

	res = gbf_project_client_get_project_root (priv->prj, &tmp);	
	if (res != GBF_PROJECT_CLIENT_OK)
		return;

	tmp2 = strrchr (tmp, G_DIR_SEPARATOR);
	if (tmp2 == NULL) 
		tmp2 = tmp;
	else
		tmp2++;

	node_data = gbf_pt_node_data_new (GBF_PROJECT_TREE_FOLDER, g_strdup (tmp2));
	priv->etree_root = e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), NULL, 0, node_data);
	g_hash_table_insert (priv->dirs, g_strdup (""), priv->etree_root);
	CORBA_free (tmp);

	res = gbf_project_client_get_sources (priv->prj, &priv->sources);
	if (res != GBF_PROJECT_CLIENT_OK)
		return;
	
	for (i = 0; i < priv->sources->_length; i++) {
		GBF_Source *source = &priv->sources->_buffer[i];
		ETreePath parent = gbf_pt_find_parent (tree, source->path);
		
		node_data = gbf_pt_node_data_new (GBF_PROJECT_TREE_FILE, source);
		
		e_tree_memory_node_insert (E_TREE_MEMORY (priv->etree_model), parent, -1, node_data);
	}
}

static int 
gbf_pt_col_count (ETreeModel *model, void *data)
{
	g_print ("col_count\n");
	
	return 1;
}

static void *
gbf_pt_dup_val (ETreeModel *model, int col, const void *val, void *data)
{
	void *new_val = NULL;
	
	g_print ("dup_val\n");

	switch (col) {
	case 0:
		new_val = g_strdup ((char *)val);
		break;
	}

	return new_val;
}

static void 
gbf_pt_free_val (ETreeModel *model, int col, void *val, void *data)
{
	g_print ("free_val\n");

	switch (col) {
	case 0:
		g_free (val);
		break;
	}	
}

static void *
gbf_pt_init_val (ETreeModel *model, int col, void *data)
{
	void *init_val = NULL;
	
	g_print ("init_val\n");

	switch (col) {
	case 0:
		init_val = NULL;
		break;
	}
	
	return init_val;
}

static gboolean
gbf_pt_val_is_empty (ETreeModel *model, int col, const void *val, void *data)
{
	g_print ("is_empty\n");

	return FALSE;
}

static char *
gbf_pt_val_to_string (ETreeModel *model, int col, const void *val, void *data)
{
	g_print ("val_to_string\n");

	return g_strdup (val);
}

static GdkPixbuf *
gbf_pt_icon_at  (ETreeModel *model, ETreePath path, void *data)
{
	GbfProjectTreeNodeData *node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (model), path);
	
	switch (node_data->type) {
	case GBF_PROJECT_TREE_FILE:
		return image_for_filename (((GBF_Source*)node_data->data)->name);
	case GBF_PROJECT_TREE_FOLDER:
		return image_for_folder ();
	}

	return NULL;
}

static void *
gbf_pt_val_at  (ETreeModel *model, ETreePath path, int col, void *data)
{
	GbfProjectTreeNodeData *node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (model), path);
	GBF_Source *source = NULL;
	void *val_at = NULL;

	switch (node_data->type) {
	case GBF_PROJECT_TREE_FILE:
		source = node_data->data;
		val_at = g_strdup (source->name);
		break;
	case GBF_PROJECT_TREE_FOLDER:
		val_at = g_strdup (node_data->data);
		break;
	}
	
	return val_at;
}

static void 
gbf_pt_set_val_at (ETreeModel *model, ETreePath path, int col, const void *val, void *data)
{
	g_print ("set_val_at\n");

	switch (col) {
	case 0:
		break;
	}
}

static gboolean 
gbf_pt_is_editable (ETreeModel *model, ETreePath path, int col, void *data)
{
	gboolean editable = FALSE;
	
	switch (col) {
	case 0:
		editable = FALSE;
		break;
	}

	return editable;
}

GbfProjectClient *
gbf_project_tree_get_project (GbfProjectTree *tree)
{
	g_return_val_if_fail (tree != NULL, NULL);
	g_return_val_if_fail (GBF_IS_PROJECT_TREE (tree), NULL);

	return tree->priv->prj;
}

void 
gbf_project_tree_set_project (GbfProjectTree *tree, GbfProjectClient *prj)
{
	GbfProjectTreePrivate *priv;
	BonoboListener *listener;
	CORBA_Object source;
	CORBA_Environment ev;

	g_return_if_fail (tree != NULL);
	g_return_if_fail (GBF_IS_PROJECT_TREE (tree));
    
	priv = tree->priv;

	CORBA_exception_init (&ev);	
	if (priv->prj) {
		source = Bonobo_Unknown_queryInterface (priv->prj->objref, 
							"IDL:Bonobo/EventSource:1.0", 
							&ev);
		Bonobo_EventSource_removeListener (source, priv->listener_id, &ev);
		gtk_object_unref (GTK_OBJECT (priv->prj));
	}
    
	priv->prj = prj;

	if (prj) {
		gtk_object_ref (GTK_OBJECT (prj));
		if (GTK_OBJECT_FLOATING (GTK_OBJECT (prj)))
			gtk_object_sink (GTK_OBJECT (prj));

		listener = bonobo_listener_new (NULL, NULL);
		gtk_signal_connect (GTK_OBJECT (listener), "event_notify",
				    GTK_SIGNAL_FUNC (event_cb), tree);

		source = Bonobo_Unknown_queryInterface (prj->objref, "IDL:Bonobo/EventSource:1.0", &ev);

		if (!CORBA_Object_is_nil (source, &ev) && ev._major == CORBA_NO_EXCEPTION) {
			priv->listener_id = Bonobo_EventSource_addListener (source, BONOBO_OBJREF (listener), &ev);
		} else {
			g_error ("couldn't get event source for project");
		}

	}

	gbf_pt_load_be_data (tree);
}

BonoboEventSource *
gbf_project_tree_get_event_source (GbfProjectTree *tree)
{
	return tree->priv->event_source;
}

void
gbf_project_tree_destroy (GtkObject *obj) 
{
	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}

static void
event_cb (BonoboListener *listener, const char *event,
	  CORBA_any *any, CORBA_Environment *ev,
	  gpointer user_data)
{
	GbfProjectTree *tree = GBF_PROJECT_TREE (user_data);
	GbfProjectTreePrivate *priv;
	
	priv = tree->priv;

	g_print ("Event: %s\n", event);
	if (!strcmp (event, "project-changed"))
		gbf_pt_load_be_data (tree);
}

void
double_click_cb (ETree *et, int row, ETreePath path, int col, GdkEvent *event, gpointer data)
{
	GbfProjectTree *tree = GBF_PROJECT_TREE (data);
	GbfProjectTreeNodeData *node_data;
	GBF_Source *source = NULL;

	node_data = e_tree_memory_node_get_data (E_TREE_MEMORY (tree->priv->etree_model), path);

	if (node_data->type == GBF_PROJECT_TREE_FILE) {
		BonoboArg *arg = bonobo_arg_new (BONOBO_ARG_STRING);
		source = node_data->data;
		arg->_type = TC_GNOME_Development_Source;
		arg->_value = source;
		CORBA_any_set_release (arg, FALSE);

		bonobo_event_source_notify_listeners (tree->priv->event_source,
						      "file-selected",
						      arg, NULL);
		bonobo_arg_release (arg);
	}
}
