/* 
 * Copyright (C) 1999 Robert Wilhelm
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "gnome_chess.h"

#include "portab.h"
#include "makros.h"
#include "pro.h"
#include "position.h"
#include "cho1.h"
#include "movlist.h"

#define WHITE 0
#define BLACK 1

typedef struct {
  char piece;
  char from;
  char to;
  int  check : 1;
  int  mate  : 1;
  int  take  : 1;
  int  color  : 1;
  int  reserved: 4;
  char ascii[8];
  position *pos;
} ply;

typedef struct {
	int  number;
	ply *p[2];
} move;

static void gnomechess_movelist_select_move (GtkWidget * widget,
						int row,int column,
						GdkEventButton * event,
						GList *move_list);

MoveList* gnomechess_movelist_new() {
	MoveList *movelist;
	GtkWidget *vbox, *scrolled_window, *clist, *result;
	static char *titles[] ={ _("Number"), _("White"), _("Black")};
	int i;

	movelist = malloc(sizeof(MoveList));

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_set_usize (vbox, 200, -1 );

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),       
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_widget_show(scrolled_window);
	gtk_widget_set_usize (scrolled_window, 150, PIECE_SIZE * 8 );
	gtk_container_add(GTK_CONTAINER(vbox), scrolled_window);

	clist = gtk_clist_new_with_titles(3,titles);
	gtk_clist_set_selection_mode(GTK_CLIST (clist),GTK_SELECTION_BROWSE);
	gtk_clist_column_titles_passive(GTK_CLIST(clist));
	gtk_signal_connect (GTK_OBJECT (clist), "select_row",
		      (GtkSignalFunc) gnomechess_movelist_select_move, NULL);

	for (i = 0; i < 3; i++)
		gtk_clist_set_column_width (GTK_CLIST (clist), i, 50);

	gtk_widget_set_usize (clist, 150, PIECE_SIZE * 8 );
	gtk_container_add(GTK_CONTAINER(scrolled_window), clist);
	gtk_widget_show(clist);

	result = gtk_label_new(_(""));
	gtk_widget_show(result);
	gtk_box_pack_start(GTK_BOX(vbox), result, TRUE, TRUE, 0);

	movelist->vbox = vbox;
	movelist->clist = clist;
	movelist->result = result;

	gnomechess_movelist_clear(movelist);
	return movelist;
}

static move *gnomechess_movelist_addtolist(MoveList *movelist) {
	move *mv;
	char    *text[10] = { "", "","" };
	char text2[10];

	mv = malloc(sizeof(move));
	mv->number = (movelist->current+1)/2;
	mv->p[0] = NULL;
	mv->p[1] = NULL;
	sprintf(text2,"%d",mv->number);
	gtk_clist_append(GTK_CLIST(movelist->clist), text);
	gtk_clist_set_text (GTK_CLIST(movelist->clist), movelist->length/2, 0, text2);
	gtk_clist_set_row_data(GTK_CLIST(movelist->clist), movelist->length/2, mv);
	gtk_clist_select_row(GTK_CLIST(movelist->clist), movelist->length/2, 0);
	gtk_clist_moveto(GTK_CLIST(movelist->clist),movelist->length/2, 0, 1.0, 1.0);
	return mv;
}

void gnomechess_movelist_add(MoveList *movelist,int piece,int from, int to) {
	move *mv;
	ply *p;
	char *str;

	p = malloc(sizeof(ply));

	if (!p) abort();
	p->piece = piece;
	p->from = from;
	p->to   = to;
	p->color = WPIECE(piece) ? WHITE : BLACK;
	p->pos = gnomechess_position_copy(currPositionPtr);
	if (prefs->movelist_notation == 0) {
		str = move_to_san(currPositionPtr, p->from, p->to);
		strcpy(p->ascii, str);
		free(str);
	} else {
		piece_move_to_ascii(p->ascii,p->piece,p->from,p->to);
	}

	movelist->length++;
	movelist->current++;
	if (WPIECE(piece)) {
		mv = gnomechess_movelist_addtolist(movelist);
		mv->p[0] = p;
		gtk_clist_set_text (GTK_CLIST(movelist->clist), movelist->length/2, 1, p->ascii);
	} else {
		mv = gtk_clist_get_row_data(GTK_CLIST(movelist->clist), (movelist->length/2)-1);
		if (mv == NULL) {
			mv = gnomechess_movelist_addtolist(movelist);
			movelist->length++;
			movelist->current++;
		}
		mv->p[1] = p;
		gtk_clist_set_text (GTK_CLIST(movelist->clist), (movelist->length/2)-1, 2, p->ascii);
	}
}

void gnomechess_movelist_movetoply(MoveList *movelist,unsigned int plynum) {
	int row;
	int col;
	move *m;

	row = (plynum -1)/ 2;
	if (plynum % 2 == 0)
		col = 2;
	else 
		col = 1;

	m = gtk_clist_get_row_data(GTK_CLIST(movelist->clist), row);
	gnomechess_position_destroy(&currPositionPtr);
	if (col == 2) {
		currPositionPtr = gnomechess_position_copy(m->p[1]->pos);
	} else {
		if (plynum == 1 && m->p[0] == NULL) {
			if (plynum < movelist->current) {
				gnomechess_movelist_move_start(movelist);
			} else {
				gnomechess_movelist_movetoply(movelist, plynum + 1);
			}
			return;
		}
		currPositionPtr = gnomechess_position_copy(m->p[0]->pos);
	}
	movelist->current = plynum;

	gtk_clist_select_row(GTK_CLIST(movelist->clist), row, 0);
	gtk_clist_moveto(GTK_CLIST(movelist->clist),row, 0, 1.0, 1.0);
}

void gnomechess_movelist_move_forward(MoveList *movelist) {
	if (movelist->current == movelist->length) return;
	
	gnomechess_movelist_movetoply(movelist, movelist->current+1);
}

void gnomechess_movelist_move_back(MoveList *movelist) {
	if (movelist->current == 0) return;

	if (movelist->current == 1) {
		gnomechess_movelist_move_start(movelist);
		return;
	}
	gnomechess_movelist_movetoply(movelist, movelist->current-1);
}

void gnomechess_movelist_move_start(MoveList *movelist) {
	if (movelist->current == 0) return;

	gtk_clist_unselect_row(GTK_CLIST(movelist->clist), (movelist->current-1)/2, 0);
	movelist->current = 0;
	gnomechess_position_destroy(&currPositionPtr);
	currPositionPtr = gnomechess_position_copy(
						(position*)gtk_object_get_user_data(GTK_OBJECT(movelist->clist)));
	gtk_clist_moveto(GTK_CLIST(movelist->clist),0, 0, 1.0, 1.0);
}

void gnomechess_movelist_move_end(MoveList *movelist) {
	movelist->current = movelist->length;
	gnomechess_movelist_movetoply(movelist, movelist->current);
}

static void
gnomechess_movelist_select_move
(GtkWidget * widget,int row,int column,GdkEventButton * event,GList *move_list) {

	if (debug)
		printf("select move %d %d\n",row,column);

	if (column == 1 || column == 2){
		gnomechess_movelist_movetoply(board.movelist, row * 2 + column);
		update_board();
	}	
}

int gnomechess_movelist_currply(MoveList *movelist) {
	return movelist->current;
}

int gnomechess_movelist_maxply(MoveList *movelist) {
	return movelist->length;
}

void gnomechess_movelist_set_result(MoveList *movelist, gchar *result) {
	gtk_label_set_text(GTK_LABEL(movelist->result), result);
}

GtkWidget* gnomechess_movelist_getview(MoveList *movelist) {
	return movelist->vbox;
}

void gnomechess_movelist_clear_from(MoveList *movelist, unsigned int plynum) {
	int i, adjust = 0;
	move *mv;

	if ((plynum % 2) == 0) {
		mv = gtk_clist_get_row_data(GTK_CLIST(movelist->clist), (plynum/2)-1);
		gtk_clist_set_text (GTK_CLIST(movelist->clist), (plynum/2)-1, 2, "");
		free(mv->p[1]);
		mv->p[1] = NULL;
		plynum++;
		adjust = 1;
	}
	for (i=plynum;i<=movelist->length;i+=2) {
		gtk_clist_remove(GTK_CLIST(movelist->clist), plynum/2);
	}
	if (adjust) plynum-=2; else plynum--;
	movelist->length = plynum;
	gnomechess_movelist_movetoply(movelist, plynum);
}

void gnomechess_movelist_clear_initial(MoveList *movelist, position *pos)  {
	position *pos2;

	movelist->current = 0;
	movelist->length =  0;
	pos2  = gnomechess_position_copy(pos);
	if (movelist->clist) {
		gtk_clist_clear( GTK_CLIST(movelist->clist));
		gtk_object_set_user_data(GTK_OBJECT(movelist->clist),(gpointer)(pos2));
	}
}

void gnomechess_movelist_clear(MoveList *movelist)  {

	gnomechess_position_set_initial(currPositionPtr);
	gnomechess_movelist_clear_initial(movelist, currPositionPtr);
}

void generate_notation(char *buf, size_t len)
{
	int number, linelen;
	char str[256];
	char *str2;
	char *result;
	move *mv;
	int i;
	position *pos;
	
	mv = NULL;
	number = 1;
	buf[0] = '\0';
	linelen = 0;

	pos = gnomechess_position_new_initial();

	/* Write the seven tag roster */
	strcat(buf,"[Event \"?\"]\n");
	strcat(buf,"[Site \"?\"]\n");
	strcat(buf,"[Round \"?\"]\n");
	strcat(buf,"[Date \"????.??.??\"]\n");
	gtk_label_get(GTK_LABEL(board.whitename), &str2);
	snprintf(str,sizeof(str),"[White \"%s\"]\n",str2);
	strcat(buf,str);
	gtk_label_get(GTK_LABEL(board.blackname), &str2);
	snprintf(str,sizeof(str),"[Black \"%s\"]\n",str2);
	strcat(buf,str);
	gtk_label_get(GTK_LABEL(board.movelist->result), &result);
	if (!strcmp(result,""))
		snprintf(str,sizeof(str),"[Result \"?\"]\n\n");		
	else
		snprintf(str,sizeof(str),"[Result \"%s\"]\n\n",result);
	strcat(buf,str);

	/* Write the moves */
	for( i = 0;  i<board.movelist->length; i++)  {
		if (i%2 == 0) {
			mv = gtk_clist_get_row_data(GTK_CLIST(board.movelist->clist), i/2);
			snprintf(str,sizeof(str),"%d",number);
			strcat(buf,str);
			strcat(buf,". ");
			linelen += strlen(str) + 2;
			number++;
		}
		gnomechess_position_make_move(pos,mv->p[i%2]->from, mv->p[i%2]->to);
		str2 = move_to_san(pos, mv->p[i%2]->from, mv->p[i%2]->to);
		strcat(buf,str2);
		strcat(buf," ");
		free(str2);
		linelen += strlen(str2) + 1;
		if (linelen >= 70) {
			strcat(buf,"\n");
			linelen = 0;
		}
	}
	if (strcmp(str2,""))
		snprintf(str,sizeof(str)," %s\n",result);
	strcat(buf,str);
}

