/* 
 * Copyright (C) 1999 JP Rosevear
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "gnome_chess.h"
#include "position.h"
#include "makros.h"

position* currPositionPtr;

static void gnomechess_position_set_empty(position *pos);

position* gnomechess_position_new() {
	position *pos;
	
	pos = malloc(sizeof(position));
	gnomechess_position_set_empty(pos);
	return pos;
}

position* gnomechess_position_new_initial() {
	position *pos;

	pos = gnomechess_position_new();
	gnomechess_position_set_initial(pos);

	return pos;
}

position* gnomechess_position_copy(position *pos ) {
	position *cpPos;

	cpPos = gnomechess_position_new();
	memcpy(cpPos, pos, sizeof(position));
	memcpy(cpPos->square, pos->square, sizeof(gchar)*120);
	return cpPos;
}

static void gnomechess_position_set_empty(position *pos) {
     unsigned int a;            

     for (a = 0 ; a < 120 ; a++ )  {
          pos->square[a]= EMPTY;
     }

     for (a=0;a<10;a++)  {
          pos->square[a]     = BORDER;
          pos->square[a+10]  = BORDER;
          pos->square[a+100] = BORDER;
          pos->square[a+110] = BORDER;
          pos->square[a*10]  = BORDER;
          pos->square[a*10+9]= BORDER;
     }

     pos->wki =
     pos->wth =
     pos->wta =
     pos->ski =
     pos->sth =
     pos->sta = 0;
     pos->vf = EMPTY;

     pos->tomove = NONE;  

}

void gnomechess_position_set_initial(position *pos) {
    unsigned int a;

	/* The white pieces */
    	pos->square[A1] = WR;
	pos->square[B1] = WN;
	pos->square[C1] = WB;
	pos->square[D1] = WQ;
    	pos->square[E1] = WK;
	pos->square[F1] = WB;
	pos->square[G1] = WN;
	pos->square[H1] = WR;

	/* The black pieces */
	pos->square[A8] = BR;
	pos->square[B8] = BN;
	pos->square[C8] = BB;
	pos->square[D8] = BQ;
	pos->square[E8] = BK;
	pos->square[F8] = BB;
	pos->square[G8] = BN;
	pos->square[H8] = BR;

	/* Pawns on the 2nd and 7th ranks */
	for (a = A2; a <= H2 ;a++)  pos->square[a] = WP;
	for (a = A7; a <= H7 ;a++)  pos->square[a] = BP;

	/* The rest is blank */
	for (a = A3; a <= H3 ;a++)  pos->square[a] = EMPTY;
	for (a = A4; a <= H4 ;a++)  pos->square[a] = EMPTY;
	for (a = A5; a <= H5 ;a++)  pos->square[a] = EMPTY;
	for (a = A6; a <= H6 ;a++)  pos->square[a] = EMPTY;

	/* White to move, white king on E1 and black king on E8 */
	pos->wta = 0;
	pos->wth = 0;
	pos->wki = E1;
	pos->sta = 0;
	pos->sth = 0;
	pos->ski = E8;
	pos->vf = EMPTY;
	pos->tomove = WHITE;
}

/* 
 * Move Functions
 *
 * A set of functions to make a move in the context of the position 
 * passed in.
 *
 */
static void gnomechess_position_move_white_castle_short(position *pos) {
        pos->square[E1] = pos->square[H1] = EMPTY;
        pos->square[F1] = WR;
        pos->square[G1] = WK;
}

static void gnomechess_position_move_white_castle_long(position *pos) {
        pos->square[E1] = pos->square[A1] = EMPTY;
        pos->square[C1] = WK;
        pos->square[D1] = WR;
}

static void gnomechess_position_move_black_castle_short(position *pos) {
        pos->square[E8] = pos->square[H8] = EMPTY;
        pos->square[F8] = BR;
        pos->square[G8] = BK;
}

static void gnomechess_position_move_black_castle_long(position *pos) {
        pos->square[E8] = pos->square[A8] = EMPTY;
        pos->square[C8] = BK;
        pos->square[D8] = BR;
}

static void gnomechess_position_move_white(position *pos, int from, int to) {
	gshort piece,zfeld;

	piece = pos->square[from];

	switch (piece) {
		case WP :
			/* If we are promoting a pawn */
			if ( to & 128) {
				zfeld = (to & 7) + A8;            
				piece = ((to & 127) >> 3 ) + WP - 1;

				pos->vf = pos->square[zfeld];
				pos->square[from] = EMPTY;
				pos->square[zfeld]  = piece;
				pos->ep = EMPTY;
				return;
			}

			/* If we are capturing en passent */
			if((to-from) !=10) {
				if((to-from) !=20) {
					if(pos->square[to] == EMPTY) {
						pos->square[to-10] = EMPTY;
						pos->ep = EMPTY;
						pos->square[to] = WP;
						pos->square[from] = EMPTY;
						pos->vf = EMPTY;
						return;
					}
				}
			}

			pos->vf = pos->square[to];
			pos->square[to] = piece;
			pos->square[from]  = EMPTY;

			if ((to-from) == 20) 	
				pos->ep = to;
			else  
				pos->ep = EMPTY;
			return;

		case WK :
			pos->ep=EMPTY;
			pos->wki = to;
			pos->wta += 1;
			pos->wth += 1;

			/*  If we are not castling */
			if (from != E1 || abs(to-from) != 2) {
				pos->vf = pos->vf = pos->square[to];
				pos->square[to] = piece;
				pos->square[from]  = EMPTY;
				return;
			}

			/*  If we are castling */
			switch  (to) {
				case G1 :
					gnomechess_position_move_white_castle_short(pos);
					break;  
				case C1 :
					gnomechess_position_move_white_castle_long(pos);
					break;  
				default :       
					abort();
			}
			return;

		default :
			pos->ep=EMPTY;
			pos->vf = pos->square[to];
			pos->square[to] = piece;
			pos->square[from]  = EMPTY;

			/* Indicate if a rook has moved */
			if ( piece == WR && from == A1) pos->wta += 1;
			if ( piece == WR && from == H1) pos->wth += 1;
			return;
	}
}

static void gnomechess_position_move_black(position *pos, int from, int to) {
	gshort piece,zfeld;

	piece = pos->square[from];

	switch (piece) {
		case BP :
			/* If we are promoting a pawn */
			if ( to & 128) {
				zfeld = (to & 7) + A1;
				piece = ((to & 127) >> 3) + BP - 1;
				pos->vf = pos->square[zfeld];
				pos->square[from]    = EMPTY;
				pos->square[zfeld]  = piece;
				pos->ep = EMPTY;
				return;
			}

			/* If we are capturing en passent */
			if((from-to) !=10) {
				if((from-to) !=20) {
					if(pos->square[to] == EMPTY) {
						pos->square[to+10] = EMPTY;
						pos->ep = EMPTY;
						pos->square[to] = BP;
						pos->square[from] = EMPTY;
						pos->vf = EMPTY;
						return;
					}
				}
			}

			pos->vf = pos->square[to];
			pos->square[to] = piece;
			pos->square[from]  = EMPTY;

			if ((from-to) == 20) 	
				pos->ep = to;
			else                   
				pos->ep = EMPTY;	
			return;

		case BK :
			pos->ep=EMPTY;
			pos->ski = to;
			pos->sta += 1;
			pos->sth += 1;

			/*  If we are not castling */
			if (from != E8 || abs(to-from) != 2) {
				pos->vf = pos->square[to];
				pos->square[to] = piece;
				pos->square[from]  = EMPTY;
				return;
			}

			/*  If we are castling */
			switch  (to)    {
				case G8 :       
					gnomechess_position_move_black_castle_short(pos);
					break; 
				case C8 :
					gnomechess_position_move_black_castle_long(pos);
					break; 
				default :
					abort();
			}
			return;

		default :
			pos->ep=EMPTY;
			pos->vf = pos->square[to];
			pos->square[to] = piece;
			pos->square[from]  = EMPTY;

			/* Indicate if a rook has moved */
			if ( piece == BR && from == A8) pos->sta += 1;
			if ( piece == BR && from == H8) pos->sth += 1;
			return;
	}
}

void gnomechess_position_move(position *pos, int from, int to) {

	switch (currPositionPtr->tomove) {
		case WHITE:
			gnomechess_position_move_white(pos, from,to);
			pos->tomove = BLACK;
			break;
		case BLACK:
			gnomechess_position_move_black(pos, from,to);
			pos->tomove = WHITE;
			break;
        	default :
			abort();
	}
}

/* 
 * Move Reverse Functions
 *
 * A set of functions to reverse a previous move in the context of the position 
 * passed in.
 *
 */
static void gnomechess_position_move_reverse_castling_white_short(position *pos) {
        pos->square[E1]=WK;
        pos->square[F1]=pos->square[G1]=EMPTY;
        pos->square[H1]=WR;
}

static void gnomechess_position_move_reverse_castling_white_long(position *pos) {
        pos->square[A1]=WR;
        pos->square[C1]=pos->square[D1]=EMPTY;
        pos->square[E1]=WK;
}

static void gnomechess_position_move_reverse_castling_black_short(position *pos) {
        pos->square[E8] = BK;
        pos->square[F8] = pos->square[G8] = EMPTY;
        pos->square[H8] = BR;
}

static void gnomechess_position_move_reverse_castling_black_long(position *pos){
        pos->square[A8]=BR;
        pos->square[D8]=pos->square[C8]=EMPTY;
        pos->square[E8]=BK;
}

static void gnomechess_position_move_reverse_promotion_white(position *pos, int from, int to) {
	pos->square[from] = WP;
	pos->square[(to & 7) + A8] = pos->vf;
}

static void gnomechess_position_move_reverse_promotion_black(position *pos, int from, int to) {
	pos->square[from] = BP;
	pos->square[(to & 7) + A1] = pos->vf;
}

void gnomechess_position_move_reverse_white(position *pos, int from, int to) {
   int fi;

   pos->tomove = WHITE;     /*      change tomove      */

   if (to & 128) {        /*    Promotion     */
      gnomechess_position_move_reverse_promotion_white(pos,from,to);
      return;
   }

        fi = pos->square[to];

        /*    look for castling   */
        if (fi == WK)
		{
        	pos->wki = from;

        	if (from != E1)  {  /*    no castling   */
               		pos->square[from]  = fi;
               		pos->square[to] = currPositionPtr->vf;
               		return;
        	}
        	if (abs(from-to) != 2)  {  /*  no castling  */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->vf;
               		return;
        	}

        	if (to == G1) {
			gnomechess_position_move_reverse_castling_white_short(pos);
			return;
		} else if (to == C1) {
			gnomechess_position_move_reverse_castling_white_long(pos);
			return;
		}
        	abort();
	}

        if (fi == WP)
	{
        	if ((to-from) !=10) {
          		if ((to-from) !=20) {
            			if (pos->vf == EMPTY) {
                			pos->square[to-10] = BP;
                			pos->square[to] = EMPTY;
                			pos->square[from] = WP;
          			        return;
        			}
			}
		}
        	pos->square[from] = fi;
        	pos->square[to]= pos->vf;
		return;
	}
        pos->square[from]  = fi;
        pos->square[to] = pos->vf;
}

void gnomechess_position_move_reverse_black(position *pos, int from, int to) {
	int fi;

	pos->tomove = BLACK;      /* change tomove */

	if (to & 128) {        /*    Promotion  */
		gnomechess_position_move_reverse_promotion_black(pos,from,to);
		return;
	}

        fi = pos->square[to];

        /* look for castling */
        if (fi == BK)
		{
        	pos->ski = from;

        	if (from != E8)  {  /*    no castling   */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->vf;
               		return;
        	}
        	if (abs(from-to) != 2)  {  /*    no castling  */
               		pos->square[from]  = fi;
               		pos->square[to] = pos->vf;
               		return;
        	}

        	if (to == G8) {
			gnomechess_position_move_reverse_castling_black_short(pos);
			return;
		} else if (to == C8) {
			gnomechess_position_move_reverse_castling_black_long(pos);
			return;
		}
        	abort();
	}
        if (fi == BP)
	{
        	if ((from-to) !=10) {
          		if ((from-to) !=20) {
            			if (currPositionPtr->vf == EMPTY) {
                			pos->square[to+10] = WP;
                			pos->square[to] = EMPTY;
                			pos->square[from] = BP;
          			        return;
        			}
			}
		}
        	pos->square[from] = fi;
        	pos->square[to]= pos->vf;
		return;
	}
	pos->square[from]  = fi;
        pos->square[to] = pos->vf;
}

void gnomechess_position_move_reverse(position *pos, int from, int to) {

	switch (currPositionPtr->tomove) {
		case WHITE:
			gnomechess_position_move_reverse_black(pos, from,to);
			break;
		case BLACK:
			gnomechess_position_move_reverse_white(pos, from,to);
			break;
        	default :
			abort();
	}
}

void gnomechess_position_destroy(position **pos) {
	if (*pos)
		free(*pos);
	*pos = NULL;
}
