/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* timer.c
 *
 * Copyright (C) 2001  JP Rosevear
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: JP Rosevear
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include "timer.h"

struct _TimerPrivate {
	gint sec;
	guint handlerid;
};

enum {
	EXPIRED_SIGNAL,
	LAST_SIGNAL
};
static gint signals [LAST_SIGNAL] = { 0 };

static GtkLabelClass *parent_class = NULL;

static void class_init (TimerClass *klass);
static void init (Timer *timer);
static void destroy (GtkObject *obj);

static gboolean timer_cb (gpointer data);

GtkType
timer_get_type (void)
{
  static GtkType type = 0;

  if (type == 0)
    {
      static const GtkTypeInfo info =
      {
        "Timer",
        sizeof (Timer),
        sizeof (TimerClass),
        (GtkClassInitFunc) class_init,
        (GtkObjectInitFunc) init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      type = gtk_type_unique (gtk_label_get_type (), &info);
    }

  return type;
}

static void
class_init (TimerClass *klass)
{
	GtkObjectClass *object_class;

	object_class = GTK_OBJECT_CLASS (klass);

	parent_class = gtk_type_class (gtk_label_get_type ());

	signals [EXPIRED_SIGNAL] =
		gtk_signal_new ("expired", GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (TimerClass, expired),
				gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);

	object_class->destroy = destroy;
}


static void
init (Timer *timer)
{
	TimerPrivate *priv;

	priv = g_new0 (TimerPrivate, 1);

	timer->priv = priv;

	priv->sec = 0;
	priv->handlerid = -1;

	gtk_label_set_text (GTK_LABEL (timer), "0:00");
}



GtkWidget *
timer_new (gint secs) 
{
	GtkWidget *timer;

	timer = GTK_WIDGET(gtk_type_new(timer_get_type()));
	timer_set(TIMER(timer), secs);	

	return timer;
}

gint 
timer_get (Timer *timer) 
{
	TimerPrivate *priv;
	
	g_return_val_if_fail (timer != NULL, 0);
	g_return_val_if_fail (IS_TIMER (timer), 0);

	priv = timer->priv;
	
	return priv->sec;
}

void 
timer_set (Timer *timer, gint secs) 
{
	TimerPrivate *priv;
	char t[10];
	char s;

	g_return_if_fail (timer != NULL);
	g_return_if_fail (IS_TIMER (timer));

	priv = timer->priv;

	priv->sec = secs;
	if (priv->sec < 0)
		s = '-';
	else
		s = ' ';		
	sprintf (t, "%c%d:%.2d", s, abs (priv->sec / 60), abs (priv->sec % 60));
	gtk_label_set_text (GTK_LABEL (timer), t);
}

void 
timer_start (Timer *timer) 
{
	TimerPrivate *priv;
	
	g_return_if_fail (timer != NULL);
	g_return_if_fail (IS_TIMER (timer));

	priv = timer->priv;

	if (priv->handlerid != -1)
		return;

	priv->handlerid = gtk_timeout_add (1000, timer_cb, timer);
}

void 
timer_stop (Timer *timer) 
{
	TimerPrivate *priv;
	
	g_return_if_fail (timer != NULL);
	g_return_if_fail (IS_TIMER (timer));

	priv = timer->priv;

	if (priv->handlerid == -1)
		return;

	gtk_timeout_remove (priv->handlerid);
	priv->handlerid = -1;
}

static gboolean
timer_cb (gpointer data) 
{
	Timer *timer = TIMER (data);
	TimerPrivate *priv;
	
	priv = timer->priv;

	priv->sec--;
	timer_set (timer, priv->sec);

	if (priv->sec <= 0)
		gtk_signal_emit (GTK_OBJECT (timer), signals [EXPIRED_SIGNAL]);

	return TRUE;
}

static void
destroy (GtkObject *obj)
{
	Timer *timer = TIMER (obj);

	timer_stop (timer);

	GTK_OBJECT_CLASS (parent_class)->destroy (obj);
}
