
#include <src/Manager.h>
#include <gee/hashmap.h>
#include <gee/hashset.h>
#include <gee/map.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>
#include <gee/collection.h>
#include <gio/gio.h>
#include "src/Constants.h"
#include "src/Device.h"
#include "src/Scanner.h"
#include "src/Terrestrial/TerrestrialScanner.h"
#include "src/Satellite/SatelliteScanner.h"
#include "src/Cable/CableScanner.h"
#include "src/ChangeType.h"
#include "src/GConfStore.h"
#include "src/ChannelList.h"
#include "src/Channel.h"
#include "src/Schedule.h"
#include <dbus/dbus-glib.h>




struct _DVBManagerPrivate {
	GeeHashMap* scanners;
	GeeHashMap* recorders;
	GeeHashMap* channellists;
	GeeHashMap* devices;
	GeeHashMap* epgscanners;
	GeeHashSet* schedules;
	guint device_group_counter;
};

#define DVB_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), DVB_TYPE_MANAGER, DVBManagerPrivate))
enum  {
	DVB_MANAGER_DUMMY_PROPERTY
};
static DVBManager* dvb_manager_instance;
static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self);
static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir);
static DBusGConnection* dvb_manager_get_dbus_connection (void);
static DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id);
static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device);
static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner);
static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend);
static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device);
static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer dvb_manager_parent_class = NULL;
static gboolean _dbus_dvb_manager_GetScannerForDevice (DVBManager* self, guint adapter, guint frontend, char*** result, GError** error);
static gboolean _dbus_dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, GArray** dbus_result, GError** error);
static gboolean _dbus_dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, guint group_id, char** result, GError** error);
static gboolean _dbus_dvb_manager_GetRecorder (DVBManager* self, guint group_id, char** result, GError** error);
static gboolean _dbus_dvb_manager_AddDeviceToNewGroup (DVBManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, gboolean* result, GError** error);
static gboolean _dbus_dvb_manager_AddDeviceToExistingGroup (DVBManager* self, guint adapter, guint frontend, guint group_id, gboolean* result, GError** error);
static gboolean _dbus_dvb_manager_RemoveDeviceFromGroup (DVBManager* self, guint adapter, guint frontend, guint group_id, gboolean* result, GError** error);
static gboolean _dbus_dvb_manager_GetChannelList (DVBManager* self, guint group_id, char** result, GError** error);
static gboolean _dbus_dvb_manager_GetDeviceGroupMembers (DVBManager* self, guint group_id, char*** result, GError** error);
static gboolean _dbus_dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, guint adapter, guint frontend, char** result, GError** error);
static gboolean _dbus_dvb_manager_GetSchedule (DVBManager* self, guint group_id, guint channel_sid, char** result, GError** error);
static void dvb_manager_finalize (GObject* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);


static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_BOOLEAN__POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);
static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data);

DVBManager* dvb_manager_get_instance (void) {
	/* TODO make thread-safe*/
	if (dvb_manager_instance == NULL) {
		DVBManager* _tmp0;
		_tmp0 = NULL;
		dvb_manager_instance = (_tmp0 = dvb_manager_new (), (dvb_manager_instance == NULL ? NULL : (dvb_manager_instance = (g_object_unref (dvb_manager_instance), NULL))), _tmp0);
	}
	return dvb_manager_instance;
}


static void _dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed (DVBScanner* _sender, gpointer self) {
	dvb_manager_on_scanner_destroyed (self, _sender);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @returns: Object path and interface to the scanner service
         *
         * Get the object path of the channel scanner for this device.
         */
char** dvb_manager_GetScannerForDevice (DVBManager* self, guint adapter, guint frontend, int* result_length1) {
	char* path;
	DVBDevice* device;
	DVBDevice* reg_dev;
	char* dbusiface;
	DVBAdapterType _tmp8;
	char** _tmp18;
	const char* _tmp17;
	const char* _tmp16;
	char** _tmp19;
	char** _tmp20;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	device = NULL;
	reg_dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (reg_dev == NULL) {
		DVBDevice* _tmp0;
		/* Create new device*/
		_tmp0 = NULL;
		device = (_tmp0 = dvb_device_new (adapter, frontend, TRUE), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp0);
	} else {
		DVBDeviceGroup* _tmp1;
		DVBEPGScanner* _tmp2;
		DVBEPGScanner* epgscanner;
		DVBDevice* _tmp4;
		DVBDevice* _tmp3;
		/* Stop epgscanner for device if there's any*/
		_tmp1 = NULL;
		_tmp2 = NULL;
		epgscanner = (_tmp2 = dvb_manager_get_epg_scanner (self, (_tmp1 = dvb_manager_get_device_group_of_device (self, reg_dev))), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), _tmp2);
		if (epgscanner != NULL) {
			dvb_epg_scanner_stop (epgscanner);
		}
		/* Assign existing device*/
		_tmp4 = NULL;
		_tmp3 = NULL;
		device = (_tmp4 = (_tmp3 = reg_dev, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp4);
		(epgscanner == NULL ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL)));
	}
	dbusiface = NULL;
	_tmp8 = dvb_device_get_Type (device);
	if (_tmp8 == DVB_ADAPTER_TYPE_DVB_T)
	do {
		char* _tmp5;
		_tmp5 = NULL;
		dbusiface = (_tmp5 = g_strdup ("org.gnome.DVB.Scanner.Terrestrial"), (dbusiface = (g_free (dbusiface), NULL)), _tmp5);
		break;
	} while (0); else if (_tmp8 == DVB_ADAPTER_TYPE_DVB_S)
	do {
		char* _tmp6;
		_tmp6 = NULL;
		dbusiface = (_tmp6 = g_strdup ("org.gnome.DVB.Scanner.Satellite"), (dbusiface = (g_free (dbusiface), NULL)), _tmp6);
		break;
	} while (0); else if (_tmp8 == DVB_ADAPTER_TYPE_DVB_C)
	do {
		char* _tmp7;
		_tmp7 = NULL;
		dbusiface = (_tmp7 = g_strdup ("org.gnome.DVB.Scanner.Cable"), (dbusiface = (g_free (dbusiface), NULL)), _tmp7);
		break;
	} while (0);
	if (!gee_map_contains (GEE_MAP (self->priv->scanners), path)) {
		DVBScanner* scanner;
		DVBAdapterType _tmp12;
		DBusGConnection* conn;
		scanner = NULL;
		_tmp12 = dvb_device_get_Type (device);
		if (_tmp12 == DVB_ADAPTER_TYPE_DVB_T)
		do {
			DVBScanner* _tmp9;
			_tmp9 = NULL;
			scanner = (_tmp9 = DVB_SCANNER (dvb_terrestrial_scanner_new (device)), (scanner == NULL ? NULL : (scanner = (g_object_unref (scanner), NULL))), _tmp9);
			break;
		} while (0); else if (_tmp12 == DVB_ADAPTER_TYPE_DVB_S)
		do {
			DVBScanner* _tmp10;
			_tmp10 = NULL;
			scanner = (_tmp10 = DVB_SCANNER (dvb_satellite_scanner_new (device)), (scanner == NULL ? NULL : (scanner = (g_object_unref (scanner), NULL))), _tmp10);
			break;
		} while (0); else if (_tmp12 == DVB_ADAPTER_TYPE_DVB_C)
		do {
			DVBScanner* _tmp11;
			_tmp11 = NULL;
			scanner = (_tmp11 = DVB_SCANNER (dvb_cable_scanner_new (device)), (scanner == NULL ? NULL : (scanner = (g_object_unref (scanner), NULL))), _tmp11);
			break;
		} while (0);
		g_signal_connect_object (scanner, "destroyed", ((GCallback) (_dvb_manager_on_scanner_destroyed_dvb_scanner_destroyed)), self, 0);
		gee_map_set (GEE_MAP (self->priv->scanners), path, scanner);
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char** _tmp13;
			char** _tmp14;
			char** _tmp15;
			_tmp13 = NULL;
			_tmp14 = NULL;
			_tmp15 = NULL;
			return (_tmp15 = (_tmp14 = (_tmp13 = g_new0 (char*, 0 + 1), _tmp13), (*result_length1) = 0, _tmp14), (scanner == NULL ? NULL : (scanner = (g_object_unref (scanner), NULL))), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), (path = (g_free (path), NULL)), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (reg_dev == NULL ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL))), (dbusiface = (g_free (dbusiface), NULL)), _tmp15);
		}
		dbus_g_connection_register_g_object (conn, path, G_OBJECT (scanner));
		g_debug ("Manager.vala:129: Created new Scanner D-Bus service for adapter %u, frontend %u (%s)", adapter, frontend, dbusiface);
		(scanner == NULL ? NULL : (scanner = (g_object_unref (scanner), NULL)));
		(conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)));
	}
	_tmp18 = NULL;
	_tmp17 = NULL;
	_tmp16 = NULL;
	_tmp19 = NULL;
	_tmp20 = NULL;
	return (_tmp20 = (_tmp19 = (_tmp18 = g_new0 (char*, 2 + 1), _tmp18[0] = (_tmp16 = path, (_tmp16 == NULL ? NULL : g_strdup (_tmp16))), _tmp18[1] = (_tmp17 = dbusiface, (_tmp17 == NULL ? NULL : g_strdup (_tmp17))), _tmp18), (*result_length1) = 2, _tmp19), (path = (g_free (path), NULL)), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (reg_dev == NULL ? NULL : (reg_dev = (g_object_unref (reg_dev), NULL))), (dbusiface = (g_free (dbusiface), NULL)), _tmp20);
}


/**
         * @returns: Device groups' ID
         */
guint* dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, int* result_length1) {
	guint* _tmp1;
	gint devs_length1;
	gint _tmp0;
	guint* devs;
	gint i;
	guint* _tmp2;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	_tmp1 = NULL;
	devs = (_tmp1 = g_new0 (guint, (_tmp0 = gee_map_get_size (GEE_MAP (self->priv->devices)))), devs_length1 = _tmp0, _tmp1);
	i = 0;
	{
		GeeSet* key_collection;
		GeeIterator* key_it;
		key_collection = gee_map_get_keys (GEE_MAP (self->priv->devices));
		key_it = gee_iterable_iterator (GEE_ITERABLE (key_collection));
		while (gee_iterator_next (key_it)) {
			guint key;
			key = GPOINTER_TO_UINT (gee_iterator_get (key_it));
			{
				devs[i] = key;
				i++;
			}
		}
		(key_it == NULL ? NULL : (key_it = (g_object_unref (key_it), NULL)));
		(key_collection == NULL ? NULL : (key_collection = (g_object_unref (key_collection), NULL)));
	}
	_tmp2 = NULL;
	return (_tmp2 = devs, (*result_length1) = devs_length1, _tmp2);
}


/**
         * @group_id: ID of device group
         * @returns: Name of adapter type the group holds
         * or an empty string when group with given id doesn't exist.
         */
char* dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, guint group_id) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char* type_str;
		DVBAdapterType _tmp4;
		char* _tmp5;
		devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
		type_str = NULL;
		_tmp4 = dvb_device_group_get_Type (devgroup);
		if (_tmp4 == DVB_ADAPTER_TYPE_DVB_T)
		do {
			char* _tmp0;
			_tmp0 = NULL;
			type_str = (_tmp0 = g_strdup ("DVB-T"), (type_str = (g_free (type_str), NULL)), _tmp0);
			break;
		} while (0); else if (_tmp4 == DVB_ADAPTER_TYPE_DVB_S)
		do {
			char* _tmp1;
			_tmp1 = NULL;
			type_str = (_tmp1 = g_strdup ("DVB-S"), (type_str = (g_free (type_str), NULL)), _tmp1);
			break;
		} while (0); else if (_tmp4 == DVB_ADAPTER_TYPE_DVB_C)
		do {
			char* _tmp2;
			_tmp2 = NULL;
			type_str = (_tmp2 = g_strdup ("DVB-C"), (type_str = (g_free (type_str), NULL)), _tmp2);
			break;
		} while (0); else
		do {
			char* _tmp3;
			_tmp3 = NULL;
			type_str = (_tmp3 = g_strdup (""), (type_str = (g_free (type_str), NULL)), _tmp3);
			break;
		} while (0);
		_tmp5 = NULL;
		return (_tmp5 = type_str, (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), _tmp5);
	}
	return g_strdup ("");
}


/**
         * @group_id: ID of device group
         * @returns: Object path of the device's recorder
         * 
         * Returns the object path to the device's recorder.
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
char* dvb_manager_GetRecorder (DVBManager* self, guint group_id) {
	char* path;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_RECORDER_PATH, group_id);
	if (!gee_map_contains (GEE_MAP (self->priv->recorders), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* device;
		DVBRecorder* recorder;
		DBusGConnection* conn;
		g_debug ("Manager.vala:181: Creating new Recorder for group %u", group_id);
		device = dvb_manager_get_device_group_if_exists (self, group_id);
		if (device == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (path = (g_free (path), NULL)), _tmp0);
		}
		recorder = dvb_recorder_new (device);
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp1;
			_tmp1 = NULL;
			return (_tmp1 = g_strdup (""), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (recorder == NULL ? NULL : (recorder = (g_object_unref (recorder), NULL))), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), (path = (g_free (path), NULL)), _tmp1);
		}
		dbus_g_connection_register_g_object (conn, path, G_OBJECT (recorder));
		gee_map_set (GEE_MAP (self->priv->recorders), GUINT_TO_POINTER (group_id), recorder);
		(device == NULL ? NULL : (device = (g_object_unref (device), NULL)));
		(recorder == NULL ? NULL : (recorder = (g_object_unref (recorder), NULL)));
		(conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)));
	}
	return path;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @channels_conf: Path to channels.conf for this device
         * @recordings_dir: Path where the recordings should be stored
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new DeviceGroup and new DVB device whereas the
         * DVB device is the reference device of this group (i.e.
         * all other devices of this group will inherit the settings
         * of the reference device).
         */
gboolean dvb_manager_AddDeviceToNewGroup (DVBManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir) {
	DVBDevice* device;
	DVBDeviceGroup* _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (DVB_IS_MANAGER (self), FALSE);
	g_return_val_if_fail (channels_conf != NULL, FALSE);
	g_return_val_if_fail (recordings_dir != NULL, FALSE);
	device = dvb_manager_create_device (adapter, frontend, channels_conf, recordings_dir);
	if (device == NULL) {
		gboolean _tmp0;
		return (_tmp0 = FALSE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp0);
	}
	/* Check if device is already assigned to other group*/
	if (dvb_manager_device_is_in_any_group (self, device)) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp1);
	}
	self->priv->device_group_counter++;
	_tmp2 = NULL;
	dvb_manager_add_device_group (self, (_tmp2 = dvb_device_group_new (self->priv->device_group_counter, device)));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	g_signal_emit_by_name (G_OBJECT (self), "changed", self->priv->device_group_counter, ((guint) (DVB_CHANGE_TYPE_ADDED)));
	return (_tmp3 = TRUE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp3);
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when the device has been registered successfully
         *
         * Creates a new device and adds it to the specified DeviceGroup.
         * The new device will inherit all settings from the group's
         * reference device.
         */
gboolean dvb_manager_AddDeviceToExistingGroup (DVBManager* self, guint adapter, guint frontend, guint group_id) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), FALSE);
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		DVBDevice* device;
		DVBDeviceGroup* devgroup;
		/* When the device is already registered we
		 might see some errors if the device is
		 currently in use*/
		device = dvb_device_new (adapter, frontend, TRUE);
		if (device == NULL) {
			gboolean _tmp0;
			return (_tmp0 = FALSE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp0);
		}
		if (dvb_manager_device_is_in_any_group (self, device)) {
			gboolean _tmp1;
			g_debug ("Device with adapter %u, frontend %u is" "already part of a group", adapter, frontend);
			return (_tmp1 = FALSE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), _tmp1);
		}
		g_debug ("Manager.vala:262: Adding device with adapter %u, frontend %u to group %u", adapter, frontend, group_id);
		devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
		if (dvb_device_group_add (devgroup, device)) {
			DVBGConfStore* _tmp2;
			gboolean _tmp3;
			_tmp2 = NULL;
			dvb_gconf_store_add_device_to_group ((_tmp2 = dvb_gconf_store_get_instance ()), device, devgroup);
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
			g_signal_emit_by_name (G_OBJECT (self), "group-changed", group_id, adapter, frontend, ((guint) (DVB_CHANGE_TYPE_ADDED)));
			return (_tmp3 = TRUE, (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), _tmp3);
		}
		(device == NULL ? NULL : (device = (g_object_unref (device), NULL)));
		(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
	}
	return FALSE;
}


/**
         * @adapter: Number of the device's adapter
         * @frontend: Number of the device's frontend
         * @group_id: ID of device group
         * @returns: TRUE when device has been removed successfully
         *
         * Removes the device from the specified group. If the group contains
         * no devices after the removal it's removed as well.
         */
gboolean dvb_manager_RemoveDeviceFromGroup (DVBManager* self, guint adapter, guint frontend, guint group_id) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), FALSE);
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		DVBDevice* dev;
		devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
		dev = dvb_device_new (adapter, frontend, FALSE);
		if (dvb_device_group_contains (devgroup, dev)) {
			if (dvb_device_group_remove (devgroup, dev)) {
				DVBEPGScanner* epgscanner;
				DVBGConfStore* _tmp0;
				gboolean _tmp2;
				/* Stop epgscanner, because it might use the
				 device we want to unregister*/
				epgscanner = dvb_manager_get_epg_scanner (self, devgroup);
				if (epgscanner != NULL) {
					dvb_epg_scanner_stop (epgscanner);
				}
				_tmp0 = NULL;
				dvb_gconf_store_remove_device_from_group ((_tmp0 = dvb_gconf_store_get_instance ()), dev, devgroup);
				(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
				g_signal_emit_by_name (G_OBJECT (self), "group-changed", group_id, adapter, frontend, ((guint) (DVB_CHANGE_TYPE_DELETED)));
				/* Group has no devices anymore, delete it*/
				if (dvb_device_group_get_size (devgroup) == 0) {
					if (gee_map_remove (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
						DVBGConfStore* _tmp1;
						/* Remove EPG scanner, too*/
						if (epgscanner != NULL) {
							gee_map_remove (GEE_MAP (self->priv->epgscanners), GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)));
						}
						_tmp1 = NULL;
						dvb_gconf_store_remove_device_group ((_tmp1 = dvb_gconf_store_get_instance ()), devgroup);
						(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
						g_signal_emit_by_name (G_OBJECT (self), "changed", group_id, ((guint) (DVB_CHANGE_TYPE_DELETED)));
					}
				} else {
					/* We still have a device, start EPG scanner again*/
					if (epgscanner != NULL) {
						dvb_epg_scanner_start (epgscanner);
					}
				}
				return (_tmp2 = TRUE, (epgscanner == NULL ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL))), (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), (dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL))), _tmp2);
			}
		}
		(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
		(dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL)));
	}
	return FALSE;
}


/**
         * @group_id: ID of device group
         * @returns: Object path to the ChannelList service for this device
         *
         * The device group must be registered with AddDeviceToNewGroup () first.
         */
char* dvb_manager_GetChannelList (DVBManager* self, guint group_id) {
	char* path;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_CHANNEL_LIST_PATH, group_id);
	if (!gee_map_contains (GEE_MAP (self->priv->channellists), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* device;
		DVBChannelList* _tmp1;
		DVBChannelList* channels;
		DBusGConnection* conn;
		g_debug ("Manager.vala:341: Creating new ChannelList D-Bus service for group %u", group_id);
		device = dvb_manager_get_device_group_if_exists (self, group_id);
		if (device == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			return (_tmp0 = g_strdup (""), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (path = (g_free (path), NULL)), _tmp0);
		}
		_tmp1 = NULL;
		channels = (_tmp1 = dvb_device_group_get_Channels (device), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		conn = dvb_manager_get_dbus_connection ();
		if (conn == NULL) {
			char* _tmp2;
			_tmp2 = NULL;
			return (_tmp2 = g_strdup (""), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (channels == NULL ? NULL : (channels = (g_object_unref (channels), NULL))), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), (path = (g_free (path), NULL)), _tmp2);
		}
		dbus_g_connection_register_g_object (conn, path, G_OBJECT (channels));
		gee_map_set (GEE_MAP (self->priv->channellists), GUINT_TO_POINTER (group_id), channels);
		(device == NULL ? NULL : (device = (g_object_unref (device), NULL)));
		(channels == NULL ? NULL : (channels = (g_object_unref (channels), NULL)));
		(conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)));
	}
	return path;
}


/**
         * @group_id: ID of device group
         * @returns: List of paths to the devices that are part of
         * the specified group (e.g. /dev/dvb/adapter0/frontend0)
         */
char** dvb_manager_GetDeviceGroupMembers (DVBManager* self, guint group_id, int* result_length1) {
	gint groupdevs_length1;
	char** groupdevs;
	char** _tmp4;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	groupdevs = (groupdevs_length1 = 0, NULL);
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		char** _tmp1;
		gint _tmp0;
		gint i;
		devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
		_tmp1 = NULL;
		groupdevs = (_tmp1 = g_new0 (char*, (_tmp0 = dvb_device_group_get_size (devgroup)) + 1), (groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, ((GDestroyNotify) (g_free))), NULL)), groupdevs_length1 = _tmp0, _tmp1);
		i = 0;
		{
			DVBDeviceGroup* dev_collection;
			GeeIterator* dev_it;
			dev_collection = devgroup;
			dev_it = gee_iterable_iterator (GEE_ITERABLE (dev_collection));
			while (gee_iterator_next (dev_it)) {
				DVBDevice* dev;
				dev = ((DVBDevice*) (gee_iterator_get (dev_it)));
				{
					char* _tmp2;
					_tmp2 = NULL;
					groupdevs[i] = (_tmp2 = g_strdup_printf (DVB_CONSTANTS_DVB_DEVICE_PATH, dvb_device_get_Adapter (dev), dvb_device_get_Frontend (dev)), (groupdevs[i] = (g_free (groupdevs[i]), NULL)), _tmp2);
					i++;
					(dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL)));
				}
			}
			(dev_it == NULL ? NULL : (dev_it = (g_object_unref (dev_it), NULL)));
		}
		(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
	} else {
		char** _tmp3;
		_tmp3 = NULL;
		groupdevs = (_tmp3 = g_new0 (char*, 0 + 1), (groupdevs = (_vala_array_free (groupdevs, groupdevs_length1, ((GDestroyNotify) (g_free))), NULL)), groupdevs_length1 = 0, _tmp3);
	}
	_tmp4 = NULL;
	return (_tmp4 = groupdevs, (*result_length1) = groupdevs_length1, _tmp4);
}


/**
         * @adapter: Adapter of device
         * @frontend: Frontend of device
         * @returns: The name of the device or "Unknown"
         *
         * The device must be part of group, otherwise "Unknown"
         * is returned.
         */
char* dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* dev;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	dev = dvb_manager_get_registered_device (self, adapter, frontend);
	if (dev == NULL) {
		char* _tmp0;
		_tmp0 = NULL;
		return (_tmp0 = g_strdup ("Unknown"), (dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL))), _tmp0);
	} else {
		const char* _tmp1;
		char* _tmp2;
		_tmp1 = NULL;
		_tmp2 = NULL;
		return (_tmp2 = (_tmp1 = dvb_device_get_Name (dev), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL))), _tmp2);
	}
	(dev == NULL ? NULL : (dev = (g_object_unref (dev), NULL)));
}


char* dvb_manager_GetSchedule (DVBManager* self, guint group_id, guint channel_sid) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		DVBDeviceGroup* devgroup;
		devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
		if (dvb_channel_list_contains (dvb_device_group_get_Channels (devgroup), channel_sid)) {
			char* path;
			char* _tmp4;
			path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCHEDULE_PATH, group_id, channel_sid);
			if (!gee_collection_contains (GEE_COLLECTION (self->priv->schedules), path)) {
				DBusGConnection* conn;
				DVBSchedule* _tmp2;
				DVBChannel* _tmp1;
				DVBSchedule* _tmp3;
				DVBSchedule* schedule;
				conn = dvb_manager_get_dbus_connection ();
				if (conn == NULL) {
					char* _tmp0;
					_tmp0 = NULL;
					return (_tmp0 = g_strdup (""), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), (path = (g_free (path), NULL)), (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), _tmp0);
				}
				_tmp2 = NULL;
				_tmp1 = NULL;
				_tmp3 = NULL;
				schedule = (_tmp3 = (_tmp2 = dvb_channel_get_Schedule ((_tmp1 = dvb_channel_list_get (dvb_device_group_get_Channels (devgroup), channel_sid))), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), _tmp3);
				dbus_g_connection_register_g_object (conn, path, G_OBJECT (schedule));
				gee_collection_add (GEE_COLLECTION (self->priv->schedules), path);
				(conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)));
				(schedule == NULL ? NULL : (schedule = (g_object_unref (schedule), NULL)));
			}
			_tmp4 = NULL;
			return (_tmp4 = path, (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), _tmp4);
		}
		(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
	}
	return g_strdup ("");
}


/**
         * @returns: Whether the device has been added successfully
         *
         * Register device, create Recorder and ChannelList D-Bus service
         */
gboolean dvb_manager_add_device_group (DVBManager* self, DVBDeviceGroup* device) {
	char* rec_path;
	char* channels_path;
	DVBGConfStore* _tmp2;
	gboolean _tmp3;
	g_return_val_if_fail (DVB_IS_MANAGER (self), FALSE);
	g_return_val_if_fail (DVB_IS_DEVICE_GROUP (device), FALSE);
	g_debug ("Manager.vala:438: Adding device group %u with %d devices", dvb_device_group_get_Id (device), dvb_device_group_get_size (device));
	gee_map_set (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (dvb_device_group_get_Id (device)), device);
	rec_path = dvb_manager_GetRecorder (self, dvb_device_group_get_Id (device));
	if (_vala_strcmp0 (rec_path, "") == 0) {
		gboolean _tmp0;
		return (_tmp0 = FALSE, (rec_path = (g_free (rec_path), NULL)), _tmp0);
	}
	channels_path = dvb_manager_GetChannelList (self, dvb_device_group_get_Id (device));
	if (_vala_strcmp0 (channels_path, "") == 0) {
		gboolean _tmp1;
		return (_tmp1 = FALSE, (rec_path = (g_free (rec_path), NULL)), (channels_path = (g_free (channels_path), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	dvb_gconf_store_add_device_group ((_tmp2 = dvb_gconf_store_get_instance ()), device);
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	if (dvb_device_group_get_Id (device) > self->priv->device_group_counter) {
		self->priv->device_group_counter = dvb_device_group_get_Id (device);
	}
	return (_tmp3 = TRUE, (rec_path = (g_free (rec_path), NULL)), (channels_path = (g_free (channels_path), NULL)), _tmp3);
}


DVBRecorder* dvb_manager_get_recorder_for_device_group (DVBManager* self, DVBDeviceGroup* device) {
	guint id;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	g_return_val_if_fail (DVB_IS_DEVICE_GROUP (device), NULL);
	id = dvb_device_group_get_Id (device);
	if (gee_map_contains (GEE_MAP (self->priv->recorders), GUINT_TO_POINTER (id))) {
		return ((DVBRecorder*) (gee_map_get (GEE_MAP (self->priv->recorders), GUINT_TO_POINTER (id))));
	} else {
		return NULL;
	}
}


DVBEPGScanner* dvb_manager_get_epg_scanner (DVBManager* self, DVBDeviceGroup* devgroup) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	g_return_val_if_fail (DVB_IS_DEVICE_GROUP (devgroup), NULL);
	return ((DVBEPGScanner*) (gee_map_get (GEE_MAP (self->priv->epgscanners), GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)))));
}


void dvb_manager_create_and_start_epg_scanner (DVBManager* self, DVBDeviceGroup* devgroup) {
	DVBEPGScanner* epgscanner;
	g_return_if_fail (DVB_IS_MANAGER (self));
	g_return_if_fail (DVB_IS_DEVICE_GROUP (devgroup));
	g_debug ("Manager.vala:472: Creating new EPG scanner for device group %u", dvb_device_group_get_Id (devgroup));
	epgscanner = dvb_epg_scanner_new (devgroup);
	dvb_epg_scanner_start (epgscanner);
	gee_map_set (GEE_MAP (self->priv->epgscanners), GUINT_TO_POINTER (dvb_device_group_get_Id (devgroup)), epgscanner);
	(epgscanner == NULL ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL)));
}


static DVBDevice* dvb_manager_create_device (guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir) {
	GError * inner_error;
	GFile* channelsfile;
	GFile* recdir;
	DVBDevice* device;
	DVBChannelList* channels;
	DVBDevice* _tmp2;
	g_return_val_if_fail (channels_conf != NULL, NULL);
	g_return_val_if_fail (recordings_dir != NULL, NULL);
	inner_error = NULL;
	/* TODO Check if adapter and frontend exists*/
	channelsfile = g_file_new_for_path (channels_conf);
	recdir = g_file_new_for_path (recordings_dir);
	device = dvb_device_new (adapter, frontend, TRUE);
	dvb_device_set_RecordingsDirectory (device, recdir);
	channels = NULL;
	{
		DVBChannelList* _tmp0;
		_tmp0 = NULL;
		channels = (_tmp0 = dvb_channel_list_restore_from_file (channelsfile, dvb_device_get_Type (device), &inner_error), (channels == NULL ? NULL : (channels = (g_object_unref (channels), NULL))), _tmp0);
		if (inner_error != NULL) {
			goto __catch20_g_error;
		}
	}
	goto __finally20;
	__catch20_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DVBDevice* _tmp1;
			g_critical (e->message);
			_tmp1 = NULL;
			return (_tmp1 = NULL, (e == NULL ? NULL : (e = (g_error_free (e), NULL))), (channelsfile == NULL ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL))), (recdir == NULL ? NULL : (recdir = (g_object_unref (recdir), NULL))), (device == NULL ? NULL : (device = (g_object_unref (device), NULL))), (channels == NULL ? NULL : (channels = (g_object_unref (channels), NULL))), _tmp1);
		}
	}
	__finally20:
	;
	dvb_device_set_Channels (device, channels);
	_tmp2 = NULL;
	return (_tmp2 = device, (channelsfile == NULL ? NULL : (channelsfile = (g_object_unref (channelsfile), NULL))), (recdir == NULL ? NULL : (recdir = (g_object_unref (recdir), NULL))), (channels == NULL ? NULL : (channels = (g_object_unref (channels), NULL))), _tmp2);
}


static DBusGConnection* dvb_manager_get_dbus_connection (void) {
	GError * inner_error;
	DBusGConnection* conn;
	inner_error = NULL;
	conn = NULL;
	{
		DBusGConnection* _tmp0;
		_tmp0 = NULL;
		conn = (_tmp0 = dbus_g_bus_get (DBUS_BUS_SESSION, &inner_error), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), _tmp0);
		if (inner_error != NULL) {
			goto __catch21_g_error;
		}
	}
	goto __finally21;
	__catch21_g_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			DBusGConnection* _tmp1;
			g_error (e->message);
			_tmp1 = NULL;
			return (_tmp1 = NULL, (e == NULL ? NULL : (e = (g_error_free (e), NULL))), (conn == NULL ? NULL : (conn = (dbus_g_connection_unref (conn), NULL))), _tmp1);
		}
	}
	__finally21:
	;
	return conn;
}


static DVBDeviceGroup* dvb_manager_get_device_group_if_exists (DVBManager* self, guint group_id) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	;
	if (gee_map_contains (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))) {
		return ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
	} else {
		return NULL;
	}
}


static gboolean dvb_manager_device_is_in_any_group (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), FALSE);
	g_return_val_if_fail (DVB_IS_DEVICE (device), FALSE);
	{
		GeeSet* group_id_collection;
		GeeIterator* group_id_it;
		group_id_collection = gee_map_get_keys (GEE_MAP (self->priv->devices));
		group_id_it = gee_iterable_iterator (GEE_ITERABLE (group_id_collection));
		while (gee_iterator_next (group_id_it)) {
			guint group_id;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (group_id_it));
			{
				DVBDeviceGroup* devgroup;
				devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
				if (dvb_device_group_contains (devgroup, device)) {
					gboolean _tmp0;
					return (_tmp0 = TRUE, (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), (group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL))), (group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL))), _tmp0);
				}
				(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
			}
		}
		(group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL)));
		(group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL)));
	}
	return FALSE;
}


static void dvb_manager_on_scanner_destroyed (DVBManager* self, DVBScanner* scanner) {
	guint adapter;
	guint frontend;
	char* path;
	DVBDeviceGroup* devgroup;
	g_return_if_fail (DVB_IS_MANAGER (self));
	g_return_if_fail (DVB_IS_SCANNER (scanner));
	adapter = dvb_device_get_Adapter (dvb_scanner_get_Device (scanner));
	frontend = dvb_device_get_Frontend (dvb_scanner_get_Device (scanner));
	g_debug ("Manager.vala:532: Destroying scanner for adapter %u, frontend %u", adapter, frontend);
	path = g_strdup_printf (DVB_CONSTANTS_DBUS_SCANNER_PATH, adapter, frontend);
	gee_map_remove (GEE_MAP (self->priv->scanners), path);
	/* Start epgscanner for device again if there was one*/
	devgroup = dvb_manager_get_device_group_of_device (self, dvb_scanner_get_Device (scanner));
	if (devgroup != NULL) {
		DVBEPGScanner* epgscanner;
		epgscanner = dvb_manager_get_epg_scanner (self, devgroup);
		if (epgscanner != NULL) {
			dvb_epg_scanner_start (epgscanner);
		}
		(epgscanner == NULL ? NULL : (epgscanner = (g_object_unref (epgscanner), NULL)));
	}
	path = (g_free (path), NULL);
	(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
}


static DVBDevice* dvb_manager_get_registered_device (DVBManager* self, guint adapter, guint frontend) {
	DVBDevice* fake_device;
	DVBDevice* _tmp1;
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	fake_device = dvb_device_new (adapter, frontend, FALSE);
	{
		GeeSet* group_id_collection;
		GeeIterator* group_id_it;
		group_id_collection = gee_map_get_keys (GEE_MAP (self->priv->devices));
		group_id_it = gee_iterable_iterator (GEE_ITERABLE (group_id_collection));
		while (gee_iterator_next (group_id_it)) {
			guint group_id;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (group_id_it));
			{
				DVBDeviceGroup* devgroup;
				devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
				if (dvb_device_group_contains (devgroup, fake_device)) {
					{
						DVBDeviceGroup* device_collection;
						GeeIterator* device_it;
						device_collection = devgroup;
						device_it = gee_iterable_iterator (GEE_ITERABLE (device_collection));
						while (gee_iterator_next (device_it)) {
							DVBDevice* device;
							device = ((DVBDevice*) (gee_iterator_get (device_it)));
							{
								if (dvb_device_equal (fake_device, device)) {
									DVBDevice* _tmp0;
									_tmp0 = NULL;
									return (_tmp0 = device, (device_it == NULL ? NULL : (device_it = (g_object_unref (device_it), NULL))), (devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL))), (group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL))), (group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL))), (fake_device == NULL ? NULL : (fake_device = (g_object_unref (fake_device), NULL))), _tmp0);
								}
								(device == NULL ? NULL : (device = (g_object_unref (device), NULL)));
							}
						}
						(device_it == NULL ? NULL : (device_it = (g_object_unref (device_it), NULL)));
					}
				}
				(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
			}
		}
		(group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL)));
		(group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL)));
	}
	_tmp1 = NULL;
	return (_tmp1 = NULL, (fake_device == NULL ? NULL : (fake_device = (g_object_unref (fake_device), NULL))), _tmp1);
}


static DVBDeviceGroup* dvb_manager_get_device_group_of_device (DVBManager* self, DVBDevice* device) {
	g_return_val_if_fail (DVB_IS_MANAGER (self), NULL);
	g_return_val_if_fail (DVB_IS_DEVICE (device), NULL);
	{
		GeeSet* group_id_collection;
		GeeIterator* group_id_it;
		group_id_collection = gee_map_get_keys (GEE_MAP (self->priv->devices));
		group_id_it = gee_iterable_iterator (GEE_ITERABLE (group_id_collection));
		while (gee_iterator_next (group_id_it)) {
			guint group_id;
			group_id = GPOINTER_TO_UINT (gee_iterator_get (group_id_it));
			{
				DVBDeviceGroup* devgroup;
				devgroup = ((DVBDeviceGroup*) (gee_map_get (GEE_MAP (self->priv->devices), GUINT_TO_POINTER (group_id))));
				if (dvb_device_group_contains (devgroup, device)) {
					{
						DVBDeviceGroup* grp_device_collection;
						GeeIterator* grp_device_it;
						grp_device_collection = devgroup;
						grp_device_it = gee_iterable_iterator (GEE_ITERABLE (grp_device_collection));
						while (gee_iterator_next (grp_device_it)) {
							DVBDevice* grp_device;
							grp_device = ((DVBDevice*) (gee_iterator_get (grp_device_it)));
							{
								if (dvb_device_equal (grp_device, device)) {
									DVBDeviceGroup* _tmp0;
									_tmp0 = NULL;
									return (_tmp0 = devgroup, (grp_device == NULL ? NULL : (grp_device = (g_object_unref (grp_device), NULL))), (grp_device_it == NULL ? NULL : (grp_device_it = (g_object_unref (grp_device_it), NULL))), (group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL))), (group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL))), _tmp0);
								}
								(grp_device == NULL ? NULL : (grp_device = (g_object_unref (grp_device), NULL)));
							}
						}
						(grp_device_it == NULL ? NULL : (grp_device_it = (g_object_unref (grp_device_it), NULL)));
					}
				}
				(devgroup == NULL ? NULL : (devgroup = (g_object_unref (devgroup), NULL)));
			}
		}
		(group_id_it == NULL ? NULL : (group_id_it = (g_object_unref (group_id_it), NULL)));
		(group_id_collection == NULL ? NULL : (group_id_collection = (g_object_unref (group_id_collection), NULL)));
	}
	return NULL;
}


DVBManager* dvb_manager_construct (GType object_type) {
	DVBManager * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


DVBManager* dvb_manager_new (void) {
	return dvb_manager_construct (DVB_TYPE_MANAGER);
}


static GObject * dvb_manager_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	DVBManagerClass * klass;
	GObjectClass * parent_class;
	DVBManager * self;
	klass = DVB_MANAGER_CLASS (g_type_class_peek (DVB_TYPE_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = DVB_MANAGER (obj);
	{
		GeeHashMap* _tmp0;
		GeeHashMap* _tmp1;
		GeeHashMap* _tmp2;
		GeeHashMap* _tmp3;
		GeeHashMap* _tmp4;
		GeeHashSet* _tmp5;
		_tmp0 = NULL;
		self->priv->scanners = (_tmp0 = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, DVB_TYPE_SCANNER, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_str_hash, g_str_equal, g_direct_equal), (self->priv->scanners == NULL ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL))), _tmp0);
		_tmp1 = NULL;
		self->priv->recorders = (_tmp1 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_RECORDER, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->recorders == NULL ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL))), _tmp1);
		_tmp2 = NULL;
		self->priv->channellists = (_tmp2 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_CHANNEL_LIST, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->channellists == NULL ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL))), _tmp2);
		_tmp3 = NULL;
		self->priv->devices = (_tmp3 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_DEVICE_GROUP, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->devices == NULL ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL))), _tmp3);
		_tmp4 = NULL;
		self->priv->epgscanners = (_tmp4 = gee_hash_map_new (G_TYPE_UINT, NULL, NULL, DVB_TYPE_EPG_SCANNER, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_hash, g_direct_equal, g_direct_equal), (self->priv->epgscanners == NULL ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL))), _tmp4);
		_tmp5 = NULL;
		self->priv->schedules = (_tmp5 = gee_hash_set_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_str_hash, g_str_equal), (self->priv->schedules == NULL ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL))), _tmp5);
		self->priv->device_group_counter = ((guint) (0));
	}
	return obj;
}


static gboolean _dbus_dvb_manager_GetScannerForDevice (DVBManager* self, guint adapter, guint frontend, char*** result, GError** error) {
	int result_length1;
	*result = dvb_manager_GetScannerForDevice (self, adapter, frontend, &result_length1);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetRegisteredDeviceGroups (DVBManager* self, GArray** dbus_result, GError** error) {
	guint* result;
	int result_length1;
	result = dvb_manager_GetRegisteredDeviceGroups (self, &result_length1);
	(*dbus_result) = g_array_new (TRUE, TRUE, sizeof (guint));
	g_array_append_vals ((*dbus_result), result, result_length1);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetTypeOfDeviceGroup (DVBManager* self, guint group_id, char** result, GError** error) {
	*result = dvb_manager_GetTypeOfDeviceGroup (self, group_id);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetRecorder (DVBManager* self, guint group_id, char** result, GError** error) {
	*result = dvb_manager_GetRecorder (self, group_id);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_AddDeviceToNewGroup (DVBManager* self, guint adapter, guint frontend, const char* channels_conf, const char* recordings_dir, gboolean* result, GError** error) {
	*result = dvb_manager_AddDeviceToNewGroup (self, adapter, frontend, channels_conf, recordings_dir);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_AddDeviceToExistingGroup (DVBManager* self, guint adapter, guint frontend, guint group_id, gboolean* result, GError** error) {
	*result = dvb_manager_AddDeviceToExistingGroup (self, adapter, frontend, group_id);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_RemoveDeviceFromGroup (DVBManager* self, guint adapter, guint frontend, guint group_id, gboolean* result, GError** error) {
	*result = dvb_manager_RemoveDeviceFromGroup (self, adapter, frontend, group_id);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetChannelList (DVBManager* self, guint group_id, char** result, GError** error) {
	*result = dvb_manager_GetChannelList (self, group_id);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetDeviceGroupMembers (DVBManager* self, guint group_id, char*** result, GError** error) {
	int result_length1;
	*result = dvb_manager_GetDeviceGroupMembers (self, group_id, &result_length1);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetNameOfRegisteredDevice (DVBManager* self, guint adapter, guint frontend, char** result, GError** error) {
	*result = dvb_manager_GetNameOfRegisteredDevice (self, adapter, frontend);
	return !error || !*error;
}


static gboolean _dbus_dvb_manager_GetSchedule (DVBManager* self, guint group_id, guint channel_sid, char** result, GError** error) {
	*result = dvb_manager_GetSchedule (self, group_id, channel_sid);
	return !error || !*error;
}


static void dvb_manager_class_init (DVBManagerClass * klass) {
	dvb_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (DVBManagerPrivate));
	G_OBJECT_CLASS (klass)->constructor = dvb_manager_constructor;
	G_OBJECT_CLASS (klass)->finalize = dvb_manager_finalize;
	g_signal_new ("changed", DVB_TYPE_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT, G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);
	g_signal_new ("group_changed", DVB_TYPE_MANAGER, G_SIGNAL_RUN_LAST, 0, NULL, NULL, g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT, G_TYPE_NONE, 4, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT);
	static const DBusGMethodInfo dvb_manager_dbus_methods[] = {
{ (GCallback) _dbus_dvb_manager_GetScannerForDevice, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_POINTER_POINTER, 0 },
{ (GCallback) _dbus_dvb_manager_GetRegisteredDeviceGroups, g_cclosure_user_marshal_BOOLEAN__POINTER_POINTER, 86 },
{ (GCallback) _dbus_dvb_manager_GetTypeOfDeviceGroup, g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER, 153 },
{ (GCallback) _dbus_dvb_manager_GetRecorder, g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER, 227 },
{ (GCallback) _dbus_dvb_manager_AddDeviceToNewGroup, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER, 292 },
{ (GCallback) _dbus_dvb_manager_AddDeviceToExistingGroup, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER, 414 },
{ (GCallback) _dbus_dvb_manager_RemoveDeviceFromGroup, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER, 517 },
{ (GCallback) _dbus_dvb_manager_GetChannelList, g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER, 617 },
{ (GCallback) _dbus_dvb_manager_GetDeviceGroupMembers, g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER, 685 },
{ (GCallback) _dbus_dvb_manager_GetNameOfRegisteredDevice, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_POINTER_POINTER, 761 },
{ (GCallback) _dbus_dvb_manager_GetSchedule, g_cclosure_user_marshal_BOOLEAN__UINT_UINT_POINTER_POINTER, 852 },
}
;
	static const DBusGObjectInfo dvb_manager_dbus_object_info = { 0, dvb_manager_dbus_methods, 11, "org.gnome.DVB.Manager\0GetScannerForDevice\0S\0adapter\0I\0u\0frontend\0I\0u\0result\0O\0F\0N\0as\0\0org.gnome.DVB.Manager\0GetRegisteredDeviceGroups\0S\0result\0O\0F\0N\0au\0\0org.gnome.DVB.Manager\0GetTypeOfDeviceGroup\0S\0group_id\0I\0u\0result\0O\0F\0N\0s\0\0org.gnome.DVB.Manager\0GetRecorder\0S\0group_id\0I\0u\0result\0O\0F\0N\0s\0\0org.gnome.DVB.Manager\0AddDeviceToNewGroup\0S\0adapter\0I\0u\0frontend\0I\0u\0channels_conf\0I\0s\0recordings_dir\0I\0s\0result\0O\0F\0N\0b\0\0org.gnome.DVB.Manager\0AddDeviceToExistingGroup\0S\0adapter\0I\0u\0frontend\0I\0u\0group_id\0I\0u\0result\0O\0F\0N\0b\0\0org.gnome.DVB.Manager\0RemoveDeviceFromGroup\0S\0adapter\0I\0u\0frontend\0I\0u\0group_id\0I\0u\0result\0O\0F\0N\0b\0\0org.gnome.DVB.Manager\0GetChannelList\0S\0group_id\0I\0u\0result\0O\0F\0N\0s\0\0org.gnome.DVB.Manager\0GetDeviceGroupMembers\0S\0group_id\0I\0u\0result\0O\0F\0N\0as\0\0org.gnome.DVB.Manager\0GetNameOfRegisteredDevice\0S\0adapter\0I\0u\0frontend\0I\0u\0result\0O\0F\0N\0s\0\0org.gnome.DVB.Manager\0GetSchedule\0S\0group_id\0I\0u\0channel_sid\0I\0u\0result\0O\0F\0N\0s\0\0", "org.gnome.DVB.Manager\0Changed\0org.gnome.DVB.Manager\0GroupChanged\0", "" };
	dbus_g_object_type_install_info (DVB_TYPE_MANAGER, &dvb_manager_dbus_object_info);
}


static void dvb_manager_instance_init (DVBManager * self) {
	self->priv = DVB_MANAGER_GET_PRIVATE (self);
}


static void dvb_manager_finalize (GObject* obj) {
	DVBManager * self;
	self = DVB_MANAGER (obj);
	(self->priv->scanners == NULL ? NULL : (self->priv->scanners = (g_object_unref (self->priv->scanners), NULL)));
	(self->priv->recorders == NULL ? NULL : (self->priv->recorders = (g_object_unref (self->priv->recorders), NULL)));
	(self->priv->channellists == NULL ? NULL : (self->priv->channellists = (g_object_unref (self->priv->channellists), NULL)));
	(self->priv->devices == NULL ? NULL : (self->priv->devices = (g_object_unref (self->priv->devices), NULL)));
	(self->priv->epgscanners == NULL ? NULL : (self->priv->epgscanners = (g_object_unref (self->priv->epgscanners), NULL)));
	(self->priv->schedules == NULL ? NULL : (self->priv->schedules = (g_object_unref (self->priv->schedules), NULL)));
	G_OBJECT_CLASS (dvb_manager_parent_class)->finalize (obj);
}


GType dvb_manager_get_type (void) {
	static GType dvb_manager_type_id = 0;
	if (dvb_manager_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (DVBManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBManager), 0, (GInstanceInitFunc) dvb_manager_instance_init, NULL };
		dvb_manager_type_id = g_type_register_static (G_TYPE_OBJECT, "DVBManager", &g_define_type_info, 0);
	}
	return dvb_manager_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}



static void g_cclosure_user_marshal_VOID__UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_VOID__UINT_UINT) ((marshal_data ? marshal_data : cc->callback)));
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), data2);
}


static void g_cclosure_user_marshal_VOID__UINT_UINT_UINT_UINT (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef void (*GMarshalFunc_VOID__UINT_UINT_UINT_UINT) (gpointer data1, guint arg_1, guint arg_2, guint arg_3, guint arg_4, gpointer data2);
	register GMarshalFunc_VOID__UINT_UINT_UINT_UINT callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (n_param_values == 5);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_VOID__UINT_UINT_UINT_UINT) ((marshal_data ? marshal_data : cc->callback)));
	callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_uint (param_values + 3), g_value_get_uint (param_values + 4), data2);
}


static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef gboolean (*GMarshalFunc_BOOLEAN__UINT_UINT_POINTER_POINTER) (gpointer data1, guint arg_1, guint arg_2, gpointer arg_3, gpointer arg_4, gpointer data2);
	register GMarshalFunc_BOOLEAN__UINT_UINT_POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	gboolean v_return;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (return_value != NULL);
	g_return_if_fail (n_param_values == 5);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_BOOLEAN__UINT_UINT_POINTER_POINTER) ((marshal_data ? marshal_data : cc->callback)));
	v_return = callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_pointer (param_values + 3), g_value_get_pointer (param_values + 4), data2);
	g_value_set_boolean (return_value, v_return);
}


static void g_cclosure_user_marshal_BOOLEAN__POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef gboolean (*GMarshalFunc_BOOLEAN__POINTER_POINTER) (gpointer data1, gpointer arg_1, gpointer arg_2, gpointer data2);
	register GMarshalFunc_BOOLEAN__POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	gboolean v_return;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (return_value != NULL);
	g_return_if_fail (n_param_values == 3);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_BOOLEAN__POINTER_POINTER) ((marshal_data ? marshal_data : cc->callback)));
	v_return = callback (data1, g_value_get_pointer (param_values + 1), g_value_get_pointer (param_values + 2), data2);
	g_value_set_boolean (return_value, v_return);
}


static void g_cclosure_user_marshal_BOOLEAN__UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef gboolean (*GMarshalFunc_BOOLEAN__UINT_POINTER_POINTER) (gpointer data1, guint arg_1, gpointer arg_2, gpointer arg_3, gpointer data2);
	register GMarshalFunc_BOOLEAN__UINT_POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	gboolean v_return;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (return_value != NULL);
	g_return_if_fail (n_param_values == 4);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_BOOLEAN__UINT_POINTER_POINTER) ((marshal_data ? marshal_data : cc->callback)));
	v_return = callback (data1, g_value_get_uint (param_values + 1), g_value_get_pointer (param_values + 2), g_value_get_pointer (param_values + 3), data2);
	g_value_set_boolean (return_value, v_return);
}


static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef gboolean (*GMarshalFunc_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER) (gpointer data1, guint arg_1, guint arg_2, const char* arg_3, const char* arg_4, gpointer arg_5, gpointer arg_6, gpointer data2);
	register GMarshalFunc_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	gboolean v_return;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (return_value != NULL);
	g_return_if_fail (n_param_values == 7);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_BOOLEAN__UINT_UINT_STRING_STRING_POINTER_POINTER) ((marshal_data ? marshal_data : cc->callback)));
	v_return = callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_string (param_values + 3), g_value_get_string (param_values + 4), g_value_get_pointer (param_values + 5), g_value_get_pointer (param_values + 6), data2);
	g_value_set_boolean (return_value, v_return);
}


static void g_cclosure_user_marshal_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER (GClosure * closure, GValue * return_value, guint n_param_values, const GValue * param_values, gpointer invocation_hint, gpointer marshal_data) {
	typedef gboolean (*GMarshalFunc_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER) (gpointer data1, guint arg_1, guint arg_2, guint arg_3, gpointer arg_4, gpointer arg_5, gpointer data2);
	register GMarshalFunc_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER callback;
	register GCClosure * cc;
	register gpointer data1, data2;
	gboolean v_return;
	cc = ((GCClosure *) (closure));
	g_return_if_fail (return_value != NULL);
	g_return_if_fail (n_param_values == 6);
	if (G_CCLOSURE_SWAP_DATA (closure)) {
		data1 = closure->data;
		data2 = param_values->data[0].v_pointer;
	} else {
		data1 = param_values->data[0].v_pointer;
		data2 = closure->data;
	}
	callback = ((GMarshalFunc_BOOLEAN__UINT_UINT_UINT_POINTER_POINTER) ((marshal_data ? marshal_data : cc->callback)));
	v_return = callback (data1, g_value_get_uint (param_values + 1), g_value_get_uint (param_values + 2), g_value_get_uint (param_values + 3), g_value_get_pointer (param_values + 4), g_value_get_pointer (param_values + 5), data2);
	g_value_set_boolean (return_value, v_return);
}



