/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <src/Event.h>
#include "src/Utils.h"
#include <gobject/gvaluecollector.h>




enum  {
	DVB_EVENT_DUMMY_PROPERTY
};
static void _g_slist_free_dvb_event_audio_component_unref (GSList* self);
static void _g_slist_free_dvb_event_video_component_unref (GSList* self);
static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self);
static gint64 dvb_event_get_end_timestamp (DVBEvent* self);
enum  {
	DVB_EVENT_AUDIO_COMPONENT_DUMMY_PROPERTY
};
static gpointer dvb_event_audio_component_parent_class = NULL;
static void dvb_event_audio_component_finalize (DVBEventAudioComponent* obj);
enum  {
	DVB_EVENT_VIDEO_COMPONENT_DUMMY_PROPERTY
};
static gpointer dvb_event_video_component_parent_class = NULL;
static void dvb_event_video_component_finalize (DVBEventVideoComponent* obj);
enum  {
	DVB_EVENT_TELETEXT_COMPONENT_DUMMY_PROPERTY
};
static gpointer dvb_event_teletext_component_parent_class = NULL;
static void dvb_event_teletext_component_finalize (DVBEventTeletextComponent* obj);
static gpointer dvb_event_parent_class = NULL;
static void dvb_event_finalize (DVBEvent* obj);



static void _g_slist_free_dvb_event_audio_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_audio_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_video_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_video_component_unref, NULL);
	g_slist_free (self);
}


static void _g_slist_free_dvb_event_teletext_component_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) dvb_event_teletext_component_unref, NULL);
	g_slist_free (self);
}


DVBEvent* dvb_event_construct (GType object_type) {
	DVBEvent* self;
	GSList* _tmp0;
	GSList* _tmp1;
	GSList* _tmp2;
	self = (DVBEvent*) g_type_create_instance (object_type);
	_tmp0 = NULL;
	self->audio_components = (_tmp0 = NULL, (self->audio_components == NULL) ? NULL : (self->audio_components = (_g_slist_free_dvb_event_audio_component_unref (self->audio_components), NULL)), _tmp0);
	_tmp1 = NULL;
	self->video_components = (_tmp1 = NULL, (self->video_components == NULL) ? NULL : (self->video_components = (_g_slist_free_dvb_event_video_component_unref (self->video_components), NULL)), _tmp1);
	_tmp2 = NULL;
	self->teletext_components = (_tmp2 = NULL, (self->teletext_components == NULL) ? NULL : (self->teletext_components = (_g_slist_free_dvb_event_teletext_component_unref (self->teletext_components), NULL)), _tmp2);
	self->year = (guint) 0;
	self->month = (guint) 0;
	self->hour = (guint) 0;
	self->day = (guint) 0;
	self->minute = (guint) 0;
	self->second = (guint) 0;
	self->duration = (guint) 0;
	self->running_status = DVB_EVENT_RUNNING_STATUS_UNDEFINED;
	return self;
}


DVBEvent* dvb_event_new (void) {
	return dvb_event_construct (DVB_TYPE_EVENT);
}


/**
         * Whether the event has started and ended in the past
         */
gboolean dvb_event_has_expired (DVBEvent* self) {
	time_t _tmp2 = {0};
	time_t _tmp0 = {0};
	struct tm _tmp1 = {0};
	struct tm current_utc;
	gint64 current_time;
	gint64 end_timestamp;
	g_return_val_if_fail (self != NULL, FALSE);
	current_utc = (gmtime_r ((_tmp2 = (time (&_tmp0), _tmp0), &_tmp2), &_tmp1), _tmp1);
	/* set day light saving time to undefined
	 otherwise mktime will add an hour,
	 because it respects dst*/
	current_utc.tm_isdst = -1;
	current_time = (gint64) mktime (&current_utc);
	end_timestamp = dvb_event_get_end_timestamp (self);
	return end_timestamp < current_time;
}


gboolean dvb_event_is_running (DVBEvent* self) {
	time_t _tmp2 = {0};
	time_t _tmp0 = {0};
	struct tm _tmp1 = {0};
	struct tm time_now;
	struct tm time_start;
	gint64 timestamp_now;
	gint64 timestamp_start;
	g_return_val_if_fail (self != NULL, FALSE);
	time_now = (gmtime_r ((_tmp2 = (time (&_tmp0), _tmp0), &_tmp2), &_tmp1), _tmp1);
	time_start = dvb_event_get_utc_start_time (self);
	timestamp_now = (gint64) timegm (&time_now);
	timestamp_start = (gint64) timegm (&time_start);
	if ((timestamp_now - timestamp_start) >= 0) {
		/* Has started, check if it's still running*/
		return !dvb_event_has_expired (self);
	} else {
		/* Has not started, yet*/
		return FALSE;
	}
}


char* dvb_event_to_string (DVBEvent* self) {
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	char* text;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	text = (_tmp2 = g_strconcat (_tmp0 = g_strdup_printf ("ID: %u\nDate: %04u-%02u-%02u %02u:%02u:%02u\n", self->id, self->year, self->month, self->day, self->hour, self->minute, self->second), _tmp1 = g_strdup_printf ("Duration: %u\nName: %s\nDescription: %s\n", self->duration, self->name, self->description), NULL), _tmp1 = (g_free (_tmp1), NULL), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
	{
		gint i;
		i = 0;
		for (; i < g_slist_length (self->audio_components); i++) {
			char* _tmp4;
			char* _tmp3;
			_tmp4 = NULL;
			_tmp3 = NULL;
			text = (_tmp4 = g_strconcat (text, _tmp3 = g_strdup_printf ("%s ", ((DVBEventAudioComponent*) g_slist_nth_data (self->audio_components, (guint) i))->type), NULL), text = (g_free (text), NULL), _tmp4);
			_tmp3 = (g_free (_tmp3), NULL);
		}
	}
	return text;
}


struct tm dvb_event_get_local_start_time (DVBEvent* self) {
	struct tm utc_time;
	time_t utc_timestamp;
	struct tm _tmp0 = {0};
	struct tm local_time;
	/* Initialize time zone and set values*/
	utc_time = dvb_event_get_utc_start_time (self);
	utc_timestamp = timegm (&utc_time);
	local_time = (localtime_r (&utc_timestamp, &_tmp0), _tmp0);
	return local_time;
}


struct tm dvb_event_get_utc_start_time (DVBEvent* self) {
	struct tm utc_time;
	utc_time = dvb_utils_create_utc_time ((gint) self->year, (gint) self->month, (gint) self->day, (gint) self->hour, (gint) self->minute, (gint) self->second);
	return utc_time;
}


/**
         * @returns: UNIX time stamp
         */
static gint64 dvb_event_get_end_timestamp (DVBEvent* self) {
	struct tm end_time;
	gint64 before;
	gint64 after;
	g_return_val_if_fail (self != NULL, 0LL);
	end_time = dvb_utils_create_utc_time ((gint) self->year, (gint) self->month, (gint) self->day, (gint) self->hour, (gint) self->minute, (gint) self->second);
	before = (gint64) mktime (&end_time);
	end_time.tm_sec = end_time.tm_sec + ((gint) self->duration);
	after = (gint64) mktime (&end_time);
	g_assert ((after - before) == self->duration);
	return after;
}


/**
         * @returns: negative value if event1 starts before event2,
         * positive value if event1 starts after event2 and zero else
         *
         * Compare the starting time of two events
         */
gint dvb_event_compare (DVBEvent* event1, DVBEvent* event2) {
	gboolean _tmp0;
	gint64 event1_time;
	gint64 event2_time;
	_tmp0 = FALSE;
	if (event1 == NULL) {
		_tmp0 = event2 == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		return 0;
	} else {
		gboolean _tmp2;
		_tmp2 = FALSE;
		if (event1 == NULL) {
			_tmp2 = event2 != NULL;
		} else {
			_tmp2 = FALSE;
		}
		if (_tmp2) {
			return +1;
		} else {
			gboolean _tmp4;
			_tmp4 = FALSE;
			if (event1 != NULL) {
				_tmp4 = event2 == NULL;
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				return -1;
			}
		}
	}
	event1_time = dvb_event_get_end_timestamp (event1);
	event2_time = dvb_event_get_end_timestamp (event2);
	if (event1_time < event2_time) {
		return -1;
	} else {
		if (event1_time > event2_time) {
			return +1;
		} else {
			return 0;
		}
	}
}


/**
         * @returns: TRUE if event1 and event2 represent the same event,
         * else FALSE
         *
         * event1 and event2 must be part of the same transport stream
         */
gboolean dvb_event_equal (DVBEvent* event1, DVBEvent* event2) {
	gboolean _tmp0;
	_tmp0 = FALSE;
	if (event1 == NULL) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = event2 == NULL;
	}
	if (_tmp0) {
		return FALSE;
	}
	return event1->id == event2->id;
}


DVBEventAudioComponent* dvb_event_audio_component_construct (GType object_type) {
	DVBEventAudioComponent* self;
	self = (DVBEventAudioComponent*) g_type_create_instance (object_type);
	return self;
}


DVBEventAudioComponent* dvb_event_audio_component_new (void) {
	return dvb_event_audio_component_construct (DVB_EVENT_TYPE_AUDIO_COMPONENT);
}


static void dvb_event_value_audio_component_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void dvb_event_value_audio_component_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		dvb_event_audio_component_unref (value->data[0].v_pointer);
	}
}


static void dvb_event_value_audio_component_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = dvb_event_audio_component_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer dvb_event_value_audio_component_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* dvb_event_value_audio_component_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		DVBEventAudioComponent* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = dvb_event_audio_component_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* dvb_event_value_audio_component_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	DVBEventAudioComponent** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = dvb_event_audio_component_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* dvb_event_param_spec_audio_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	DVBEventParamSpecAudioComponent* spec;
	g_return_val_if_fail (g_type_is_a (object_type, DVB_EVENT_TYPE_AUDIO_COMPONENT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer dvb_event_value_get_audio_component (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_AUDIO_COMPONENT), NULL);
	return value->data[0].v_pointer;
}


void dvb_event_value_set_audio_component (GValue* value, gpointer v_object) {
	DVBEventAudioComponent* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_AUDIO_COMPONENT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, DVB_EVENT_TYPE_AUDIO_COMPONENT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		dvb_event_audio_component_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		dvb_event_audio_component_unref (old);
	}
}


static void dvb_event_audio_component_class_init (DVBEventAudioComponentClass * klass) {
	dvb_event_audio_component_parent_class = g_type_class_peek_parent (klass);
	DVB_EVENT_AUDIO_COMPONENT_CLASS (klass)->finalize = dvb_event_audio_component_finalize;
}


static void dvb_event_audio_component_instance_init (DVBEventAudioComponent * self) {
	self->ref_count = 1;
}


static void dvb_event_audio_component_finalize (DVBEventAudioComponent* obj) {
	DVBEventAudioComponent * self;
	self = DVB_EVENT_AUDIO_COMPONENT (obj);
	self->type = (g_free (self->type), NULL);
}


GType dvb_event_audio_component_get_type (void) {
	static GType dvb_event_audio_component_type_id = 0;
	if (dvb_event_audio_component_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { dvb_event_value_audio_component_init, dvb_event_value_audio_component_free_value, dvb_event_value_audio_component_copy_value, dvb_event_value_audio_component_peek_pointer, "p", dvb_event_value_audio_component_collect_value, "p", dvb_event_value_audio_component_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (DVBEventAudioComponentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_event_audio_component_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEventAudioComponent), 0, (GInstanceInitFunc) dvb_event_audio_component_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		dvb_event_audio_component_type_id = g_type_register_fundamental (g_type_fundamental_next (), "DVBEventAudioComponent", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return dvb_event_audio_component_type_id;
}


gpointer dvb_event_audio_component_ref (gpointer instance) {
	DVBEventAudioComponent* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void dvb_event_audio_component_unref (gpointer instance) {
	DVBEventAudioComponent* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		DVB_EVENT_AUDIO_COMPONENT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


DVBEventVideoComponent* dvb_event_video_component_construct (GType object_type) {
	DVBEventVideoComponent* self;
	self = (DVBEventVideoComponent*) g_type_create_instance (object_type);
	return self;
}


DVBEventVideoComponent* dvb_event_video_component_new (void) {
	return dvb_event_video_component_construct (DVB_EVENT_TYPE_VIDEO_COMPONENT);
}


static void dvb_event_value_video_component_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void dvb_event_value_video_component_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		dvb_event_video_component_unref (value->data[0].v_pointer);
	}
}


static void dvb_event_value_video_component_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = dvb_event_video_component_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer dvb_event_value_video_component_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* dvb_event_value_video_component_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		DVBEventVideoComponent* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = dvb_event_video_component_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* dvb_event_value_video_component_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	DVBEventVideoComponent** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = dvb_event_video_component_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* dvb_event_param_spec_video_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	DVBEventParamSpecVideoComponent* spec;
	g_return_val_if_fail (g_type_is_a (object_type, DVB_EVENT_TYPE_VIDEO_COMPONENT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer dvb_event_value_get_video_component (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_VIDEO_COMPONENT), NULL);
	return value->data[0].v_pointer;
}


void dvb_event_value_set_video_component (GValue* value, gpointer v_object) {
	DVBEventVideoComponent* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_VIDEO_COMPONENT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, DVB_EVENT_TYPE_VIDEO_COMPONENT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		dvb_event_video_component_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		dvb_event_video_component_unref (old);
	}
}


static void dvb_event_video_component_class_init (DVBEventVideoComponentClass * klass) {
	dvb_event_video_component_parent_class = g_type_class_peek_parent (klass);
	DVB_EVENT_VIDEO_COMPONENT_CLASS (klass)->finalize = dvb_event_video_component_finalize;
}


static void dvb_event_video_component_instance_init (DVBEventVideoComponent * self) {
	self->ref_count = 1;
}


static void dvb_event_video_component_finalize (DVBEventVideoComponent* obj) {
	DVBEventVideoComponent * self;
	self = DVB_EVENT_VIDEO_COMPONENT (obj);
	self->aspect_ratio = (g_free (self->aspect_ratio), NULL);
}


GType dvb_event_video_component_get_type (void) {
	static GType dvb_event_video_component_type_id = 0;
	if (dvb_event_video_component_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { dvb_event_value_video_component_init, dvb_event_value_video_component_free_value, dvb_event_value_video_component_copy_value, dvb_event_value_video_component_peek_pointer, "p", dvb_event_value_video_component_collect_value, "p", dvb_event_value_video_component_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (DVBEventVideoComponentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_event_video_component_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEventVideoComponent), 0, (GInstanceInitFunc) dvb_event_video_component_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		dvb_event_video_component_type_id = g_type_register_fundamental (g_type_fundamental_next (), "DVBEventVideoComponent", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return dvb_event_video_component_type_id;
}


gpointer dvb_event_video_component_ref (gpointer instance) {
	DVBEventVideoComponent* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void dvb_event_video_component_unref (gpointer instance) {
	DVBEventVideoComponent* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		DVB_EVENT_VIDEO_COMPONENT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


DVBEventTeletextComponent* dvb_event_teletext_component_construct (GType object_type) {
	DVBEventTeletextComponent* self;
	self = (DVBEventTeletextComponent*) g_type_create_instance (object_type);
	return self;
}


DVBEventTeletextComponent* dvb_event_teletext_component_new (void) {
	return dvb_event_teletext_component_construct (DVB_EVENT_TYPE_TELETEXT_COMPONENT);
}


static void dvb_event_value_teletext_component_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void dvb_event_value_teletext_component_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		dvb_event_teletext_component_unref (value->data[0].v_pointer);
	}
}


static void dvb_event_value_teletext_component_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = dvb_event_teletext_component_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer dvb_event_value_teletext_component_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* dvb_event_value_teletext_component_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		DVBEventTeletextComponent* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = dvb_event_teletext_component_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* dvb_event_value_teletext_component_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	DVBEventTeletextComponent** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = dvb_event_teletext_component_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* dvb_event_param_spec_teletext_component (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	DVBEventParamSpecTeletextComponent* spec;
	g_return_val_if_fail (g_type_is_a (object_type, DVB_EVENT_TYPE_TELETEXT_COMPONENT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer dvb_event_value_get_teletext_component (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_TELETEXT_COMPONENT), NULL);
	return value->data[0].v_pointer;
}


void dvb_event_value_set_teletext_component (GValue* value, gpointer v_object) {
	DVBEventTeletextComponent* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_EVENT_TYPE_TELETEXT_COMPONENT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, DVB_EVENT_TYPE_TELETEXT_COMPONENT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		dvb_event_teletext_component_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		dvb_event_teletext_component_unref (old);
	}
}


static void dvb_event_teletext_component_class_init (DVBEventTeletextComponentClass * klass) {
	dvb_event_teletext_component_parent_class = g_type_class_peek_parent (klass);
	DVB_EVENT_TELETEXT_COMPONENT_CLASS (klass)->finalize = dvb_event_teletext_component_finalize;
}


static void dvb_event_teletext_component_instance_init (DVBEventTeletextComponent * self) {
	self->ref_count = 1;
}


static void dvb_event_teletext_component_finalize (DVBEventTeletextComponent* obj) {
	DVBEventTeletextComponent * self;
	self = DVB_EVENT_TELETEXT_COMPONENT (obj);
	self->type = (g_free (self->type), NULL);
}


GType dvb_event_teletext_component_get_type (void) {
	static GType dvb_event_teletext_component_type_id = 0;
	if (dvb_event_teletext_component_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { dvb_event_value_teletext_component_init, dvb_event_value_teletext_component_free_value, dvb_event_value_teletext_component_copy_value, dvb_event_value_teletext_component_peek_pointer, "p", dvb_event_value_teletext_component_collect_value, "p", dvb_event_value_teletext_component_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (DVBEventTeletextComponentClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_event_teletext_component_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEventTeletextComponent), 0, (GInstanceInitFunc) dvb_event_teletext_component_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		dvb_event_teletext_component_type_id = g_type_register_fundamental (g_type_fundamental_next (), "DVBEventTeletextComponent", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return dvb_event_teletext_component_type_id;
}


gpointer dvb_event_teletext_component_ref (gpointer instance) {
	DVBEventTeletextComponent* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void dvb_event_teletext_component_unref (gpointer instance) {
	DVBEventTeletextComponent* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		DVB_EVENT_TELETEXT_COMPONENT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void dvb_value_event_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void dvb_value_event_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		dvb_event_unref (value->data[0].v_pointer);
	}
}


static void dvb_value_event_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = dvb_event_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer dvb_value_event_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* dvb_value_event_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		DVBEvent* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = dvb_event_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* dvb_value_event_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	DVBEvent** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = dvb_event_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* dvb_param_spec_event (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	DVBParamSpecEvent* spec;
	g_return_val_if_fail (g_type_is_a (object_type, DVB_TYPE_EVENT), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer dvb_value_get_event (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_TYPE_EVENT), NULL);
	return value->data[0].v_pointer;
}


void dvb_value_set_event (GValue* value, gpointer v_object) {
	DVBEvent* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, DVB_TYPE_EVENT));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, DVB_TYPE_EVENT));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		dvb_event_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		dvb_event_unref (old);
	}
}


static void dvb_event_class_init (DVBEventClass * klass) {
	dvb_event_parent_class = g_type_class_peek_parent (klass);
	DVB_EVENT_CLASS (klass)->finalize = dvb_event_finalize;
}


static void dvb_event_instance_init (DVBEvent * self) {
	self->ref_count = 1;
}


static void dvb_event_finalize (DVBEvent* obj) {
	DVBEvent * self;
	self = DVB_EVENT (obj);
	self->name = (g_free (self->name), NULL);
	self->description = (g_free (self->description), NULL);
	self->extended_description = (g_free (self->extended_description), NULL);
	(self->audio_components == NULL) ? NULL : (self->audio_components = (_g_slist_free_dvb_event_audio_component_unref (self->audio_components), NULL));
	(self->video_components == NULL) ? NULL : (self->video_components = (_g_slist_free_dvb_event_video_component_unref (self->video_components), NULL));
	(self->teletext_components == NULL) ? NULL : (self->teletext_components = (_g_slist_free_dvb_event_teletext_component_unref (self->teletext_components), NULL));
}


GType dvb_event_get_type (void) {
	static GType dvb_event_type_id = 0;
	if (dvb_event_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { dvb_value_event_init, dvb_value_event_free_value, dvb_value_event_copy_value, dvb_value_event_peek_pointer, "p", dvb_value_event_collect_value, "p", dvb_value_event_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (DVBEventClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) dvb_event_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (DVBEvent), 0, (GInstanceInitFunc) dvb_event_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		dvb_event_type_id = g_type_register_fundamental (g_type_fundamental_next (), "DVBEvent", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return dvb_event_type_id;
}


gpointer dvb_event_ref (gpointer instance) {
	DVBEvent* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void dvb_event_unref (gpointer instance) {
	DVBEvent* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		DVB_EVENT_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}




