/*
 * Copyright (C) 2008,2009 Sebastian Pölsterl
 *
 * This file is part of GNOME DVB Daemon.
 *
 * GNOME DVB Daemon is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GNOME DVB Daemon is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNOME DVB Daemon.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gio/gio.h>
#include <time.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib.h>




#define DVB_UTILS_BUFFER_SIZE 4096
const char* dvb_utils_get_nick_from_enum (GType enumtype, gint val);
gboolean dvb_utils_get_value_by_name_from_enum (GType enumtype, const char* name, gint* evalue);
const char* dvb_utils_get_name_by_value_from_enum (GType enumtype, gint val);
static void _g_slist_free_g_object_unref (GSList* self);
void dvb_utils_mkdirs (GFile* directory, GError** error);
char* dvb_utils_remove_nonalphanums (const char* text);
gint64 dvb_utils_difftime (const struct tm* t1, const struct tm* t2);
struct tm dvb_utils_create_time (gint year, gint month, gint day, gint hour, gint minute, gint second);
struct tm dvb_utils_create_utc_time (gint year, gint month, gint day, gint hour, gint minute, gint second);
char* dvb_utils_read_file_contents (GFile* file, GError** error);
void dvb_utils_delete_dir_recursively (GFile* dir, GError** error);
DBusGConnection* dvb_utils_get_dbus_connection (void);



inline const char* dvb_utils_get_nick_from_enum (GType enumtype, gint val) {
	const char* result;
	GEnumClass* eclass;
	GEnumValue* eval;
	eclass = (GEnumClass*) g_type_class_ref (enumtype);
	eval = g_enum_get_value (eclass, val);
	if (eval == NULL) {
		g_critical ("Utils.vala:31: Enum has no value %d", val);
		result = NULL;
		(eclass == NULL) ? NULL : (eclass = (g_type_class_unref (eclass), NULL));
		return result;
	} else {
		result = eval->value_nick;
		(eclass == NULL) ? NULL : (eclass = (g_type_class_unref (eclass), NULL));
		return result;
	}
	(eclass == NULL) ? NULL : (eclass = (g_type_class_unref (eclass), NULL));
}


inline gboolean dvb_utils_get_value_by_name_from_enum (GType enumtype, const char* name, gint* evalue) {
	gboolean result;
	GEnumClass* enumclass;
	GEnumValue* eval;
	g_return_val_if_fail (name != NULL, FALSE);
	enumclass = (GEnumClass*) g_type_class_ref (enumtype);
	eval = g_enum_get_value_by_name (enumclass, name);
	if (eval == NULL) {
		g_critical ("Utils.vala:43: Enum has no member named %s", name);
		result = FALSE;
		(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
		return result;
	} else {
		(*evalue) = eval->value;
		result = TRUE;
		(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
		return result;
	}
	(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
}


inline const char* dvb_utils_get_name_by_value_from_enum (GType enumtype, gint val) {
	const char* result;
	GEnumClass* enumclass;
	GEnumValue* eval;
	enumclass = (GEnumClass*) g_type_class_ref (enumtype);
	eval = g_enum_get_value (enumclass, val);
	if (eval == NULL) {
		g_critical ("Utils.vala:56: Enum has no value %d", val);
		result = NULL;
		(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
		return result;
	} else {
		result = eval->value_name;
		(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
		return result;
	}
	(enumclass == NULL) ? NULL : (enumclass = (g_type_class_unref (enumclass), NULL));
}


static void _g_slist_free_g_object_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) g_object_unref, NULL);
	g_slist_free (self);
}


void dvb_utils_mkdirs (GFile* directory, GError** error) {
	GError * _inner_error_;
	GSList* create_dirs;
	GFile* _tmp0_;
	GFile* current_dir;
	g_return_if_fail (directory != NULL);
	_inner_error_ = NULL;
	create_dirs = NULL;
	_tmp0_ = NULL;
	current_dir = (_tmp0_ = directory, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
	while (TRUE) {
		GFile* _tmp1_;
		GFile* _tmp2_;
		if (!(current_dir != NULL)) {
			break;
		}
		if (g_file_query_exists (current_dir, NULL)) {
			break;
		}
		_tmp1_ = NULL;
		create_dirs = g_slist_prepend (create_dirs, (_tmp1_ = current_dir, (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_)));
		_tmp2_ = NULL;
		current_dir = (_tmp2_ = g_file_get_parent (current_dir), (current_dir == NULL) ? NULL : (current_dir = (g_object_unref (current_dir), NULL)), _tmp2_);
	}
	{
		GSList* dir_collection;
		GSList* dir_it;
		dir_collection = create_dirs;
		for (dir_it = dir_collection; dir_it != NULL; dir_it = dir_it->next) {
			GFile* _tmp4_;
			GFile* dir;
			_tmp4_ = NULL;
			dir = (_tmp4_ = (GFile*) dir_it->data, (_tmp4_ == NULL) ? NULL : g_object_ref (_tmp4_));
			{
				char* _tmp3_;
				_tmp3_ = NULL;
				g_debug ("Utils.vala:74: Creating %s", _tmp3_ = g_file_get_path (dir));
				_tmp3_ = (g_free (_tmp3_), NULL);
				g_file_make_directory (dir, NULL, &_inner_error_);
				if (_inner_error_ != NULL) {
					g_propagate_error (error, _inner_error_);
					(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
					(create_dirs == NULL) ? NULL : (create_dirs = (_g_slist_free_g_object_unref (create_dirs), NULL));
					(current_dir == NULL) ? NULL : (current_dir = (g_object_unref (current_dir), NULL));
					return;
				}
				(dir == NULL) ? NULL : (dir = (g_object_unref (dir), NULL));
			}
		}
	}
	(create_dirs == NULL) ? NULL : (create_dirs = (_g_slist_free_g_object_unref (create_dirs), NULL));
	(current_dir == NULL) ? NULL : (current_dir = (g_object_unref (current_dir), NULL));
}


char* dvb_utils_remove_nonalphanums (const char* text) {
	char* result;
	GError * _inner_error_;
	GRegex* regex;
	char* new_text;
	g_return_val_if_fail (text != NULL, NULL);
	_inner_error_ = NULL;
	regex = NULL;
	{
		GRegex* _tmp0_;
		GRegex* _tmp1_;
		_tmp0_ = g_regex_new ("\\W", 0, 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch38_g_regex_error;
			}
			goto __finally38;
		}
		_tmp1_ = NULL;
		regex = (_tmp1_ = _tmp0_, (regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL)), _tmp1_);
	}
	goto __finally38;
	__catch38_g_regex_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			const char* _tmp2_;
			g_error ("Utils.vala:84: RegexError: %s", e->message);
			_tmp2_ = NULL;
			result = (_tmp2_ = text, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_));
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			return result;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally38:
	if (_inner_error_ != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	new_text = NULL;
	{
		char* _tmp3_;
		char* _tmp4_;
		_tmp3_ = g_regex_replace_literal (regex, text, (glong) (-1), 0, "_", 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_REGEX_ERROR) {
				goto __catch39_g_regex_error;
			}
			goto __finally39;
		}
		_tmp4_ = NULL;
		new_text = (_tmp4_ = _tmp3_, new_text = (g_free (new_text), NULL), _tmp4_);
	}
	goto __finally39;
	__catch39_g_regex_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			const char* _tmp5_;
			g_error ("Utils.vala:92: RegexError: %s", e->message);
			_tmp5_ = NULL;
			result = (_tmp5_ = text, (_tmp5_ == NULL) ? NULL : g_strdup (_tmp5_));
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
			new_text = (g_free (new_text), NULL);
			return result;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally39:
	if (_inner_error_ != NULL) {
		(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
		new_text = (g_free (new_text), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	result = new_text;
	(regex == NULL) ? NULL : (regex = (g_regex_unref (regex), NULL));
	return result;
}


/**
     * @returns: Difference in seconds
     */
inline gint64 dvb_utils_difftime (const struct tm* t1, const struct tm* t2) {
	gint64 result;
	gint64 ts1;
	gint64 ts2;
	gint64 diff;
	ts1 = (gint64) mktime (&(*t1));
	ts2 = (gint64) mktime (&(*t2));
	diff = ts1 - ts2;
	if (diff < 0) {
		result = (-1) * diff;
		return result;
	} else {
		result = diff;
		return result;
	}
}


/**
     * Creates Time of local time
     */
inline struct tm dvb_utils_create_time (gint year, gint month, gint day, gint hour, gint minute, gint second) {
	struct tm result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	time_t _tmp7_ = {0};
	time_t _tmp5_ = {0};
	struct tm _tmp6_ = {0};
	struct tm t;
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (year >= 1900) {
		_tmp4_ = month >= 1;
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		_tmp3_ = day >= 1;
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		_tmp2_ = hour >= 0;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = minute >= 0;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = second >= 0;
	} else {
		_tmp0_ = FALSE;
	}
	g_assert (_tmp0_);
	/* Create Time with some initial value, otherwise time is wrong*/
	t = (localtime_r ((_tmp7_ = (time (&_tmp5_), _tmp5_), &_tmp7_), &_tmp6_), _tmp6_);
	t.tm_year = year - 1900;
	t.tm_mon = month - 1;
	t.tm_mday = day;
	t.tm_hour = hour;
	t.tm_min = minute;
	t.tm_sec = second;
	result = t;
	return result;
}


/**
     * Creates Time of UTC time
     */
inline struct tm dvb_utils_create_utc_time (gint year, gint month, gint day, gint hour, gint minute, gint second) {
	struct tm result;
	gboolean _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	gboolean _tmp3_;
	gboolean _tmp4_;
	time_t _tmp7_ = {0};
	time_t _tmp5_ = {0};
	struct tm _tmp6_ = {0};
	struct tm t;
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	_tmp3_ = FALSE;
	_tmp4_ = FALSE;
	if (year >= 1900) {
		_tmp4_ = month >= 1;
	} else {
		_tmp4_ = FALSE;
	}
	if (_tmp4_) {
		_tmp3_ = day >= 1;
	} else {
		_tmp3_ = FALSE;
	}
	if (_tmp3_) {
		_tmp2_ = hour >= 0;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = minute >= 0;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = second >= 0;
	} else {
		_tmp0_ = FALSE;
	}
	g_assert (_tmp0_);
	/* Create Time with some initial value, otherwise time is wrong*/
	t = (gmtime_r ((_tmp7_ = (time (&_tmp5_), _tmp5_), &_tmp7_), &_tmp6_), _tmp6_);
	t.tm_year = year - 1900;
	t.tm_mon = month - 1;
	t.tm_mday = day;
	t.tm_hour = hour;
	t.tm_min = minute;
	t.tm_sec = second;
	t.tm_isdst = -1;
	/* undefined*/
	result = t;
	return result;
}


/* TODO throw error*/
char* dvb_utils_read_file_contents (GFile* file, GError** error) {
	char* result;
	GError * _inner_error_;
	char* attrs;
	GFileInfo* info;
	GFileInputStream* stream;
	GString* sb;
	gchar* _tmp6_;
	gint buffer_size;
	gint buffer_length1;
	gchar* buffer;
	glong bytes_read;
	const char* _tmp9_;
	g_return_val_if_fail (file != NULL, NULL);
	_inner_error_ = NULL;
	attrs = g_strdup_printf ("%s,%s", G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_ATTRIBUTE_ACCESS_CAN_READ);
	info = NULL;
	{
		GFileInfo* _tmp0_;
		GFileInfo* _tmp1_;
		_tmp0_ = g_file_query_info (file, attrs, 0, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch40_g_error;
			goto __finally40;
		}
		_tmp1_ = NULL;
		info = (_tmp1_ = _tmp0_, (info == NULL) ? NULL : (info = (g_object_unref (info), NULL)), _tmp1_);
	}
	goto __finally40;
	__catch40_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Utils.vala:166: Could not retrieve attributes: %s", e->message);
			result = NULL;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			return result;
		}
	}
	__finally40:
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return NULL;
	}
	if (g_file_info_get_file_type (info) != G_FILE_TYPE_REGULAR) {
		char* _tmp2_;
		_tmp2_ = NULL;
		g_critical ("Utils.vala:171: %s is not a regular file", _tmp2_ = g_file_get_path (file));
		_tmp2_ = (g_free (_tmp2_), NULL);
		result = NULL;
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return result;
	}
	if (!g_file_info_get_attribute_boolean (info, G_FILE_ATTRIBUTE_ACCESS_CAN_READ)) {
		char* _tmp3_;
		_tmp3_ = NULL;
		g_critical ("Utils.vala:176: Cannot read %s", _tmp3_ = g_file_get_path (file));
		_tmp3_ = (g_free (_tmp3_), NULL);
		result = NULL;
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		return result;
	}
	stream = NULL;
	{
		GFileInputStream* _tmp4_;
		GFileInputStream* _tmp5_;
		_tmp4_ = g_file_read (file, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch41_g_error;
			goto __finally41;
		}
		_tmp5_ = NULL;
		stream = (_tmp5_ = _tmp4_, (stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL)), _tmp5_);
	}
	goto __finally41;
	__catch41_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_critical ("Utils.vala:184: Could not read file: %s", e->message);
			result = NULL;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
			return result;
		}
	}
	__finally41:
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
		return NULL;
	}
	sb = g_string_new ("");
	_tmp6_ = NULL;
	buffer = (_tmp6_ = g_new0 (gchar, DVB_UTILS_BUFFER_SIZE), buffer_length1 = DVB_UTILS_BUFFER_SIZE, buffer_size = buffer_length1, _tmp6_);
	bytes_read = 0L;
	while (TRUE) {
		gssize _tmp7_;
		_tmp7_ = g_input_stream_read ((GInputStream*) stream, buffer, (gsize) DVB_UTILS_BUFFER_SIZE, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			attrs = (g_free (attrs), NULL);
			(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
			(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
			(sb == NULL) ? NULL : (sb = (g_string_free (sb, TRUE), NULL));
			buffer = (g_free (buffer), NULL);
			return NULL;
		}
		if (!((bytes_read = (glong) _tmp7_) > 0)) {
			break;
		}
		{
			gint i;
			i = 0;
			{
				gboolean _tmp8_;
				_tmp8_ = TRUE;
				while (TRUE) {
					if (!_tmp8_) {
						i++;
					}
					_tmp8_ = FALSE;
					if (!(i < bytes_read)) {
						break;
					}
					g_string_append_c (sb, buffer[i]);
				}
			}
		}
	}
	g_input_stream_close ((GInputStream*) stream, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		attrs = (g_free (attrs), NULL);
		(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
		(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
		(sb == NULL) ? NULL : (sb = (g_string_free (sb, TRUE), NULL));
		buffer = (g_free (buffer), NULL);
		return NULL;
	}
	_tmp9_ = NULL;
	result = (_tmp9_ = sb->str, (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
	attrs = (g_free (attrs), NULL);
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
	(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
	(sb == NULL) ? NULL : (sb = (g_string_free (sb, TRUE), NULL));
	buffer = (g_free (buffer), NULL);
	return result;
	attrs = (g_free (attrs), NULL);
	(info == NULL) ? NULL : (info = (g_object_unref (info), NULL));
	(stream == NULL) ? NULL : (stream = (g_object_unref (stream), NULL));
	(sb == NULL) ? NULL : (sb = (g_string_free (sb, TRUE), NULL));
	buffer = (g_free (buffer), NULL);
}


void dvb_utils_delete_dir_recursively (GFile* dir, GError** error) {
	GError * _inner_error_;
	char* attrs;
	GFileEnumerator* files;
	GFileEnumerator* _tmp0_;
	GFileEnumerator* _tmp1_;
	GFileInfo* childinfo;
	char* _tmp5_;
	g_return_if_fail (dir != NULL);
	_inner_error_ = NULL;
	attrs = g_strdup_printf ("%s,%s", G_FILE_ATTRIBUTE_STANDARD_TYPE, G_FILE_ATTRIBUTE_STANDARD_NAME);
	files = NULL;
	_tmp0_ = g_file_enumerate_children (dir, attrs, 0, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		attrs = (g_free (attrs), NULL);
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		return;
	}
	_tmp1_ = NULL;
	files = (_tmp1_ = _tmp0_, (files == NULL) ? NULL : (files = (g_object_unref (files), NULL)), _tmp1_);
	if (files == NULL) {
		attrs = (g_free (attrs), NULL);
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		return;
	}
	childinfo = NULL;
	while (TRUE) {
		GFileInfo* _tmp2_;
		GFileInfo* _tmp3_;
		guint32 type;
		GFile* child;
		_tmp2_ = g_file_enumerator_next_file (files, NULL, &_inner_error_);
		if (_inner_error_ != NULL) {
			g_propagate_error (error, _inner_error_);
			attrs = (g_free (attrs), NULL);
			(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
			(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
			return;
		}
		_tmp3_ = NULL;
		if (!((childinfo = (_tmp3_ = _tmp2_, (childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL)), _tmp3_)) != NULL)) {
			break;
		}
		type = g_file_info_get_attribute_uint32 (childinfo, G_FILE_ATTRIBUTE_STANDARD_TYPE);
		child = g_file_get_child (dir, g_file_info_get_name (childinfo));
		switch (type) {
			case G_FILE_TYPE_DIRECTORY:
			{
				dvb_utils_delete_dir_recursively (child, &_inner_error_);
				if (_inner_error_ != NULL) {
					g_propagate_error (error, _inner_error_);
					(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
					attrs = (g_free (attrs), NULL);
					(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
					(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
					return;
				}
				break;
			}
			case G_FILE_TYPE_REGULAR:
			{
				char* _tmp4_;
				_tmp4_ = NULL;
				g_debug ("Utils.vala:223: Deleting file %s", _tmp4_ = g_file_get_path (child));
				_tmp4_ = (g_free (_tmp4_), NULL);
				g_file_delete (child, NULL, &_inner_error_);
				if (_inner_error_ != NULL) {
					g_propagate_error (error, _inner_error_);
					(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
					attrs = (g_free (attrs), NULL);
					(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
					(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
					return;
				}
				break;
			}
		}
		(child == NULL) ? NULL : (child = (g_object_unref (child), NULL));
	}
	_tmp5_ = NULL;
	g_debug ("Utils.vala:229: Deleting directory %s", _tmp5_ = g_file_get_path (dir));
	_tmp5_ = (g_free (_tmp5_), NULL);
	g_file_delete (dir, NULL, &_inner_error_);
	if (_inner_error_ != NULL) {
		g_propagate_error (error, _inner_error_);
		attrs = (g_free (attrs), NULL);
		(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
		(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
		return;
	}
	attrs = (g_free (attrs), NULL);
	(files == NULL) ? NULL : (files = (g_object_unref (files), NULL));
	(childinfo == NULL) ? NULL : (childinfo = (g_object_unref (childinfo), NULL));
}


inline DBusGConnection* dvb_utils_get_dbus_connection (void) {
	DBusGConnection* result;
	GError * _inner_error_;
	DBusGConnection* conn;
	_inner_error_ = NULL;
	conn = NULL;
	{
		DBusGConnection* _tmp0_;
		DBusGConnection* _tmp1_;
		_tmp0_ = dbus_g_bus_get (DBUS_BUS_SESSION, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch42_g_error;
			goto __finally42;
		}
		_tmp1_ = NULL;
		conn = (_tmp1_ = _tmp0_, (conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL)), _tmp1_);
	}
	goto __finally42;
	__catch42_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_error ("Utils.vala:238: Could not get D-Bus session bus: %s", e->message);
			result = NULL;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
			return result;
		}
	}
	__finally42:
	if (_inner_error_ != NULL) {
		(conn == NULL) ? NULL : (conn = (dbus_g_connection_unref (conn), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	result = conn;
	return result;
}




