# Infrared Remote Control Properties for GNOME
# Copyright (C) 2008 Fluendo Embedded S.L. (www.fluendo.com)
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
'''
Facilities for handling the D-BUS driven configuration backend.
'''

import dbus, dbus.service, gobject, shutil
import errno, os, os.path, re, pty, signal, tempfile

from gettext               import gettext as _
from gnome_lirc_properties import config, lirc
from StringIO              import StringIO

# Modern flavors of dbus bindings have that symbol in dbus.lowlevel,
# for old flavours the internal _dbus_bindings module must be used.

try:
    from dbus.lowlevel import HANDLER_RESULT_NOT_YET_HANDLED
except ImportError:
    from _dbus_bindings import HANDLER_RESULT_NOT_YET_HANDLED

class AccessDeniedException(dbus.DBusException):
    '''
    This exception is raised when some operation is not permitted.
    '''

    _dbus_error_name = 'org.gnome.LircProperties.AccessDeniedException'

class UnsupportedException(dbus.DBusException):
    '''
    This exception is raised when some operation is not supported.
    '''

    _dbus_error_name = 'org.gnome.LircProperties.UnsupportedException'

class UsageError(dbus.DBusException):
    '''
    This exception is raised when some operation was not used properly.
    '''

    _dbus_error_name = 'org.gnome.LircProperties.UsageError'

class PolicyKitService(dbus.service.Object):
    def _check_permission(self, sender, action=config.POLICY_KIT_ACTION):
        '''
        Verifies if the specified action is permitted, and raises
        an AccessDeniedException if not.

        The caller should use ObtainAuthorization() to get permission.
        '''

        try:
            if sender:
                kit = dbus.SystemBus().get_object('org.freedesktop.PolicyKit', '/')
                pid = dbus.UInt32(os.getpid())

                if 'yes' != kit.IsProcessAuthorized(action, pid, False):
                    raise AccessDeniedException('Process not authorized by PolicyKit')
                if 'yes' != kit.IsSystemBusNameAuthorized(action, sender, False):
                    raise AccessDeniedException('Session not authorized by PolicyKit')

        except AccessDeniedException:
            raise

        except dbus.DBusException, ex:
            raise AccessDeniedException(ex.message)

class ExternalToolDriver(PolicyKitService):
    INTERFACE_NAME = 'org.gnome.LircProperties.ExternalToolDriver'

    def __spawn_external_tool(self):
        pid, fd = pty.fork()

        if 0 == pid:
            self._on_run_external_tool()
            assert False # should not be reached

        os.waitpid(pid, os.P_NOWAIT)

        return pid, fd

    def __io_handler(self, fd, condition):
        if condition & gobject.IO_IN:
            chunk = os.read(fd, 4096)
            self._on_next_chunk(chunk)

            self.__line_buffer += chunk

            while True:
                linebreak = self.__line_buffer.find('\n')

                if linebreak < 0: break

                line = self.__line_buffer[:linebreak].rstrip()
                self.__line_buffer = self.__line_buffer[linebreak + 1:]

                self._on_next_line(line)

        if condition & gobject.IO_HUP:
            self._on_hangup()
            return False

        return True

    def _send_response(self, response='\n'):
        os.write(self.__fd, response)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='', out_signature='',
                         sender_keyword='sender')
    def Execute(self, sender=None):
        self._check_permission(sender)

        self.__line_buffer = ''
        self.__pid, self.__fd = self.__spawn_external_tool()

        gobject.io_add_watch(self.__fd,
                             gobject.IO_IN | gobject.IO_HUP,
                             self.__io_handler)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='', out_signature='',
                         sender_keyword='sender')
    def Proceed(self, sender=None):
        self._check_permission(sender)
        self._send_response('\n')

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='', out_signature='',
                         sender_keyword='sender')
    def Release(self, sender=None):
        self._check_permission(sender)

        try:
            if -1 != os.waitpid(self.__pid, os.P_NOWAIT):
                print 'Terminating child process %d...' % self.__pid
                os.kill(self.__pid, signal.SIGTERM)

        except OSError, ex:
            if ex.errno != errno.ESRCH:
                print 'Cannot terminate process %d: %s' % (self.__pid, ex.message)

        self.__pid = 0
        self.remove_from_connection()

    @dbus.service.signal(dbus_interface=INTERFACE_NAME, signature='')
    def ReportProgress(self): pass
    @dbus.service.signal(dbus_interface=INTERFACE_NAME, signature='s')
    def ReportSuccess(self, message): pass
    @dbus.service.signal(dbus_interface=INTERFACE_NAME, signature='s')
    def ReportFailure(self, message): pass
    @dbus.service.signal(dbus_interface=INTERFACE_NAME, signature='ss')
    def RequestAction(self, title, details): pass

    def _on_run_external_tool(self): pass
    def _on_next_chunk(self, line): pass
    def _on_next_line(self, line): pass
    def _on_hangup(self): pass

    _pid = property(lambda self: self.__pid)

class IrRecordDriver(ExternalToolDriver):
    # Following strings are some known error messages of irrecord 0.5,
    # as shipped with Ubuntu 7.10 on 2008-02-13:

    _errors = {
        'could not init hardware': _('Could not initialize hardware.'),
        'gap not found, can\'t continue': _('No key presses recognized, gap not found.'),
        'no data for 10 secs, aborting': _('No key presses recognized, aborting.'),
    }

    # Following strings indicate state changes in irrecord 0.5,
    # as shipped with Ubuntu 7.10 on 2008-02-13:

    _token_intro_text       = 'This program will record the signals'
    _token_hold_one_button  = 'Hold down an arbitrary button.'
    _token_random_buttons   = 'Now start pressing buttons on your remote control.'
    _token_next_key         = 'Please enter the name for the next button'
    _token_wait_toggle_mask = 'If you can\'t see any dots appear'
    _token_finished         = 'Successfully written config file.'

    # Last instance index for automatic object path creation:
    __last_instance = 0

    def __init__(self, connection, driver, device,
                 filename='lircd.conf', path=None):
        assert not os.path.isabs(filename)

        if not path:
            IrRecordDriver.__last_instance += 1
            path = '/IrRecordDriver%d' % self.__last_instance

        self._workdir = tempfile.mkdtemp(prefix='gnome-lirc-properties-')
        self._cmdargs = [config.LIRC_IRRECORD, '--driver=%s' % driver]
        self._filename = filename

        if device:
            self._cmdargs.append('--device=%s' % device)
        if filename:
            self._cmdargs.append(filename)

        super(IrRecordDriver, self).__init__(connection, path)

    def _on_run_external_tool(self):
        os.chdir(self._workdir)
        self._prepare_workdir()

        args, env = self._cmdargs, {'LC_ALL': 'C'}
        print 'running %s in %s...' % (args[0], self._workdir)
        print 'arguments: %r' % args[1:]

        os.execve(args[0], filter(None, args), env)

    def _prepare_workdir(self):
        pass

    def _cleanup_workdir(self):
        print 'cleaning up %s...' % self._workdir

        for name in os.listdir(self._workdir):
            os.unlink(os.path.join(self._workdir, name))

    def _on_hangup(self):
        if list(self.locations):
            self.ReportFailure(_('Custom remote configuration aborted unexpectedly.'))
            self.Release()

    def _find_error_messages(self, line):
        for token, message in self._errors.items():
            if line.find(token) >= 0:
                self.ReportFailure(message)
                self.Release()
                return True

        return False

    def __del__(self):
        self._cleanup_workdir()
        os.rmdir(self._workdir)

class DetectParametersDriver(IrRecordDriver):
    def __init__(self, connection, driver, device):
        super(DetectParametersDriver, self).__init__(connection, driver, device)
        self.__report_progress = False

    def _prepare_workdir(self):
        if os.path.exists(self._filename):
            os.unlink(self._filename)

    def _on_next_line(self, line):
        print '%d:%s' % (self._pid, line)

        # Identify known error messages:

        if self._find_error_messages(line):
            return

        # Try to catch state changes:

        if line.startswith(self._token_intro_text):
            self._send_response()
            return

        if line.startswith(self._token_hold_one_button):
            self.RequestAction(_('Hold down an arbitrary button.'), '')
            return

        if line.startswith(self._token_random_buttons):
            self.RequestAction(
                _('Press random buttons on your remote control.'),
                _('It is very important that you press many different ' +
                  'buttons and hold them down for approximately one ' +
                  'second. Each button should move the progress bar ' +
                  'by at least one, but in no case by more than ten ' +
                  'steps.'))

            return

        if line.startswith(self._token_next_key):
            self._send_response()
            return

        if line.startswith(self._token_wait_toggle_mask):
            self.RequestAction(
                _('Press a button repeatedly as fast as possible.'),
                _('Make sure you keep pressing the <b>same</b> button and that you ' +
                  '<b>do not hold</b> the button down!\nWait a bit between button ' +
                  'presses, if you can not see any progress.'))

            return

        if line.startswith(self._token_finished):
            filename = os.path.join(self._workdir, self._filename)
            configuration = open(filename).read()
            self.ReportSuccess(configuration)
            self.Release()
            return

    def _on_next_chunk(self, chunk):
        if '.' == chunk:
            self.ReportProgress()

class LearnKeyCodeDriver(IrRecordDriver):
    def __init__(self, connection, driver, device, configuration, keys):
        super(LearnKeyCodeDriver, self).__init__(connection, driver, device)
        self.__keys, self.__configuration = keys, configuration

    def _prepare_workdir(self):
        open(self._filename, 'w').write(self.__configuration)

    def _on_next_line(self, line):
        print '%d:%s' % (self._pid, line)

        # Identify known error messages:

        if self._find_error_messages(line):
            return

        # Try to catch state changes:

        if line.startswith(self._token_intro_text):
            self._send_response()
            return

        if line.startswith(self._token_next_key):
            if not self.__keys:
                self._send_response()
                return

            self._send_response('%s\n' % self.__keys.pop(0))
            return

        if line.startswith(self._token_finished):
            configuration = self._find_configuration()

            if configuration: self.ReportSuccess(configuration)
            else: self.ReportFailure(_('Cannot find recorded key codes'))

            self.Release()
            return

    def _find_configuration(self):
        for suffix in '.new', '.conf', '':
            filename = os.path.join(self._workdir, self._filename + suffix)

            if os.path.isfile(filename):
                return open(filename).read()

        return None

    def _on_run_external_tool(self):
        irrecord = os.popen('%s --help' % config.LIRC_IRRECORD)

        if -1 == irrecord.read().find('--resume'):
            # TODO: Should be an UnsupportedException in LearnKeyCode(),
            # but when raising that exception from there, the execeptions
            # message property also contains a stack trace.
            self.ReportFailure(_(
                'Installed lirc package doesn\'t support self-contained ' +
                'key code yet. Update to latest lirc package if possible.'))

        self._cmdargs.insert(1, '--resume')

        super(LearnKeyCodeDriver, self)._on_run_external_tool()


class BackendService(PolicyKitService):
    '''
    A D-Bus service that PolicyKit controls access to.
    '''

    INTERFACE_NAME = 'org.gnome.LircProperties.Mechanism'
    SERVICE_NAME   = 'org.gnome.LircProperties.Mechanism'
    IDLE_TIMEOUT   =  30

    # These are extra fields set by our GUI:

    __re_receiver_directive = re.compile(r'^\s*RECEIVER_(VENDOR|MODEL)=')

    # These are used by the Debian/Ubuntu packages, as of 2008-02-12.
    # The "REMOTE_" prefix is made optional, since it only was introduced
    # with lirc 0.8.3~pre1-0ubuntu4 of Hardy Heron.

    __re_remote_directive = re.compile(r'^\s*(?:REMOTE_)?(DRIVER|DEVICE|MODULES|' +
                                       r'LIRCD_ARGS|LIRCD_CONF|VENDOR|MODEL)=')
    __re_start_lircd      = re.compile(r'^\s*START_LIRCD=')

    def __init__(self, connection, path='/'):
        super(BackendService, self).__init__(connection, path)

        self.__loop = gobject.MainLoop()
        self.__timeout = 0

        connection.add_message_filter(self.__message_filter)

    def __message_filter(self, connection, message):
        if self.__timeout: self.__start_idle_timeout()
        return HANDLER_RESULT_NOT_YET_HANDLED

    def __start_idle_timeout(self):
        if self.__timeout:
            gobject.source_remove(self.__timeout)

        self.__timeout = gobject.timeout_add(self.IDLE_TIMEOUT * 1000,
                                             self.__timeout_cb)

    def __timeout_cb(self):
        # Keep service alive, as long as additional objects are exported:
        if self.connection.list_exported_child_objects('/'):
            return True

        print 'Terminating %s due inactivity.' % self.SERVICE_NAME
        self.__loop.quit()

        return False

    def run(self):
        '''
        Creates a GLib main loop for keeping the service alive.
        '''

        print 'Running %s.' % self.SERVICE_NAME
        print 'Terminating it after %d seconds of inactivity.' % self.IDLE_TIMEOUT

        self.__start_idle_timeout()
        self.__loop.run()

    def _write_hardware_configuration(self, remote_values=None, receiver_values=None):
        if not remote_values: remote_values = dict()
        if not receiver_values: receiver_values = dict()

        oldfile = os.path.join(config.LIRC_CONFDIR, 'hardware.conf')
        newfile = '%s.tmp' % oldfile

        if not os.path.isfile(oldfile):
            raise UnsupportedException('Cannot find %s script' % oldfile)

        print 'Updating %s...' % oldfile

        output = file(newfile, 'w')
        for line in file(oldfile, 'r'):

            # Deal with lines starting with REMOTE_,
            # replacing their values with ours:

            match = self.__re_remote_directive.match(line)

            if match:
                # Remove entry from the dict, so we know what was written.
                value = remote_values.pop(match.group(1), None)

                if None != value:
                    print match.group(0), value
                    print >>output, ('%s"%s"' % (match.group(0), value))
                    continue

            # Deal with lines starting with RECEIVER_,
            # replacing their values with ours:

            match = self.__re_receiver_directive.match(line)

            if match:
                # Remove entry from the dict, so we know what was written.
                value = receiver_values.pop(match.group(1), None)

                if None != value:
                    print >>output, ('%s"%s"' % (match.group(0), value))
                    continue

            # Deal with the START_LIRCD line:
            # Unconditionally set it to "true" since we rely on lircd starting up.

            match = self.__re_start_lircd.match(line)

            if match:
                print >>output, ('%strue' % match.group(0))
                continue

            output.write(line)

        # Write out any values that were not already in the file,
        # and therefore just replaced:

        if remote_values:
            print >>output, '\n# Remote settings required by gnome-lirc-properties'
        for key, value in remote_values.items():
            print >>output, ('REMOTE_%s="%s"' % (key, value))

        if receiver_values:
            print >>output, '\n# Receiver settings required by gnome-lirc-properties'
        for key, value in receiver_values.items():
            print >>output, ('RECEIVER_%s="%s"' % (key, value))

        # Replace old file with new contents:

        os.unlink(oldfile)
        os.rename(newfile, oldfile)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='sssss', out_signature='',
                         sender_keyword='sender')
    def WriteReceiverConfiguration(self, vendor, product,
                                   driver, device, modules,
                                   sender=None):
        '''
        Update the /etc/lirc/hardware.conf file, so that lircd is started as specified.
        '''

        self._check_permission(sender)

        remote_values = {
            'DRIVER': driver,
            'DEVICE': device,
            'MODULES': modules,
            'LIRCD_ARGS': '',
            'LIRCD_CONF': '',
        }

        receiver_values = {
            'VENDOR': vendor,
            'MODEL': product,
        }

        self._write_hardware_configuration(remote_values, receiver_values)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='s', out_signature='',
                         sender_keyword='sender')
    def WriteRemoteConfiguration(self, contents, sender=None):
        '''
        Write the contents to the system lircd.conf file.
        PolicyKit will not allow this function to be called without sudo/root
        access, and will ask the user to authenticate if necessary, when
        the application calls PolicyKit's ObtainAuthentication().
        '''

        self._check_permission(sender)

        if not contents:
            raise UsageError('Bad IR remote configuration file')

        # Parse contents:

        hwdb = lirc.RemotesDatabase()
        hwdb.read(StringIO(contents))
        remote = len(hwdb) and hwdb[0]

        # Update hardware.conf with choosen remote:

        if remote:
            values = {
                'VENDOR': remote.vendor or _('Unknown'),
                'MODEL': remote.product or remote.name
            }

            self._write_hardware_configuration(remote_values=values)

        # Write remote configuration:

        filename = os.path.join(config.LIRC_CONFDIR, 'lircd.conf')

        print 'Updating %s...' % filename
        file(filename, 'w').write(contents)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='', out_signature='',
                         sender_keyword='sender')
    def ManageLircDaemon(self, action, sender=None):
        '''
        Starts the LIRC daemon.
        '''

        self._check_permission(sender)

        print 'Managing lircd: %s...' % action
        args = '/etc/init.d/lirc', action

        os.spawnv(os.P_WAIT, args[0], args)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='ss', out_signature='o',
                         sender_keyword='sender')
    def DetectParameters(self, driver, device, sender=None):
        '''
        Detects parameters of the IR remote by running irrecord.
        '''

        self._check_permission(sender)

        return DetectParametersDriver(self.connection, driver, device)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='sssas', out_signature='o',
                         sender_keyword='sender')
    def LearnKeyCode(self, driver, device, configuration, keys, sender=None):
        '''
        Learn the scan code of some IR remote key by running irrecord.
        '''

        self._check_permission(sender)

        return LearnKeyCodeDriver(self.connection,
                                  driver, device,
                                  configuration,
                                  keys)

    @dbus.service.method(dbus_interface=INTERFACE_NAME,
                         in_signature='s', out_signature='',
                         sender_keyword='sender')
    def InstallRemoteDatabase(self, filename, sender=None):
        '''
        Update the customized receiver database.
        '''

        # Copy the tarball with updates into our data folder
        # to allow atomic replacement of the old tarball:
        tarball = os.path.join(config.PACKAGE_DIR, 'remotes-update.tar.gz')
        shutil.copyfile(filename, tarball)

        # Now practice the atomic replacement:
        os.rename(tarball, config.LIRC_REMOTES_TARBALL)

def get_service_bus():
    '''
    Retrieves a reference to the D-BUS system bus.
    '''

    return dbus.SystemBus()

def get_service(bus=None):
    '''
    Retrieves a reference to the D-BUS driven configuration service.
    '''

    if not bus:
        bus = get_service_bus()

    service = bus.get_object(BackendService.SERVICE_NAME, '/')

    return service

if __name__ == '__main__':
    # Integrate DBus with GLib main loops.

    from dbus.mainloop.glib import DBusGMainLoop
    DBusGMainLoop(set_as_default=True)

    # Get system bus and acquire service name.

    bus = get_service_bus()
    name = dbus.service.BusName(BackendService.SERVICE_NAME, bus)

    # Run the service.

    BackendService(bus).run()

