#ifndef __ADDRESS_CONDUIT_H__
#define __ADDRESS_CONDUIT_H__

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <pi-appinfo.h>
#include <pi-address.h>
#include <glib.h>
#include <gnome.h>
#include <errno.h>

#include <gpilotd/gnome-pilot-conduit.h>
#include <gpilotd/gnome-pilot-conduit-standard-abs.h>

#include "card.h"

#define CC_DEBUG

#ifdef CC_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "address: "##format, ##args)
#else
#define LOG(format,args...)
#endif

#define OBJ_DATA_CONDUIT "conduit_data"
#define OBJ_DATA_CONFIG  "conduit_config"
#define CONFIG_PREFIX    "/gnome-pilot.d/address-conduit/Pilot_%u/"


typedef struct _GCardLocalRecord GCardLocalRecord;

struct _GCardLocalRecord {
	LocalRecord      local;
	GCardLocalRecord *next;
	Card             *gcard;
	struct Address   *addr;
	gint              category;
};

typedef struct _ConduitData ConduitData;
struct _ConduitData {
	struct AddressAppInfo ai;
	GList *records;
	GnomePilotDBInfo *dbi;
};

#define GET_CONDUIT_CFG(s) ((ConduitCfg*)gtk_object_get_data(GTK_OBJECT(s),OBJ_DATA_CONFIG))
#define GET_CONDUIT_DATA(s) ((ConduitData*)gtk_object_get_data(GTK_OBJECT(s),OBJ_DATA_CONDUIT))

typedef struct IterateData {
	int flag;
	int archived;
	GCardLocalRecord *prev;
	GCardLocalRecord *first;
} IterateData;

typedef struct LoadInfo {
	gint id;
	gint secret;
	time_t mtime;
} LoadInfo;


typedef struct _ConduitCfg ConduitCfg;
struct _ConduitCfg 
{
	GnomePilotConduitSyncType  sync_type;   /* only used by capplet */
	guint32  pilotId;
	gchar   *filename;
	gboolean open_secret;
};


static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar prefix[256];
	g_return_if_fail(c!=NULL);
	
	g_snprintf(prefix,255,CONFIG_PREFIX,pilotId);
 
	*c = g_new0(ConduitCfg,1);
	gnome_config_push_prefix(prefix);
	(*c)->sync_type = GnomePilotConduitSyncTypeCustom; /* this will be reset by capplet */
	(*c)->filename = gnome_config_get_string("filename");
	(*c)->open_secret = gnome_config_get_bool("open_secret=FALSE");
	gnome_config_pop_prefix();
	
	if((*c)->filename==NULL) {
		(*c)->filename=g_strdup_printf("%s/pilot/Address.gcrd",g_get_home_dir());
	} 
	(*c)->pilotId = pilotId;
}

static void 
save_configuration(ConduitCfg *c) 
{
	gchar prefix[256];

	g_return_if_fail(c!=NULL);
	g_snprintf(prefix,255,CONFIG_PREFIX,c->pilotId);
	
	gnome_config_push_prefix(prefix);
	gnome_config_set_string("filename",c->filename);
	gnome_config_set_bool("open_secret",c->open_secret);
	gnome_config_pop_prefix();
	
	gnome_config_sync();
	gnome_config_drop_all();
}

static void 
copy_configuration(ConduitCfg *d, ConduitCfg *c)
{
	g_return_if_fail(c!=NULL);
	g_return_if_fail(d!=NULL);
	d->sync_type=c->sync_type;
	if(d->filename) g_free(d->filename);
	d->filename = g_strdup(c->filename);
	d->open_secret = c->open_secret;
	d->pilotId = c->pilotId;
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) 
{
	ConduitCfg *d;
	g_return_val_if_fail(c!=NULL,NULL);
	d = g_new0(ConduitCfg,1);
	copy_configuration(d,c);
	return d;
}


/** this method frees all data from the conduit config */
static void 
destroy_configuration(ConduitCfg **c) 
{
	g_return_if_fail(c!=NULL);
	if((*c)->filename) g_free((*c)->filename);
	g_free(*c);
	*c = NULL;
}

#endif
