/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 *
 * GNOME Power Manager Brightness Applet
 * Copyright (C) 2006 Benjamin Canou <bookeldor@gmail.com>
 * Copyright (C) 2007 Richard Hughes <richard@hughsie.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <panel-applet.h>
#include <gtk/gtk.h>
#include <gtk/gtkbox.h>
#include <libgnomeui/gnome-help.h>
#include <gdk/gdkkeysyms.h>
#include <libdbus-watch.h>

#include "brightness-applet.h"
#include "../src/gpm-common.h"

static void      gpm_brightness_applet_class_init (GpmBrightnessAppletClass *klass);
static void      gpm_brightness_applet_init       (GpmBrightnessApplet *applet);

G_DEFINE_TYPE (GpmBrightnessApplet, gpm_brightness_applet, PANEL_TYPE_APPLET)

static void      gpm_applet_get_icon              (GpmBrightnessApplet *applet);
static void      gpm_applet_check_size            (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_draw_cb               (GpmBrightnessApplet *applet);
static void      gpm_applet_change_background_cb  (GpmBrightnessApplet *applet,
						   PanelAppletBackgroundType arg1,
						   GdkColor *arg2, GdkPixmap *arg3, gpointer data);
static void      gpm_applet_theme_change_cb (GtkIconTheme *icon_theme, gpointer data);
static void      gpm_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event);
static gboolean  gpm_applet_destroy_popup_cb      (GpmBrightnessApplet *applet);
static void      gpm_applet_update_tooltip        (GpmBrightnessApplet *applet);
static void      gpm_applet_update_popup_level    (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_plus_cb               (GtkWidget *w, GpmBrightnessApplet *applet);
static gboolean  gpm_applet_minus_cb              (GtkWidget *w, GpmBrightnessApplet *applet);
static gboolean  gpm_applet_key_press_cb          (GpmBrightnessApplet *applet, GdkEventKey *event);
static gboolean  gpm_applet_scroll_cb             (GpmBrightnessApplet *applet, GdkEventScroll *event);
static gboolean  gpm_applet_slide_cb              (GtkWidget *w, GpmBrightnessApplet *applet);
static void      gpm_applet_create_popup          (GpmBrightnessApplet *applet);
static gboolean  gpm_applet_popup_cb              (GpmBrightnessApplet *applet, GdkEventButton *event);
static void      gpm_applet_dialog_about_cb       (BonoboUIComponent *uic, gpointer data, const gchar *verbname);
static gboolean  gpm_applet_bonobo_cb             (PanelApplet *_applet, const gchar *iid, gpointer data);
static void      gpm_applet_destroy_cb            (GtkObject *object);

#define GPM_BRIGHTNESS_APPLET_OAFID		"OAFIID:GNOME_BrightnessApplet"
#define GPM_BRIGHTNESS_APPLET_FACTORY_OAFID	"OAFIID:GNOME_BrightnessApplet_Factory"
#define GPM_BRIGHTNESS_APPLET_ICON		"gpm-brightness-lcd"
#define GPM_BRIGHTNESS_APPLET_ICON_DISABLED	"gpm-brightness-lcd-disabled"
#define GPM_BRIGHTNESS_APPLET_NAME		_("Power Manager Brightness Applet")
#define GPM_BRIGHTNESS_APPLET_DESC		_("Adjusts laptop panel brightness.")
#define PANEL_APPLET_VERTICAL(p)					\
	 (((p) == PANEL_APPLET_ORIENT_LEFT) || ((p) == PANEL_APPLET_ORIENT_RIGHT))

/**
 * gpm_applet_get_brightness:
 * Return value: Success value, or zero for failure
 **/
static gboolean
gpm_applet_get_brightness (GpmBrightnessApplet *applet)
{
	GError  *error = NULL;
	gboolean ret;
	guint policy_brightness;

	if (applet->proxy == NULL) {
		printf ("WARNING: not connected\n");
		return FALSE;
	}

	ret = dbus_g_proxy_call (applet->proxy, "GetBrightness", &error,
				 G_TYPE_INVALID,
				 G_TYPE_UINT, &policy_brightness,
				 G_TYPE_INVALID);
	if (error) {
		printf ("DEBUG: ERROR: %s\n", error->message);
		g_error_free (error);
	}
	if (ret == TRUE) {
		applet->level = policy_brightness;
	} else {
		/* abort as the DBUS method failed */
		printf ("WARNING: GetBrightness failed!\n");
	}

	return ret;
}

/**
 * gpm_applet_set_brightness:
 * Return value: Success value, or zero for failure
 **/
static gboolean
gpm_applet_set_brightness (GpmBrightnessApplet *applet)
{
	GError  *error = NULL;
	gboolean ret;

	if (applet->proxy == NULL) {
		printf ("WARNING: not connected");
		return FALSE;
	}

	ret = dbus_g_proxy_call (applet->proxy, "SetBrightness", &error,
				 G_TYPE_UINT, applet->level,
				 G_TYPE_INVALID,
				 G_TYPE_INVALID);
	if (error) {
		printf ("DEBUG: ERROR: %s", error->message);
		g_error_free (error);
	}
	if (ret == FALSE) {
		/* abort as the DBUS method failed */
		printf ("WARNING: SetBrightness failed!");
	}

	return ret;
}

/**
 * gpm_applet_get_icon:
 * @applet: Brightness applet instance
 *
 * retrieve an icon from stock with a size adapted to panel
 **/
static void
gpm_applet_get_icon (GpmBrightnessApplet *applet)
{
	const gchar *icon;

	/* free */
	if (applet->icon != NULL) {
		g_object_unref (applet->icon);
		applet->icon = NULL;
	}

	if (applet->size <= 2) {
		return;
	}

	/* get icon */
	if (applet->proxy == NULL) {
		icon = GPM_BRIGHTNESS_APPLET_ICON_DISABLED;
	} else if (applet->call_worked == FALSE) {
		icon = GPM_BRIGHTNESS_APPLET_ICON_DISABLED;
	} else {
		icon = GPM_BRIGHTNESS_APPLET_ICON;
	}

	applet->icon = gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
						 icon, applet->size - 2, 0, NULL);

	if (applet->icon == NULL) {
		printf ("WARNING: Cannot find %s!\n", icon);
	} else {
		printf ("DEBUG: got icon %s!\n", icon);
		/* update size cache */
		applet->icon_height = gdk_pixbuf_get_height (applet->icon);
		applet->icon_width = gdk_pixbuf_get_width (applet->icon);
	}
}

/**
 * gpm_applet_check_size:
 * @applet: Brightness applet instance
 *
 * check if panel size has changed and applet adapt size
 **/
static void
gpm_applet_check_size (GpmBrightnessApplet *applet)
{
	/* we don't use the size function here, but the yet allocated size because the
	   size value is false (kind of rounded) */
	if (PANEL_APPLET_VERTICAL(panel_applet_get_orient (PANEL_APPLET (applet)))) {
		if (applet->size != GTK_WIDGET(applet)->allocation.width) {
			applet->size = GTK_WIDGET(applet)->allocation.width;
			gpm_applet_get_icon (applet);
			gtk_widget_set_size_request (GTK_WIDGET(applet), applet->size, applet->icon_height + 2);
		}
	} else {
		if (applet->size != GTK_WIDGET(applet)->allocation.height) {
			applet->size = GTK_WIDGET(applet)->allocation.height;
			gpm_applet_get_icon (applet);
			gtk_widget_set_size_request (GTK_WIDGET(applet), applet->icon_width + 2, applet->size);
		}
	}
}

/**
 * gpm_applet_draw_cb:
 * @applet: Brightness applet instance
 *
 * draws applet content (background + icon)
 **/
static gboolean
gpm_applet_draw_cb (GpmBrightnessApplet *applet)
{
	gint w, h, bg_type;
	GdkColor color;
	GdkGC *gc;
	GdkPixmap *background;

	if (GTK_WIDGET(applet)->window == NULL) {
		return FALSE;
	}

	/* retrieve applet size */
	gpm_applet_get_icon (applet);
	gpm_applet_check_size (applet);
	if (applet->size <= 2) {
		return FALSE;
	}

	/* if no icon, then don't try to display */
	if (applet->icon == NULL) {
		return FALSE;
	}

	w = GTK_WIDGET(applet)->allocation.width;
	h = GTK_WIDGET(applet)->allocation.height;

	gc = gdk_gc_new (GTK_WIDGET(applet)->window);

	/* draw pixmap background */
	bg_type = panel_applet_get_background (PANEL_APPLET (applet), &color, &background);
	if (bg_type == PANEL_PIXMAP_BACKGROUND && !applet->popped) {
		/* fill with given background pixmap */
		gdk_draw_drawable (GTK_WIDGET(applet)->window, gc, background, 0, 0, 0, 0, w, h);
	}
	
	/* draw color background */
	if (bg_type == PANEL_COLOR_BACKGROUND && !applet->popped) {
		gdk_gc_set_rgb_fg_color (gc,&color);
		gdk_gc_set_fill (gc,GDK_SOLID);
		gdk_draw_rectangle (GTK_WIDGET(applet)->window, gc, TRUE, 0, 0, w, h);
	}

	/* fill with selected color if popped */
	if (applet->popped) {
		color = gtk_rc_get_style (GTK_WIDGET(applet))->bg[GTK_STATE_SELECTED];
		gdk_gc_set_rgb_fg_color (gc,&color);
		gdk_gc_set_fill (gc,GDK_SOLID);
		gdk_draw_rectangle (GTK_WIDGET(applet)->window, gc, TRUE, 0, 0, w, h);
	}

	/* draw icon at center */
	gdk_draw_pixbuf (GTK_WIDGET(applet)->window, gc, applet->icon,
			 0, 0, (w - applet->icon_width)/2, (h - applet->icon_height)/2,
			 applet->icon_width, applet->icon_height,
			 GDK_RGB_DITHER_NONE, 0, 0);

	return TRUE;
}

/**
 * gpm_applet_change_background_cb:
 *
 * Enqueues an expose event (don't know why it's not the default behaviour)
 **/
static void
gpm_applet_change_background_cb (GpmBrightnessApplet *applet,
				 PanelAppletBackgroundType arg1,
				 GdkColor *arg2, GdkPixmap *arg3, gpointer data)
{
	gtk_widget_queue_draw (GTK_WIDGET (applet));
}

/**
 * gpm_applet_destroy_popup_cb:
 * @applet: Brightness applet instance
 *
 * destroys the popup (called if orientation has changed)
 **/
static gboolean
gpm_applet_destroy_popup_cb (GpmBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_parent (applet->popup, NULL);
		gtk_widget_destroy (applet->popup);
		applet->popup = NULL;
	}
	return TRUE;
}

/**
 * gpm_applet_update_tooltip:
 * @applet: Brightness applet instance
 *
 * sets tooltip's content (percentage or disabled)
 **/
static void
gpm_applet_update_tooltip (GpmBrightnessApplet *applet)
{
	static gchar buf[101];
	if (applet->popped == FALSE) {
		if (applet->proxy == NULL) {
			snprintf (buf, 100, _("Cannot connect to gnome-power-manager"));
		} else if (applet->call_worked == FALSE) {
			snprintf (buf, 100, _("Cannot get laptop panel brightness"));
		} else {
			snprintf (buf, 100, _("LCD brightness : %d%%"), applet->level);
		}
		gtk_tooltips_set_tip (applet->tooltip, GTK_WIDGET(applet), buf, NULL);
	} else {
		gtk_tooltips_set_tip (applet->tooltip, GTK_WIDGET(applet), NULL, NULL);
	}
}

/**
 * gpm_applet_update_popup_level:
 * @applet: Brightness applet instance
 * @get_hw: set UI value from HW value
 * @set_hw: set HW value from UI value
 *
 * updates popup and hardware level of brightness
 * FALSE FAlSE -> set UI from cached value
 * TRUE  FAlSE -> set UI from HW value
 * TRUE  FALSE -> set HW from UI value, then set UI from HW value
 * FALSE TRUE  -> set HW from UI value
 **/
static void
gpm_applet_update_popup_level (GpmBrightnessApplet *applet)
{
	if (applet->popup != NULL) {
		gtk_widget_set_sensitive (applet->btn_plus, applet->level < 100);
		gtk_widget_set_sensitive (applet->btn_minus, applet->level > 0);
		gtk_range_set_value (GTK_RANGE(applet->slider), (guint) applet->level);
	}
	gpm_applet_update_tooltip (applet);
}

/**
 * gpm_applet_plus_cb:
 * @widget: The sending widget (plus button)
 * @applet: Brightness applet instance
 *
 * callback for the plus button
 **/
static gboolean
gpm_applet_plus_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	if (applet->level < 100) {
		applet->level++;
	}
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_minus_cb:
 * @widget: The sending widget (minus button)
 * @applet: Brightness applet instance
 *
 * callback for the minus button
 **/
static gboolean
gpm_applet_minus_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	if (applet->level > 0) {
		applet->level--;
	}
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_slide_cb:
 * @widget: The sending widget (slider)
 * @applet: Brightness applet instance
 *
 * callback for the slider
 **/
static gboolean
gpm_applet_slide_cb (GtkWidget *w, GpmBrightnessApplet *applet)
{
	applet->level = gtk_range_get_value (GTK_RANGE(applet->slider));
	applet->call_worked = gpm_applet_set_brightness (applet);
	gpm_applet_update_popup_level (applet);
	return TRUE;
}

/**
 * gpm_applet_slide_cb:
 * @applet: Brightness applet instance
 * @event: The key press event
 *
 * callback handling keyboard
 * mainly escape to unpop and arrows to change brightness
 **/
static gboolean
gpm_applet_key_press_cb (GpmBrightnessApplet *applet, GdkEventKey *event)
{
	int i;
	
	switch (event->keyval) {
	case GDK_KP_Enter:
	case GDK_ISO_Enter:
	case GDK_3270_Enter:
	case GDK_Return:
	case GDK_space:
	case GDK_KP_Space:
	case GDK_Escape:
		/* if yet popped, release focus and hide then redraw applet unselected */
		if (applet->popped) {
			gdk_keyboard_ungrab (GDK_CURRENT_TIME);
			gdk_pointer_ungrab (GDK_CURRENT_TIME);
			gtk_grab_remove (GTK_WIDGET(applet));
			gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_NORMAL);
			gtk_widget_hide (applet->popup);
			applet->popped = FALSE;
			gpm_applet_draw_cb (applet);
			gpm_applet_update_tooltip (applet);
			return TRUE;
		} else {
			return FALSE;
		}
		break;
	case GDK_Page_Up:
		for (i = 0;i < 10;i++) {
			gpm_applet_plus_cb (NULL, applet);
		}
		return TRUE;
		break;
	case GDK_Left:
	case GDK_Up:
		gpm_applet_plus_cb (NULL, applet);
		return TRUE;
		break;
	case GDK_Page_Down:
		for (i = 0;i < 10;i++) {
			gpm_applet_minus_cb (NULL, applet);
		}
		return TRUE;
		break;
	case GDK_Right:
	case GDK_Down:
		gpm_applet_minus_cb (NULL, applet);
		return TRUE;
		break;
	default:
		return FALSE;
		break;
	}

	return FALSE;
}

/**
 * gpm_applet_scroll_cb:
 * @applet: Brightness applet instance
 * @event: The scroll event
 *
 * callback handling mouse scrolls, either when the applet
 * is not popped and the mouse is over the applet, or when
 * the applet is popped and no matter where the mouse is.
 **/
static gboolean
gpm_applet_scroll_cb (GpmBrightnessApplet *applet, GdkEventScroll *event)
{
	int i;

	if (event->type == GDK_SCROLL) {
		if (event->direction == GDK_SCROLL_UP) {
			for (i = 0;i < 5;i++) {
				gpm_applet_plus_cb (NULL, applet);
			}
			
		} else {
			for (i = 0;i < 5;i++) {
				gpm_applet_minus_cb (NULL, applet);
			}
		}
		return TRUE;
	}

	return FALSE;
}

/**
 * gpm_applet_create_popup:
 * @applet: Brightness applet instance
 *
 * cretes a new popup according to orientation of panel
 **/
static void
gpm_applet_create_popup (GpmBrightnessApplet *applet)
{
	static GtkWidget *box, *frame;
	gint orientation = panel_applet_get_orient (PANEL_APPLET (PANEL_APPLET (applet)));

	gpm_applet_destroy_popup_cb (applet);

	/* slider */
	if (PANEL_APPLET_VERTICAL(orientation)) {
		applet->slider = gtk_hscale_new_with_range (0, 100, 1);
		gtk_widget_set_size_request (applet->slider, 100, -1);
	} else {
		applet->slider = gtk_vscale_new_with_range (0, 100, 1);
		gtk_widget_set_size_request (applet->slider, -1, 100);
	}
	gtk_range_set_inverted (GTK_RANGE(applet->slider), TRUE);
	gtk_scale_set_draw_value (GTK_SCALE(applet->slider), FALSE);
	gtk_range_set_value (GTK_RANGE(applet->slider), applet->level);
	g_signal_connect (G_OBJECT(applet->slider), "value-changed", G_CALLBACK(gpm_applet_slide_cb), applet);

	/* minus button */
	applet->btn_minus = gtk_button_new_with_label ("\342\210\222"); /* U+2212 MINUS SIGN */
	gtk_button_set_relief (GTK_BUTTON(applet->btn_minus), GTK_RELIEF_NONE);
	g_signal_connect (G_OBJECT(applet->btn_minus), "pressed", G_CALLBACK(gpm_applet_minus_cb), applet);

	/* plus button */
	applet->btn_plus = gtk_button_new_with_label ("+");
	gtk_button_set_relief (GTK_BUTTON(applet->btn_plus), GTK_RELIEF_NONE);
	g_signal_connect (G_OBJECT(applet->btn_plus), "pressed", G_CALLBACK(gpm_applet_plus_cb), applet);

	/* box */
	if (PANEL_APPLET_VERTICAL(orientation)) {
		box = gtk_hbox_new (FALSE, 1);
	} else {
		box = gtk_vbox_new (FALSE, 1);
	}
	gtk_box_pack_start (GTK_BOX(box), applet->btn_plus, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->slider, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(box), applet->btn_minus, FALSE, FALSE, 0);

	/* frame */
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_OUT);
	gtk_container_add (GTK_CONTAINER(frame), box);

	/* window */
	applet->popup = gtk_window_new (GTK_WINDOW_POPUP);
	GTK_WIDGET_UNSET_FLAGS (applet->popup, GTK_TOPLEVEL);
	gtk_widget_set_parent (applet->popup, GTK_WIDGET(applet));
	gtk_container_add (GTK_CONTAINER(applet->popup), frame);
}

/**
 * gpm_applet_popup_cb:
 * @applet: Brightness applet instance
 *
 * pops and unpops
 **/
static gboolean
gpm_applet_popup_cb (GpmBrightnessApplet *applet, GdkEventButton *event)
{
	gint orientation, x, y;

	/* react only to left mouse button */
	if (event->button != 1) {
		return FALSE;
	}

	/* if yet popped, release focus and hide then redraw applet unselected */
	if (applet->popped) {
		gdk_keyboard_ungrab (GDK_CURRENT_TIME);
		gdk_pointer_ungrab (GDK_CURRENT_TIME);
		gtk_grab_remove (GTK_WIDGET(applet));
		gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_NORMAL);
		gtk_widget_hide (applet->popup);
		applet->popped = FALSE;
		gpm_applet_draw_cb (applet);
		gpm_applet_update_tooltip (applet);
		return TRUE;
	}

	/* update UI for current brightness */
	gpm_applet_update_popup_level (applet);

	/* otherwise pop */
	applet->popped = TRUE;
	gpm_applet_draw_cb (applet);

	/* create a new popup (initial or if panel parameters changed) */
	if (applet->popup == NULL) {
		gpm_applet_create_popup (applet);
	}

	/* update UI for current brightness */
	gpm_applet_update_popup_level (applet);

	gtk_widget_show_all (applet->popup);

	/* retrieve geometry parameters and move window appropriately */
	orientation = panel_applet_get_orient (PANEL_APPLET (PANEL_APPLET (applet)));
	gdk_window_get_origin (GTK_WIDGET(applet)->window, &x, &y);

	switch (orientation) {
	case PANEL_APPLET_ORIENT_DOWN:
		x += GTK_WIDGET(applet)->allocation.x
			+ GTK_WIDGET(applet)->allocation.width/2;
		y += GTK_WIDGET(applet)->allocation.y
			+ GTK_WIDGET(applet)->allocation.height;
		x -= applet->popup->allocation.width/2;
		break;
	case PANEL_APPLET_ORIENT_UP:
		x += GTK_WIDGET(applet)->allocation.x
			+ GTK_WIDGET(applet)->allocation.width/2;
		y += GTK_WIDGET(applet)->allocation.y;
		x -= applet->popup->allocation.width/2;
		y -= applet->popup->allocation.height;
		break;
	case PANEL_APPLET_ORIENT_RIGHT:
		y += GTK_WIDGET(applet)->allocation.y
			+ GTK_WIDGET(applet)->allocation.height/2;
		x += GTK_WIDGET(applet)->allocation.x
			+ GTK_WIDGET(applet)->allocation.width;
		y -= applet->popup->allocation.height/2;
		break;
	case PANEL_APPLET_ORIENT_LEFT:
		y += GTK_WIDGET(applet)->allocation.y
			+ GTK_WIDGET(applet)->allocation.height/2;
		x += GTK_WIDGET(applet)->allocation.x;
		x -= applet->popup->allocation.width;
		y -= applet->popup->allocation.height/2;
		break;
	default:
		g_assert_not_reached ();
	}

	gtk_window_move (GTK_WINDOW (applet->popup), x, y);

	/* grab input */
	gtk_widget_grab_focus (GTK_WIDGET(applet));
	gtk_grab_add (GTK_WIDGET(applet));
	gdk_pointer_grab (GTK_WIDGET(applet)->window, TRUE,
			  GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK |
			  GDK_POINTER_MOTION_MASK,
			  NULL, NULL, GDK_CURRENT_TIME);
	gdk_keyboard_grab (GTK_WIDGET(applet)->window,
			   TRUE, GDK_CURRENT_TIME);
	gtk_widget_set_state (GTK_WIDGET(applet), GTK_STATE_SELECTED);

	return TRUE;
}

/**
 * gpm_applet_theme_change_cb:
 *
 * Updtes icon when theme changes
 **/
static void
gpm_applet_theme_change_cb (GtkIconTheme *icon_theme, gpointer data)
{
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET (data);
	gpm_applet_get_icon (applet);
}

/**
 * gpm_applet_stop_scroll_events_cb:
 *
 * Prevents scroll events from reaching the tooltip
 **/
static void
gpm_applet_stop_scroll_events_cb (GtkWidget *widget, GdkEvent  *event)
{
	if (event->type == GDK_SCROLL)
		g_signal_stop_emission_by_name (widget, "event-after");
}

/**
 * gpm_applet_dialog_about_cb:
 *
 * displays about dialog
 **/
static void
gpm_applet_dialog_about_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GtkAboutDialog *about;

	GdkPixbuf *logo =
		gtk_icon_theme_load_icon (gtk_icon_theme_get_default (),
					  GPM_BRIGHTNESS_APPLET_ICON,
					  128, 0, NULL);

	static const gchar *authors[] = {
		"Benjamin Canou <bookeldor@gmail.com>",
		"Richard Hughes <richard@hughsie.com>",
		NULL
	};
	const char *documenters [] = {
		NULL
	};
	const char *license[] = {
		 N_("Licensed under the GNU General Public License Version 2"),
		 N_("Power Manager is free software; you can redistribute it and/or\n"
		   "modify it under the terms of the GNU General Public License\n"
		   "as published by the Free Software Foundation; either version 2\n"
		   "of the License, or (at your option) any later version."),
		 N_("Power Manager is distributed in the hope that it will be useful,\n"
		   "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		   "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		   "GNU General Public License for more details."),
		 N_("You should have received a copy of the GNU General Public License\n"
		   "along with this program; if not, write to the Free Software\n"
		   "Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA\n"
		   "02110-1301, USA.")
	};
	const char *translator_credits = NULL;
	char	   *license_trans;

	license_trans = g_strconcat (_(license[0]), "\n\n", _(license[1]), "\n\n",
				     _(license[2]), "\n\n", _(license[3]), "\n", NULL);

	about = (GtkAboutDialog*) gtk_about_dialog_new ();
	gtk_about_dialog_set_name (about, GPM_BRIGHTNESS_APPLET_NAME);
	gtk_about_dialog_set_version (about, VERSION);
	gtk_about_dialog_set_copyright (about, _("Copyright \xc2\xa9 2006 Benjamin Canou"));
	gtk_about_dialog_set_comments (about, GPM_BRIGHTNESS_APPLET_DESC);
	gtk_about_dialog_set_authors (about, authors);
	gtk_about_dialog_set_documenters (about, documenters);
	gtk_about_dialog_set_translator_credits (about, translator_credits);
	gtk_about_dialog_set_logo (about, logo);
	gtk_about_dialog_set_license (about, license_trans);
	gtk_about_dialog_set_website (about, GPM_HOMEPAGE_URL);

	g_signal_connect (G_OBJECT(about), "response",
			  G_CALLBACK(gtk_widget_destroy), NULL);

	gtk_widget_show (GTK_WIDGET(about));

	g_free (license_trans);
	gdk_pixbuf_unref (logo);
}

/**
 * help_cb:
 *
 * open gpm help
 **/
static void
help_cb (BonoboUIComponent *uic, gpointer data, const gchar *verbname)
{
	GError *error = NULL;
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET(data);
	GnomeProgram *program = gnome_program_get ();

	gnome_help_display_with_doc_id (program, "gnome-power-manager",
					"gnome-power-manager.xml",
					"applets-brightness", &error);
	if (error != NULL) {
		GtkWidget *dialog =
			gtk_message_dialog_new (GTK_WINDOW (GTK_WIDGET(applet)->parent),
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_CLOSE,
						error->message);
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_error_free (error);
	}
}

/**
 * gpm_applet_destroy_cb:
 * @object: Class instance to destroy
 **/
static void
gpm_applet_destroy_cb (GtkObject *object)
{
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET(object);

	if (applet->watch != NULL) {
		g_object_unref (applet->watch);
	}
	if (applet->icon != NULL) {
		gdk_pixbuf_unref (applet->icon);
	}
}

/**
 * gpm_brightness_applet_class_init:
 * @klass: Class instance
 **/
static void
gpm_brightness_applet_class_init (GpmBrightnessAppletClass *class)
{
	/* nothing to do here */
}

static void
brightness_changed_cb (DBusGProxy          *proxy,
		       guint	            brightness,
		       GpmBrightnessApplet *applet)
{
	printf ("DEBUG: BrightnessChanged detected: %i\n", brightness);
	applet->level = brightness;
}

/**
 * gpm_brightness_applet_dbus_connect:
 **/
gboolean
gpm_brightness_applet_dbus_connect (GpmBrightnessApplet *applet)
{
	GError *error = NULL;

	if (applet->connection == NULL) {
		printf ("DEBUG: get connection\n");
		g_clear_error (&error);
		applet->connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
		if (error != NULL) {
			printf ("WARNING: Could not connect to DBUS daemon: %s", error->message);
			g_error_free (error);
			applet->connection = NULL;
			return FALSE;
		}
	}
	if (applet->proxy == NULL) {
		printf ("DEBUG: get proxy\n");
		g_clear_error (&error);
		applet->proxy = dbus_g_proxy_new_for_name_owner (applet->connection,
							 GPM_DBUS_SERVICE,
							 GPM_DBUS_PATH_BACKLIGHT,
							 GPM_DBUS_INTERFACE_BACKLIGHT,
							 &error);
		if (error != NULL) {
			printf ("WARNING: Cannot connect, maybe the daemon is not running: %s\n", error->message);
			g_error_free (error);
			applet->proxy = NULL;
			return FALSE;
		}
		dbus_g_proxy_add_signal (applet->proxy, "BrightnessChanged",
					 G_TYPE_UINT, G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (applet->proxy, "BrightnessChanged",
					     G_CALLBACK (brightness_changed_cb),
					     applet, NULL);
		/* reset, we might be starting race */
		applet->call_worked = TRUE;
	}
	return TRUE;
}

/**
 * gpm_brightness_applet_dbus_disconnect:
 **/
gboolean
gpm_brightness_applet_dbus_disconnect (GpmBrightnessApplet *applet)
{
	if (applet->proxy != NULL) {
		printf ("DEBUG: removing proxy\n");
		g_object_unref (applet->proxy);
		applet->proxy = NULL;
	}
	return TRUE;
}

/**
 * watch_connection_cb:
 * @proxy: The dbus raw proxy
 * @status: The status of the service, where TRUE is connected
 * @screensaver: This class instance
 **/
static void
watch_connection_cb (DbusWatch           *watch,
		     gboolean	          status,
		     GpmBrightnessApplet *applet)
{
	if (status) {
		gpm_brightness_applet_dbus_connect (applet);
		gpm_applet_update_tooltip (applet);
		gpm_applet_draw_cb (applet);
	} else {
		gpm_brightness_applet_dbus_disconnect (applet);
		gpm_applet_update_tooltip (applet);
		gpm_applet_draw_cb (applet);
	}
}

/**
 * gpm_brightness_applet_init:
 * @applet: Brightness applet instance
 **/
static void
gpm_brightness_applet_init (GpmBrightnessApplet *applet)
{
	/* initialize fields */
	applet->size = 0;
	applet->call_worked = TRUE;
	applet->popped = FALSE;
	applet->popup = NULL;
	applet->icon = NULL;
	applet->connection = NULL;
	applet->proxy = NULL;
	applet->tooltip = gtk_tooltips_new ();

	applet->watch = dbus_watch_new ();
	g_signal_connect (applet->watch, "connection-changed",
			  G_CALLBACK (watch_connection_cb), applet);
	dbus_watch_assign (applet->watch, DBUS_WATCH_SESSION, GPM_DBUS_SERVICE);
	gpm_brightness_applet_dbus_connect (applet);

	/* coldplug */
	applet->call_worked = gpm_applet_get_brightness (applet);
	gpm_applet_update_popup_level (applet);

	/* prepare */
	panel_applet_set_flags (PANEL_APPLET (applet), PANEL_APPLET_EXPAND_MINOR);

	/* show */
	gtk_widget_show_all (GTK_WIDGET(applet));

	/* set appropriate size and load icon accordingly */
	gpm_applet_draw_cb (applet);

	/* connect */
	g_signal_connect (G_OBJECT(applet), "button-press-event",
			  G_CALLBACK(gpm_applet_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "scroll-event",
			  G_CALLBACK(gpm_applet_scroll_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "key-press-event",
			  G_CALLBACK(gpm_applet_key_press_cb), NULL);

	/* We use g_signal_connect_after because letting the panel draw
	 * the background is the only way to have the correct
	 * background when a theme defines a background picture. */
	g_signal_connect_after (G_OBJECT(applet), "expose-event",
				G_CALLBACK(gpm_applet_draw_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "change-background",
			  G_CALLBACK(gpm_applet_change_background_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "change-orient",
			  G_CALLBACK(gpm_applet_draw_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "change-orient",
			  G_CALLBACK(gpm_applet_destroy_popup_cb), NULL);

	g_signal_connect (G_OBJECT(applet), "destroy",
			  G_CALLBACK(gpm_applet_destroy_cb), NULL);

	/* prevent scroll events from reaching the tooltip */
	g_signal_connect (G_OBJECT (applet), "event-after", G_CALLBACK (gpm_applet_stop_scroll_events_cb), NULL);

	g_signal_connect (gtk_icon_theme_get_default (), "changed", G_CALLBACK (gpm_applet_theme_change_cb), applet);
}

/**
 * gpm_applet_bonobo_cb:
 * @_applet: GpmBrightnessApplet instance created by the bonobo factory
 * @iid: Bonobo id
 *
 * the function called by bonobo factory after creation
 **/
static gboolean
gpm_applet_bonobo_cb (PanelApplet *_applet, const gchar *iid, gpointer data)
{
	GpmBrightnessApplet *applet = GPM_BRIGHTNESS_APPLET(_applet);

	static BonoboUIVerb verbs [] = {
		BONOBO_UI_VERB ("About", gpm_applet_dialog_about_cb),
		BONOBO_UI_VERB ("Help", help_cb),
		BONOBO_UI_VERB_END
	};

	if (strcmp (iid, GPM_BRIGHTNESS_APPLET_OAFID) != 0) {
		return FALSE;
	}

	panel_applet_setup_menu_from_file (PANEL_APPLET (applet),
					   DATADIR,
					   "GNOME_BrightnessApplet.xml",
					   NULL, verbs, applet);
	gpm_applet_draw_cb (applet);
	return TRUE;
}

/**
 * this generates a main with a bonobo factory
 **/
PANEL_APPLET_BONOBO_FACTORY
 (/* the factory iid */
 GPM_BRIGHTNESS_APPLET_FACTORY_OAFID,
 /* generates brighness applets instead of regular gnome applets  */
 GPM_TYPE_BRIGHTNESS_APPLET,
 /* the applet name and version */
 "BrightnessApplet", VERSION,
 /* our callback (with no user data) */
 gpm_applet_bonobo_cb, NULL);
