/* font-dialog.c - functions that do things to fonts.
 *
 * Copyright (C) 1998 Chris Lahey.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "config.h"
#include "gnome-font-dialog.h"
#include "gnome-print-i18n.h"
#include <gtk/gtkcombo.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklistitem.h>
#include <gtk/gtkframe.h>
#include <gtk/gtktogglebutton.h>
#include <libgnomeui/gnome-stock.h>

/* This is the default text shown in the preview entry, though the user
   can set it. Remember that some fonts only have capital letters. */
#define PREVIEW_TEXT "abcdefghijk ABCDEFGHIJK"

/* This is the initial and maximum height of the preview entry (it expands
   when large font sizes are selected). Initial height is also the minimum. */
#define INITIAL_PREVIEW_HEIGHT 44
#define MAX_PREVIEW_HEIGHT 300

/* These are what we use as the standard font sizes, for the size clist.
   Note that when using points we still show these integer point values but
   we work internally in decipoints (and decipoint values can be typed in). */
const static char *font_sizes[] = {
  "8", "9", "10", "11", "12", "13", "14", "16", "18", "20", "22", "24", "26", "28",
  "32", "36", "40", "48", "56", "64", "72"
};

typedef struct _GnomeFontWeightTab GnomeFontWeightTab;

struct _GnomeFontWeightTab {
  const char *name;
  int code;
};

static const GnomeFontWeightTab weight_tab[] = {
  { "Extra Light", GNOME_FONT_EXTRA_LIGHT },

  { "Thin", GNOME_FONT_THIN },

  { "Light", GNOME_FONT_LIGHT },

  { "Regular", GNOME_FONT_BOOK },

  { "Medium", GNOME_FONT_MEDIUM },

  { "Semibold", GNOME_FONT_SEMI },

  { "Bold", GNOME_FONT_BOLD },

  { "Heavy", GNOME_FONT_HEAVY },
 
  { "Extra Bold", GNOME_FONT_EXTRABOLD },

  { "Black", GNOME_FONT_BLACK },

  { "Extra Black", GNOME_FONT_EXTRABLACK }
};

static void    gnome_font_selection_class_init	     (GnomeFontSelectionClass *klass);
static void    gnome_font_selection_init	     (GnomeFontSelection *fontsel);
static void    gnome_font_selection_destroy	     (GtkObject      *object);
static void    gnome_font_selection_update_preview   (GnomeFontSelection *fontsel);
static void    gnome_font_selection_changed          (GtkWidget *widget, gpointer data);

/* FontSelectionDialog */
static void    gnome_font_selection_dialog_class_init  (GnomeFontSelectionDialogClass *klass);
static void    gnome_font_selection_dialog_init	     (GnomeFontSelectionDialog *fontseldiag);

static GnomeDialogClass *font_selection_parent_class = NULL;
static GtkTableClass    *font_selection_dialog_parent_class = NULL;

GtkType
gnome_font_selection_get_type()
{
  static GtkType font_selection_type = 0;
  
  if(!font_selection_type)
    {
      static const GtkTypeInfo fontsel_type_info =
      {
	"GnomeFontSelection",
	sizeof (GnomeFontSelection),
	sizeof (GnomeFontSelectionClass),
	(GtkClassInitFunc) gnome_font_selection_class_init,
	(GtkObjectInitFunc) gnome_font_selection_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      font_selection_type = gtk_type_unique (GTK_TYPE_TABLE,
					     &fontsel_type_info);
    }
  
  return font_selection_type;
}

static void
gnome_font_selection_class_init(GnomeFontSelectionClass *klass)
{
  GtkObjectClass *object_class;
  
  object_class = (GtkObjectClass *) klass;
  
  font_selection_parent_class = gtk_type_class (GTK_TYPE_TABLE);
  
  object_class->destroy = gnome_font_selection_destroy;
}

static void
gnome_font_selection_init(GnomeFontSelection *fontsel)
{
  GnomeFontClass *klass;
  GtkWidget *combo;
  GtkWidget *listitem;
  GtkWidget *check_button;
  GtkWidget *entry;
  GList *list; /* Of type gchar */
  gint i;
  GtkTable *table = GTK_TABLE( fontsel );

  table->homogeneous = FALSE;
  gtk_table_resize (table, 2, 3);

  combo = gtk_combo_new();
  klass = gtk_type_class (gnome_font_get_type ());
  list = gnome_font_family_list (klass);
  gtk_combo_set_popdown_strings( GTK_COMBO( combo ), list );
  gnome_font_family_list_free (list);
  gtk_table_attach( GTK_TABLE( table ), combo, 0, 1, 0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->family_name = combo;
  gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), "Times" );
  gtk_widget_show( combo );
  gtk_signal_connect( GTK_OBJECT( GTK_COMBO( combo )->entry ),
		      "changed",
		      GTK_SIGNAL_FUNC( gnome_font_selection_changed ),
		      fontsel );

  combo = gtk_combo_new();
  for ( i = 0; i < sizeof(font_sizes) / sizeof(font_sizes[0]); i++ )
    {
      listitem = gtk_list_item_new_with_label ( font_sizes[ i ] );
      gtk_widget_show( listitem );
      gtk_container_add( GTK_CONTAINER( GTK_COMBO( combo )->list ), listitem );
    }
  gtk_table_attach( GTK_TABLE( table ), combo, 1, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->font_size = combo;
  gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), "12" );
  gtk_widget_show( combo );
  gtk_signal_connect( GTK_OBJECT( GTK_COMBO( combo )->entry ),
		      "changed",
		      GTK_SIGNAL_FUNC( gnome_font_selection_changed ),
		      fontsel );
  
  combo = gtk_combo_new();
  for ( i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++ )
    {
      listitem = gtk_list_item_new_with_label ( weight_tab[ i ].name );
      gtk_widget_show( listitem );
      gtk_container_add( GTK_CONTAINER( GTK_COMBO( combo )->list ), listitem );
    }
  gtk_table_attach( GTK_TABLE( table ), combo, 0, 1, 1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->font_weight = combo;
  gtk_entry_set_text( GTK_ENTRY( GTK_COMBO( combo )->entry ), "Regular" );
  gtk_widget_show( combo );
  gtk_signal_connect( GTK_OBJECT( GTK_COMBO( combo )->entry ),
		      "changed",
		      GTK_SIGNAL_FUNC( gnome_font_selection_changed ),
		      fontsel );
  
  check_button = gtk_check_button_new_with_label( _("Italic") );
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( check_button ), FALSE );
  gtk_table_attach( GTK_TABLE( table ), check_button, 1, 2, 1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->italic = check_button;
  gtk_widget_show( check_button );
  gtk_signal_connect( GTK_OBJECT( check_button ),
		      "toggled",
		      GTK_SIGNAL_FUNC( gnome_font_selection_changed ),
		      fontsel );

  entry = gtk_entry_new();
  gtk_table_attach( GTK_TABLE( table ), entry, 0, 2, 2, 3,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->preview = entry;
  gtk_widget_show( entry );
  gnome_font_selection_update_preview( fontsel );
}

static void
gnome_font_selection_changed( GtkWidget *widget, gpointer data )
{
  GnomeFontSelection *fontsel = GNOME_FONT_SELECTION( data );
  gnome_font_selection_update_preview( fontsel );
}

static void
gnome_font_selection_update_preview( GnomeFontSelection *fontsel )
{
  GnomeDisplayFont *font = gnome_font_selection_get_font( fontsel );
  GtkWidget *preview;
  GtkStyle *style;
  gint text_height, new_height;
  gchar *text;
  
  style = gtk_style_new ();
  gdk_font_unref (style->font);
  style->font = font->gdk_font;
  gdk_font_ref (style->font);
  
  preview = fontsel->preview;
  gtk_widget_set_style (preview, style);
  gtk_style_unref(style);
  
  text_height = preview->style->font->ascent
    + preview->style->font->descent;
  /* We don't ever want to be over MAX_PREVIEW_HEIGHT pixels high. */
  new_height = text_height + 20;
  if (new_height < INITIAL_PREVIEW_HEIGHT)
    new_height = INITIAL_PREVIEW_HEIGHT;
  if (new_height > MAX_PREVIEW_HEIGHT)
    new_height = MAX_PREVIEW_HEIGHT;
  
  if ((preview->requisition.height < text_height + 10)
      || (preview->requisition.height > text_height + 40))
    gtk_widget_set_usize(preview, -1, new_height);
  
  /* This sets the preview text, if it hasn't been set already. */
  text = gtk_entry_get_text(GTK_ENTRY(fontsel->preview));
  if (strlen(text) == 0)
    gtk_entry_set_text(GTK_ENTRY(fontsel->preview), PREVIEW_TEXT);
  gtk_entry_set_position(GTK_ENTRY(fontsel->preview), 0);
#if 0  
  /* If this is a 2-byte font display a message to say it may not be
     displayed properly. */
  xfs = GDK_FONT_XFONT(fontsel->font);
  if (xfs->min_byte1 != 0 || xfs->max_byte1 != 0)
    gtk_label_set_text(GTK_LABEL(fontsel->message_label),
		       _("This is a 2-byte font and may not be displayed correctly."));
#endif
}

GtkWidget *
gnome_font_selection_new()
{
  GnomeFontSelection *fontsel;
  
  fontsel = gtk_type_new (gnome_font_selection_get_type());
  
  return GTK_WIDGET (fontsel);
}

static void
gnome_font_selection_destroy (GtkObject *object)
{
  GnomeFontSelection *fontsel;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_FONT_SELECTION (object));
  
  fontsel = GNOME_FONT_SELECTION (object);
  
  if (GTK_OBJECT_CLASS (font_selection_parent_class)->destroy)
    (* GTK_OBJECT_CLASS (font_selection_parent_class)->destroy) (object);
}


/*****************************************************************************
 * These functions are the main public interface for getting/setting the font.
 *****************************************************************************/

GnomeDisplayFont*
gnome_font_selection_get_font (GnomeFontSelection *fontsel)
{
  gchar *family_chars;
  gchar *temp;
  gint font_size;
  GnomeFontWeight font_weight;
  GnomeDisplayFont *display_font;
  gboolean italic = FALSE;
  gint i;
  
  g_return_val_if_fail (fontsel != NULL, NULL);

  family_chars = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->family_name )->entry ), 0, -1 );
  
  temp = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->font_size )->entry ), 0, -1 );
  font_size = atoi( temp );
  g_free ( temp );
  
  temp = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->font_weight )->entry ), 0, -1 );
  font_weight = GNOME_FONT_BOOK;
  for (i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++)
    if (!g_strcasecmp (temp, weight_tab[i].name))
      {
	font_weight = weight_tab[i].code;
	break;
      }
  g_free ( temp );

  italic = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( fontsel->italic ) );
  
  display_font = gnome_get_display_font( family_chars,
					 font_weight,
					 italic,
					 font_size,
					 1);
  g_free( family_chars );

  return display_font;
}


/*****************************************************************************
 * GtkFontSelectionDialog
 *****************************************************************************/

guint
gnome_font_selection_dialog_get_type (void)
{
  static guint font_selection_dialog_type = 0;
  
  if (!font_selection_dialog_type)
    {
      GtkTypeInfo fontsel_diag_info =
      {
	"GnomeFontSelectionDialog",
	sizeof (GnomeFontSelectionDialog),
	sizeof (GnomeFontSelectionDialogClass),
	(GtkClassInitFunc) gnome_font_selection_dialog_class_init,
	(GtkObjectInitFunc) gnome_font_selection_dialog_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      font_selection_dialog_type = gtk_type_unique (gnome_dialog_get_type(), &fontsel_diag_info);
    }
  
  return font_selection_dialog_type;
}

static void
gnome_font_selection_dialog_class_init (GnomeFontSelectionDialogClass *klass)
{
  GtkObjectClass *object_class;
  
  object_class = (GtkObjectClass*) klass;
  
  font_selection_dialog_parent_class = gtk_type_class (gnome_dialog_get_type());
}

static void
gnome_font_selection_dialog_init (GnomeFontSelectionDialog *fontseldiag)
{
  GtkWidget *frame;
  gnome_dialog_append_button ( GNOME_DIALOG(fontseldiag),
			       GNOME_STOCK_BUTTON_OK);

  gnome_dialog_append_button ( GNOME_DIALOG(fontseldiag),
			       GNOME_STOCK_BUTTON_CANCEL);

  gnome_dialog_set_default( GNOME_DIALOG( fontseldiag ), 0 );

  frame = gtk_frame_new (_("Select font"));
  
  fontseldiag->fontsel = gnome_font_selection_new();
  gtk_container_add (GTK_CONTAINER (frame), fontseldiag->fontsel);
  
  gtk_container_set_border_width (GTK_CONTAINER (fontseldiag), 4);
  gtk_window_set_policy(GTK_WINDOW(fontseldiag), FALSE, TRUE, TRUE);
  
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (fontseldiag)->vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show_all (frame);  
}

GtkWidget*
gnome_font_selection_dialog_new	(const gchar	  *title)
{
  GnomeFontSelectionDialog *fontseldiag;
  
  fontseldiag = gtk_type_new (GNOME_TYPE_FONT_SELECTION_DIALOG);
  gtk_window_set_title (GTK_WINDOW (fontseldiag),
			title ? title : _("Font Selection"));
  
  return GTK_WIDGET (fontseldiag);
}

GnomeDisplayFont*
gnome_font_selection_dialog_get_font	(GnomeFontSelectionDialog *fsd)
{
  return gnome_font_selection_get_font(GNOME_FONT_SELECTION(fsd->fontsel));
}
