#include "config.h"
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include <libart_lgpl/art_affine.h>
#include <libgnomeprint/gnome-print-ps.h>


static void gnome_print_ps_class_init (GnomePrintPsClass *klass);

static void
gnome_print_ps_init (GnomePrintPs *ps);

static void
gnome_print_ps_finalize (GtkObject *object);

static GnomePrintContextClass *parent_class = NULL;

GtkType
gnome_print_ps_get_type (void)
{
  static GtkType ps_type = 0;

  if (!ps_type)
    {
      GtkTypeInfo ps_info =
      {
	"GnomePrintPs",
	sizeof (GnomePrintPs),
	sizeof (GnomePrintPsClass),
	(GtkClassInitFunc) gnome_print_ps_class_init,
	(GtkObjectInitFunc) gnome_print_ps_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      ps_type = gtk_type_unique (gnome_print_context_get_type (), &ps_info);
    }

  return ps_type;
}

/* The implementations of the PostScript paint methods, autogenned */

int
gnome_print_ps_newpath (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "newpath\n");
}

int
gnome_print_ps_moveto (GnomePrintContext *pc, double x, double y)
{
  return gnome_print_context_fprintf (pc,
                                      "%g %g moveto\n",
                                      x,
                                      y);
}

int
gnome_print_ps_lineto (GnomePrintContext *pc, double x, double y)
{
  return gnome_print_context_fprintf (pc,
                                      "%g %g lineto\n",
                                      x,
                                      y);
}

int
gnome_print_ps_curveto (GnomePrintContext *pc, double x1, double y1, double x2, double y2, double x3, double y3)
{
  return gnome_print_context_fprintf (pc,
                                      "%g %g %g %g %g %g curveto\n",
                                      x1,
                                      y1,
                                      x2,
                                      y2,
                                      x3,
                                      y3);
}

int
gnome_print_ps_closepath (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "closepath\n");
}

int
gnome_print_ps_setrgbcolor (GnomePrintContext *pc,
			    double r, double g, double b)
{
  GnomePrintPs *ps = GNOME_PRINT_PS (pc);

  /* I think there's a problem:
   * gsave, setcolor, grestore means a setrgbcolor is required;
   * ps->r, etc. are the "last set" color, not the "current color"
   * so you need a color stack to do this optimization.
   */
#if 0
  /* if the current color is the color the user wants to set, noop */
  if (ps->r == r &&
      ps->g == g &&
      ps->b == b)
    return 0;
#endif

  ps->r = r; ps->g = g; ps->b = b;

  return gnome_print_context_fprintf (pc,
                                      "%g %g %g setrgbcolor\n",
                                      r,
                                      g,
                                      b);
}

int
gnome_print_ps_fill (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "fill\n");
}

int
gnome_print_ps_eofill (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "eofill\n");
}

int
gnome_print_ps_setlinewidth (GnomePrintContext *pc, double width)
{
  return gnome_print_context_fprintf (pc,
                                      "%g setlinewidth\n",
                                      width);
}

int
gnome_print_ps_setmiterlimit (GnomePrintContext *pc, double limit)
{
  return gnome_print_context_fprintf (pc,
                                      "%g setmiterlimit\n",
                                      limit);
}

int
gnome_print_ps_setlinejoin (GnomePrintContext *pc, int jointype)
{
  return gnome_print_context_fprintf (pc,
                                      "%d setlinejoin\n",
                                      jointype);
}

int
gnome_print_ps_setlinecap (GnomePrintContext *pc, int captype)
{
  return gnome_print_context_fprintf (pc,
                                      "%d setlinecap\n",
                                      captype);
}

int
gnome_print_ps_setdash (GnomePrintContext *pc, int n_values, double *values, double offset)
{
  return gnome_print_context_fprintf (pc,
                                      "%d %g %g setdash\n",
                                      n_values,
                                      values,
                                      offset);
}

int
gnome_print_ps_strokepath (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "strokepath\n");
}

int
gnome_print_ps_stroke (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "stroke\n");
}

static int
gnome_print_ps_setfont_raw (GnomePrintContext *pc, GnomeFontUnsized *unsized,
			    double size)
{
  GnomePrintPs *ps;
  char *pfa;
  int i;
  char *fontname;

  g_return_val_if_fail (pc != NULL, -1);

  ps = GNOME_PRINT_PS (pc);

  if (unsized == NULL)
    return -1;

  fontname = gnome_font_unsized_get_name (unsized);
  for (i = 0; i < ps->n_builtins; i++)
    if (!strcmp (fontname, ps->builtins[i]))
      break;

  if (i == ps->n_builtins)
    {
      fontname = gnome_font_unsized_get_glyph_name (unsized);
      for (i = 0; i < ps->n_fonts; i++)
	if (!strcmp (fontname, ps->fonts[i]))
	  break;
      if (i == ps->n_fonts)
	{
	  pfa = gnome_font_unsized_get_pfa (unsized);
	  if (pfa == NULL)
	    return -1;
	  
	  if (gnome_print_context_fprintf (pc, "%s", pfa) < 0)
	    return -1;
	  
	  if (ps->n_fonts == ps->n_fonts_max)
	    ps->fonts = g_realloc (ps->fonts, sizeof (char *) *
				   (ps->n_fonts_max <<= 1));
	  ps->fonts[ps->n_fonts++] = fontname;
	  g_free (pfa);
	}
    }

  if (ps->current_font && strcmp (ps->current_font, fontname) == 0 &&
      ps->current_font_size == size)
    return 0; /* this font is already set */

  ps->current_font = fontname;
  ps->current_font_size = size;

  return gnome_print_context_fprintf (pc,
				      "/%s findfont %g scalefont setfont\n",
				      fontname, size);

}

/* todo: just make this call the _raw variant on the unsized font */
int
gnome_print_ps_setfont (GnomePrintContext *pc, GnomeFont *font)
{
  GnomePrintPs *ps;
  char *pfa;
  int i;
  char *fontname;

  g_return_val_if_fail (pc != NULL, -1);

  ps = GNOME_PRINT_PS (pc);

  if (font == NULL)
    return -1;

  fontname = gnome_font_get_name (font);
  for (i = 0; i < ps->n_builtins; i++)
    if (!strcmp (fontname, ps->builtins[i]))
      break;

  if (i == ps->n_builtins)
    {
      fontname = gnome_font_get_glyph_name (font);
      for (i = 0; i < ps->n_fonts; i++)
	if (!strcmp (fontname, ps->fonts[i]))
	  break;
      if (i == ps->n_fonts)
	{
	  pfa = gnome_font_get_pfa (font);
	  if (pfa == NULL)
	    return -1;
	  
	  if (gnome_print_context_fprintf (pc, "%s", pfa) < 0)
	    return -1;
	  
	  if (ps->n_fonts == ps->n_fonts_max)
	    ps->fonts = g_realloc (ps->fonts, sizeof (char *) *
				   (ps->n_fonts_max <<= 1));
	  ps->fonts[ps->n_fonts++] = fontname;
	  g_free (pfa);
	}
    }

  /* This doesn't consider gsave/grestore... */
#if 0
  if (ps->current_font && strcmp (ps->current_font, fontname) == 0 &&
      ps->current_font_size == font->size)
    return 0; /* this font is already set */
#endif

  ps->current_font = fontname;
  ps->current_font_size = font->size;

  /* invalidate the settings for gnome-text */
  ps->cur_font = 0;
  ps->cur_size = 0;

  return gnome_print_context_fprintf (pc,
				      "/%s findfont %g scalefont setfont\n",
				      fontname, font->size);

}

int
gnome_print_ps_show (GnomePrintContext *pc, char *text)
{
  int i;

  if (gnome_print_context_fprintf (pc, "(") < 0)
    return -1;
  for (i = 0; text[i]; i++)
    {
      /* todo: maybe we should test for non-ascii characters here, and
	 format as octal */
      if (text[i] == '(' || text[i] == ')' || text[i] == '\\') 
	{
	  if (gnome_print_context_fprintf (pc, "\\%c", text[i]) < 0)
	    return -1;
	}
      else
	{
	  if (gnome_print_context_fprintf (pc, "%c", text[i]) < 0)
	    return -1;
	}
    }
  return gnome_print_context_fprintf (pc, ") show\n");
}

int
gnome_print_ps_concat (GnomePrintContext *pc, double matrix[6])
{
  char str[128];

  art_affine_to_string (str, matrix);
  return gnome_print_context_fprintf (pc,
                                      "%s\n", str);
}

int
gnome_print_ps_setmatrix (GnomePrintContext *pc, double matrix[6])
{
  return gnome_print_context_fprintf (pc,
                                      "[ %g %g %g %g %g %g ] setmatrix\n",
                                      matrix[0],
                                      matrix[1],
                                      matrix[2],
                                      matrix[3],
                                      matrix[4],
                                      matrix[5]);
}

int
gnome_print_ps_gsave (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "gsave\n");
}

int
gnome_print_ps_grestore (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "grestore\n");
}

int
gnome_print_ps_clip (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "clip\n");
}

int
gnome_print_ps_eoclip (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "eoclip\n");
}

static int
gnome_print_ps_image (GnomePrintContext *pc, char *data, int width, int height, int rowstride, int bytes_per_pixel)
{
  int status;
  char linebuf[80];
  int x, y;
  int pos;
  int startline, ix;
  unsigned char b;
  const char tohex[16] = "0123456789abcdef";
  int bytes_per_line;

  bytes_per_line = width * bytes_per_pixel;

  status = gnome_print_context_fprintf (pc, "/buf %d string def\n%d %d 8\n",
					bytes_per_line,
					width, height);
  if (status < 0)
    return status;

  status = gnome_print_context_fprintf (pc, "[ %d 0 0 %d 0 %d ]\n",
					width,
					-height,
					height);
  if (status < 0)
    return status;

  status = gnome_print_context_fprintf (pc, "{ currentfile buf readhexstring pop }\n");
  if (status < 0)
    return status;

  if (bytes_per_pixel == 1)
    status = gnome_print_context_fprintf (pc, "image\n");
  else if (bytes_per_pixel == 3)
    status = gnome_print_context_fprintf (pc, "false %d colorimage\n",
					  bytes_per_pixel);
  if (status < 0)
    return status;

  pos = 0;
  startline = 0;
  for (y = 0; y < height; y++)
    {
      ix = startline;
      for (x = 0; x < bytes_per_line; x++)
	{
	  b = data[ix++];
	  linebuf[pos++] = tohex[b >> 4];
	  linebuf[pos++] = tohex[b & 15];
	  if (pos == 72)
	    {
	      linebuf[pos++] = '\n';
	      if (gnome_print_context_write_file (pc, linebuf, pos) < pos)
		return -1;
	      pos = 0;
	    }
	}
      startline += rowstride;
    }
  if (pos)
    {
      linebuf[pos++] = '\n';
      if (gnome_print_context_write_file (pc, linebuf, pos) < pos)
	return -1;
    }
  return 0;
}

int
gnome_print_ps_grayimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
  return  gnome_print_ps_image (pc, data, width, height, rowstride, 1);
}

int
gnome_print_ps_rgbimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
  return gnome_print_ps_image (pc, data, width, height, rowstride, 3);
}

int
gnome_print_ps_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
  int i;
  int attr_idx;
  int cur_size;
  int cur_xscale;
  int last_size;
  double scale_factor;
  GnomeTextFontHandle cur_font, last_font;
  GnomePrintPs *ps = GNOME_PRINT_PS (pc);
  GnomeTextGlyphAttrEl *attrs = line->attrs;
  int x;
  int glyph;
  gboolean open;

  cur_font = ps->cur_font;
  last_font = cur_font;
  cur_size = ps->cur_size;
  last_size = cur_size;

  cur_xscale = 1000;
  scale_factor = cur_size * cur_xscale * 1e-9 * GNOME_TEXT_SCALE;

  open = 0;
  x = 0;
  attr_idx = 0;
  for (i = 0; i < line->n_glyphs; i++)
    {
      while (attrs[attr_idx].glyph_pos == i)
	{
	  switch (attrs[attr_idx].attr)
	    {
	    case GNOME_TEXT_GLYPH_FONT:
	      cur_font = attrs[attr_idx].attr_val;
	      break;
	    case GNOME_TEXT_GLYPH_SIZE:
	      cur_size = attrs[attr_idx].attr_val;
	      scale_factor = cur_size * cur_xscale * 1e-9 * GNOME_TEXT_SCALE;
	      break;
	    default:
	      break;
	    }
	  attr_idx++;
	}
      if (cur_size != last_size || cur_font != last_font)
	{
#ifdef VERBOSE
	  g_print ("cur_size = %d, expands to %g\n",
		   cur_size, ps->current_font_size);
#endif
	  if (open)
	    gnome_print_context_fprintf (pc, ") show\n");
	  gnome_print_ps_setfont_raw (pc, gnome_text_get_font (cur_font),
				      cur_size * 0.001);
	  open = 0;
	  last_size = cur_size;
	  last_font = cur_font;
	}
#ifdef VERBOSE
      g_print ("x = %d, glyph x = %d\n",
	       x, line->glyphs[i].x);
#endif
      if (abs (line->glyphs[i].x - x) > 1)
	{
	  gnome_print_context_fprintf (pc, "%s%g 0 rmoveto\n",
				       open ? ") show " : "",
				       ((line->glyphs[i].x - x) * 1.0 /
					GNOME_TEXT_SCALE));
	  open = 0;
	  x = line->glyphs[i].x;
	}
      glyph = line->glyphs[i].glyph_num;
      if (!open)
	gnome_print_context_fprintf (pc, "(");
      if (glyph >= ' ' && glyph < 0x7f)
	if (glyph == '(' || glyph == ')' || glyph == '\\')
	  gnome_print_context_fprintf (pc, "\\%c", glyph);
	else
	  gnome_print_context_fprintf (pc, "%c", glyph);
      else
	gnome_print_context_fprintf (pc, "\\%03o", glyph);
      open = 1;
      x += floor (gnome_text_get_width (cur_font, glyph) * scale_factor + 0.5);
    }
  if (open)
    gnome_print_context_fprintf (pc, ") show\n");
  ps->cur_font = cur_font;
  ps->cur_size = cur_size;
  return 0;
}

int
gnome_print_ps_showpage (GnomePrintContext *pc)
{
  return gnome_print_context_fprintf (pc,
                                      "showpage\n");
}

static int
gnome_print_ps_close (GnomePrintContext *pc)
{
  gnome_print_context_fprintf (pc,
			       "%%%%EOF\n");
  return gnome_print_context_close_file (pc);
}

static void
gnome_print_ps_class_init (GnomePrintPsClass *class)
{
  GtkObjectClass *object_class;
  GnomePrintContextClass *pc_class;

  object_class = (GtkObjectClass *)class;
  pc_class = (GnomePrintContextClass *)class;

  parent_class = gtk_type_class (gnome_print_context_get_type ());

  object_class->finalize = gnome_print_ps_finalize;

  /* initialization code, autogenned */
  pc_class->newpath = gnome_print_ps_newpath;
  pc_class->moveto = gnome_print_ps_moveto;
  pc_class->lineto = gnome_print_ps_lineto;
  pc_class->curveto = gnome_print_ps_curveto;
  pc_class->closepath = gnome_print_ps_closepath;
  pc_class->setrgbcolor = gnome_print_ps_setrgbcolor;
  pc_class->fill = gnome_print_ps_fill;
  pc_class->eofill = gnome_print_ps_eofill;
  pc_class->setlinewidth = gnome_print_ps_setlinewidth;
  pc_class->setmiterlimit = gnome_print_ps_setmiterlimit;
  pc_class->setlinejoin = gnome_print_ps_setlinejoin;
  pc_class->setlinecap = gnome_print_ps_setlinecap;
  pc_class->setdash = gnome_print_ps_setdash;
  pc_class->strokepath = gnome_print_ps_strokepath;
  pc_class->stroke = gnome_print_ps_stroke;
  pc_class->setfont = gnome_print_ps_setfont;
  pc_class->show = gnome_print_ps_show;
  pc_class->concat = gnome_print_ps_concat;
  pc_class->setmatrix = gnome_print_ps_setmatrix;
  pc_class->gsave = gnome_print_ps_gsave;
  pc_class->grestore = gnome_print_ps_grestore;
  pc_class->clip = gnome_print_ps_clip;
  pc_class->eoclip = gnome_print_ps_eoclip;
  pc_class->grayimage = gnome_print_ps_grayimage;
  pc_class->rgbimage = gnome_print_ps_rgbimage;
  pc_class->textline = gnome_print_ps_textline;
  pc_class->showpage = gnome_print_ps_showpage;

  pc_class->close = gnome_print_ps_close;
}

/* These are the PostScript 35, assumed to be in the printer. */
static char *gnome_print_ps_builtins[] = {
  "AvantGarde-Book",
  "AvantGarde-BookOblique",
  "AvantGarde-Demi",
  "AvantGarde-DemiOblique",
  "Bookman-Demi",
  "Bookman-DemiItalic",
  "Bookman-Light",
  "Bookman-LightItalic",
  "Courier",
  "Courier-Bold",
  "Courier-BoldOblique",
  "Courier-Oblique",
  "Helvetica",
  "Helvetica-Bold",
  "Helvetica-BoldOblique",
  "Helvetica-Narrow",
  "Helvetica-Narrow-Bold",
  "Helvetica-Narrow-BoldOblique",
  "Helvetica-Narrow-Oblique",
  "Helvetica-Oblique",
  "NewCenturySchlbk-Bold",
  "NewCenturySchlbk-BoldItalic",
  "NewCenturySchlbk-Italic",
  "NewCenturySchlbk-Roman",
  "Palatino-Bold",
  "Palatino-BoldItalic",
  "Palatino-Italic",
  "Palatino-Roman",
  "Symbol",
  "Times-Bold",
  "Times-BoldItalic",
  "Times-Italic",
  "Times-Roman",
  "ZapfChancery-MediumItalic",
  "ZapfDingbats"
};

static void
gnome_print_ps_init (GnomePrintPs *ps)
{
  int i;

  ps->n_fonts = 0;
  ps->n_fonts_max = 16;
  ps->fonts = g_new (char *, ps->n_fonts_max);

  ps->n_builtins = sizeof(gnome_print_ps_builtins) / sizeof(char *);
  ps->n_builtins_max = sizeof(gnome_print_ps_builtins) / sizeof(char *);
  ps->builtins = g_new (char *, ps->n_builtins_max);

  for (i = 0; i < sizeof(gnome_print_ps_builtins) / sizeof(char *); i++)
    ps->builtins[i] = gnome_print_ps_builtins[i];

  ps->r = 0;
  ps->g = 0;
  ps->b = 0;

  ps->current_font = NULL;
  ps->current_font_size = 0;
}

GnomePrintPs *
gnome_print_ps_new (GnomePrinter *printer)
{
  GnomePrintPs *ps;
  GnomePrintContext *pc;

  ps = gtk_type_new (gnome_print_ps_get_type ());

  gnome_print_context_open_file (GNOME_PRINT_CONTEXT (ps), printer->filename);

  pc = GNOME_PRINT_CONTEXT (ps);

  gnome_print_context_fprintf (pc,
			       "%%!PS-Adobe-2.0\n"
			       "%%%% Creator: Gnome Print Version %s\n"
			       "%%%% DocumentName: %s\n"
			       "%%%% Author: %s\n"
			       "%%%%EndComments\n\n\n",
			       VERSION,
			       "Document Name Goes Here",
			       "Author Goes Here");
  return ps;
}


static void
gnome_print_ps_finalize (GtkObject *object)
{
  GnomePrintPs *ps;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_PRINT_PS (object));

  ps = GNOME_PRINT_PS (object);

  g_free (ps->fonts);

  (* GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

