/*
 *  Copyright (C) 2000 Helix Code Inc.
 *
 *  Authors: Michael Zucchi <notzed@helixcode.com>
 *           Miguel de Icaza (miguel@gnu.org)
 *
 *  A system print preview window.  Based on print-preview.c
 *  from gnumeric.
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * gnome-print-master.c: Master interface for printing
 * Based on print.c and print-preview.c from gnumeric.
 *
 * Authors:
 *    Miguel de Icaza (miguel@gnu.org)
 *    Michael Zucchi <zucchi#helixcode.com>
 *
 * Given the large memory usage of an entire sheet on
 * a canvas, we have now taken a new approach: we keep in
 * a GNOME Print Metafile each page.  And we render this
 * metafile into the printing context on page switch.
 */
#include <config.h>
#include <gnome.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeprint/gnome-printer.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-meta.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-printer-dialog.h>

#include "gnome-print-master-preview.h"

/* Note: these images are only here temporarily.
   It will be changed to use different disk-based images once I work
   out the interface (probably like EOG) - Z */

/* XPM */
static char * stock_zoom_in_xpm[] = {
"24 24 163 2",
"  	c None",
". 	c #000000",
"+ 	c #30302D",
"@ 	c #7E7E75",
"# 	c #B3B3A5",
"$ 	c #CFCFBE",
"% 	c #D2D2C1",
"& 	c #BABAAA",
"* 	c #848479",
"= 	c #31312D",
"- 	c #87877F",
"; 	c #D7D7CC",
"> 	c #DADACD",
", 	c #DADACC",
"' 	c #DADACB",
") 	c #DCDCCC",
"! 	c #DFDFCF",
"~ 	c #DEDECD",
"{ 	c #D9D9C8",
"] 	c #7E7E73",
"^ 	c #A6A69D",
"/ 	c #E2E2D9",
"( 	c #E9E9E2",
"_ 	c #EAEAE2",
": 	c #E6E6DB",
"< 	c #E3E3D5",
"[ 	c #E3E3D3",
"} 	c #DFDFCE",
"| 	c #D9D9C7",
"1 	c #D4D4C3",
"2 	c #929285",
"3 	c #86867E",
"4 	c #E0E0D8",
"5 	c #F0F0EB",
"6 	c #F8F8F5",
"7 	c #F7F7F3",
"8 	c #F0F0E9",
"9 	c #DADACA",
"0 	c #D3D3C2",
"a 	c #CECEBD",
"b 	c #C9C9B8",
"c 	c #C4C4B4",
"d 	c #717168",
"e 	c #30302C",
"f 	c #D6D6CA",
"g 	c #E7E7DF",
"h 	c #F6F6F3",
"i 	c #FEFEFD",
"j 	c #FCFCFA",
"k 	c #F5F5EF",
"l 	c #D6D6C6",
"m 	c #C4C4B3",
"n 	c #BFBFAE",
"o 	c #B9B9AA",
"p 	c #282824",
"q 	c #D9D9CC",
"r 	c #E6E6DD",
"s 	c #F3F3EE",
"t 	c #FAFAF7",
"u 	c #F9F9F5",
"v 	c #D0D0C0",
"w 	c #C8C8B9",
"x 	c #C3C3B3",
"y 	c #BEBEAE",
"z 	c #B9B9A9",
"A 	c #B4B4A4",
"B 	c #68685F",
"C 	c #B3B3A6",
"D 	c #E4E4D8",
"E 	c #EDEDE4",
"F 	c #F2F2EA",
"G 	c #EFEFE6",
"H 	c #E7E7DC",
"I 	c #CACABA",
"J 	c #BEBEAD",
"K 	c #B8B8A9",
"L 	c #B3B3A4",
"M 	c #AEAE9F",
"N 	c #909083",
"O 	c #D1D1C0",
"P 	c #DBDBCC",
"Q 	c #E2E2D4",
"R 	c #A9A99A",
"S 	c #A1A192",
"T 	c #D5D5C3",
"U 	c #E2E2D1",
"V 	c #A8A899",
"W 	c #A3A395",
"X 	c #9B9B8D",
"Y 	c #BDBDAE",
"Z 	c #DBDBCA",
"` 	c #D2D2C2",
" .	c #CDCDBD",
"..	c #C8C8B8",
"+.	c #B7B7A7",
"@.	c #B2B2A2",
"#.	c #ADAD9E",
"$.	c #A3A394",
"%.	c #9E9E8F",
"&.	c #828275",
"*.	c #DADAC8",
"=.	c #D5D5C4",
"-.	c #D0D0BF",
";.	c #CBCBBA",
">.	c #C6C6B5",
",.	c #C1C1B0",
"'.	c #B1B1A2",
").	c #ACAC9D",
"!.	c #A7A798",
"~.	c #A2A294",
"{.	c #9D9D8F",
"].	c #98988A",
"^.	c #57574F",
"/.	c #CACAB9",
"(.	c #C5C5B5",
"_.	c #C0C0B0",
":.	c #BBBBAB",
"<.	c #A2A293",
"[.	c #9D9D8E",
"}.	c #989889",
"|.	c #1F1F1C",
"1.	c #7B7B71",
"2.	c #C5C5B4",
"3.	c #C0C0AF",
"4.	c #BABAAB",
"5.	c #B5B5A6",
"6.	c #A6A697",
"7.	c #A1A193",
"8.	c #9C9C8E",
"9.	c #979789",
"0.	c #929284",
"a.	c #53534B",
"b.	c #8D8D82",
"c.	c #BFBFAF",
"d.	c #B5B5A5",
"e.	c #B0B0A0",
"f.	c #ABAB9C",
"g.	c #9C9C8D",
"h.	c #969689",
"i.	c #919184",
"j.	c #65655C",
"k.	c #6E6E65",
"l.	c #B4B4A5",
"m.	c #AFAFA0",
"n.	c #AAAA9B",
"o.	c #A5A596",
"p.	c #A0A092",
"q.	c #969688",
"r.	c #919183",
"s.	c #241D14",
"t.	c #211100",
"u.	c #262623",
"v.	c #8C8C80",
"w.	c #7F7F74",
"x.	c #55554E",
"y.	c #E9D1B8",
"z.	c #A6825C",
"A.	c #582D00",
"B.	c #0B0600",
"C.	c #160B00",
"D.	c #4D2800",
"E.	c #371C00",
"F.	c #372B1F",
"              . . . . . .                       ",
"          . . . . . . . . . .                   ",
"        . . + @ # $ % & * = . .                 ",
"      . . - ; > , ' ) ! ~ { ] . .               ",
"    . . ^ / ( _ : < [ } | 1 $ 2 . .             ",
"  . . 3 4 5 6 7 8 . . 9 0 a b c d . .           ",
"  . e f g h i j k . . l $ b m n o p .           ",
". . @ q r s t u 8 . . v w x y z A B . .         ",
". . C , D E F G H . . I x J K L M N . .         ",
". . O P Q . . . . . . . . . . M R S . .         ",
". . T } U . . . . . . . . . . V W X . .         ",
". . Y } Z l `  .... . +.@.#.V $.%.&.. .         ",
". . * *.=.-.;.>.,.. . '.).!.~.{.].^.. .         ",
"  . e 1 $ /.(._.:.. . ).!.<.[.}.2 |..           ",
"  . . 1./.2.3.4.5.. . 6.7.8.9.0.a.. .           ",
"    . . b.c.& d.e.f.6.S g.h.i.j.. . .           ",
"      . . k.l.m.n.o.p.X q.r.a.. . s.t..         ",
"        . . u.j.v.[.}.w.x.|.. . s.y.z.A.B.      ",
"          . . . . . . . . . . . C.z.y.z.A.B.    ",
"              . . . . . .       . D.z.y.z.A..   ",
"                                  B.A.z.y.z.E.. ",
"                                    B.A.z.y.z.. ",
"                                      B.D.z.y.. ",
"                                        . C.F.. "};


/* XPM */
static char * stock_zoom_out_xpm[] = {
"24 24 176 2",
"  	c None",
". 	c #000000",
"+ 	c #30302D",
"@ 	c #7E7E75",
"# 	c #B3B3A5",
"$ 	c #CFCFBE",
"% 	c #D2D2C1",
"& 	c #BABAAA",
"* 	c #848479",
"= 	c #31312D",
"- 	c #87877F",
"; 	c #D7D7CC",
"> 	c #DADACD",
", 	c #DADACC",
"' 	c #DADACB",
") 	c #DCDCCC",
"! 	c #DFDFCF",
"~ 	c #DEDECD",
"{ 	c #D9D9C8",
"] 	c #7E7E73",
"^ 	c #A6A69D",
"/ 	c #E2E2D9",
"( 	c #E9E9E2",
"_ 	c #EAEAE2",
": 	c #E6E6DB",
"< 	c #E3E3D5",
"[ 	c #E3E3D3",
"} 	c #DFDFCE",
"| 	c #D9D9C7",
"1 	c #D4D4C3",
"2 	c #929285",
"3 	c #86867E",
"4 	c #E0E0D8",
"5 	c #F0F0EB",
"6 	c #F8F8F5",
"7 	c #F7F7F3",
"8 	c #F0F0E9",
"9 	c #EAEADF",
"0 	c #E2E2D4",
"a 	c #DADACA",
"b 	c #D3D3C2",
"c 	c #CECEBD",
"d 	c #C9C9B8",
"e 	c #C4C4B4",
"f 	c #717168",
"g 	c #30302C",
"h 	c #D6D6CA",
"i 	c #E7E7DF",
"j 	c #F6F6F3",
"k 	c #FEFEFD",
"l 	c #FCFCFA",
"m 	c #F5F5EF",
"n 	c #DFDFD0",
"o 	c #D6D6C6",
"p 	c #C4C4B3",
"q 	c #BFBFAE",
"r 	c #B9B9AA",
"s 	c #282824",
"t 	c #D9D9CC",
"u 	c #E6E6DD",
"v 	c #F3F3EE",
"w 	c #FAFAF7",
"x 	c #F9F9F5",
"y 	c #E4E4D9",
"z 	c #D9D9CA",
"A 	c #D0D0C0",
"B 	c #C8C8B9",
"C 	c #C3C3B3",
"D 	c #BEBEAE",
"E 	c #B9B9A9",
"F 	c #B4B4A4",
"G 	c #68685F",
"H 	c #B3B3A6",
"I 	c #E4E4D8",
"J 	c #EDEDE4",
"K 	c #F2F2EA",
"L 	c #EFEFE6",
"M 	c #E7E7DC",
"N 	c #DCDCCF",
"O 	c #D2D2C3",
"P 	c #CACABA",
"Q 	c #BEBEAD",
"R 	c #B8B8A9",
"S 	c #B3B3A4",
"T 	c #AEAE9F",
"U 	c #909083",
"V 	c #D1D1C0",
"W 	c #DBDBCC",
"X 	c #A9A99A",
"Y 	c #A1A192",
"Z 	c #D5D5C3",
"` 	c #E2E2D1",
" .	c #A8A899",
"..	c #A3A395",
"+.	c #9B9B8D",
"@.	c #BDBDAE",
"#.	c #DBDBCA",
"$.	c #D2D2C2",
"%.	c #CDCDBD",
"&.	c #C8C8B8",
"*.	c #C1C1B2",
"=.	c #BCBCAC",
"-.	c #B7B7A7",
";.	c #B2B2A2",
">.	c #ADAD9E",
",.	c #A3A394",
"'.	c #9E9E8F",
").	c #828275",
"!.	c #DADAC8",
"~.	c #D5D5C4",
"{.	c #D0D0BF",
"].	c #CBCBBA",
"^.	c #C6C6B5",
"/.	c #C1C1B0",
"(.	c #BBBBAC",
"_.	c #B6B6A7",
":.	c #B1B1A2",
"<.	c #ACAC9D",
"[.	c #A7A798",
"}.	c #A2A294",
"|.	c #9D9D8F",
"1.	c #98988A",
"2.	c #57574F",
"3.	c #CACAB9",
"4.	c #C5C5B5",
"5.	c #C0C0B0",
"6.	c #BBBBAB",
"7.	c #B6B6A6",
"8.	c #B1B1A1",
"9.	c #A2A293",
"0.	c #9D9D8E",
"a.	c #989889",
"b.	c #1F1F1C",
"c.	c #7B7B71",
"d.	c #C5C5B4",
"e.	c #C0C0AF",
"f.	c #BABAAB",
"g.	c #B5B5A6",
"h.	c #B0B0A1",
"i.	c #ABAB9C",
"j.	c #A6A697",
"k.	c #A1A193",
"l.	c #9C9C8E",
"m.	c #979789",
"n.	c #929284",
"o.	c #53534B",
"p.	c #8D8D82",
"q.	c #BFBFAF",
"r.	c #B5B5A5",
"s.	c #B0B0A0",
"t.	c #9C9C8D",
"u.	c #969689",
"v.	c #919184",
"w.	c #65655C",
"x.	c #6E6E65",
"y.	c #B4B4A5",
"z.	c #AFAFA0",
"A.	c #AAAA9B",
"B.	c #A5A596",
"C.	c #A0A092",
"D.	c #969688",
"E.	c #919183",
"F.	c #241D14",
"G.	c #211100",
"H.	c #262623",
"I.	c #8C8C80",
"J.	c #7F7F74",
"K.	c #55554E",
"L.	c #E9D1B8",
"M.	c #A6825C",
"N.	c #582D00",
"O.	c #0B0600",
"P.	c #160B00",
"Q.	c #4D2800",
"R.	c #371C00",
"S.	c #372B1F",
"              . . . . . .                       ",
"          . . . . . . . . . .                   ",
"        . . + @ # $ % & * = . .                 ",
"      . . - ; > , ' ) ! ~ { ] . .               ",
"    . . ^ / ( _ : < [ } | 1 $ 2 . .             ",
"  . . 3 4 5 6 7 8 9 0 a b c d e f . .           ",
"  . g h i j k l m 9 n o $ d p q r s .           ",
". . @ t u v w x 8 y z A B C D E F G . .         ",
". . H , I J K L M N O P C Q R S T U . .         ",
". . V W 0 . . . . . . . . . . T X Y . .         ",
". . Z } ` . . . . . . . . . .  ...+.. .         ",
". . @.} #.o $.%.&.*.=.-.;.>. .,.'.).. .         ",
". . * !.~.{.].^./.(._.:.<.[.}.|.1.2.. .         ",
"  . g 1 $ 3.4.5.6.7.8.<.[.9.0.a.2 b..           ",
"  . . c.3.d.e.f.g.h.i.j.k.l.m.n.o.. .           ",
"    . . p.q.& r.s.i.j.Y t.u.v.w.. . .           ",
"      . . x.y.z.A.B.C.+.D.E.o.. . F.G..         ",
"        . . H.w.I.0.a.J.K.b.. . F.L.M.N.O.      ",
"          . . . . . . . . . . . P.M.L.M.N.O.    ",
"              . . . . . .       . Q.M.L.M.N..   ",
"                                  O.N.M.L.M.R.. ",
"                                    O.N.M.L.M.. ",
"                                      O.Q.M.L.. ",
"                                        . P.S.. "};


/* XPM */
static char * stock_zoom_fit_xpm[] = {
"24 24 154 2",
"  	c None",
". 	c #000000",
"+ 	c #30302D",
"@ 	c #7E7E75",
"# 	c #B3B3A5",
"$ 	c #CFCFBE",
"% 	c #D2D2C1",
"& 	c #BABAAA",
"* 	c #848479",
"= 	c #31312D",
"- 	c #87877F",
"; 	c #D7D7CC",
"> 	c #DADACD",
", 	c #DADACC",
"' 	c #DADACB",
") 	c #DCDCCC",
"! 	c #DFDFCF",
"~ 	c #DEDECD",
"{ 	c #D9D9C8",
"] 	c #7E7E73",
"^ 	c #A6A69D",
"/ 	c #E2E2D9",
"( 	c #E9E9E2",
"_ 	c #EAEAE2",
": 	c #E6E6DB",
"< 	c #E3E3D5",
"[ 	c #E3E3D3",
"} 	c #DFDFCE",
"| 	c #D9D9C7",
"1 	c #D4D4C3",
"2 	c #929285",
"3 	c #86867E",
"4 	c #E0E0D8",
"5 	c #EAEADF",
"6 	c #E2E2D4",
"7 	c #C4C4B4",
"8 	c #717168",
"9 	c #30302C",
"0 	c #D6D6CA",
"a 	c #E7E7DF",
"b 	c #DFDFD0",
"c 	c #BFBFAE",
"d 	c #B9B9AA",
"e 	c #282824",
"f 	c #D9D9CC",
"g 	c #E6E6DD",
"h 	c #F9F9F5",
"i 	c #F0F0E9",
"j 	c #E4E4D9",
"k 	c #D9D9CA",
"l 	c #D0D0C0",
"m 	c #C8C8B9",
"n 	c #B9B9A9",
"o 	c #B4B4A4",
"p 	c #68685F",
"q 	c #B3B3A6",
"r 	c #E4E4D8",
"s 	c #EFEFE6",
"t 	c #E7E7DC",
"u 	c #DCDCCF",
"v 	c #D2D2C3",
"w 	c #CACABA",
"x 	c #C3C3B3",
"y 	c #B3B3A4",
"z 	c #AEAE9F",
"A 	c #909083",
"B 	c #D1D1C0",
"C 	c #DBDBCC",
"D 	c #E8E8DB",
"E 	c #E6E6DA",
"F 	c #E2E2D5",
"G 	c #CACABB",
"H 	c #BDBDAD",
"I 	c #B8B8A8",
"J 	c #B3B3A3",
"K 	c #A9A99A",
"L 	c #A1A192",
"M 	c #D5D5C3",
"N 	c #E2E2D1",
"O 	c #DBDBCB",
"P 	c #D6D6C7",
"Q 	c #D0D0C1",
"R 	c #C3C3B2",
"S 	c #BDBDAC",
"T 	c #B7B7A8",
"U 	c #B2B2A3",
"V 	c #ADAD9E",
"W 	c #A8A899",
"X 	c #A3A395",
"Y 	c #9B9B8D",
"Z 	c #BDBDAE",
"` 	c #DBDBCA",
" .	c #CDCDBD",
"..	c #C8C8B8",
"+.	c #C1C1B2",
"@.	c #BCBCAC",
"#.	c #B7B7A7",
"$.	c #B2B2A2",
"%.	c #A3A394",
"&.	c #9E9E8F",
"*.	c #828275",
"=.	c #DADAC8",
"-.	c #D5D5C4",
";.	c #C6C6B5",
">.	c #C1C1B0",
",.	c #BBBBAC",
"'.	c #B6B6A7",
").	c #B1B1A2",
"!.	c #ACAC9D",
"~.	c #9D9D8F",
"{.	c #98988A",
"].	c #57574F",
"^.	c #B6B6A6",
"/.	c #B1B1A1",
"(.	c #989889",
"_.	c #1F1F1C",
":.	c #7B7B71",
"<.	c #CACAB9",
"[.	c #B0B0A1",
"}.	c #ABAB9C",
"|.	c #929284",
"1.	c #53534B",
"2.	c #8D8D82",
"3.	c #BFBFAF",
"4.	c #B5B5A5",
"5.	c #B0B0A0",
"6.	c #A6A697",
"7.	c #9C9C8D",
"8.	c #969689",
"9.	c #919184",
"0.	c #65655C",
"a.	c #6E6E65",
"b.	c #B4B4A5",
"c.	c #AFAFA0",
"d.	c #AAAA9B",
"e.	c #A5A596",
"f.	c #A0A092",
"g.	c #969688",
"h.	c #919183",
"i.	c #241D14",
"j.	c #211100",
"k.	c #262623",
"l.	c #8C8C80",
"m.	c #9D9D8E",
"n.	c #7F7F74",
"o.	c #55554E",
"p.	c #E9D1B8",
"q.	c #A6825C",
"r.	c #582D00",
"s.	c #0B0600",
"t.	c #160B00",
"u.	c #4D2800",
"v.	c #371C00",
"w.	c #372B1F",
"              . . . . . .                       ",
"          . . . . . . . . . .                   ",
"        . . + @ # $ % & * = . .                 ",
"      . . - ; > , ' ) ! ~ { ] . .               ",
"    . . ^ / ( _ : < [ } | 1 $ 2 . .             ",
"  . . 3 4 . . . . 5 6 . . . . 7 8 . .           ",
"  . 9 0 a . . . . 5 b . . . . c d e .           ",
". . @ f g . . h i j k l m . . n o p . .         ",
". . q , r . . s t u v w x . . y z A . .         ",
". . B C 6 D E F > v G x H I J z K L . .         ",
". . M } N ! O P Q w R S T U V W X Y . .         ",
". . Z } ` . .  ...+.@.#.$.. . %.&.*.. .         ",
". . * =.-.. . ;.>.,.'.).!.. . ~.{.].. .         ",
"  . 9 1 $ . . . . ^./.. . . . (.2 _..           ",
"  . . :.<.. . . . [.}.. . . . |.1.. .           ",
"    . . 2.3.& 4.5.}.6.L 7.8.9.0.. . .           ",
"      . . a.b.c.d.e.f.Y g.h.1.. . i.j..         ",
"        . . k.0.l.m.(.n.o._.. . i.p.q.r.s.      ",
"          . . . . . . . . . . . t.q.p.q.r.s.    ",
"              . . . . . .       . u.q.p.q.r..   ",
"                                  s.r.q.p.q.v.. ",
"                                    s.r.q.p.q.. ",
"                                      s.u.q.p.. ",
"                                        . t.w.. "};


/* XPM */
static char * stock_zoom_fit_wide_xpm[] = {
"24 24 130 2",
"  	c None",
". 	c #000000",
"+ 	c #30302D",
"@ 	c #7E7E75",
"# 	c #B3B3A5",
"$ 	c #CFCFBE",
"% 	c #D2D2C1",
"& 	c #BABAAA",
"* 	c #848479",
"= 	c #31312D",
"- 	c #87877F",
"; 	c #D7D7CC",
"> 	c #DADACD",
", 	c #DADACC",
"' 	c #DADACB",
") 	c #DCDCCC",
"! 	c #DFDFCF",
"~ 	c #DEDECD",
"{ 	c #D9D9C8",
"] 	c #7E7E73",
"^ 	c #A6A69D",
"/ 	c #E2E2D9",
"( 	c #E9E9E2",
"_ 	c #EAEAE2",
": 	c #E6E6DB",
"< 	c #E3E3D5",
"[ 	c #E3E3D3",
"} 	c #DFDFCE",
"| 	c #D9D9C7",
"1 	c #D4D4C3",
"2 	c #929285",
"3 	c #86867E",
"4 	c #E0E0D8",
"5 	c #EAEADF",
"6 	c #E2E2D4",
"7 	c #C4C4B4",
"8 	c #717168",
"9 	c #30302C",
"0 	c #D6D6CA",
"a 	c #E7E7DF",
"b 	c #C6C6B5",
"c 	c #DFDFD0",
"d 	c #BFBFAE",
"e 	c #B9B9AA",
"f 	c #282824",
"g 	c #D9D9CC",
"h 	c #E6E6DD",
"i 	c #F9F9F5",
"j 	c #F0F0E9",
"k 	c #E4E4D9",
"l 	c #D9D9CA",
"m 	c #D0D0C0",
"n 	c #C8C8B9",
"o 	c #B9B9A9",
"p 	c #B4B4A4",
"q 	c #68685F",
"r 	c #B3B3A6",
"s 	c #E4E4D8",
"t 	c #B3B3A4",
"u 	c #AEAE9F",
"v 	c #909083",
"w 	c #D1D1C0",
"x 	c #DBDBCC",
"y 	c #E8E8DB",
"z 	c #B3B3A3",
"A 	c #A9A99A",
"B 	c #A1A192",
"C 	c #D5D5C3",
"D 	c #E2E2D1",
"E 	c #ADAD9E",
"F 	c #A8A899",
"G 	c #A3A395",
"H 	c #9B9B8D",
"I 	c #BDBDAE",
"J 	c #DBDBCA",
"K 	c #A3A394",
"L 	c #9E9E8F",
"M 	c #828275",
"N 	c #DADAC8",
"O 	c #D5D5C4",
"P 	c #C1C1B0",
"Q 	c #BBBBAC",
"R 	c #B6B6A7",
"S 	c #B1B1A2",
"T 	c #ACAC9D",
"U 	c #9D9D8F",
"V 	c #98988A",
"W 	c #57574F",
"X 	c #B6B6A6",
"Y 	c #B1B1A1",
"Z 	c #989889",
"` 	c #1F1F1C",
" .	c #7B7B71",
"..	c #CACAB9",
"+.	c #B0B0A1",
"@.	c #ABAB9C",
"#.	c #929284",
"$.	c #53534B",
"%.	c #8D8D82",
"&.	c #BFBFAF",
"*.	c #B5B5A5",
"=.	c #B0B0A0",
"-.	c #A6A697",
";.	c #9C9C8D",
">.	c #969689",
",.	c #919184",
"'.	c #65655C",
").	c #6E6E65",
"!.	c #B4B4A5",
"~.	c #AFAFA0",
"{.	c #AAAA9B",
"].	c #A5A596",
"^.	c #A0A092",
"/.	c #969688",
"(.	c #919183",
"_.	c #241D14",
":.	c #211100",
"<.	c #262623",
"[.	c #8C8C80",
"}.	c #9D9D8E",
"|.	c #7F7F74",
"1.	c #55554E",
"2.	c #E9D1B8",
"3.	c #A6825C",
"4.	c #582D00",
"5.	c #0B0600",
"6.	c #160B00",
"7.	c #4D2800",
"8.	c #371C00",
"9.	c #372B1F",
"              . . . . . .                       ",
"          . . . . . . . . . .                   ",
"        . . + @ # $ % & * = . .                 ",
"      . . - ; > , ' ) ! ~ { ] . .               ",
"    . . ^ / ( _ : < [ } | 1 $ 2 . .             ",
"  . . 3 4 . . . . 5 6 . . . . 7 8 . .           ",
"  . 9 0 a . b b b 5 c b b b . d e f .           ",
". . @ g h . b i j k l m n b . o p q . .         ",
". . r , s . . . . . . . . . . t u v . .         ",
". . w x 6 y . . . . . . . . z u A B . .         ",
". . C } D ! . . . . . . . . E F G H . .         ",
". . I } J . . . . . . . . . . K L M . .         ",
". . * N O . b b P Q R S T b . U V W . .         ",
"  . 9 1 $ . b b b X Y b b b . Z 2 ` .           ",
"  . .  .... . . . +.@.. . . . #.$.. .           ",
"    . . %.&.& *.=.@.-.B ;.>.,.'.. . .           ",
"      . . ).!.~.{.].^.H /.(.$.. . _.:..         ",
"        . . <.'.[.}.Z |.1.` . . _.2.3.4.5.      ",
"          . . . . . . . . . . . 6.3.2.3.4.5.    ",
"              . . . . . .       . 7.3.2.3.4..   ",
"                                  5.4.3.2.3.8.. ",
"                                    5.4.3.2.3.. ",
"                                      5.7.3.2.. ",
"                                        . 6.9.. "};


/* XPM */
static char * stock_zoom_fit_tall_xpm[] = {
"24 24 131 2",
"  	c None",
". 	c #000000",
"+ 	c #30302D",
"@ 	c #7E7E75",
"# 	c #B3B3A5",
"$ 	c #CFCFBE",
"% 	c #D2D2C1",
"& 	c #BABAAA",
"* 	c #848479",
"= 	c #31312D",
"- 	c #87877F",
"; 	c #D7D7CC",
"> 	c #DADACD",
", 	c #DADACC",
"' 	c #DADACB",
") 	c #DCDCCC",
"! 	c #DFDFCF",
"~ 	c #DEDECD",
"{ 	c #D9D9C8",
"] 	c #7E7E73",
"^ 	c #A6A69D",
"/ 	c #E2E2D9",
"( 	c #E9E9E2",
"_ 	c #EAEAE2",
": 	c #E6E6DB",
"< 	c #E3E3D5",
"[ 	c #E3E3D3",
"} 	c #DFDFCE",
"| 	c #D9D9C7",
"1 	c #D4D4C3",
"2 	c #929285",
"3 	c #86867E",
"4 	c #E0E0D8",
"5 	c #EAEADF",
"6 	c #E2E2D4",
"7 	c #C4C4B4",
"8 	c #717168",
"9 	c #30302C",
"0 	c #D6D6CA",
"a 	c #E7E7DF",
"b 	c #C6C6B5",
"c 	c #BFBFAE",
"d 	c #B9B9AA",
"e 	c #282824",
"f 	c #D9D9CC",
"g 	c #E6E6DD",
"h 	c #F9F9F5",
"i 	c #C8C8B9",
"j 	c #B9B9A9",
"k 	c #B4B4A4",
"l 	c #68685F",
"m 	c #B3B3A6",
"n 	c #E4E4D8",
"o 	c #EFEFE6",
"p 	c #C3C3B3",
"q 	c #B3B3A4",
"r 	c #AEAE9F",
"s 	c #909083",
"t 	c #D1D1C0",
"u 	c #DBDBCC",
"v 	c #E8E8DB",
"w 	c #E6E6DA",
"x 	c #E2E2D5",
"y 	c #BDBDAD",
"z 	c #B8B8A8",
"A 	c #B3B3A3",
"B 	c #A9A99A",
"C 	c #A1A192",
"D 	c #D5D5C3",
"E 	c #E2E2D1",
"F 	c #DBDBCB",
"G 	c #D6D6C7",
"H 	c #B7B7A8",
"I 	c #B2B2A3",
"J 	c #ADAD9E",
"K 	c #A8A899",
"L 	c #A3A395",
"M 	c #9B9B8D",
"N 	c #BDBDAE",
"O 	c #DBDBCA",
"P 	c #CDCDBD",
"Q 	c #B2B2A2",
"R 	c #A3A394",
"S 	c #9E9E8F",
"T 	c #828275",
"U 	c #DADAC8",
"V 	c #D5D5C4",
"W 	c #ACAC9D",
"X 	c #9D9D8F",
"Y 	c #98988A",
"Z 	c #57574F",
"` 	c #989889",
" .	c #1F1F1C",
"..	c #7B7B71",
"+.	c #CACAB9",
"@.	c #B0B0A1",
"#.	c #ABAB9C",
"$.	c #929284",
"%.	c #53534B",
"&.	c #8D8D82",
"*.	c #BFBFAF",
"=.	c #B5B5A5",
"-.	c #B0B0A0",
";.	c #A6A697",
">.	c #9C9C8D",
",.	c #969689",
"'.	c #919184",
").	c #65655C",
"!.	c #6E6E65",
"~.	c #B4B4A5",
"{.	c #AFAFA0",
"].	c #AAAA9B",
"^.	c #A5A596",
"/.	c #A0A092",
"(.	c #969688",
"_.	c #919183",
":.	c #241D14",
"<.	c #211100",
"[.	c #262623",
"}.	c #8C8C80",
"|.	c #9D9D8E",
"1.	c #7F7F74",
"2.	c #55554E",
"3.	c #E9D1B8",
"4.	c #A6825C",
"5.	c #582D00",
"6.	c #0B0600",
"7.	c #160B00",
"8.	c #4D2800",
"9.	c #371C00",
"0.	c #372B1F",
"              . . . . . .                       ",
"          . . . . . . . . . .                   ",
"        . . + @ # $ % & * = . .                 ",
"      . . - ; > , ' ) ! ~ { ] . .               ",
"    . . ^ / ( _ : < [ } | 1 $ 2 . .             ",
"  . . 3 4 . . . . 5 6 . . . . 7 8 . .           ",
"  . 9 0 a . b b . . . . b b . c d e .           ",
". . @ f g . b h . . . . i b . j k l . .         ",
". . m , n . b o . . . . p b . q r s . .         ",
". . t u 6 v w x . . . . y z A r B C . .         ",
". . D } E ! F G . . . . H I J K L M . .         ",
". . N } O . b P . . . . Q b . R S T . .         ",
". . * U V . b b . . . . W b . X Y Z . .         ",
"  . 9 1 $ . b b . . . . b b . ` 2  ..           ",
"  . . ..+.. . . . @.#.. . . . $.%.. .           ",
"    . . &.*.& =.-.#.;.C >.,.'.).. . .           ",
"      . . !.~.{.].^./.M (._.%.. . :.<..         ",
"        . . [.).}.|.` 1.2. .. . :.3.4.5.6.      ",
"          . . . . . . . . . . . 7.4.3.4.5.6.    ",
"              . . . . . .       . 8.4.3.4.5..   ",
"                                  6.5.4.3.4.9.. ",
"                                    6.5.4.3.4.. ",
"                                      6.8.4.3.. ",
"                                        . 7.0.. "};

#define TOOLBAR_BUTTON_BASE 5
#define MOVE_INDEX          5

typedef enum {
	MODE_MOVE,
	MODE_MOVE_DRAGGING,
	MODE_ZOOM_IN,
	MODE_ZOOM_OUT
} PreviewMode;

struct _Private {
	/* the output print context, a gnomeprintpreview */
	GnomePrintPreview *output;

	int		     first_page_rendered;

	GnomeUIInfo          *toolbar;

	PreviewMode          mode;

	/*
	 * Used for dragging the sheet
	 */
	int                 drag_anchor_x, drag_anchor_y;
	int                 drag_ofs_x, drag_ofs_y;
};

static void preview_zoom_fit_cmd(GtkToggleButton *t, GnomePrintMasterPreview *pmp);

/*
 * Padding in points around the simulated page
 */
#define PAGE_PAD 4

static void
render_page (GnomePrintMasterPreview *pmp, int page)
{
	const GnomePaper *paper;
	const char *paper_name;
	struct _Private *pp = pmp->private;

	gtk_object_unref (GTK_OBJECT (pp->output));
	pp->output = NULL;

	/*
	 * Create the preview printing context
	 */
	paper = pmp->master->paper;
	paper_name = gnome_paper_name (paper);
	pp->output = (GnomePrintPreview *)gnome_print_preview_new (pmp->canvas, paper_name);

	/*
	 * Reset scrolling region always
	 */
	gnome_canvas_set_scroll_region (
		pmp->canvas,
		0 - PAGE_PAD,
		0 - PAGE_PAD,
		gnome_paper_pswidth (paper) + PAGE_PAD,
		gnome_paper_psheight (paper) + PAGE_PAD);

	gnome_print_meta_render_from_object_page (GNOME_PRINT_CONTEXT(pp->output),
						  GNOME_PRINT_META (gnome_print_master_get_context(pmp->master)),
						  page);
}

static void
goto_page (GnomePrintMasterPreview *pmp, int page)
{
	char text[16];

	if (page == pmp->current_page)
		return;

	pmp->current_page = page;

	sprintf(text, "%d", page+1);
	gtk_entry_set_text (GTK_ENTRY (pmp->page_entry), text);

	render_page (pmp, page);
}

static void
change_page_cmd (GtkEntry *entry, GnomePrintMasterPreview *pmp)
{
	char *text = gtk_entry_get_text (entry);
	int p;

	p = atoi (text);
	p--;
	if (p < 0){
		goto_page (pmp, 0);
		return;
	}
	if (p > pmp->pagecount){
		goto_page (pmp, pmp->pagecount-1);
		return;
	}
	goto_page (pmp, p);
}

static void
do_zoom (GnomePrintMasterPreview *pmp, int factor)
{
	double zoom;

	if (factor > 0)
		zoom = pmp->canvas->pixels_per_unit * 1.25;
	else if (factor < 0)
		zoom = pmp->canvas->pixels_per_unit / 1.25;
	else
		zoom = 1.0;

	gnome_canvas_set_pixels_per_unit (pmp->canvas, zoom);
}

/* Button press handler for the print preview canvas */
static gint
preview_canvas_button_press (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GnomePrintMasterPreview *pmp = data;
	struct _Private *pp = pmp->private;
	int retval;

	retval = FALSE;

	switch (pp->mode) {
	case MODE_MOVE:
		if (event->button != 1)
			break;

		pp->mode = MODE_MOVE_DRAGGING;
		pp->drag_anchor_x = event->x;
		pp->drag_anchor_y = event->y;
		gnome_canvas_get_scroll_offsets (GNOME_CANVAS (widget),
						 &pp->drag_ofs_x,
						 &pp->drag_ofs_y);

		gdk_pointer_grab (widget->window, FALSE,
				  (GDK_POINTER_MOTION_MASK
				   | GDK_POINTER_MOTION_HINT_MASK
				   | GDK_BUTTON_RELEASE_MASK),
				  NULL,
				  NULL, /*cursor_get (GNUMERIC_CURSOR_HAND_CLOSED),*/
				  event->time);

		retval = TRUE;
		break;

	default:
		break;
	}

	return retval;
}

/* Drags the print preview canvas to the specified position */
static void
drag_to (GnomePrintMasterPreview *pmp, int x, int y)
{
	struct _Private *pp = pmp->private;
	int dx, dy;

	dx = pp->drag_anchor_x - x;
	dy = pp->drag_anchor_y - y;

	/* Right now this will suck for diagonal movement.  GtkLayout does not
	 * have a way to scroll itself diagonally, i.e. you have to change the
	 * vertical and horizontal adjustments independently, leading to ugly
	 * visual results.  The canvas freezes and thaws the layout in case of
	 * diagonal movement, forcing it to repaint everything.
	 *
	 * This will be resolved in the canvas.
	 */
	gnome_canvas_scroll_to (pmp->canvas, pp->drag_ofs_x + dx, pp->drag_ofs_y + dy);
}

/* Button release handler for the print preview canvas */
static gint
preview_canvas_button_release (GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GnomePrintMasterPreview *pmp = data;
	struct _Private *pp = pmp->private;
	int retval;

	retval = FALSE;

	switch (pp->mode) {
	case MODE_MOVE_DRAGGING:
		if (event->button != 1)
			break;

		drag_to (pmp, event->x, event->y);
		pp->mode = MODE_MOVE;

		gdk_pointer_ungrab (event->time);
		retval = TRUE;
		break;

	default:
		break;
	}

	return retval;
}

/* Motion notify handler for the print preview canvas */
static gint
preview_canvas_motion (GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
	GnomePrintMasterPreview *pmp = data;
	struct _Private *pp = pmp->private;
	int retval;
	gint x, y;
	GdkModifierType mods;

	retval = FALSE;

	switch (pp->mode) {
	case MODE_MOVE_DRAGGING:
		if (event->is_hint)
			gdk_window_get_pointer (widget->window, &x, &y, &mods);
		else {
			x = event->x;
			y = event->y;
		}

		drag_to (pmp, x, y);
		retval = TRUE;
		break;

	default:
		break;
	}

	return retval;
}


static void
preview_close_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	gtk_object_destroy (GTK_OBJECT (pmp));
}

static void
preview_file_print_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	gnome_print_master_print(pmp->master);
	/* should we clean ourselves up now? */
}

static void
preview_first_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	goto_page (pmp, 0);
}

static void
preview_next_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	int current_page = pmp->current_page;

	if (current_page+2 > pmp->pagecount)
		return;
	goto_page (pmp, current_page+1);
}

static void
preview_prev_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	int current_page = pmp->current_page;

	if (current_page < 1)
		return;
	goto_page (pmp, current_page-1);
}

static void
preview_last_page_cmd (void *unused, GnomePrintMasterPreview *pmp)
{
	goto_page (pmp, pmp->pagecount-1);
}

static void
preview_zoom_in_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	do_zoom(pmp, 1);
}

static void
preview_zoom_out_cmd (GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	do_zoom(pmp, -1);
}

static void
preview_zoom_fit_cmd(GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	double zoomx, zoomy;
	int width = GTK_WIDGET(pmp->canvas)->allocation.width;
	int height = GTK_WIDGET(pmp->canvas)->allocation.height;
	
	zoomx = width/(gnome_paper_pswidth (pmp->master->paper)+5.0+PAGE_PAD);
	zoomy = height/(gnome_paper_psheight (pmp->master->paper)+5.0+PAGE_PAD);
	
	if (zoomy<zoomx)
		zoomx = zoomy;
	gnome_canvas_set_pixels_per_unit (pmp->canvas, zoomx);
}

static void
preview_zoom_fit_wide_cmd(GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	double zoom;
	int width = GTK_WIDGET(pmp->canvas)->allocation.width;
	
	/* include paper decoration */
	zoom = width/(gnome_paper_pswidth (pmp->master->paper)+5.0+PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (pmp->canvas, zoom);
}

static void
preview_zoom_fit_tall_cmd(GtkToggleButton *t, GnomePrintMasterPreview *pmp)
{
	double zoom;
	int height = GTK_WIDGET(pmp->canvas)->allocation.height;
	
	/* include paper decoration */
	zoom = height/(gnome_paper_psheight (pmp->master->paper)+5.0+PAGE_PAD);
	gnome_canvas_set_pixels_per_unit (pmp->canvas, zoom);
}

static gint
preview_canvas_key (GtkWidget *widget, GdkEventKey *event, void *data)
{
	GnomePrintMasterPreview *pmp = data;
	int x,y;
	int height, width;
	int domove=0;

	gnome_canvas_get_scroll_offsets (pmp->canvas, &x, &y);
	height = GTK_WIDGET(pmp->canvas)->allocation.height;
	width = GTK_WIDGET(pmp->canvas)->allocation.width;

	switch (event->keyval) {
	case '1':
		preview_zoom_fit_cmd(0, pmp);
		break;
	case '2':		/* as good a key as any? */
		preview_zoom_fit_wide_cmd(0, pmp);
		break;
	case '3':
		preview_zoom_fit_tall_cmd(0, pmp);
		break;
	case '+':
	case '=':
	case GDK_KP_Add:
		do_zoom(pmp, 1);
		break;
	case '-':
	case '_':
	case GDK_KP_Subtract:
		do_zoom(pmp, -1);
		break;
	case GDK_KP_Right:
	case GDK_Right:
		if (event->state & GDK_SHIFT_MASK)
			x+=width;
		else
			x+=10;
		domove=1;
		break;
	case GDK_KP_Left:
	case GDK_Left:
		if (event->state & GDK_SHIFT_MASK)
			x-=width;
		else
			x-=10;
		domove=1;
		break;
	case GDK_KP_Up:
	case GDK_Up:
		if (event->state & GDK_SHIFT_MASK)
			goto page_up;
		y-=10;
		domove=1;
		break;
	case GDK_KP_Down:
	case GDK_Down:
		if (event->state & GDK_SHIFT_MASK)
			goto page_down;
		y+=10;
		domove=1;
		break;
	case GDK_KP_Page_Up:
	case GDK_Page_Up:
	case GDK_Delete:
	case GDK_KP_Delete:
	case GDK_BackSpace:
	page_up:
		if (y<=0) {
			if (pmp->current_page>0) {
				goto_page(pmp, pmp->current_page-1);
				y = GTK_LAYOUT(pmp->canvas)->height-height;
			}
		} else {
			y-=height;
		}
		domove=1;
		break;
	case GDK_KP_Page_Down:
	case GDK_Page_Down:
	case ' ':
	page_down:
		if (y>=GTK_LAYOUT(pmp->canvas)->height-height) {
			if (pmp->current_page<pmp->pagecount-1) {
				goto_page(pmp, pmp->current_page+1);
				y=0;
			}
		} else {
			y+=height;
		}
		domove=1;
		break;
	case GDK_KP_Home:
	case GDK_Home:
		goto_page (pmp, 0);
		y=0;
		domove=1;
		break;
	case GDK_KP_End:
	case GDK_End:
		goto_page (pmp, pmp->pagecount-1);
		y=0;
		domove=1;
		break;
	case GDK_Escape:
		gtk_object_destroy (GTK_OBJECT (pmp));
		return TRUE;
	default:
		return FALSE;
	}

	if (domove)
		gnome_canvas_scroll_to (pmp->canvas, x, y);

	gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
	return TRUE;
}

static void
create_preview_canvas (GnomePrintMasterPreview *pmp)
{
	GtkWidget *box, *status;
	struct _Private *pp = pmp->private;
	const GnomePaper *paper;
	const char *paper_name;
	GnomeCanvasItem *i;

	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	gtk_widget_push_visual (gdk_rgb_get_visual ());

	pmp->scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	pmp->canvas = GNOME_CANVAS (gnome_canvas_new_aa ());
	gnome_canvas_set_pixels_per_unit (pmp->canvas, 1.0);

	gtk_signal_connect (GTK_OBJECT (pmp->canvas), "button_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_press),
			    pmp);
	gtk_signal_connect (GTK_OBJECT (pmp->canvas), "button_release_event",
			    GTK_SIGNAL_FUNC (preview_canvas_button_release),
			    pmp);
	gtk_signal_connect (GTK_OBJECT (pmp->canvas), "motion_notify_event",
			    GTK_SIGNAL_FUNC (preview_canvas_motion),
			    pmp);
	gtk_signal_connect (GTK_OBJECT (pmp->canvas), "key_press_event",
			    GTK_SIGNAL_FUNC (preview_canvas_key),
			    pmp);

	gtk_container_add (GTK_CONTAINER (pmp->scrolled_window), GTK_WIDGET (pmp->canvas));

	/*
	 * Create the preview printing context
	 */
	paper = pmp->master->paper;
	paper_name = gnome_paper_name (paper);
	pp->output = (GnomePrintPreview *)gnome_print_preview_new (pmp->canvas, paper_name);

	/*
	 * Now add some padding above and below and put a simulated
	 * page on the background
	 */
	i = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (gnome_canvas_root (pmp->canvas)),
		gnome_canvas_rect_get_type (),
		"x1",   	 0.0,
		"y1",   	 0.0,
		"x2",   	 (double) gnome_paper_pswidth (paper),
		"y2",   	 (double) gnome_paper_psheight (paper),
		"fill_color",    "white",
		"outline_color", "black",
		"width_pixels",  1,
		NULL);
	gnome_canvas_item_lower_to_bottom (i);
	i = gnome_canvas_item_new (
		GNOME_CANVAS_GROUP (gnome_canvas_root (pmp->canvas)),
		gnome_canvas_rect_get_type (),
		"x1",   	 3.0,
		"y1",   	 3.0,
		"x2",   	 (double) gnome_paper_pswidth (paper) + 3,
		"y2",   	 (double) gnome_paper_psheight (paper) + 3,
		"fill_color",    "black",
		NULL);
	gnome_canvas_item_lower_to_bottom (i);
	gnome_canvas_set_scroll_region (
		pmp->canvas,
		0 - PAGE_PAD,
		0 - PAGE_PAD,
		gnome_paper_pswidth (paper) + PAGE_PAD,
		gnome_paper_psheight (paper) + PAGE_PAD);

	box = gtk_vbox_new (FALSE, 0);
	status = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (status), gtk_label_new (_("Page: ")), FALSE, FALSE, 0);
	pmp->page_entry = gtk_entry_new ();
	gtk_widget_set_usize (pmp->page_entry, 40, 0);
	gtk_signal_connect (GTK_OBJECT (pmp->page_entry), "activate", change_page_cmd, pmp);
	gtk_box_pack_start (GTK_BOX (status), pmp->page_entry, FALSE, FALSE, 0);
	pmp->last = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (status), pmp->last, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (box), status, FALSE, FALSE, 3);
	gtk_box_pack_start (GTK_BOX (box), pmp->scrolled_window, TRUE, TRUE, 0);
	gnome_app_set_contents (GNOME_APP(pmp), box);
	gtk_widget_show_all (box);

	gtk_widget_grab_focus (GTK_WIDGET (pmp->canvas));

	return;
}


static GnomeUIInfo preview_file_menu [] = {
	GNOMEUIINFO_MENU_PRINT_ITEM (preview_file_print_cmd, NULL),
	GNOMEUIINFO_MENU_CLOSE_ITEM (preview_close_cmd, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo preview_view_menu [] = {
	GNOMEUIINFO_ITEM_STOCK (
		N_("_First page"), N_("Shows the first page"),
		preview_first_page_cmd, GNOME_STOCK_PIXMAP_FIRST),
	GNOMEUIINFO_ITEM_STOCK (
		N_("_Previous page"), N_("Shows the previous page"),
		preview_prev_page_cmd, GNOME_STOCK_PIXMAP_BACK),
	GNOMEUIINFO_ITEM_STOCK (
		N_("_Next page"), N_("Shows the next page"),
		preview_next_page_cmd, GNOME_STOCK_PIXMAP_FORWARD),
	GNOMEUIINFO_ITEM_STOCK (
		N_("_Last page"), N_("Shows the last page"),
		preview_last_page_cmd, GNOME_STOCK_PIXMAP_LAST),

	GNOMEUIINFO_SEPARATOR,

	{ GNOME_APP_UI_ITEM, N_("Zoom _in"), N_("Zooms in"), preview_zoom_in_cmd },
	{ GNOME_APP_UI_ITEM, N_("Zoom _out"), N_("Zooms out"), preview_zoom_out_cmd },

	GNOMEUIINFO_END
};

static GnomeUIInfo top_menu [] = {
	GNOMEUIINFO_MENU_FILE_TREE (preview_file_menu),
	{ GNOME_APP_UI_SUBTREE, N_("_View"), NULL, preview_view_menu },
	GNOMEUIINFO_END
};

static GnomeUIInfo toolbar [] = {
	GNOMEUIINFO_ITEM_STOCK (
		N_("Print"), N_("Prints the current file"),
		preview_file_print_cmd, GNOME_STOCK_PIXMAP_PRINT),
	GNOMEUIINFO_ITEM_STOCK (
		N_("First"), N_("Shows the first page"),
		preview_first_page_cmd, GNOME_STOCK_PIXMAP_FIRST),
	GNOMEUIINFO_ITEM_STOCK (
		N_("Back"), N_("Shows the previous page"),
		preview_prev_page_cmd, GNOME_STOCK_PIXMAP_BACK),
	GNOMEUIINFO_ITEM_STOCK (
		N_("Next"), N_("Shows the next page"),
		preview_next_page_cmd, GNOME_STOCK_PIXMAP_FORWARD),
	GNOMEUIINFO_ITEM_STOCK (
		N_("Last"), N_("Shows the last page"),
		preview_last_page_cmd, GNOME_STOCK_PIXMAP_LAST),

	GNOMEUIINFO_ITEM (N_("Zoom in"), N_("Zooms the page in"), preview_zoom_in_cmd, stock_zoom_in_xpm),
	GNOMEUIINFO_ITEM (N_("Zoom out"), N_("Zooms the page out"), preview_zoom_out_cmd, stock_zoom_out_xpm),
	GNOMEUIINFO_ITEM (N_("Fit"), N_("Zooms to fit the whole page"), preview_zoom_fit_cmd, stock_zoom_fit_xpm),
	GNOMEUIINFO_ITEM (N_("Fit Wide"), N_("Zooms to fit the width of the page"), preview_zoom_fit_wide_cmd, stock_zoom_fit_wide_xpm),
	GNOMEUIINFO_ITEM (N_("Fit Tall"), N_("Zooms to fit the height of the page"), preview_zoom_fit_tall_cmd, stock_zoom_fit_tall_xpm),
	GNOMEUIINFO_END
};

static void
create_toplevel (GnomePrintMasterPreview *pmp)
{
	gint width, height;
	const GnomePaper *paper;
	struct _Private *pp;

	g_return_if_fail (pmp != NULL);

	pp = pmp->private;

	paper  = pmp->master->paper;
	width  = gnome_paper_pswidth  (paper) + PAGE_PAD * 3;
	height = gnome_paper_psheight (paper) + PAGE_PAD * 3;

	if (width > gdk_screen_width () - 40)
		width = gdk_screen_width () - 40;

	if (height > gdk_screen_height () - 100)
		height = gdk_screen_height () - 100;

	gtk_widget_set_usize (GTK_WIDGET(pmp), width, height);
	gtk_window_set_policy (GTK_WINDOW (pmp), TRUE, TRUE, FALSE);

	gnome_app_create_menus_with_data (GNOME_APP(pmp), top_menu, pmp);

	pp->toolbar = g_malloc (sizeof (toolbar));
	memcpy (pp->toolbar, toolbar, sizeof (toolbar));

	gnome_app_create_toolbar_with_data (GNOME_APP(pmp), pp->toolbar, pmp);
}

/*
  The GnomePrintMasterPreview object
*/

static void gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *class);
static void gnome_print_master_preview_init       (GnomePrintMasterPreview      *gspaper);

static GnomeAppClass *parent_class;

guint
gnome_print_master_preview_get_type (void)
{
	static guint print_master_preview_type = 0;
	
	if (!print_master_preview_type) {
		GtkTypeInfo print_master_preview_info = {
			"GnomePrintMasterPreview",
			sizeof (GnomePrintMasterPreview),
			sizeof (GnomePrintMasterPreviewClass),
			(GtkClassInitFunc) gnome_print_master_preview_class_init,
			(GtkObjectInitFunc) gnome_print_master_preview_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		print_master_preview_type = gtk_type_unique (gnome_app_get_type (), &print_master_preview_info);
	}
	
	return print_master_preview_type;
}

static void
gnome_print_master_preview_finalize (GtkObject *object)
{
	GnomePrintMasterPreview *pmp = GNOME_PRINT_MASTER_PREVIEW(object);
	struct _Private *pp = pmp->private;

	if (pp->output != NULL)
		gtk_object_unref (GTK_OBJECT (pp->output));
	g_free (pp->toolbar);
	g_free (pp);

	if (pmp->master != NULL)
		gtk_object_unref(GTK_OBJECT(pmp->master));

	GTK_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_print_master_preview_class_init (GnomePrintMasterPreviewClass *class)
{
	GtkObjectClass *object_class;
	
	object_class = (GtkObjectClass *) class;
	parent_class = gtk_type_class (gnome_app_get_type ());

	object_class->finalize = gnome_print_master_preview_finalize;
}

static void
gnome_print_master_preview_init (GnomePrintMasterPreview *pmp)
{
	struct _Private *pp;

	pp = pmp->private = g_malloc0(sizeof(*pp));
	pmp->current_page = -1;
}

/**
 * gnome_print_master_preview_new:
 * @gpm: A GnomePrintMaster which has been printed to and completed.
 * @title: Window title for the preview window.
 * 
 * Create a new preview window widget.  The preview has inbuilt
 * buttons for zooming and panning, and menu's to print.
 * 
 * Return value: A newly created GnomePrintMasterPreview widget.
 **/
GnomePrintMasterPreview *
gnome_print_master_preview_new (GnomePrintMaster *gpm, char *title)
{
	GnomePrintMasterPreview *pmp;
	struct _Private *pp;
	char text[16];

	pmp = GNOME_PRINT_MASTER_PREVIEW (gtk_type_new(gnome_print_master_preview_get_type()));

	gnome_app_construct(GNOME_APP(pmp), "preview_window", title);

	pp = pmp->private;
	pmp->master = gpm;
	gtk_object_ref(GTK_OBJECT(gpm));

	create_toplevel(pmp);
	create_preview_canvas(pmp);

	/* this zooms to fit, once we know how big the window actually is */
	gtk_signal_connect(GTK_OBJECT(pmp->canvas), "realize",
			   GTK_SIGNAL_FUNC(preview_zoom_fit_cmd), pmp);

	pmp->pagecount = gnome_print_master_get_pages(gpm);

	goto_page (pmp, 0);
	
	sprintf(text, "/%d", pmp->pagecount);
	gtk_label_set_text (GTK_LABEL (pmp->last), text);

	return pmp;
}

