/*
 * gnome-print-preview.c: A preview driver for GnomePrint that renders
 * into a GnomeCanvas and a control object for the pages rendered.
 *
 * Authors:
 *   Miguel de Icaza (miguel@gnu.org)
 *   Lauris Kaplinski (lauris@ariman.ee)
 *
 * (C) 1999, 2000 International GNOME Support
 */

#include <config.h>
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include <libgnomeui/gnome-canvas-image.h>
#include <libgnomeprint/gnome-print-private.h>
#include <libgnomeprint/gnome-print-preview.h>
#include <libgnomeprint/gnome-print-preview-private.h>
#include <libgnomeprint/gnome-canvas-bpath.h>
#include <libgnomeprint/gnome-canvas-hacktext.h>
#include <libgnomeprint/gnome-canvas-clipgroup.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gt1-parset1.h>
#include <libart_lgpl/art_pixbuf.h>
#include <libgnome/gnome-paper.h>

#include <libgnomeprint/gp-gc.h>

struct _GnomePrintPreviewPrivate {
	GPGC * gc;

	/* Current page displayed */
	int top_page;
	int current_page;

	/* The root group, with a translation setup */
	GnomeCanvasItem *root;

	/* The current page */
	GnomeCanvasItem *page;
	
	GHashTable *font_hash;
};

static GnomePrintContextClass *print_preview_parent_class;

static int
gpp_newpath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_newpath (pp->priv->gc);

	return 1;
}

static int
gpp_moveto (GnomePrintContext *pc, double x, double y)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	
	gp_gc_moveto (pp->priv->gc, x, y);

	return 1;
}

static int
gpp_lineto (GnomePrintContext *pc, double x, double y)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	g_return_val_if_fail (gp_gc_has_currentpoint (pp->priv->gc), -1);
	
	gp_gc_lineto (pp->priv->gc, x, y);

	return 1;
}

static int
gpp_curveto (GnomePrintContext *pc,
	     double x1, double y1,
	     double x2, double y2,
	     double x3, double y3)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	g_return_val_if_fail (gp_gc_has_currentpoint (pp->priv->gc), -1);
	
	gp_gc_curveto (pp->priv->gc, x1, y1, x2, y2, x3, y3);

	return 1;
}

static int
gpp_closepath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

#if 0
	/* fixme: ? */
	g_return_val_if_fail (gp_path_length (priv->gc->path) > 2, -1);
#endif

	gp_gc_closepath (pp->priv->gc);

	return 1;
}

static int
gpp_setrgbcolor (GnomePrintContext *pc, double r, double g, double b)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_rgbcolor (pp->priv->gc, r, g, b);

	return 1;
}

static int
gpp_stroke (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item;

	/* fixme: currentpath invariants */

	group = (GnomeCanvasGroup *) gp_gc_get_data (priv->gc);
	g_assert (group != NULL);
	g_assert (GNOME_IS_CANVAS_GROUP (group));

	/* Fixme: Can we assume that linewidth == width_units? */
	/* Probably yes, as object->page ctm == identity */

	item = gnome_canvas_item_new (group,
		gnome_canvas_bpath_get_type (),
		"bpath",	gp_gc_get_currentpath (priv->gc),
		"width_units",	gp_gc_get_linewidth (priv->gc),
		"cap_style",	gp_gc_get_linecap (priv->gc),
		"join_style",	gp_gc_get_linejoin (priv->gc),
		"outline_color_rgba", gp_gc_get_rgba (priv->gc),
		"miterlimit",	gp_gc_get_miterlimit (priv->gc),
		"dash",		gp_gc_get_dash (priv->gc),
		NULL);

	/* fixme: is this correct? */
	gp_gc_newpath (priv->gc);

	return 1;
}

static int
gpp_strokepath (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_strokepath (pp->priv->gc);

	return 1;
}

static int
gpp_create_fill (GnomePrintPreviewPrivate * priv, ArtWindRule wind)
{
	GPPath * closedpath;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item;
	
	/* fixme: currentpath invariants */

	closedpath = gp_path_close_all (gp_gc_get_currentpath (priv->gc));
	if (gp_path_is_empty (closedpath)) {
		g_warning ("gpp_create_fill:closepath is empty");
		gp_path_unref (closedpath);
		return -1;
	}

	group = (GnomeCanvasGroup *) gp_gc_get_data (priv->gc);
	g_assert (group != NULL);
	g_assert (GNOME_IS_CANVAS_GROUP (group));

	item = gnome_canvas_item_new (group,
		gnome_canvas_bpath_get_type (),
		"bpath", closedpath,
		"outline_color", NULL,
		"fill_color_rgba", gp_gc_get_rgba (priv->gc),
		"wind", wind,
		NULL);

	gp_path_unref (closedpath);

	gp_gc_newpath (priv->gc);

	return 1;
}

static int
gpp_fill (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	return gpp_create_fill (pp->priv, ART_WIND_RULE_NONZERO);
}

static int
gpp_eofill (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	return gpp_create_fill (pp->priv, ART_WIND_RULE_ODDEVEN);
}

static int
gpp_setlinewidth (GnomePrintContext *pc, double width)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_linewidth (pp->priv->gc, width);

	return 1;
}

static int
gpp_setmiterlimit (GnomePrintContext *pc, double limit)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_miterlimit (pp->priv->gc, limit);

	return 1;
}

static int
gpp_setlinejoin (GnomePrintContext *pc, int jointype)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	switch (jointype) {
	case 0:
		gp_gc_set_linejoin (pp->priv->gc, ART_PATH_STROKE_JOIN_MITER);
		break;
	case 1:
		gp_gc_set_linejoin (pp->priv->gc, ART_PATH_STROKE_JOIN_ROUND);
		break;
	case 2:
		gp_gc_set_linejoin (pp->priv->gc, ART_PATH_STROKE_JOIN_BEVEL);
		break;
	default:
		return -1;
		break;
	}

	return 1;
}

static int
gpp_setlinecap (GnomePrintContext *pc, int captype)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	switch (captype) {
	case 0:
		gp_gc_set_linecap (pp->priv->gc, ART_PATH_STROKE_CAP_BUTT);
		break;
	case 1:
		gp_gc_set_linecap (pp->priv->gc, ART_PATH_STROKE_CAP_ROUND);
		break;
	case 2:
		gp_gc_set_linecap (pp->priv->gc, ART_PATH_STROKE_CAP_SQUARE);
		break;
	default:
		return -1;
		break;
	}

	return 1;
}

static int
gpp_setdash (GnomePrintContext *pc, int n_values, const double *values, double offset)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_dash (pp->priv->gc, n_values, values, offset);

	return 1;
}

static int
gpp_setfont (GnomePrintContext *pc, GnomeFont *font)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_font (pp->priv->gc, font);

	return 1;
}

static int
gpp_concat (GnomePrintContext *pc, const double matrix [6])
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_concat (pp->priv->gc, matrix);

	return 1;
}

/* fixme: should we prepeng root page matrix? */
#ifdef SETMATRIX_DEPRECATED
static int
gpp_setmatrix (GnomePrintContext *pc, const double matrix[6])
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	g_warning ("setmatrix is deprecated");

	gp_gc_setmatrix (pp->priv->gc, matrix);

	return 1;
}
#endif

static int
gpp_gsave (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_gsave (pp->priv->gc);

	return 1;
}

static int
gpp_grestore (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_grestore (pp->priv->gc);

	return 1;
}

static int
gpp_create_clip (GnomePrintPreviewPrivate * priv, ArtWindRule wind)
{
	GPPath * closedpath;
	GnomeCanvasGroup * group;
	GnomeCanvasItem * clip;

	closedpath = gp_path_close_all (gp_gc_get_currentpath (priv->gc));
	if (gp_path_is_empty (closedpath)) {
		g_warning ("gpp_create_clip: closedpath is empty");
		gp_path_unref (closedpath);
		return -1;
	}

	group = (GnomeCanvasGroup *) gp_gc_get_data (priv->gc);

	clip = gnome_canvas_item_new (group,
		gnome_canvas_clipgroup_get_type (),
		"path", closedpath,
		"wind", wind,
		NULL);

	gp_path_unref (closedpath);

	gp_gc_set_data (priv->gc, clip);

	gp_gc_clip (priv->gc);

	return 1;
}

static int
gpp_clip (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	return gpp_create_clip (pp->priv, ART_WIND_RULE_NONZERO);
}

static int
gpp_eoclip (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	return gpp_create_clip (pp->priv, ART_WIND_RULE_ODDEVEN);
}

static int
gpp_image (GnomePrintContext *pc, const char *data, int width, int height, int rowstride, int bytes_per_pixel)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomeCanvasGroup * group;
	GnomeCanvasItem *item;
	ArtPixBuf *pixbuf;
	int size, bpp;
	void *dup;
	
	/*
	 * We do convert gray scale images to RGB
	 */

	if (bytes_per_pixel == 1)
		bpp = 3;
	else
		bpp = bytes_per_pixel;
	
	size = width * height * bpp;
	dup = art_alloc (size);
	if (!dup)
		return -1;

	if (bytes_per_pixel == 3){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 4){
		memcpy (dup, data, size);
		pixbuf = art_pixbuf_new_rgba (dup, width, height, rowstride);
	} else if (bytes_per_pixel == 1){
		const char *source;
		char *target;
		int  ix, iy;

		source = data;
		target = dup;

		for (iy = 0; iy < height; iy++){
			for (ix = 0; ix < width; ix++){
				*target++ = *source;
				*target++ = *source;
				*target++ = *source;
				source++;
			}
		}
		pixbuf = art_pixbuf_new_rgb (dup, width, height, rowstride * 3);
	} else
		return -1;

	group = (GnomeCanvasGroup *) gp_gc_get_data (pp->priv->gc);

	item = gnome_canvas_item_new (group,
		gnome_canvas_image_get_type (),
		"pixbuf", pixbuf,
		"x",      0.0,
		"y",      0.0,
		"width",  1.0,
		"height", 1.0,
		"anchor", GTK_ANCHOR_SW,
		NULL);


	/* Apply the transformation for the image */
	{
		double transform [6];
		double flip [6];
#if 0
		double ident [6];
		
		art_affine_identity (ident);
		art_affine_flip (flip, ident, FALSE, TRUE);
		art_affine_multiply (transform, flip, g->affine);
		art_affine_multiply (transform, transform, g->page_affine);
#else
		art_affine_scale (flip, 1.0, -1.0);
#endif
		art_affine_multiply (transform, flip, gp_gc_get_ctm (pp->priv->gc));

		gnome_canvas_item_affine_relative (item, transform);
	}
	
	return 1;
}

static int
gpp_grayimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	return  gpp_image (pc, data, width, height, rowstride, 1);
}

static int
gpp_rgbimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	return gpp_image (pc, data, width, height, rowstride, 3);
}


static int
gpp_rgbaimage (GnomePrintContext *pc, const char *data, int width, int height, int rowstride)
{
	return gpp_image (pc, data, width, height, rowstride, 4);
}


static int
gpp_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
	g_warning ("Implement textline");

	return -1;
}

static int
gpp_showpage (GnomePrintContext *pc)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
#if 0
	/*
	 * Reset affine and path
	 */
	art_affine_identity (gc (priv)->affine);

	/*
	 * Kill current path
	 */
	gc_clear_bpath (gc (priv));
#endif
#ifdef SETMATRIX_DEPRECATED
	gp_gc_setmatrix (pp->priv->gc, NULL);
#endif	
	gp_gc_newpath (pp->priv->gc);

	return 0;
}

static int
gpp_beginpage (GnomePrintContext *pc, const char *name_of_this_page)
{
	return 0;
}

#if 0
static Gt1LoadedFont *
fetch_font (GnomePrintPreviewPrivate *priv, const char *pfb_file, const char *afm_file)
{
	Gt1LoadedFont *font;
	
	if (!priv->font_hash){
		priv->font_hash = g_hash_table_new (g_str_hash, g_str_equal);
	}

	font = g_hash_table_lookup (priv->font_hash, pfb_file);
	if (font == (void *) -1)
		return NULL;

	if (font)
		return font;

	if (font == NULL){
		font = gt1_load_font (pfb_file);
		if (font == NULL)
			return NULL;

		g_hash_table_insert (priv->font_hash, g_strdup (pfb_file), font);
	}
	return font;
}
#endif

static int
gpp_show_ucs4 (GnomePrintContext *pc, guint32 * buf, gint length)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item;
	const ArtPoint * cp;
	const GnomeFont * font;
	ArtPoint p;
	double transform [6], a[6], inverse[6];
	GnomeGlyphList * gl;
	gint i;

	font = gp_gc_get_font (pp->priv->gc);

	/*
	 * The X and Y positions were computed to be the base
	 * with the translation already done for the new
	 * Postscript->Canvas translation
	 */
	
	cp = gp_gc_get_currentpoint (pp->priv->gc);

	memcpy (transform, gp_gc_get_ctm (priv->gc), sizeof (transform));
	art_affine_scale (a, 1.0, -1.0);
	art_affine_multiply (transform, a, transform);

	art_affine_invert (inverse, transform);
	art_affine_point (&p, cp, inverse);
	group = (GnomeCanvasGroup *) gp_gc_get_data (pp->priv->gc);

	gl = gnome_glyphlist_from_text_dumb ((GnomeFont *) font, gp_gc_get_rgba (pp->priv->gc),
					     0.0, 0.0, "");
	for (i = 0; i < length; i++) {
		gint glyph;
		glyph = gnome_font_lookup_default (font, GUINT32_FROM_BE (buf[i]));
		if (glyph > 0) gnome_glyphlist_glyph (gl, glyph);
	}

	item = gnome_canvas_item_new (group,
		gnome_canvas_hacktext_get_type (),
		"x",           p.x,
		"y",           p.y,
		"glyphlist",   gl,
		NULL);

	gnome_glyphlist_unref (gl);
#if 1
	gnome_canvas_item_affine_absolute (item, transform);
#endif
#if 0
	priv->current_page++;
	
	gc_clear_bpath (gc (priv));
#endif

	/* fixme: is this correct */
	gp_gc_newpath (pp->priv->gc);

	return 0;
}

static int
gpp_glyphlist (GnomePrintContext *pc, GnomeGlyphList * glyphlist)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);
	GnomePrintPreviewPrivate *priv = pp->priv;
	GnomeCanvasGroup *group;
	GnomeCanvasItem *item;
	const ArtPoint * cp;
	ArtPoint p;
	double transform [6], a[6], inverse[6];


	/*
	 * The X and Y positions were computed to be the base
	 * with the translation already done for the new
	 * Postscript->Canvas translation
	 */
	
	cp = gp_gc_get_currentpoint (pp->priv->gc);

	memcpy (transform, gp_gc_get_ctm (priv->gc), sizeof (transform));
	art_affine_scale (a, 1.0, -1.0);
	art_affine_multiply (transform, a, transform);

	art_affine_invert (inverse, transform);
	art_affine_point (&p, cp, inverse);

	group = (GnomeCanvasGroup *) gp_gc_get_data (pp->priv->gc);

	item = gnome_canvas_item_new (group,
		gnome_canvas_hacktext_get_type (),
		"x",           p.x,
		"y",           p.y,
		"glyphlist",        glyphlist,
		NULL);

	gnome_canvas_item_affine_absolute (item, transform);

	/* fixme: is this correct */
	gp_gc_newpath (pp->priv->gc);

	return 0;
}

static int
gpp_setopacity (GnomePrintContext *pc, double opacity)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (pc);

	gp_gc_set_opacity (pp->priv->gc, opacity);

	return 0;
}

static int
gpp_close (GnomePrintContext *pc)
{
	return 0;
}

static gboolean
kill_font (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	gt1_unload_font (value);
	return TRUE;
}

static void
gpp_destroy (GtkObject *object)
{
	GnomePrintPreview *pp = GNOME_PRINT_PREVIEW (object);
	GnomePrintPreviewPrivate *priv = pp->priv;

#if 0
	GSList *l;

	for (l = priv->context_list; l; l = l->next)
		gc_free (l->data);
	g_slist_free (priv->context_list);
#endif
	gp_gc_unref (priv->gc);

	if (pp->canvas)
		gtk_object_unref (GTK_OBJECT (pp->canvas));

	if (priv->font_hash){
		g_hash_table_foreach_remove (priv->font_hash, kill_font, NULL);
		g_hash_table_destroy (priv->font_hash);
	}

	if (priv->page)
		gtk_object_destroy (GTK_OBJECT (priv->page));

	if (priv->root)
		gtk_object_destroy (GTK_OBJECT (priv->root));
	
	g_free (priv);

	GTK_OBJECT_CLASS (print_preview_parent_class)->destroy (object);
}

static void
gpp_class_init (GnomePrintPreviewClass *class)
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;
	GnomePrintContextClass *pc_class = (GnomePrintContextClass *) class;

	print_preview_parent_class = gtk_type_class (gnome_print_context_get_type ());

	object_class->destroy = gpp_destroy;
	
	pc_class->newpath = gpp_newpath;
	pc_class->moveto = gpp_moveto;
	pc_class->lineto = gpp_lineto;
	pc_class->curveto = gpp_curveto;
	pc_class->closepath = gpp_closepath;
	pc_class->setrgbcolor = gpp_setrgbcolor;
	pc_class->fill = gpp_fill;
	pc_class->eofill = gpp_eofill;
	pc_class->setlinewidth = gpp_setlinewidth;
	pc_class->setmiterlimit = gpp_setmiterlimit;
	pc_class->setlinejoin = gpp_setlinejoin;
	pc_class->setlinecap = gpp_setlinecap;
	pc_class->setdash = gpp_setdash;
	pc_class->strokepath = gpp_strokepath;
	pc_class->stroke = gpp_stroke;
	pc_class->setfont = gpp_setfont;
	pc_class->show_ucs4 = gpp_show_ucs4;
	pc_class->concat = gpp_concat;
#ifdef SETMATRIX_DEPRECATED
	pc_class->setmatrix = gpp_setmatrix;
#endif		
	pc_class->gsave = gpp_gsave;
	pc_class->grestore = gpp_grestore;
	pc_class->clip = gpp_clip;
	pc_class->eoclip = gpp_eoclip;
	pc_class->grayimage = gpp_grayimage;
	pc_class->rgbimage = gpp_rgbimage;
	pc_class->rgbaimage = gpp_rgbaimage;
	pc_class->textline = gpp_textline;
	pc_class->showpage = gpp_showpage;
	pc_class->beginpage = gpp_beginpage;
	pc_class->setopacity = gpp_setopacity;
	pc_class->glyphlist = gpp_glyphlist;
	
	pc_class->close = gpp_close;
}

static void
gpp_init (GnomePrintPreview *preview)
{
	GnomePrintPreviewPrivate *priv;
	
	priv = preview->priv = g_new0 (GnomePrintPreviewPrivate, 1);

	priv->gc = gp_gc_new ();
}

static void
clear_val (GtkObject *object, void **val)
{
	*val = NULL;
}

/**
 * gnome_print_preview_construct:
 * @preview: the #GnomePrintPreview object to construct
 * @canvas: Canvas on which the preview will render
 * @paper_info: a GnomePaper information
 *
 * Constructs the @preview object.
 */
void
gnome_print_preview_construct (GnomePrintPreview *preview,
			       GnomeCanvas *canvas,
			       const GnomePaper *paper_info)
{
	GnomeCanvasGroup * group;
	double page2root[6];

	g_return_if_fail (preview != NULL);
	g_return_if_fail (GNOME_IS_PRINT_PREVIEW (preview));
	g_return_if_fail (canvas != NULL);
	g_return_if_fail (GNOME_IS_CANVAS (canvas));
	g_return_if_fail (paper_info != NULL);

	gtk_object_ref (GTK_OBJECT (canvas));
	preview->canvas = canvas;

	if (getenv ("GNOME_PRINT_DEBUG_WIDE"))
		gnome_canvas_set_scroll_region (
			canvas,	-900, -900, 900, 900);
	else
		gnome_canvas_set_scroll_region (
			canvas, 0, 0,
			gnome_paper_pswidth (paper_info),
			gnome_paper_psheight (paper_info));

	preview->priv->root = gnome_canvas_item_new (
		gnome_canvas_root (preview->canvas),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL);

	preview->priv->page = gnome_canvas_item_new (
		gnome_canvas_root (preview->canvas),
		gnome_canvas_group_get_type (),
		"x", 0.0,
		"y", 0.0,
		NULL);

	gtk_signal_connect (GTK_OBJECT (preview->priv->page), "destroy",
			    GTK_SIGNAL_FUNC (clear_val), &preview->priv->page);
	gtk_signal_connect (GTK_OBJECT (preview->priv->root), "destroy",
			    GTK_SIGNAL_FUNC (clear_val), &preview->priv->root);
	/*
	 * Setup base group
	 */

	group = GNOME_CANVAS_GROUP (preview->priv->page);

	gp_gc_set_data (preview->priv->gc, group);

	/*
	 * Setup the affines
	 */
#if 0
	art_affine_identity (gc (preview->priv)->affine);
	art_affine_translate (page2root, 0, -gnome_paper_psheight (paper_info));
	art_affine_flip (page2root,
			 page2root,
			 FALSE, TRUE);
#else
	art_affine_scale (page2root, 1.0, -1.0);
	page2root[5] = gnome_paper_psheight (paper_info);
#endif

	gnome_canvas_item_affine_absolute (preview->priv->page, page2root);
}

/**
 * gnome_print_preview_new:
 * @canvas: Canvas on which we display the print preview
 * @paper_size: A valid name for a paper size
 *
 * Creates a new PrintPreview object that use the @canvas GnomeCanvas 
 * as its rendering backend.
 *
 * Returns: A GnomePrintContext suitable for using with the GNOME print API.
 */
GnomePrintContext *
gnome_print_preview_new (GnomeCanvas *canvas, const char *paper_size)
{
	GnomePrintPreview *preview;
	const GnomePaper *paper_info;
	
	g_return_val_if_fail (canvas != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_CANVAS (canvas), NULL);
	g_return_val_if_fail (paper_size != NULL, NULL);

	paper_info = gnome_paper_with_name (paper_size);
	if (paper_info == NULL)
		g_return_val_if_fail (FALSE, NULL);

	preview = gtk_type_new (gnome_print_preview_get_type ());
	if (preview == NULL)
		return NULL;

	gnome_print_preview_construct (preview, canvas, paper_info);
	return GNOME_PRINT_CONTEXT (preview);
}

/**
 * gnome_print_preview_get_type:
 *
 * GTK type identification routine for #GnomePrintPreview
 *
 * Returns: The Gtk type for the #GnomePrintPreview object
 */
GtkType
gnome_print_preview_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"GnomePrintPreview",
			sizeof (GnomePrintPreview),
			sizeof (GnomePrintPreviewClass),
			(GtkClassInitFunc) gpp_class_init,
			(GtkObjectInitFunc) gpp_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		type = gtk_type_unique (gnome_print_context_get_type (), &info);
	}
	return type;
}

