/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/*
 * gnome-print-pdf-type1.c: Type 1 font handling for PDF
 *
 * Authors:
 *   Chema Celorio (chema@celorio.com)
 *
 * (C) 2000 Jose M Celorio1528
 *
 * References :
 * 
 * [1] Portable Document Format Referece Manual, Version 1.3 (March 11, 1999)
 *
 */


/* __FUNCTION__ is not defined in Irix. thanks, <drk@sgi.com>*/
#ifndef __GNUC__
  #define __FUNCTION__   ""
#endif
#define debug(section,str) if (TRUE) printf ("%s:%d (%s) %s\n", __FILE__, __LINE__, __FUNCTION__, str); 
	
#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <ctype.h>
#include <stdio.h>

#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-pdf-type1.h>
#include <libgnomeprint/gnome-font-private.h>
#include <libgnomeprint/gnome-print-private.h>

#define EOL "\r\n"
#define GNOME_PRINT_PDF_FONT_UNDEFINED 999

/**
 * text_utils_search_real:
 * @buffer: buffer to do the searching in
 * @buffer_length: length of the buffer
 * @search_text: text to search for
 * @search_text_length: lenght of the search text
 * @case_sensitive: TRUE if the search is case sensitive, FALSE otherwise
 * 
 * Searches for "search_text" in "buffer" and returns the location where it is found
 * it can search buffers that contain ascii ceros. That is why you need to provide
 * buffer & search text length
 *
 * Return Value: 
 **/
static gint
text_utils_search_real (const gchar *buffer, gint buffer_length,
												const gchar *search_text, gint search_text_length,
												gboolean case_sensitive)
{
	gint p1;
	gint p2 = 0;
	gint case_sensitive_mask;

	case_sensitive_mask = case_sensitive?0:32;
	
	for (p1=0; p1 < buffer_length; p1 ++)
	{
		if ((buffer[p1]     |case_sensitive_mask)==
		    (search_text[p2]|case_sensitive_mask))
		{
			p2++;
			if (p2==search_text_length)
				return p1 - search_text_length + 1;
		}
		else
			p2 = 0;
	}
	
	return -1;
}


/**
 * gnome_print_pdf_type1_get_length: 
 * @pfb: pfb buffer
 * 
 * returns the numeric value of 2 bytes starting at pfb
 *
 * Return Value: 
 **/
static gint
gp_t1_get_length (guchar *pfb)
{
	guchar msb;
	guchar lsb;
	gint resp;

  lsb = pfb [0];
	msb = pfb [1];

	resp = (msb * 256) + lsb;

	return resp;
}

#define PDF_FONT_TAG_1 "%!"
#define PDF_FONT_TAG_2 "currentfile eexec"
#define PDF_FONT_TAG_3 "0000000000"

/**
 * gnome_print_pdf_type1_determine_lengths:
 * @pfb: font's pfb buffer
 * @pfb_size: size of pfb
 * @length1: length of the 1st section
 * @length2: 2nd
 * @length3: 3rd
 * 
 * given a pdf, it determines the lengths of the 1st,2nd and 3rd section
 * 
 * Return Value: TRUE on success, FALSE on error
 **/
static gboolean
gp_t1_determine_lengths (guchar *pfb, gint pfb_size, gint *length1, gint *length2, gint *length3)
{
	gint pos1;
	gint pos2;
	gint pos3;
	gint temp;

	debug (FALSE, "");

	
	g_return_val_if_fail (pfb != NULL, FALSE);

	/* Search for %! */
	pos1 = text_utils_search_real (pfb, pfb_size,
																 PDF_FONT_TAG_1, strlen (PDF_FONT_TAG_1),
																 TRUE);
	if (pos1 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_1);
		return FALSE;
	}

	/* Search for "currentfile eexec"*/
	pos2 = text_utils_search_real (pfb, pfb_size,
																 PDF_FONT_TAG_2, strlen (PDF_FONT_TAG_2),
																 TRUE);
	if (pos2 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_2);
		return FALSE;
	}

	/* Search for 00000 */
	pos3 = text_utils_search_real (pfb, pfb_size,
																 PDF_FONT_TAG_3, strlen (PDF_FONT_TAG_3),
																 TRUE);

	if (pos3 == -1) {
		g_warning ("Could not find %s\n", PDF_FONT_TAG_3);
		return FALSE;
	}

	temp = pos2 + strlen (PDF_FONT_TAG_2) - pos1 + 1;

	*length1 = temp;
	*length2 = pos3 - pos2 - strlen (PDF_FONT_TAG_2) - 1 - 12;
                       /*12 are the 2 x 6 bytes block init */
	*length3 = 0;

	debug (FALSE, "end");

	
	return TRUE;
}

/**
 * gnome_print_pdf_type1_get_lengths: 
 * @pfb: pfb buffer
 * @length1: returns the length of the 1st section
 * @length2: 2nd section
 * @length3: 3rd section
 * 
 * reads the lengths of the 3 sections of the pdf based on the blocks identifiers
 * it also performs basic checking of the blocks identiiers
 *
 * Return Value: TRUE on success, FALSE otherwise
 **/
	
static gboolean
gp_t1_get_lengths (guchar *pfb, gint pfb_size, gint *len1, gint *len2, gint *len3)
{
	gint length1;
	gint length2;
	gint length3;

	debug (FALSE, "");
	
	g_return_val_if_fail (pfb != NULL, FALSE);
	
	if ((pfb [0] != 0x80) ||
			(pfb [1] != 0x01)) {
		g_warning ("Expected 0x80,0x01 at the begining of the pfb\n");
		return FALSE;
	}
	if ((pfb [4] != 0x00) ||
			(pfb [5] != 0x00)) {
		g_warning ("Expected 0x00,0x00 at the begining of the pfb\n");
		return FALSE;
	}

	*len1 = gp_t1_get_length (pfb + 2);

	if ((pfb [ *len1 + 6]     != 0x80) ||
			(pfb [ *len1 + 6 + 1] != 0x02)) {
		g_warning ("Expected 0x80,0x02 at the midle of the pfb\n");
		return FALSE;
	}
	if ((pfb [ *len1 + 6 + 4] != 0x00) ||
			(pfb [ *len1 + 6 + 5] != 0x00)) {
		g_warning ("Expected 0x00,0x00 at the middle of the pfb\n");
		return FALSE;
	}

	*len2 = gp_t1_get_length (pfb + *len1 + 2 + 6);
	*len3 = 0;

	if (!gp_t1_determine_lengths (pfb, pfb_size, &length1, &length2, &length3)) {
		g_warning ("Could not determine lengths from font file");
		return -1;
	}
	if ((*len1 != length1) ||
			(*len2 != length2) ||
			(*len3 != length3) ) {
		g_warning ("The lengthsdo not match [%i,%i,%i] [%i,%i,%i]",
							 *len1, *len2, *len3, length1, length2, length3);
		return -1;
	}
	
	
	debug (FALSE, "end");
	
	return TRUE;
}


/**
 * gnome_print_pdf_type1_read_pfb:
 * @file_name: full path of the .pfb file
 * @pfb_size: return the length read here
 * 
 * reads a pfb from filename 
 * 
 * Return Value: pointer to the malloced buffer, caller must free
 **/
static guchar*
gp_t1_read_pfb (const gchar* file_name, gint *pfb_size_)
{
	FILE *f;
	gint pfb_size_max;
	gint bytes_read;
	gint pfb_size;
	guchar *pfb;

	*pfb_size_ = 0;
	pfb_size = 0;
	
	g_return_val_if_fail (file_name != NULL, NULL);
	
	f = fopen (file_name, "r");
  if (f == NULL)
    {
      g_warning ("Couldn't open font file %s\n", file_name);
      return NULL;
    }

  pfb_size = 0;
  pfb_size_max = 32768;
  pfb = g_new (guchar, pfb_size_max);
  while (1)
    {
      bytes_read = fread (pfb + pfb_size, 1, pfb_size_max - pfb_size, f);
      if (bytes_read == 0)
				break;
      pfb_size += bytes_read;
      pfb = g_realloc (pfb, pfb_size_max <<= 1);
    }

	*pfb_size_ = pfb_size;
	
	return pfb;
}


/**
 * gnome_print_pdf_font_parse:
 * @file_name: full path of the pfb of the font
 * @body_: return the cleaned structure here, the caller must free it later
 * @length: the length of the block
 * @length1: length of the 1st section of the font
 * @length2: "" 2nd ""
 * @length3: "" 3rd ""
 * @body_length: The length of @body
 * 
 * Parsed the type1 font pointed by *file_name and cleans it by removing
 * the block identifiers and the 3rd secciont (section of 512 0's). It also
 * calculates the lengths of the 3 sections of the font
 *
 * Return Value: 0 on success, -1 otherwise
 **/
static gint
gnome_print_pdf_font_parse (const gchar *file_name,
														gchar **body_,
														gint *length, gint *len1, gint *len2, gint *len3,
														gint *body_length)
{
	gint ret = 0;
	guchar *pfb;
	guchar *pfb_clean;
	gint pfb_size;

	debug (FALSE, "");

	*length = 0;
	*len1 = 0;
	*len2 = 0;
	*len3 = 0;

	pfb = gp_t1_read_pfb (file_name, &pfb_size);

	if (!pfb_size) {
		g_warning ("Could not read the font in \"%s\"\n", file_name);
		return -1;
	}

	if (!gp_t1_get_lengths (pfb, pfb_size, len1, len2, len3)) {
		g_warning ("Could not get lengths from font file");
		return -1;
	}
	/*
	pfa_encoded_size = gnome_print_encode_ascii85_wcs (pfa_size - pos1);
	pfa_encoded = g_new (gchar, pfa_encoded_size);
	real_size = gnome_print_encode_ascii85 (pfa + pos1,
	pfa_encoded,
	pos3 - pos2 - strlen (PDF_FONT_TAG_2) - 1*
	pfa_size - pos1);
	g_print ("Real Size %i\n", real_size);
*/

	*length = *len1 + *len2 + strlen (EOL);
	*len3 = 0;
	
	*body_length = *length;

  pfb_clean = g_new (guchar, *len1 + *len2 + 1);
	
	memcpy (pfb_clean, pfb + 6, *len1);
	memcpy (pfb_clean + *len1, pfb + 6 + *len1 + 6, *len2);
	
	*body_ = pfb_clean;
	*body_length = *len1 + *len2;

	g_free (pfb);

	debug (FALSE, "end");
	
	return ret;
}

/**
 * gnome_print_pdf_font_type1_embed:
 * @pc: Print Context to embed in
 * @font: Font to embed
 * 
 * Embeds in the print context a the type1 font
 *
 * Return Value: 0 on success, -1 on error
 **/
gint
gnome_print_pdf_font_type1_embed (GnomePrintContext *pc,
																	GnomePrintPdfFont *font)
{
	GnomePrintPdf *pdf;
	const GnomeFontFace *face;
	const gchar *file_name;
	guint object_number;
	gchar *body;
	gint length;
	gint length1;
	gint length2;
	gint length3;
	gint ret = 0;
	gint written;
	gint body_length;
	gboolean ascii85 = FALSE;
	
	debug (FALSE, "");

	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), -1);
	g_return_val_if_fail (font != NULL, -1);
	face = gnome_font_get_face (font->gnome_font);
	g_return_val_if_fail (GNOME_IS_FONT_FACE (face), -1);
	pdf = GNOME_PRINT_PDF (pc);
	g_return_val_if_fail (GNOME_IS_PRINT_PDF (pdf), -1);

	file_name = face->private->pfb_fn;

	gnome_print_pdf_font_parse (file_name,
															&body,
															&length,
															&length1,
															&length2,
															&length3,
															&body_length);

	if ((length == 0) ||
			(length1 == 0)){
		g_warning ("Could not embed the font\n");
		return -1;
	}
	
	object_number = font->object_number_pfb;

	ret += gnome_print_pdf_object_start (pc, object_number);
	if (ascii85)
		ret += gnome_print_pdf_write  (pc,"/Filter /ASCII85Decode" EOL);
	ret += gnome_print_pdf_write  (pc,
																 "/Length %i" EOL
																 "/Length1 %i" EOL
																 "/Length2 %i" EOL
																 "/Length3 %i" EOL,
																 length,
																 length1,
																 length2,
																 length3);
	ret += gnome_print_pdf_write  (pc,
																 ">>" EOL
																 "stream" EOL);
	written = gnome_print_context_write_file (pc, body, body_length);
	gnome_print_pdf_add_bytes_written (pdf, written);
	ret += written;
	ret += gnome_print_pdf_write  (pc, EOL "endstream" EOL);
	ret += gnome_print_pdf_write  (pc, EOL "endobj" EOL);
	ret += gnome_print_pdf_object_end   (pc, object_number, TRUE);

	g_free (body);

	return ret;
}

#define EEXEC_C1 ((unsigned short)52845)
#define EEXEC_C2 ((unsigned short)22719)
/* REF: adobe type 1 font format, pag 63 */
static int
decrypt_eexec (char *plaintext, const char *ciphertext, int ciphertext_size)
{
  int i;
  unsigned short r;
  unsigned char cipher;
  unsigned char plain;

  r = 55665;  /* initial key */

  for (i = 0; i < ciphertext_size; i++)
    {
      cipher = ciphertext[i];
      plain = (cipher ^ (r>>8));
      r = (cipher + r) * EEXEC_C1 + EEXEC_C2;
      if (i >= 4)
	plaintext[i - 4] = plain;
    }
  return ciphertext_size - 4;
}

/**
 * gp_t1_get_body_from_pfb:
 * @file_name: full path of the font
 * @body: return the body of the font here
 * @body_length: which has this length
 * 
 * get the body of the font (section 2 of 3) unencrypted
 * 
 * Return Value: TRUE on success, FALSE otherwise
 **/
static gboolean
gp_t1_get_body_from_pfb (const gchar* file_name, gchar **body, gint * body_length)
{
	gchar *pfb;
	gchar *decrypted;
	const gchar *crypted;
	gint length;
	gint length1;
	gint length2;
	gint length3;
	gint pfb_length;
	gint size;

	*body = NULL;
	if (gnome_print_pdf_font_parse (file_name,
																	&pfb,
																	&length,
																	&length1,
																	&length2,
																	&length3,
																	&pfb_length) == -1)
		return FALSE;


	crypted = pfb + length1;
	
	decrypted = g_new (gchar, length2);
	
	size = decrypt_eexec (decrypted, crypted, length2);

	g_free (pfb);

	*body_length = size;
	*body = decrypted;
	
	return TRUE;
}


/**
 * gp_t1_get_number_from_brackets:
 * @buffer: buffer pointing to a bracked value
 * @number: return the number here
 * 
 * given a pointer to a buffer like "[123]" returns 123
 * 
 * Return Value: TRUE on success, FALSE on error
 **/
static gboolean
gp_t1_get_number_from_brackets (gchar *buffer, gint *number)
{
	gint n;
	gchar *num;

	g_return_val_if_fail (buffer != NULL, FALSE);
	g_return_val_if_fail (buffer [0] == '[', FALSE);
	
	*number = 0;
	num = g_new (gchar, 15);
	for (n = 0; n < 15; n++) {
		if (buffer [n+1] == ']')
			break;
		num [n] = buffer [n+1];
	}
	num [n] = 0;
	
	if (n > 13) {
		g_free (num);
		return FALSE;
	}

	*number = atoi (num);

	return TRUE;
}
		
#define GP_STEMV_TAG "/StdHW"
#define GP_STEMH_TAG "/StdVW"

/* TODO: Save the steam for each font in something like gnome_config,
	 so that we don't have to parse the font everytime we want to get
	 it's stems. */
/**
 * gnome_print_pdf_type1_get_stems:
 * @face: Font Face to get stems from 
 * @stemv: return the stemv here
 * @stemh: return the stemh here
 * 
 * This function parsed the type1 font, and searches for the
 * "/StdHW [39] def" & "/StdVW [96] def" field, returns the numbers
 * inside
 *
 * Return Value: TRUE on succes, FALSE otherwise
 **/
gint
gnome_print_pdf_type1_get_stems (const GnomeFontFace *face, gint *stemv, gint*stemh)
{
	const gchar *file_name;
	gchar *body;
	gint body_size;
	gint pos;
	gint number;

	*stemv = 96;
	*stemh = 0;
	
	g_return_val_if_fail (GNOME_IS_FONT_FACE (face), FALSE);

	file_name = face->private->pfb_fn;

	if (!gp_t1_get_body_from_pfb (file_name, &body, &body_size)) {
		g_warning ("Cant get body from pfb");
		return FALSE;
	}

	/* Get the stemv */
	pos = text_utils_search_real (body, body_size,
																GP_STEMV_TAG, strlen (GP_STEMV_TAG),
																TRUE);
	if (pos == -1) {
		g_warning ("Could not find the \"%s\" tag", GP_STEMV_TAG);
		return FALSE;
	}
	if (!gp_t1_get_number_from_brackets (body + pos + strlen (GP_STEMV_TAG) + 1,
																			 &number)) {
		g_warning ("Could not get number from brackets");
		return FALSE;
	}
	*stemv = number;
			
	/* Get the stemh */
	pos = text_utils_search_real (body, body_size,
																GP_STEMH_TAG, strlen (GP_STEMH_TAG),
																TRUE);
	if (pos == -1) {
		g_warning ("Could not find the \"%s\" tag", GP_STEMH_TAG);
		return FALSE;
	}
	if (!gp_t1_get_number_from_brackets (body + pos + strlen (GP_STEMH_TAG) + 1,
																			 &number)) {
		g_warning ("Could not get number from brackets");
		return FALSE;
	}
	*stemh = number;
	
	return TRUE;
}
