#define __GNOME_RFONT_C__

#include <math.h>
#include <libgnomeprint/gnome-rfont.h>
#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_vpath_bpath.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_rect_svp.h>
#include <libart_lgpl/art_gray_svp.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libgnomeprint/gnome-font-private.h>
#include <libgnomeprint/gnome-pgl-private.h>

#define noVERBOSE

struct _GnomeRFont {
	GtkObject object;
	GnomeFontFace * face;
	GnomeFont * font;
	gdouble transform[6];
	GHashTable * bpaths;
	GHashTable * svps;
	GHashTable * graymaps;
	GHashTable * dimensions;
	GHashTable * displays;
	GdkFont * gdk_font;
	gchar * x_font_name;
};

struct _GnomeRFontClass {
	GtkObjectClass parent_class;
};

typedef struct {
	guchar * pixels;
	gint x0, y0;
	gint width, height, rowstride;
} GnomeRFontGrayMap;

typedef struct {
	ArtDRect ddim;
	ArtIRect idim;
} GnomeRFontDimension;

typedef struct {
	GnomeRFont * rfont;
	GHashTable * pixmaps;
	GHashTable * bitmaps;
	GdkGC * gc;
	GdkGC * bmgc;
	GdkWindow * window;
	GdkVisual * visual;
} GnomeRFontDisplay;

static GHashTable * rfonts = NULL;

static void gnome_rfont_class_init (GnomeRFontClass * klass);
static void gnome_rfont_init (GnomeRFont * rfont);

static void gnome_rfont_destroy (GtkObject * object);

static const GnomeRFontGrayMap * gnome_rfont_get_glyph_graymap (const GnomeRFont * rfont, gint glyph);
static const GnomeRFontDimension * gnome_rfont_get_glyph_dimension (const GnomeRFont * rfont, gint glyph);
static const GnomeRFontDisplay * gnome_rfont_get_display (const GnomeRFont * rfont, GdkDrawable * drawable);

static const GdkPixmap * dsp_get_glyph_pixmap (const GnomeRFontDisplay * dsp, gint glyph, guint32 rgba, guint32 background);
static const GdkBitmap * dsp_get_glyph_bitmap (const GnomeRFontDisplay * dsp, gint glyph);

static gboolean rfont_free_pixmap (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_bitmap (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_display (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_dimension (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_graymap (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_svp (gpointer key, gpointer value, gpointer data);
static gboolean rfont_free_bpath (gpointer key, gpointer value, gpointer data);

static guint rfont_hash (gconstpointer key);
static gboolean rfont_equal (gconstpointer key1, gconstpointer key2);

static GtkObjectClass * parent_class;

GtkType
gnome_rfont_get_type (void)
{
	static GtkType rfont_type = 0;
	if (!rfont_type) {
		GtkTypeInfo rfont_info = {
			"GnomeRFont",
			sizeof (GnomeRFont),
			sizeof (GnomeRFontClass),
			(GtkClassInitFunc) gnome_rfont_class_init,
			(GtkObjectInitFunc) gnome_rfont_init,
			NULL, NULL,
			NULL
		};
		rfont_type = gtk_type_unique (gtk_object_get_type (), &rfont_info);
	}
	return rfont_type;
}

static void
gnome_rfont_class_init (GnomeRFontClass * klass)
{
	GtkObjectClass * object_class;

	object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class (gtk_object_get_type ());

	object_class->destroy = gnome_rfont_destroy;
}

static void
gnome_rfont_init (GnomeRFont * rfont)
{
	art_affine_identity (rfont->transform);
	rfont->bpaths = g_hash_table_new (NULL, NULL);
	rfont->svps = g_hash_table_new (NULL, NULL);
	rfont->graymaps = g_hash_table_new (NULL, NULL);
	rfont->dimensions = g_hash_table_new (NULL, NULL);
	rfont->displays = g_hash_table_new (NULL, NULL);
}

static void
gnome_rfont_destroy (GtkObject * object)
{
	GnomeRFont * rfont;

	rfont = (GnomeRFont *) object;

	g_hash_table_remove (rfonts, rfont);

	if (rfont->x_font_name) g_free (rfont->x_font_name);
	if (rfont->gdk_font) gdk_font_unref (rfont->gdk_font);

	if (rfont->displays) {
		g_hash_table_foreach_remove (rfont->displays, rfont_free_display, NULL);
		g_hash_table_destroy (rfont->displays);
		rfont->displays = NULL;
	}
	if (rfont->dimensions) {
		g_hash_table_foreach_remove (rfont->dimensions, rfont_free_dimension, NULL);
		g_hash_table_destroy (rfont->dimensions);
		rfont->dimensions = NULL;
	}
	if (rfont->graymaps) {
		g_hash_table_foreach_remove (rfont->graymaps, rfont_free_graymap, NULL);
		g_hash_table_destroy (rfont->graymaps);
		rfont->graymaps = NULL;
	}
	if (rfont->svps) {
		g_hash_table_foreach_remove (rfont->svps, rfont_free_svp, NULL);
		g_hash_table_destroy (rfont->svps);
		rfont->svps = NULL;
	}
	if (rfont->bpaths) {
		g_hash_table_foreach_remove (rfont->bpaths, rfont_free_bpath, NULL);
		g_hash_table_destroy (rfont->bpaths);
		rfont->bpaths = NULL;
	}

	if (rfont->font) {
		gtk_object_unref (GTK_OBJECT (rfont->font));
		rfont->font = NULL;
	}

	if (((GtkObjectClass *) (parent_class))->destroy)
		(* ((GtkObjectClass *) (parent_class))->destroy) (object);
}

GnomeRFont *
gnome_font_get_rfont (GnomeFont * font, gdouble * transform)
{
	GnomeRFont search;
	GnomeRFont * rfont;

	g_return_val_if_fail (font != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_FONT (font), NULL);
	g_return_val_if_fail (transform != NULL, NULL);

	if (rfonts == NULL) {
		rfonts = g_hash_table_new (rfont_hash, rfont_equal);
	}

	search.font = font;
	memcpy (search.transform, transform, 4 * sizeof (gdouble));
	search.transform[4] = search.transform[5] = 0.0;

	rfont = g_hash_table_lookup (rfonts, &search);

	if (rfont != NULL) {
#ifdef VERBOSE
		g_print ("found cached rfont %s\n", gnome_font_get_name (font));
#endif
		gnome_rfont_ref (rfont);
		return rfont;
	}

	rfont = gtk_type_new (gnome_rfont_get_type ());

	rfont->face = (GnomeFontFace *) gnome_font_get_face (font);
	rfont->font = font;
	gtk_object_ref (GTK_OBJECT (font));
	memcpy (rfont->transform, transform, 4 * sizeof (gdouble));
	rfont->transform[4] = rfont->transform[5] = 0.0;

	g_hash_table_insert (rfonts, rfont, rfont);

	return rfont;
}

const GnomeFont *
gnome_rfont_get_font (const GnomeRFont * rfont)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	return rfont->font;
}

const GnomeFontFace *
gnome_rfont_get_face (const GnomeRFont * rfont)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	return rfont->face;
}

gdouble *
gnome_rfont_get_matrix (const GnomeRFont * rfont, gdouble * matrix)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);
	g_return_val_if_fail (matrix != NULL, NULL);

	memcpy (matrix, rfont->transform, 4 * sizeof (gdouble));

	return matrix;
}

ArtPoint *
gnome_rfont_get_stdadvance (const GnomeRFont * rfont, ArtPoint * advance)
{
	gdouble size;

	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);
	g_return_val_if_fail (advance != NULL, NULL);

	size = gnome_font_get_size (rfont->font);

	advance->x = size * (rfont->transform[0] + rfont->transform[2]);
	advance->y = size * (rfont->transform[1] + rfont->transform[3]);

	return advance;
}

ArtPoint *
gnome_rfont_get_glyph_stdadvance (const GnomeRFont * rfont, gint glyph, ArtPoint * advance)
{
	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	gnome_font_get_glyph_stdadvance (rfont->font, glyph, advance);
	art_affine_point (advance, advance, rfont->transform);

	return advance;
}

ArtDRect *
gnome_rfont_get_glyph_stdbbox (const GnomeRFont * rfont, gint glyph, ArtDRect * bbox)
{
	const GnomeRFontDimension * dim;

	g_return_val_if_fail (rfont != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (rfont), NULL);

	dim = gnome_rfont_get_glyph_dimension (rfont, glyph);

	* bbox = dim->ddim;

	return bbox;
}

const ArtBpath *
gnome_rfont_get_glyph_bpath (const GnomeRFont * rfont, gint glyph)
{
	ArtBpath * bpath;
	gdouble affine[6];
	gdouble size;

	bpath = g_hash_table_lookup (rfont->bpaths, GINT_TO_POINTER (glyph));

	if (bpath) return bpath;

	size = gnome_font_get_size (rfont->font);
	affine[0] = rfont->transform[0] * size * 0.001;
	affine[1] = rfont->transform[1] * size * 0.001;
	affine[2] = rfont->transform[2] * size * 0.001;
	affine[3] = rfont->transform[3] * size * 0.001;
	affine[4] = affine[5] = 0.0;

	bpath = (ArtBpath *) gnome_font_face_get_glyph_stdoutline (rfont->face, glyph);

	g_return_val_if_fail (bpath != NULL, NULL);

	bpath = art_bpath_affine_transform (bpath, affine);

	g_hash_table_insert (rfont->bpaths, GINT_TO_POINTER (glyph), bpath);

	return bpath;
}

const ArtSVP *
gnome_rfont_get_glyph_svp (const GnomeRFont * rfont, gint glyph)
{
	ArtSVP * svp, * svp1;
	ArtBpath * bpath;
	ArtVpath * vpath, * vpath1;

	svp = g_hash_table_lookup (rfont->svps, GINT_TO_POINTER (glyph));

	if (svp) return svp;

	/* fixme: */

	bpath = (ArtBpath *) gnome_rfont_get_glyph_bpath (rfont, glyph);

	g_return_val_if_fail (bpath != NULL, NULL);

	vpath = art_bez_path_to_vec (bpath, 0.25);
	vpath1 = art_vpath_perturb (vpath);
	art_free (vpath);
	svp = art_svp_from_vpath (vpath1);
	art_free (vpath1);
	svp1 = art_svp_uncross (svp);
	art_svp_free (svp);
	svp = art_svp_rewind_uncrossed (svp1, ART_WIND_RULE_ODDEVEN);
	art_svp_free (svp1);

	g_hash_table_insert (rfont->svps, GINT_TO_POINTER (glyph), svp);

	return svp;
}

static const GnomeRFontGrayMap *
gnome_rfont_get_glyph_graymap (const GnomeRFont * rfont, gint glyph)
{
	GnomeRFontGrayMap * gmap;
	ArtSVP * svp;
	ArtDRect bbox;
	ArtIRect ibox;

	gmap = g_hash_table_lookup (rfont->graymaps, GINT_TO_POINTER (glyph));

	if (gmap) return gmap;

	/* fixme: */

	svp = (ArtSVP *) gnome_rfont_get_glyph_svp (rfont, glyph);
	art_drect_svp (&bbox, svp);
	art_drect_to_irect (&ibox, &bbox);

	gmap = g_new (GnomeRFontGrayMap, 1);
	gmap->width = gmap->rowstride = ibox.x1 - ibox.x0;
	gmap->height = ibox.y1 - ibox.y0;
	gmap->x0 = ibox.x0;
	gmap->y0 = ibox.y0;
	gmap->pixels = g_new0 (guchar, gmap->width * gmap->height);

	art_gray_svp_aa (svp,
		ibox.x0, ibox.y0, ibox.x1, ibox.y1,
		gmap->pixels, gmap->rowstride);

	g_hash_table_insert (rfont->graymaps, GINT_TO_POINTER (glyph), gmap);

	return gmap;
}

static const GnomeRFontDimension *
gnome_rfont_get_glyph_dimension (const GnomeRFont * rfont, gint glyph)
{
	GnomeRFontDimension * dim;
	ArtSVP * svp;

	dim = g_hash_table_lookup (rfont->dimensions, GINT_TO_POINTER (glyph));

	if (dim) return dim;

	/* fixme: */

	svp = (ArtSVP *) gnome_rfont_get_glyph_svp (rfont, glyph);

	dim = g_new (GnomeRFontDimension, 1);

	art_drect_svp (&dim->ddim, svp);
	art_drect_to_irect (&dim->idim, &dim->ddim);

	g_hash_table_insert (rfont->dimensions, GINT_TO_POINTER (glyph), dim);

	return dim;
}

/* fixme: How to do it The Right Way (TM)? */

static const GnomeRFontDisplay *
gnome_rfont_get_display (const GnomeRFont * rfont, GdkDrawable * drawable)
{
	GdkWindow * toplevel;
	GdkVisual * visual;
	GnomeRFontDisplay * dsp;

#if 0
	toplevel = gdk_window_get_toplevel (drawable);
	g_return_val_if_fail (toplevel != NULL, NULL);
	visual = gdk_window_get_visual (toplevel);
	if (!visual) visual = gdk_visual_get_best ();
	g_return_val_if_fail (visual != NULL, NULL);
#else
	toplevel = NULL;
	visual = gdk_visual_get_best ();
	g_return_val_if_fail (visual != NULL, NULL);
#endif

	dsp = g_hash_table_lookup (rfont->displays, toplevel);

	if (dsp) return dsp;

	/* fixme: */

	dsp = g_new (GnomeRFontDisplay, 1);

	dsp->rfont = (GnomeRFont *) rfont;
	dsp->pixmaps = g_hash_table_new (NULL, NULL);
	dsp->bitmaps = g_hash_table_new (NULL, NULL);
	dsp->gc = gdk_gc_new (drawable);
	dsp->bmgc = NULL;
	dsp->window = toplevel;
	dsp->visual = visual;

	g_hash_table_insert (rfont->displays, toplevel, dsp);

	return dsp;
}

void
gnome_rfont_render_glyph_rgba8 (const GnomeRFont * rfont, gint glyph,
				guint32 rgba,
				gdouble x, gdouble y,
				guchar * buf,
				gint width, gint height, gint rowstride,
				guint flags)
{
	const GnomeRFontGrayMap * gmap;
	gint xp, yp, x0, y0, x1, y1;
	guint inkr, inkg, inkb, inka;
	guchar * s, * s0, * d, * d0;

	gmap = gnome_rfont_get_glyph_graymap (rfont, glyph);

	xp = (gint) floor (x + 0.5);
	yp = (gint) floor (y + 0.5);

	x0 = MAX (0, xp + gmap->x0);
	y0 = MAX (0, yp + gmap->y0);
	x1 = MIN (width, xp + gmap->x0 + gmap->width);
	y1 = MIN (height, yp + gmap->y0 + gmap->height);

	inkr = rgba >> 24;
	inkg = (rgba >> 16) & 0xff;
	inkb = (rgba >> 8) & 0xff;
	inka = rgba & 0xff;

	d0 = d = buf + y0 * rowstride + x0 * 4;
	s0 = s = gmap->pixels + (y0 - yp - gmap->y0) * gmap->rowstride + (x0 - xp - gmap->x0);

	for (y = y0; y < y1; y++) {
		for (x = x0; x < x1; x++) {
			guint bgr, bgg, bgb, bga;
			guint pixr, pixg, pixb;
			guint fgr, fgg, fgb;
			guint alpha;

			bgr = * (d + 0);
			bgg = * (d + 1);
			bgb = * (d + 2);
			bga = * (d + 3);

			alpha = (inka * (*s++) + 0x80) >> 8;

			pixr = (bgr * bga + 0x80) >> 8;
			pixg = (bgg * bga + 0x80) >> 8;
			pixb = (bgb * bga + 0x80) >> 8;

			fgr = ((inkr - pixr) * alpha + 0x80) >> 8;
			fgg = ((inkg - pixg) * alpha + 0x80) >> 8;
			fgb = ((inkb - pixb) * alpha + 0x80) >> 8;

			*d++ = pixr + fgr;
			*d++ = pixg + fgg;
			*d++ = pixb + fgb;
			*d++ = bga + (((0xff - bga) * alpha + 0x80) >> 8);
		}

		s = s0 += gmap->rowstride;
		d = d0 += rowstride;
	}
}

void
gnome_rfont_render_glyph_rgb8 (const GnomeRFont * rfont, gint glyph,
				guint32 rgba,
				gdouble x, gdouble y,
				guchar * buf,
				gint width, gint height, gint rowstride,
				guint flags)
{
	const GnomeRFontGrayMap * gmap;
	gint xp, yp, x0, y0, x1, y1;
	guint inkr, inkg, inkb, inka;
	guchar * s, * s0, * d, * d0;

	gmap = gnome_rfont_get_glyph_graymap (rfont, glyph);

	xp = (gint) floor (x + 0.5);
	yp = (gint) floor (y + 0.5);

	x0 = MAX (0, xp + gmap->x0);
	y0 = MAX (0, yp + gmap->y0);
	x1 = MIN (width, xp + gmap->x0 + gmap->width);
	y1 = MIN (height, yp + gmap->y0 + gmap->height);

	inkr = rgba >> 24;
	inkg = (rgba >> 16) & 0xff;
	inkb = (rgba >> 8) & 0xff;
	inka = rgba & 0xff;

	d0 = d = buf + y0 * rowstride + x0 * 3;
	s0 = s = gmap->pixels + (y0 - yp - gmap->y0) * gmap->rowstride + (x0 - xp - gmap->x0);

	for (y = y0; y < y1; y++) {
		for (x = x0; x < x1; x++) {
			guint bgr, bgg, bgb;
			guint fgr, fgg, fgb;
			guint alpha;

			bgr = * (d + 0);
			bgg = * (d + 1);
			bgb = * (d + 2);
			alpha = (inka * (*s++) + 0x80) >> 8;

			fgr = (inkr * alpha + 0x80) >> 8;
			fgg = (inkg * alpha + 0x80) >> 8;
			fgb = (inkb * alpha + 0x80) >> 8;

			*d++ = ((bgr * (0xff - alpha) + 0x80) >> 8) + fgr;
			*d++ = ((bgg * (0xff - alpha) + 0x80) >> 8) + fgg;
			*d++ = ((bgb * (0xff - alpha) + 0x80) >> 8) + fgb;
		}

		s = s0 += gmap->rowstride;
		d = d0 += rowstride;
	}
}

void gnome_rfont_render_pgl_rgb8 (const GnomePosGlyphList * pgl,
				    gdouble x, gdouble y,
				    guchar * buf,
				    gint width, gint height, gint rowstride,
				    guint flags)
{
	GSList * l;
	gint i;

	g_return_if_fail (pgl != NULL);
	g_return_if_fail (buf != NULL);

	for (l = pgl->strings; l != NULL; l = l->next) {
		GnomePosString * string;
		string = (GnomePosString *) l->data;
		for (i = 0; i < string->length; i++) {
			gnome_rfont_render_glyph_rgb8 (string->rfont, string->glyphs[i].glyph,
						       string->glyphs[i].color,
						       x + string->glyphs[i].x, y + string->glyphs[i].y,
						       buf,
						       width, height, rowstride,
						       flags);
		}
	}
}

/* fixme: collect all bits of cached data into single struct */

void
gnome_rfont_render_glyph_gdk_drawable (const GnomeRFont * rfont, gint glyph,
				guint32 rgba,
				gdouble x, gdouble y,
				GdkDrawable * drawable,
				guint32 background,
				guint flags)
{
	const GnomeRFontDisplay * dsp;
	GdkPixmap * pixmap;
	GdkBitmap * bitmap;
	const GnomeRFontDimension * dim;
	gint xp, yp;

	g_return_if_fail (rfont != NULL);
	g_return_if_fail (GNOME_IS_RFONT (rfont));

	dsp = gnome_rfont_get_display (rfont, drawable);
	g_return_if_fail (dsp != NULL);

	pixmap = (GdkPixmap *) dsp_get_glyph_pixmap (dsp, glyph, rgba, background);
	if (!pixmap) return;
	bitmap = (GdkBitmap *) dsp_get_glyph_bitmap (dsp, glyph);
	g_return_if_fail (bitmap != NULL);

	dim = gnome_rfont_get_glyph_dimension (rfont, glyph);

	xp = (gint) floor (x + 0.5) + dim->idim.x0;
	yp = (gint) floor (y + 0.5) + dim->idim.y0;

	gdk_gc_set_clip_mask (dsp->gc, bitmap);
	gdk_gc_set_clip_origin (dsp->gc, xp, yp);
	gdk_draw_pixmap (drawable, dsp->gc, pixmap,
		0, 0,
		xp, yp,
		dim->idim.x1 - dim->idim.x0, dim->idim.y1 - dim->idim.y0);
}

void gnome_rfont_render_pgl_gdk_drawable (const GnomePosGlyphList * pgl,
				gdouble x, gdouble y,
				GdkDrawable * drawable,
				guint32 background,
				guint flags)
{
	GSList * l;
	gint i;

	g_return_if_fail (pgl != NULL);
	g_return_if_fail (drawable != NULL);

	for (l = pgl->strings; l != NULL; l = l->next) {
		GnomePosString * string;
		string = (GnomePosString *) l->data;
		for (i = 0; i < string->length; i++) {
			gnome_rfont_render_glyph_gdk_drawable (string->rfont, string->glyphs[i].glyph,
							       string->glyphs[i].color,
							       x + string->glyphs[i].x, y + string->glyphs[i].y,
							       drawable,
							       background,
							       flags);
		}
	}
}

/* Helpers */

static gboolean
rfont_free_pixmap (gpointer key, gpointer value, gpointer data)
{
	gdk_pixmap_unref ((GdkPixmap *) value);

	return TRUE;
}

static gboolean
rfont_free_bitmap (gpointer key, gpointer value, gpointer data)
{
	gdk_bitmap_unref ((GdkBitmap *) value);

	return TRUE;
}

static gboolean
rfont_free_display (gpointer key, gpointer value, gpointer data)
{
	GnomeRFontDisplay * display;

	display = (GnomeRFontDisplay *) value;

	if (display->pixmaps) {
		g_hash_table_foreach_remove (display->pixmaps, rfont_free_pixmap, NULL);
		g_hash_table_destroy (display->pixmaps);
	}

	if (display->bitmaps) {
		g_hash_table_foreach_remove (display->bitmaps, rfont_free_bitmap, NULL);
		g_hash_table_destroy (display->bitmaps);
	}

	if (display->gc) gdk_gc_unref (display->gc);
	if (display->bmgc) gdk_gc_unref (display->bmgc);

	g_free (display);

	return TRUE;
}

static gboolean
rfont_free_dimension (gpointer key, gpointer value, gpointer data)
{
	g_free (value);

	return TRUE;
}

static gboolean
rfont_free_graymap (gpointer key, gpointer value, gpointer data)
{
	GnomeRFontGrayMap * gmap;

	gmap = (GnomeRFontGrayMap *) value;

	if (gmap->pixels) g_free (gmap->pixels);
	g_free (gmap);

	return TRUE;
}

static gboolean
rfont_free_svp (gpointer key, gpointer value, gpointer data)
{
	art_svp_free ((ArtSVP *) value);

	return TRUE;
}

static gboolean
rfont_free_bpath (gpointer key, gpointer value, gpointer data)
{
	art_free (value);

	return TRUE;
}

static guint
rfont_hash (gconstpointer key)
{
	GnomeRFont * rfont;

	rfont = (GnomeRFont *) key;

	return (guint) rfont->font;
}

static gint
rfont_equal (gconstpointer key1, gconstpointer key2)
{
	GnomeRFont * f1, * f2;

	f1 = (GnomeRFont *) key1;
	f2 = (GnomeRFont *) key2;

	if (f1->font != f2->font) return FALSE;

	return art_affine_equal (f1->transform, f2->transform);
}

static const GdkPixmap *
dsp_get_glyph_pixmap (const GnomeRFontDisplay * dsp, gint glyph, guint32 rgba, guint32 background)
{
	gint hash;
	GdkPixmap * pixmap;
	const GnomeRFontGrayMap * gmap;
	gint x, y;
	guchar * tmp, * tp;
	guint bgr, bgg, bgb;

	/*
	 * Well - we should do intelligent hashing here - this is test
	 */

	hash = glyph + (rgba & 0xf0f0f0f0) + ((background & 0xf0f0f000) >> 8);

	pixmap = g_hash_table_lookup (dsp->pixmaps, GINT_TO_POINTER (hash));

	if (pixmap) return pixmap;

	gmap = gnome_rfont_get_glyph_graymap (dsp->rfont, glyph);
	g_return_val_if_fail (gmap != NULL, NULL);

	if ((gmap->width < 1) || (gmap->height < 1)) return NULL;

	tmp = tp = g_new0 (gchar, gmap->width * gmap->height * 3);

	bgr = background >> 24;
	bgg = (background >> 16) & 0xff;
	bgb = (background >> 8) & 0xff;

	for (y = 0; y < gmap->height; y++) {
		for (x = 0; x < gmap->width; x++) {
			*tp++ = bgr;
			*tp++ = bgg;
			*tp++ = bgb;
		}
	}

	gnome_rfont_render_glyph_rgb8 (dsp->rfont, glyph, rgba, -gmap->x0, -gmap->y0,
				       tmp,
				       gmap->width, gmap->height, gmap->width * 3, 0);

	pixmap = gdk_pixmap_new (dsp->window, gmap->width, gmap->height, dsp->visual->depth);

	gdk_gc_set_clip_mask (dsp->gc, NULL);
	gdk_gc_set_function (dsp->gc, GDK_SET);
	gdk_draw_rectangle (pixmap, dsp->gc, TRUE, 0, 0, gmap->width, gmap->height);
	gdk_gc_set_function (dsp->gc, GDK_COPY);

	gdk_draw_rgb_image (pixmap, dsp->gc, 0, 0, gmap->width, gmap->height,
		GDK_RGB_DITHER_NONE, tmp, gmap->width * 3);

	g_free (tmp);

#if 0
	gdk_gc_set_function (dsp->gc, GDK_CLEAR);
	for (y = 0; y < gmap->height; y++) {
		for (x = 0; x < gmap->width; x++) {
			if (gmap->pixels[y * gmap->rowstride + x] > 0) {
				alpha = (gmap->pixels[y * gmap->rowstride + x] * fga + 0x80) >> 8;
				gdk_draw_point (pixmap, dsp->gc, x, y);
			}
		}
	}
#endif

	g_hash_table_insert (dsp->pixmaps, GINT_TO_POINTER (hash), pixmap);

	return pixmap;
}

static const GdkBitmap *
dsp_get_glyph_bitmap (const GnomeRFontDisplay * dsp, gint glyph)
{
	GdkBitmap * bitmap;
	const GnomeRFontGrayMap * gmap;
	gint x, y;

	bitmap = g_hash_table_lookup (dsp->bitmaps, GINT_TO_POINTER (glyph));

	if (bitmap) return bitmap;

	gmap = gnome_rfont_get_glyph_graymap (dsp->rfont, glyph);
	g_return_val_if_fail (gmap != NULL, NULL);

	if ((gmap->width < 1) || (gmap->height < 1)) return NULL;

	bitmap = gdk_pixmap_new (dsp->window, gmap->width, gmap->height, 1);

	if (!dsp->bmgc) {
		((GnomeRFontDisplay *) dsp)->bmgc = gdk_gc_new (bitmap);
	}

	gdk_gc_set_clip_mask (dsp->bmgc, NULL);
	gdk_gc_set_function (dsp->bmgc, GDK_CLEAR);
	gdk_draw_rectangle (bitmap, dsp->bmgc, TRUE, 0, 0, gmap->width, gmap->height);

	gdk_gc_set_function (dsp->bmgc, GDK_SET);
	for (y = 0; y < gmap->height; y++) {
		for (x = 0; x < gmap->width; x++) {
			if (gmap->pixels[y * gmap->rowstride + x] > 0) {
				gdk_draw_point (bitmap, dsp->bmgc, x, y);
			}
		}
	}

	g_hash_table_insert (dsp->bitmaps, GINT_TO_POINTER (glyph), bitmap);

	return bitmap;
}

/*
 * These are original GnomeDisplayFont routines
 *
 * We merge these eventually with GnomeRFont
 */

GHashTable *scaled_display_fonts = NULL;
GHashTable *gnome_font_family_to_x_hash = NULL;
GHashTable *gnome_font_x_to_family_hash = NULL;
GHashTable *gnome_font_weight_to_string_hash = NULL;
GHashTable *gnome_font_string_to_weight_hash = NULL;

typedef struct
{
  char *print_name;
  char *x_name;
} gnome_font_family_to_x_map;

gnome_font_family_to_x_map gnome_font_family_to_x_maps[] =
{
  {"Bitstream Charter", "charter"},
  {"Century Schoolbook L", "new century schoolbook"},
  {"Courier", "courier"},
  {"Dingbats", "dingbats"},
  {"Helvetica", "helvetica"},
  {"ITC Avant Garde Gothic", "gothic"},
  {"ITC Bookman", "bookman"},
  {"ITC Zapf Chancery", "zapf chancery"},
  {"ITC Zapf Dingbats", "zapf dingbats"},
  {"New Century Schoolbook", "new century schoolbook"},
  {"Nimbus Mono L", "nimbus"}, /* times roman? */
  {"Nimbus Roman No9 L", "nimbus"}, /* times roman? */
  {"Nimbus Sans L", "nimbus"}, /* times roman? */
  {"Palatino", "palatino"},
  {"Standard Symbols L", "symbol"},
  {"Symbol", "symbol"},
  {"Times", "times"},
  {"URW Bookman L", "bookman"},
  {"URW Chancery L", "chancery"},
  {"URW Gothic L", "gothic"},
  {"URW Palladio L", "palladio"},
  {0, 0}
};

typedef struct
{
  GnomeFontWeight weight;
  char *name;
} gnome_font_weight_to_string_map;

gnome_font_weight_to_string_map gnome_font_weight_to_string_maps [] =
{
  {GNOME_FONT_EXTRA_LIGHT, "extralight"},
  {GNOME_FONT_THIN, "thin"},
  {GNOME_FONT_LIGHT, "light"},
  {GNOME_FONT_BOOK, "regular"},
  {GNOME_FONT_MEDIUM, "medium"},
  {GNOME_FONT_SEMI, "demibold"},
  {GNOME_FONT_BOLD, "bold"},
  {GNOME_FONT_HEAVY, "heavy"},
  {GNOME_FONT_EXTRABOLD, "extrabold"},
  {GNOME_FONT_BLACK, "black"},
  {GNOME_FONT_EXTRABLACK, "extrablack"},
  {(GnomeFontWeight)0, 0}
};


static void
initialize_hashes (void)
{
  gnome_font_family_to_x_map *ftx = gnome_font_family_to_x_maps;
  gnome_font_weight_to_string_map *wtx = gnome_font_weight_to_string_maps;

  scaled_display_fonts = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_family_to_x_hash = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_x_to_family_hash = g_hash_table_new (g_str_hash, g_str_equal);
  gnome_font_weight_to_string_hash =
    g_hash_table_new (g_int_hash, g_int_equal);
  gnome_font_string_to_weight_hash =
    g_hash_table_new (g_str_hash, g_str_equal);

  /* build 2 way hashes for x vs font family names */
  while (ftx->print_name)
    {
#if 0
      debugmsg ("inserting family-->x mapping: '%s' --> '%s'\n",
		ftx->print_name, ftx->x_name);
#endif
      g_hash_table_insert (gnome_font_family_to_x_hash,
			   ftx->print_name, ftx->x_name);
      g_hash_table_insert (gnome_font_x_to_family_hash,
			   ftx->x_name, ftx->print_name);
      ftx ++;
    }


  /* build 2 way hashes for gnome font weight to x names */
  while (wtx->name)
    {
#if 0
      debugmsg ("inserting weight-->string mapping: '%d' --> '%s'\n",
	      wtx->weight, wtx->name);
#endif
      g_hash_table_insert (gnome_font_weight_to_string_hash,
			   & (wtx->weight), wtx->name);
      g_hash_table_insert (gnome_font_string_to_weight_hash,
			   wtx->name, & (wtx->weight));
      wtx ++;
    }
}


static const char *
gnome_font_family_to_x_name (const char *family)
{
  const char *x_name;
  if (! scaled_display_fonts) initialize_hashes ();

  x_name = (const char *) g_hash_table_lookup (gnome_font_family_to_x_hash, family);
  if (! x_name)
    return "helvetica";

  return x_name;
}


const char *
gnome_font_weight_to_string (GnomeFontWeight gfw)
{
  const char *x_weight;
  if (! scaled_display_fonts) initialize_hashes ();

  x_weight =
    (const char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &gfw);

  if (! x_weight)
    x_weight = "medium";

  return x_weight;
}


GnomeFontWeight
string_to_gnome_font_weight (const char *x_weight)
{
  GnomeFontWeight *gfw;

  if (! scaled_display_fonts) initialize_hashes ();

  gfw =
    (GnomeFontWeight *)
    g_hash_table_lookup (gnome_font_string_to_weight_hash, x_weight);

  return (*gfw);
}


static char *
setFontComponent (const char *font_name, unsigned int pos, const char *s)
{
  char *p;
  const char *parts[ 14 ];
  char *scratch;
  char *new_name;
  int i, len;

  if (pos > 13)
    {
      g_warning ("replaceFontComponent -- pos out of ");
      g_warning ("range: %d\n", pos);
      return g_strdup (font_name);
    }

  if (font_name != NULL && *font_name != '\0')
    scratch = g_strdup (font_name);
  else
    {
      char *d = "-*-*-*-*-*-*-*-*-*-*-*-*-*-*";
      scratch = g_strdup (d);
    }

  for (i=0, p=scratch+1; i<14 && (*p) != '\0'; i++)
    {
      parts[ i ] = p;
      for (; *p != '-' && *p != '\0'; p ++)
	;
      *p = '\0';
      p ++;
    }

  if (s == NULL)
    parts[ pos ] = "*";
  else
    parts[ pos ] = s;

  for (i=0, len=0; i<14; i++)
    len += strlen (parts[ i ]);

  new_name = (char *) g_malloc (len + 17);

  g_snprintf(new_name, len + 17, "-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s",
	  parts[ 0 ], parts[ 1 ], parts[ 2 ], parts[ 3 ], 
	  parts[ 4 ], parts[ 5 ], parts[ 6 ], parts[ 7 ], 
	  parts[ 8 ], parts[ 9 ], parts[ 10 ], parts[ 11 ],
	  parts[ 12 ], parts[ 13 ]);

  g_free (scratch);
  return new_name;
}


static char *
setFontFamily (const char *font_name, const char *s)
{
  return setFontComponent (font_name, 1, s);
}

static char *
setFontWeight (const char *font_name, const char *s)
{
  return setFontComponent (font_name, 2, s);
}

static char *
setFontSlant (const char *font_name, const char *s)
{
  return setFontComponent (font_name, 3, s);
}


static char *
setFontPixelSize (const char *font_name, int i)
{
  gchar temp[4 * sizeof (int)];
  gchar *returnval;
  g_snprintf (temp, 4 * sizeof (int), "%d", i );
  returnval = setFontComponent (font_name, 6, temp);
  return returnval;
}


static void
setComponentReplace (char **io,
		     char *(*func) (const char *font_name, const char *s),
		     const char *part)
{
  char *new_name = (*func) (*io, part);
  g_free (*io);
  (*io) = new_name;
}

static GdkFont *
find_best_x_weight (const char *x_font_name_ret, GnomeFontWeight start_weight)
{
  gchar *current_x_name = g_strdup( x_font_name_ret );
  GdkFont *current_font = gdk_font_load (current_x_name);
  GnomeFontWeight weighta, weightb;
  const char *x_weight;

  if (! scaled_display_fonts) initialize_hashes ();

  weighta = weightb = start_weight;
  if ( current_font )
    {
      g_free( current_x_name );
      return current_font;
    }
  while( weighta >= GNOME_FONT_LIGHTEST || weightb <= GNOME_FONT_HEAVIEST )
    {
      weighta --;
      weightb ++;

      x_weight =
	(const char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &weighta);

      if ( x_weight )
	{
	  setComponentReplace (&current_x_name, setFontWeight, x_weight);
	  current_font = gdk_font_load (current_x_name);
	  if ( current_font )
	    {
	      g_free( current_x_name );
	      return current_font;
	    }
	}

      x_weight =
	(const char *) g_hash_table_lookup (gnome_font_weight_to_string_hash, &weightb);

      if ( x_weight )
	{
	  setComponentReplace (&current_x_name, setFontWeight, x_weight);
	  current_font = gdk_font_load (current_x_name);
	  if ( current_font )
	    {
	      g_free( current_x_name );
	      return current_font;
	    }
	}
  
    }
  
  g_free( current_x_name );
  return NULL;
}

/* here is a horrible little routine that tries to find an X font
   that is about the same size as a printer font */
static GdkFont *find_best_x_font (const char *unsized_name,
				  GnomeFont *gnome_font,
				  char **x_font_name_ret)
{
  /* sample string used during the search for a good size */
  /*char *test_string = "abcdefghijklmnopqrstuvwxyz";*/
  /*char *test_string = "snider";*/
  /* const char *test_string = "oooowwww"; */
#if 0
  int length = strlen (test_string);

  double perfect_width; /* the printer-font width of the sample string */
  double max_width; /* we must end up below this size */

  int direction; /* either +1 or -1 -- are we looking for a bigger font
		    or a smaller one? */

  /* information about the best match found so far */
  int current_width; /* width of rendered sample string */
  int current_diff; /* difference between current_width and perfect_width */
#endif
  int current_size; /* point size */
  char *current_x_name; /* the x-style name of the current font */
  GdkFont *current_font; /* the current font */

#if 0
  g_return_val_if_fail (unsized_name, NULL);
  g_return_val_if_fail (gnome_font, NULL);
  g_return_val_if_fail (x_font_name_ret, NULL);

  perfect_width = gnome_font_get_width_string (gnome_font, test_string);
  /*space_width = gnome_font_get_width_string (gnome_font, " ");*/
  /*max_width = perfect_width + space_width / 2;*/
  max_width = perfect_width /*- 1*/;
  debugmsg( "perfect_width = %f\n", perfect_width );
#endif
  current_size = gnome_font_get_size (gnome_font);
  current_x_name = setFontPixelSize (unsized_name, current_size);
  current_font = find_best_x_weight (current_x_name, gnome_font->private->fontmap_entry->private->weight_code);

  /* start by getting some x font loaded */
  if (current_font == NULL)
    {
#if 0
      debugmsg ("couldn't load first font '%s'\n", current_x_name);
#endif
      /* is it italic? */
      if (gnome_font->private->fontmap_entry->private->italic)
	{
	  /* try oblique */
#if 0
	  debugmsg ("trying oblique...\n");
#endif
	  setComponentReplace (&current_x_name, setFontSlant, "o");
	  current_font = find_best_x_weight (current_x_name, gnome_font->private->fontmap_entry->private->weight_code);

	  if (current_font == NULL)
	    {
	      /* failed.  put italic back */
#if 0
	      debugmsg ("oblique failed: '%s'\n", current_x_name);
#endif
	      setComponentReplace (&current_x_name, setFontSlant, "i");
	    }
	}

      if (current_font == NULL)
	{
	  /* try helvetica */
#if 0
	  debugmsg ("trying hevletica...\n");
#endif
	  setComponentReplace (&current_x_name, setFontFamily, "helvetica");
	  current_font = find_best_x_weight (current_x_name, gnome_font->private->fontmap_entry->private->weight_code);

	  if (current_font == NULL)
	    {
#if 0
	      debugmsg ("helvetica failed. giving up...\n");
#endif
	      return NULL;
	    }
	}
    }
#if 0
  debugmsg ("initial name loaded: '%s'\n", current_x_name);
#endif
#if 0
  current_width = gdk_text_width (current_font, test_string, length);
  current_diff = (int) (perfect_width - current_width);

  debugmsg ("size=%d, starting diff=%d\n", current_size, current_diff);

  /* are we looking for a bigger x font or a smaller one? */
  /*
  if (current_diff > 0) direction = 1;
  else direction = -1;
  */
  if (current_diff > 0)
    {
      /*direction = (int) gnome_font->size / 6.0;*/
      direction = 10;
      /*if (direction < 1) direction = 1;*/
      debugmsg ("size=%f, direction=%d\n", gnome_font->size, direction);
    }
  else
    {
      /*direction = (int) gnome_font->size / -6.0;*/
      direction = -10;
      /*if (direction > -1) direction = -1;*/
      debugmsg ("size=%f, direction=%d\n", gnome_font->size, direction);
    }


  while (current_diff != 0)
    {
      int next_size = current_size + direction;
      char *next_x_name = setFontPixelSize (current_x_name, next_size);
      GdkFont *next_font = find_best_x_weight (next_x_name, gnome_font->fontmap_entry->weight_code);
      int next_width;
      int next_diff;

      if (next_font == NULL)
	{
	  debugmsg ("couldn't load '%s'\n", next_x_name);
	  continue; /* ??? */
	}
      next_width = gdk_text_width (next_font, test_string, length);
      next_diff = (int) (perfect_width - next_width);

      /* if we've gotten bigger than max_width, stop */
      if (next_width >= max_width && direction > 0)
	{
	  debugmsg (" bailing because next_width[%d] >= max_width[%d]\n",
		  next_width, (int) max_width);
	  break;
	}

      /* if the previous font was closer than this one, stop */
      if (iabs (next_diff) > iabs (current_diff))
	{
	  debugmsg (" bailing because next_diff[%d] > current_diff[%d]\n",
		  next_diff, current_diff);
	  break;
	}

      g_free (current_x_name);
      gdk_font_unref (current_font);

      current_size = next_size;
      current_x_name = next_x_name;
      current_font = next_font;
      current_diff = next_diff;

      debugmsg (" new size=%d, new diff=%d\n", current_size, current_diff);
    }
#endif
  (*x_font_name_ret) = current_x_name;
  return current_font;
}



/* construct an x font name and find the best point size */

static GnomeDisplayFont *create_display_font (const char *family,
					      GnomeFontWeight weight,
					      gboolean italic,
					      double points,
					      double scale)
{
	GnomeFont *gnome_font;
#if 0
  GnomeFont *closest;
#endif
#if 0
  GnomeFontUnsized *gfus;
  GnomeDisplayFont *wpdf;
#else
#if 0
  GnomeFontFace * face;
#endif
  GnomeRFont * rfont;
  gdouble transform[6];
#endif

  g_return_val_if_fail (family, NULL);

  if (! scaled_display_fonts) initialize_hashes ();

#if 0
  if ((closest = gnome_font_new_closest (family, weight, italic, 0)) == NULL)
	  return NULL;
#endif

  gnome_font = gnome_font_new_closest (family, weight, italic, ceil (points * scale));

  g_return_val_if_fail (gnome_font != NULL, NULL);

#if 0	/* This is crashing and we don't use */
          /*face in the rest of the function. (Chema)*/
  face = closest->private->fontmap_entry;
#endif

  art_affine_scale (transform, scale, -scale);
  rfont = gnome_font_get_rfont (gnome_font, transform);

	  gnome_font_unref (gnome_font);
#if 0
  wpdf = (GnomeDisplayFont *) g_malloc (sizeof (GnomeDisplayFont));
  wpdf->unsized_font = gfus;
  wpdf->gnome_font = gnome_font;
  wpdf->scale = scale;
#endif

  return rfont;
}

/* cache for create_display_font */

GnomeDisplayFont *gnome_get_display_font (const char *family,
					  GnomeFontWeight weight,
					  gboolean italic,
					  double points,
					  double scale)
{
  char key[ 1024 ];
  GnomeDisplayFont *wpdf;

  g_snprintf (key, sizeof (key), "%s.%s.%s.%d",
	      family,
	      gnome_font_weight_to_string (weight),
	      italic?"t":"f",
	      (int) ceil (points * scale));

  if (! scaled_display_fonts) initialize_hashes ();

  /*
  debugmsg ("lookup_display_font -- hash:%p key='%s'\n",
	  scaled_display_fonts,
	  key);
  */
  wpdf =
    (GnomeDisplayFont *) g_hash_table_lookup (scaled_display_fonts, key);

  if (wpdf == NULL)
    {
      wpdf = create_display_font (family, weight, italic, points, scale);
      if (wpdf)
	g_hash_table_insert (scaled_display_fonts, g_strdup (key), wpdf);
      else
	/* We should maintain a negative cache.  */;
    }

  return wpdf;
}

int
gnome_display_font_height (GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, 0);
	g_return_val_if_fail (gdf->gdk_font != NULL, 0);

	return MAX (gdf->gdk_font->ascent + gdf->gdk_font->descent,
		    gdf->font->private->size);
}

GnomeDisplayFont *gnome_font_get_display_font( GnomeFont *font )
{
  return gnome_get_display_font( font->private->fontmap_entry->private->familyname,
				 font->private->fontmap_entry->private->weight_code,
				 font->private->fontmap_entry->private->italic,
				 font->private->size,
				 font->private->scale );
}

/*
 * Here are GnomeDisplayFont methods
 */

static void
gdf_find_x_fonts (GnomeDisplayFont * gdf)
{
	const gchar * x_family;
	gchar * xfn;

	x_family = gnome_font_family_to_x_name (gdf->face->private->familyname);
	xfn = g_strdup ("-*-helvetica-*-r-*-*-12-*-*-*-*-*-*-*");

	setComponentReplace (&xfn, setFontFamily, x_family);
	setComponentReplace (&xfn, setFontWeight,
		       gnome_font_weight_to_string (gdf->face->private->weight_code));

	if (gdf->face->private->italic)
		setComponentReplace (&xfn, setFontSlant, "i");


	gdf->gdk_font = find_best_x_font (xfn, gdf->font, &(gdf->x_font_name));

	g_free (xfn);
}

const GnomeFontFace *
gnome_display_font_get_face (const GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (gdf), NULL);

	return gnome_rfont_get_face (gdf);
}

const GnomeFont *
gnome_display_font_get_font (const GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (gdf), NULL);

	return gnome_rfont_get_font (gdf);
}

gdouble
gnome_display_font_get_scale (const GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, 0.0);
	g_return_val_if_fail (GNOME_IS_RFONT (gdf), 0.0);

	return art_affine_expansion (gdf->transform);
}

const gchar *
gnome_display_font_get_x_font_name (const GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (gdf), NULL);

	if (!gdf->x_font_name) {
		gdf_find_x_fonts ((GnomeDisplayFont *) gdf);
	}

	return gdf->x_font_name;
}

GdkFont *
gnome_display_font_get_gdk_font (const GnomeDisplayFont * gdf)
{
	g_return_val_if_fail (gdf != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_RFONT (gdf), NULL);

	if (!gdf->x_font_name) {
		gdf_find_x_fonts ((GnomeDisplayFont *) gdf);
	}

	return gdf->gdk_font;
}

void
gnome_display_font_ref (GnomeDisplayFont * gdf)
{
	gtk_object_ref (GTK_OBJECT (gdf));
}

void
gnome_display_font_unref (GnomeDisplayFont * gdf)
{
	gtk_object_unref (GTK_OBJECT (gdf));
}







