/* GnomeScan - Gnome Scanning Infrastructure
 *
 * gnomescantypes.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "gnomescantypes.h"
#include "gnomescan-intl.h"

/* GTypes */
GType
gnome_scan_range_get_type	(void)
{
  return g_pointer_type_register_static ("GnomeScanRange");
}

GType
gnome_scan_geometry_get_type	(void)
{
  return g_pointer_type_register_static ("GnomeScanGeometry");
}

/********************************
 * 	   CONTEXT INFO		*
 ********************************/

/**
 * gnome_scan_context_info_copy:
 * @info: a #GnomeScanContextInfo
 * 
 * Copy a GnomeScanContextInfo.
 * 
 * Return value: a copy of @info
 */
GnomeScanContextInfo*
gnome_scan_context_info_copy (GnomeScanContextInfo *info)
{
  GnomeScanContextInfo *copy;

  copy = g_memdup (info, sizeof (GnomeScanContextInfo));
  copy->area = g_memdup (info->area, sizeof (GnomeScanArea));

  return (gpointer) copy;
}

/**
 * gnome_scan_context_info_free:
 * @info: a #GnomeScanContextInfo
 * 
 * Free a #GnomeScanContextInfo
 */
void
gnome_scan_context_info_free (GnomeScanContextInfo *info)
{
  g_free (info);
}

GType
gnome_scan_context_info_get_type (void)
{
  static GType type = 0;

  if (G_UNLIKELY (type == 0)) {
    type = g_boxed_type_register_static ("GnomeScanContextInfo",
					 (GBoxedCopyFunc) gnome_scan_context_info_copy,
					 (GBoxedFreeFunc) gnome_scan_context_info_free);
  }

  return type;   
}


/********************************
 *     	    FORECAST		*
 ********************************/

/**
 * gnome_scan_forecast_copy:
 * @forecast: a #GnomeScanForecast
 * 
 * Copy a GnomeScanForecast.
 * 
 * Return value: a copy of @forecast
 */
GnomeScanForecast*
gnome_scan_forecast_copy (GnomeScanForecast *forecast)
{
  GnomeScanForecast *copy;
  copy = g_memdup (forecast, sizeof (GnomeScanForecast));
  g_debug ("%s : %i => %i", __FUNCTION__, forecast->size, copy->size);
  return copy;
}

/**
 * gnome_scan_forecast_free:
 * @forecast: a #GnomeScanForecast
 * 
 * Free a #GnomeScanForecast
 */
void
gnome_scan_forecast_free (GnomeScanForecast *forecast)
{
  g_free (forecast);
}

GType
gnome_scan_forecast_get_type (void)
{
  static GType type = 0;

  if (G_UNLIKELY (type == 0)) {
    type = g_boxed_type_register_static ("GnomeScanForecast",
					 (GBoxedCopyFunc) gnome_scan_forecast_copy,
					 (GBoxedFreeFunc) gnome_scan_forecast_free);
  }

  return type;   
}

/********************************
 *     ACQUISITION RESULT		*
 ********************************/

/**
 * gnome_scan_result_copy:
 * @result: a #GnomeScanResult
 * 
 * Copy a GnomeScanResult.
 * 
 * Return value: a copy of @result
 */
GnomeScanResult*
gnome_scan_result_copy (GnomeScanResult *result)
{
  GnomeScanResult *copy;
  copy = g_memdup (result, sizeof (GnomeScanResult));
  copy->image = gdk_pixbuf_copy (result->image);
  return copy;
}

/**
 * gnome_scan_result_free:
 * @result: a #GnomeScanResult
 * 
 * Free a #GnomeScanResult
 */
void
gnome_scan_result_free (GnomeScanResult *result)
{
  g_object_unref (result->image);
  g_free (result);
}

GType
gnome_scan_result_get_type (void)
{
  static GType type = 0;

  if (G_UNLIKELY (type == 0)) {
    type = g_boxed_type_register_static ("GnomeScanResult",
					 (GBoxedCopyFunc) gnome_scan_result_copy,
					 (GBoxedFreeFunc) gnome_scan_result_free);
  }

  return type;   
}

/**
 * gnome_scanner_source_get_title:
 * @source: the source value
 * 
 * Return the translated source title.
 * 
 * Return value: the source title string
 */
gchar*
gnome_scanner_source_get_title (GnomeScannerSource source)
{
  static const gchar* titles[] = {
    /* translators, scanner source  */
    N_("None"),
    /* translators, scanner source  */
    N_("Automatic"),
    /* translators, scanner source  */
    N_("Flatbed"),
    /* translators, scanner source  */
    N_("Automatic Document Feeder")
  };

  return _(titles[source]);
}
