/* GSane - SANE GNOME Scan backend 
 * Copyright © 2007-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GSane is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * GSane is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GSane.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include "gsane-option-area.h"

#define GSANE_OPTION_AREA_GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE((o), GSANE_TYPE_OPTION_AREA, GSaneOptionAreaPrivate))

struct _GSaneOptionAreaPrivate {
	GSANE_OPTION_HANDLER_DEFINE_OPTION(tl_x);
	GSANE_OPTION_HANDLER_DEFINE_OPTION(tl_y);
	GSANE_OPTION_HANDLER_DEFINE_OPTION(br_x);
	GSANE_OPTION_HANDLER_DEFINE_OPTION(br_y);
	guint count;
	GnomeScanOption* origin;
	GnomeScanOption* paper_size;
	GnomeScanOption* page_orientation;
};

G_DEFINE_TYPE(GSaneOptionArea, gsane_option_area, GSANE_TYPE_OPTION_HANDLER)

static void
gsane_option_area_install_origin(GSaneOptionArea *self)
{
	GSaneOptionHandler* handler = (GSaneOptionHandler*) self;
	/* TODO: convert from non MM unit */
	GnomeScanPoint point = {
		.x = gsane_option_handler_get_double(handler, self->priv->tl_x_desc, self->priv->tl_x_index, NULL),
		.y = gsane_option_handler_get_double(handler, self->priv->tl_y_desc, self->priv->tl_y_index, NULL)
	};

	self->priv->origin = GNOME_SCAN_OPTION(gnome_scan_option_pointer_new("origin", _("Origin"), _("Coordinate of the top left corner of the paper."), GNOME_SCAN_OPTION_GROUP_FORMAT, GETTEXT_PACKAGE,
									     gnome_scan_point_dup(&point), GNOME_SCAN_OPTION_HINT_HIDDEN));
	gnome_scan_node_install_option(GNOME_SCAN_NODE(handler->scanner), self->priv->origin);
}

static void
gsane_option_area_update(GSaneOptionArea* self)
{
	GSaneOptionHandler* handler = GSANE_OPTION_HANDLER(self);
	GtkPaperSize* ps;
	GnomeScanEnumValue* value;
	gdouble tl_x, tl_y, br_x, br_y;
	GnomeScanPoint *origin;
	GtkPageOrientation orientation;

	gnome_scan_option_enum_get_value(GNOME_SCAN_OPTION_ENUM(self->priv->paper_size), &value);
	ps = g_value_get_boxed(&value->value);
	gnome_scan_option_enum_get_value(GNOME_SCAN_OPTION_ENUM(self->priv->page_orientation), &value);
	orientation = g_value_get_enum(&value->value);
	origin = gnome_scan_option_pointer_get_value(GNOME_SCAN_OPTION_POINTER(self->priv->origin));
	
	tl_x = origin->x;
	tl_y = origin->y;
	if (orientation == GTK_PAGE_ORIENTATION_PORTRAIT
	    || g_str_equal(gtk_paper_size_get_name(ps), "max")) {
		br_x = origin->x + gtk_paper_size_get_width(ps, GTK_UNIT_MM);
		br_y = origin->y + gtk_paper_size_get_height(ps, GTK_UNIT_MM);
	}
	else {
		br_x = origin->x + gtk_paper_size_get_height(ps, GTK_UNIT_MM);
		br_y = origin->y + gtk_paper_size_get_width(ps, GTK_UNIT_MM);
	}

	g_debug("Scan (%.2f ; %.2f) (%.2f ; %.2f)", tl_x, tl_y, br_x, br_y);

	/* TODO: convert to non MM unit */
	gsane_option_handler_set_double(handler, self->priv->tl_x_desc, self->priv->tl_x_index, tl_x, NULL);
	gsane_option_handler_set_double(handler, self->priv->tl_y_desc, self->priv->tl_y_index, tl_y, NULL);
	gsane_option_handler_set_double(handler, self->priv->br_x_desc, self->priv->br_x_index, br_x, NULL);
	gsane_option_handler_set_double(handler, self->priv->br_y_desc, self->priv->br_y_index, br_y, NULL);
}

static void
gsane_option_area_option_changed(GSaneOptionArea* self, GParamSpec* pspec, GObject* option)
{
	gsane_option_area_update(self);
}

static GSList*
gsane_option_area_add_paper_size(GSaneOptionArea* self, GSList* values, const gchar* paper_name, gdouble width, gdouble height, GtkUnit unit)
{
	GtkPaperSize* ps;
	GnomeScanEnumValue evalue;
	GValue* value;

	ps = gtk_paper_size_new(paper_name);

	g_debug("%s is %.2fx%.2f %s",
		gtk_paper_size_get_display_name(ps),
		gtk_paper_size_get_width(ps, unit),
		gtk_paper_size_get_height(ps, unit),
		gnome_scan_unit_to_string(unit));

	if (gtk_paper_size_get_width(ps, unit) > width
	    || gtk_paper_size_get_height(ps, unit) > height) {
		g_debug("%s is too wide", gtk_paper_size_get_display_name(ps));
		gtk_paper_size_free(ps);
		return values;
	}

	value = g_new0(GValue, 1);
	g_value_init(value, GTK_TYPE_PAPER_SIZE);
	g_value_set_boxed(value, ps);
	gnome_scan_enum_value_init(&evalue, value, gtk_paper_size_get_display_name(ps), NULL);
	values = g_slist_append(values, gnome_scan_enum_value_dup(&evalue));
	gnome_scan_enum_value_destroy(&evalue);
	g_value_reset(value);
	g_free(value);

	return values;
}

static void
gsane_option_area_install_paper_size(GSaneOptionArea* self)
{
	GSaneOptionHandler* handler = (GSaneOptionHandler*)self;
	GtkPaperSize* ps;
	GnomeScanEnumValue evalue;
	GValue* value;
	GSList* values = NULL;
	gdouble width, height;
	GnomeScanUnit unit = gsane_option_unit(self->priv->tl_x_desc);

	/* get width and height */
	switch (self->priv->tl_x_desc->type) {
	case SANE_TYPE_INT:
		width = self->priv->br_x_desc->constraint.range->max;
		height = self->priv->br_y_desc->constraint.range->max;
		break;
	case SANE_TYPE_FIXED:
		width = SANE_UNFIX(self->priv->br_x_desc->constraint.range->max);
		height = SANE_UNFIX(self->priv->br_y_desc->constraint.range->max);
		break;
	default:
		g_warning("Unsupported coordinate type.");
		return;
		break;
	}

	g_debug("area is %.2fx%.2f %s", width, height, gnome_scan_unit_to_string(unit));

	/* define max paper size */
	value = g_new0(GValue, 1);
	g_value_init(value, GTK_TYPE_PAPER_SIZE);
	ps = gtk_paper_size_new_custom("max", _("Maximum"), width, height, unit);
	g_value_set_boxed(value, ps);
	gnome_scan_enum_value_init(&evalue, value, gtk_paper_size_get_display_name(ps), NULL);
	values = g_slist_append(values, gnome_scan_enum_value_dup(&evalue));
	gnome_scan_enum_value_destroy(&evalue);

	/* A4 */
	values = gsane_option_area_add_paper_size(self, values, GTK_PAPER_NAME_A4, width, height, unit);
	/* A5 */
	values = gsane_option_area_add_paper_size(self, values, GTK_PAPER_NAME_A5, width, height, unit);
	/* US bullshit */
	values = gsane_option_area_add_paper_size(self, values, GTK_PAPER_NAME_LETTER, width, height, unit);
	values = gsane_option_area_add_paper_size(self, values, GTK_PAPER_NAME_LEGAL, width, height, unit);
	values = gsane_option_area_add_paper_size(self, values, GTK_PAPER_NAME_EXECUTIVE, width, height, unit);

	g_value_reset(value);
	g_free(value);

	self->priv->paper_size = GNOME_SCAN_OPTION(gnome_scan_option_enum_new("paper-size", _("Paper-Size"), _("Paper-Size"), GNOME_SCAN_OPTION_GROUP_FORMAT, GETTEXT_PACKAGE,
									      values->data, values, GNOME_SCAN_OPTION_HINT_PRIMARY));
	gnome_scan_node_install_option(GNOME_SCAN_NODE(handler->scanner), self->priv->paper_size);
	g_signal_connect_swapped(self->priv->paper_size, "notify::value", G_CALLBACK(gsane_option_area_option_changed), self);
}


static GSList*
gsane_option_area_add_orientation(GSaneOptionArea* self, GSList* values, GtkPageOrientation orientation, const gchar* stock_id)
{
	GnomeScanEnumValue evalue;
	GValue* value;

	value = g_new0(GValue, 1);
	g_value_init(value, GTK_TYPE_PAGE_ORIENTATION);
	g_value_set_enum(value, orientation);
	gnome_scan_enum_value_init(&evalue, value, stock_id, NULL);
	values = g_slist_append(values, gnome_scan_enum_value_dup(&evalue));
	gnome_scan_enum_value_destroy(&evalue);
	g_value_reset(value);
	g_free(value);

	return values;
}

static void
gsane_option_area_install_page_orientation(GSaneOptionArea *self)
{
	GSaneOptionHandler* handler = (GSaneOptionHandler*)self;
	GSList* values = NULL;
	GnomeScanEnumValue*value;
	GtkPaperSize *ps;
	gdouble width, height;

	gnome_scan_option_enum_get_value(GNOME_SCAN_OPTION_ENUM(self->priv->paper_size), &value);
	ps = g_value_get_boxed(&value->value);
	width = gtk_paper_size_get_width(ps, GTK_UNIT_MM);
	height= gtk_paper_size_get_height(ps, GTK_UNIT_MM);

	values = gsane_option_area_add_orientation(self, values, GTK_PAGE_ORIENTATION_PORTRAIT, GTK_STOCK_ORIENTATION_PORTRAIT);
	values = gsane_option_area_add_orientation(self, values, GTK_PAGE_ORIENTATION_LANDSCAPE, GTK_STOCK_ORIENTATION_LANDSCAPE);
	value = width <= height ? values->data : values->next->data;
	self->priv->page_orientation = GNOME_SCAN_OPTION(gnome_scan_option_enum_new("page-orientation", _("Page Orientation"), _("Page orientation"), GNOME_SCAN_OPTION_GROUP_FORMAT, GETTEXT_PACKAGE,
										    value, values, GNOME_SCAN_OPTION_HINT_PRIMARY));
	gnome_scan_node_install_option(GNOME_SCAN_NODE(handler->scanner), self->priv->page_orientation);
	g_signal_connect_swapped(self->priv->page_orientation, "notify::value", G_CALLBACK(gsane_option_area_option_changed), self);
}

static void
gsane_option_handler_handle_matched(GSaneOptionHandler *handler, const SANE_Option_Descriptor* desc, SANE_Int n, const gchar*group)
{
	GSaneOptionArea* self = GSANE_OPTION_AREA (handler);

	self->priv->count++;

	if (self->priv->count < 4)
		return;

	/* we have matched all SANE option, actually install high level options */
	gsane_option_area_install_origin(self);
	gsane_option_area_install_paper_size(self);
	gsane_option_area_install_page_orientation(self);
}

static void
gsane_option_area_handle_option(GSaneOptionHandler *handler, const SANE_Option_Descriptor* desc, SANE_Int n, const gchar*group)
{
	GSaneOptionArea* self = GSANE_OPTION_AREA (handler);
	GSaneOptionAreaClass* klass = GSANE_OPTION_AREA_GET_CLASS(self);

	GSANE_OPTION_HANDLER_MATCH_OPTION(self, klass, tl_x, desc, n, group); 
	GSANE_OPTION_HANDLER_MATCH_OPTION(self, klass, tl_y, desc, n, group); 
	GSANE_OPTION_HANDLER_MATCH_OPTION(self, klass, br_x, desc, n, group); 
	GSANE_OPTION_HANDLER_MATCH_OPTION(self, klass, br_y, desc, n, group);
}

static void
gsane_option_area_class_init(GSaneOptionAreaClass *klass)
{
	g_type_class_add_private(klass, sizeof(GSaneOptionAreaPrivate));
	GSaneOptionHandlerClass *oh_class = GSANE_OPTION_HANDLER_CLASS(klass);
	oh_class->unique	= TRUE;
	oh_class->handle_option = gsane_option_area_handle_option;

#define	gsane_option_handler_handle_tl_x	gsane_option_handler_handle_matched
#define	gsane_option_handler_handle_tl_y	gsane_option_handler_handle_matched
#define	gsane_option_handler_handle_br_x	gsane_option_handler_handle_matched
#define	gsane_option_handler_handle_br_y	gsane_option_handler_handle_matched

	GSANE_OPTION_HANDLER_CLASS_INSTALL_OPTION(klass, tl_x, "tl-x");
	GSANE_OPTION_HANDLER_CLASS_INSTALL_OPTION(klass, tl_y, "tl-y");
	GSANE_OPTION_HANDLER_CLASS_INSTALL_OPTION(klass, br_x, "br-x");
	GSANE_OPTION_HANDLER_CLASS_INSTALL_OPTION(klass, br_y, "br-y");

#undef gsane_option_handler_handle_tl_x
#undef gsane_option_handler_handle_tl_y
#undef gsane_option_handler_handle_br_x
#undef gsane_option_handler_handle_br_y

}

static void
gsane_option_area_init(GSaneOptionArea* self)
{
	self->priv = GSANE_OPTION_AREA_GET_PRIVATE(self);
	self->priv->count = 0;
}


