/*
 * GNOME Speech - Speech services for the GNOME desktop
 *
 * Copyright 2002 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * festivalspeaker.c.c: Implementation of the FestivalSynthesisDriver
 *                            object-- a GNOME Speech driver for the Festival
 *                            Speech Synthesis System
 *
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <libbonobo.h>
#include "festivalspeaker.h"
 


static GObjectClass *parent_class;

static void
festival_write (FestivalSpeaker *speaker,
		gchar *text)
{
	gint l = strlen (text);
	write (speaker->festival_input[1], text, l);
}



static FestivalSpeaker *
festival_speaker_from_servant (PortableServer_Servant *servant)
{
	return FESTIVAL_SPEAKER(bonobo_object_from_servant (servant));
}



static CORBA_long
festival_say (PortableServer_Servant servant,
	      const CORBA_char *text,
	      CORBA_Environment *ev)
{
	FestivalSpeaker *speaker = festival_speaker_from_servant (servant);
	gchar *speak_command;
  
	speak_command = g_strdup_printf ("(SayText \"%s\")\n", text);
	festival_write (speaker, speak_command);
	g_free (speak_command);
	return TRUE;
}



static CORBA_boolean
festival_stop (PortableServer_Servant servant,
	       CORBA_Environment *ev)
{
	FestivalSpeaker *speaker = festival_speaker_from_servant (servant);

	festival_write (speaker, "(audio_mode 'shutup)\n");
	return TRUE;
}



static gboolean
festival_set_rate (Speaker *speaker,
		   gdouble new_value)
{
	FestivalSpeaker *fest_speaker = FESTIVAL_SPEAKER (speaker);
	gdouble stretch;
	gchar *command;

	stretch = 200/new_value;
	command = g_strdup_printf ("(Parameter.set 'Duration_Stretch %lf)\n", stretch);
	festival_write (fest_speaker, command);
	g_free (command);
	return TRUE;
}



static gboolean
festival_set_voice (Speaker *s,
		    gdouble new_value)
{
	FestivalSpeaker *festival_speaker = FESTIVAL_SPEAKER(s);


	if (new_value == 1)
		festival_write (festival_speaker,
				"(voice_ked_diphone)\n");
	else if (new_value == 2)
		festival_write (festival_speaker,
				"(voice_kal_diphone)\n");
	else
		return FALSE;
	return TRUE;
}



static void
festival_speaker_init (FestivalSpeaker *festival_speaker)
{
	Speaker *speaker = SPEAKER (festival_speaker);

	festival_speaker->pid = -1;
	festival_speaker->festival_input[0] = festival_speaker->festival_input[1] = -1;
	festival_speaker->festival_output[0] = festival_speaker->festival_output[1] = -1;  

	/* Add supported parameters */

	speaker_add_parameter (speaker,
					"rate",
					1,
					200,
					400,
					festival_set_rate);

	/* Add voice parameter */

	speaker_add_parameter (speaker,
			       "voice",
			       1,
			       1,
			       2,
			       festival_set_voice);
}



static void
festival_speaker_finalize (GObject *obj)
{
	FestivalSpeaker *speaker = FESTIVAL_SPEAKER (obj);

	if (speaker->pid != -1) {

		/* Tell the festival binary we're done */

		festival_write (speaker, "(exit)\n");
		
		/* Wait for it to exit */

		waitpid (speaker->pid, NULL, 0);

		/* Close fds */

		close (speaker->festival_output[0]);
		close (speaker->festival_input[1]);
	}
	if (parent_class->finalize)
		parent_class->finalize (obj);
}



static void
festival_speaker_class_init (FestivalSpeakerClass *klass)
{
	SpeakerClass *class = SPEAKER_CLASS (klass);
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
  
	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = festival_speaker_finalize;

	/* Initialize parent class epv table */

	class->epv.say = festival_say;
	class->epv.stop = festival_stop;
}




BONOBO_TYPE_FUNC (FestivalSpeaker,
		  speaker_get_type (),
		  festival_speaker);



FestivalSpeaker *
festival_speaker_new (const GNOME_Speech_VoiceInfo *info)
{
	FestivalSpeaker *speaker;
	speaker = g_object_new (FESTIVAL_SPEAKER_TYPE, NULL);
  
	pipe (speaker->festival_input);
	pipe (speaker->festival_output);
	speaker->pid = fork ();
	if (!speaker->pid) {
		close (0);
		dup (speaker->festival_input[0]);
		close (1);
		close (2);
		dup (speaker->festival_output[1]);
		dup (speaker->festival_output[1]);
		close (speaker->festival_input[1]);
		close (speaker->festival_output[0]);
		execlp ("festival", "festival", NULL);
	}

	close (speaker->festival_output[1]);
	close (speaker->festival_input[0]);

	/* Setup async streaming mode */

	festival_write (speaker, "(audio_mode 'async)\n");

	
	/* Setup voice */

	if (!strcmp (info->name, "Kevin"))
		festival_write (speaker, "(voice_ked_diphone)\n");
	else if (!strcmp (info->name, "Kal"))
		festival_write (speaker, "(voice_kal_diphone)\n");
	return speaker;
}




