/* $Id: gdict-app.c,v 1.124 2005/03/03 17:08:30 vnoel Exp $ */

/*
 *  Mike Hughes <mfh@psilord.com>
 *  Papadimitriou Spiros <spapadim+@cs.cmu.edu>
 *  Bradford Hovinen <hovinen@udel.edu>
 *
 *  This code released under the GNU GPL.
 *  Read the file COPYING for more information.
 *
 *  GDict main window
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <libgnomeui/gnome-entry.h>
#include "dict.h"
#include "gdict-app.h"
#include "gdict-about.h"
#include "gdict-pref.h"
#include "gdict-pref-dialog.h"
#include "gdict-defbox.h"
#include "gdict-speller.h"
#include "gdict-utils.h"
#include "gdict-web-search.h"

#include <libgnomeprint/gnome-print-config.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprintui/gnome-print-dialog.h>

#define GDICT_DEFAULT_WIDTH  450
#define GDICT_DEFAULT_HEIGHT 330

static GObjectClass *parent_class;
gboolean  gail_loaded = FALSE;
GSList *gdict_windows = NULL;

enum
{
  TARGET_STRING,
  TARGET_TEXT,
  TARGET_COMPOUND_TEXT,
  TARGET_UTF8_STRING,
  TARGET_TEXT_BUFFER_CONTENTS
};

/* gdict_init_context
 * Initialises the context object with information on the current server
 * Returns 0 on success, -1 if the server could not be found
 */
gint
gdict_init_context (GDictWindow *gdict) 
{
  g_return_if_fail (GDICT_IS_WINDOW (gdict));

  if (gdict->context) 
    dict_context_destroy (gdict->context);

  gdict->context = dict_context_new (gdict->pref->server, gdict->pref->port);
  gdict->context->command = dict_disconnect_command_new ();
  gdict->defbox->context = gdict->context;

#ifdef ENABLE_IPV6
  if (gdict->context->host6info)
    return 0;
#endif

  if (gdict->context->hostinfo)
    return 0;
  else
    return -1;
}

void
gdict_socket_error_cb (GtkWidget *widget, const gchar *message, gpointer data) 
{
  GDictWindow *gdict = data;
  GtkWidget *dialog;
  dialog = gtk_message_dialog_new_with_markup (NULL, GTK_DIALOG_DESTROY_WITH_PARENT,
                 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                 "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
                 _("Connection error"), message); 
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
}

gint
gdict_lookup_spelling (GDictWindow *gdict, gchar *text, gboolean pattern) 
{
  GDictSpeller *speller = GDICT_SPELLER(gdict->speller);
  GtkWidget *check;

  g_return_val_if_fail (text != NULL, 0);

  if (pattern) 
    speller->strat = "re";
  else
    speller->strat = gdict->pref->dfl_strat;

  check = gtk_ui_manager_get_widget (gdict->ui, "/MenuBar/ViewMenu/ViewSpellings");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (check), TRUE);
  
  if (gdict_speller_lookup (gdict, speller, text) == -1)
    return -1;

  return 0;
}

static gboolean
is_pattern (gchar *text) 
{
  if (strpbrk (text, "*|{}()[]"))
    return TRUE;
  else
    return FALSE;
}

void
gdict_not_online (GDictWindow *gdict) 
{
  GtkWidget *dialog;
  
  dialog = gtk_message_dialog_new_with_markup (GTK_WINDOW (gdict),
                 GTK_DIALOG_DESTROY_WITH_PARENT,
                 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
                 "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
                 _("Unable to perform requested operation."),
                 _("Either the server you are using is not available \n"
                 "or you are not connected to the Internet."));

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);
  gdk_window_set_cursor (GTK_WIDGET(gdict)->window, NULL);
}

void
gdict_lookup_definition (GDictWindow *gdict, gchar *text) 
{
  gint retval;
  gchar *word_entry_text;

  g_return_if_fail (text != NULL);

  /* If needed update the the word_entry */
  word_entry_text = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
  if ((word_entry_text == NULL) ||
      (*word_entry_text == 0)   ||
      (strcmp (word_entry_text, text) != 0)) {
    gnome_entry_prepend_history (GNOME_ENTRY (gdict->gnome_word_entry), 1, text);
    gtk_entry_set_text (GTK_ENTRY (gdict->word_entry), text);
  }

  g_free (word_entry_text);

  if (gdict->pref->smart && is_pattern (text)) {
    retval = gdict_lookup_spelling (gdict, text, TRUE);
  }
  else {
    retval = gdict_defbox_lookup (gdict->defbox, text);
    if (!retval)
      gtk_widget_show (GTK_WIDGET(gdict));
  }

  if (retval)
    gdict_not_online (gdict);
}

void
gdict_lookup_web (GDictWindow *gdict, gchar *text)
{
  gchar *command = NULL;
  int status;
  GError *error = NULL;
  GSList *sitelist;
  GDictWebSearchSite *site;
  gchar *browser_command;
  gchar *uri, *uri_list = NULL;

  for (sitelist = gdict->web_search_sites; sitelist != NULL; sitelist = g_slist_next (sitelist)) {
    site = sitelist->data;
    if (site->enabled) {
      uri = g_strdup_printf (site->url, text);
      if (!uri_list)
        uri_list = g_strdup(uri);
      else {
        gchar *tmp;
        tmp = g_strdup_printf ("%s %s", uri_list, uri);
        g_free (uri_list);
        uri_list = tmp;
      }
      g_free (uri);
    }
  }

  browser_command = gconf_client_get_string (gdict_get_gconf_client (), 
               "/desktop/gnome/url-handlers/http/command",
               NULL);

  if (g_ascii_strncasecmp (browser_command, "epiphany", 8) == 0)
    command = g_strdup_printf("epiphany --new-tab %s", uri_list);
  else
    command = g_strdup_printf(browser_command, uri_list);

  g_spawn_command_line_async (command, &error);
  
  g_free (browser_command);
  g_free (command);
  g_free (uri_list);
}

static void
gdict_lookup_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  gchar *text = NULL;

  text = gtk_editable_get_chars (GTK_EDITABLE (gdict->word_entry), 0, -1);
  if (gdict->search_type == SEARCH_TYPE_REGULAR)
    gdict_lookup_definition (gdict, text);
  else
    gdict_lookup_web (gdict, text);

  g_free (text);
}

static void
gdict_menu_item_set_sensitive (GDictWindow *gdict, char *path, gboolean sensitive)
{
  GtkAction *action;

  g_return_if_fail (path);
  action = gtk_ui_manager_get_action (gdict->ui, path);
  gtk_action_set_sensitive (action, sensitive);
}

void
gdict_menus_set_sensitive (GDictWindow *gdict, gboolean found_definition)
{
  gdict_menu_item_set_sensitive (gdict, "/MenuBar/DictionaryMenu/Print", found_definition);
  gdict_menu_item_set_sensitive (gdict, "/MenuBar/EditMenu/Find", found_definition);
  if (gdict->edit_actions)
    gtk_action_group_set_sensitive (gdict->edit_actions, found_definition);
}

static void
gdict_lookup_button_drag_cb (GtkWidget *widget, GdkDragContext *context, gint x, gint y,
           GtkSelectionData *sd, guint info, guint t, gpointer data)
{
  gchar *text;
  GDictWindow *gdict = data;

  text = gtk_selection_data_get_text (sd);

  if (text) {
    gtk_entry_set_text (GTK_ENTRY (gdict->word_entry), text);
    gnome_entry_prepend_history (GNOME_ENTRY (gdict->gnome_word_entry), 1, text);
    gdict_lookup_definition (gdict, text);
    g_free (text);
  }
}

static void 
gdict_print_dialog_cb (GtkDialog *widget, int arg, gpointer data)
{
  GDictWindow *gdict = data;

  gtk_widget_hide (GTK_WIDGET(widget));
  switch (arg) {
  case GNOME_PRINT_DIALOG_RESPONSE_PRINT:
    gdict_defbox_prepare_print (gdict->defbox);
    gnome_print_job_print ((GnomePrintJob *) gdict->print_job);
  case GNOME_PRINT_DIALOG_RESPONSE_CANCEL:
    g_object_unref ((GnomePrintJob *) gdict->print_job);
    break;
  case GNOME_PRINT_DIALOG_RESPONSE_PREVIEW:
    gdict_defbox_prepare_print (gdict->defbox);
    gdict_defbox_print_preview (gdict->defbox);
    break;
  }
}

static void
gdict_print_cb (GtkAction *action, GtkWindow *window)
{
  GDictWindow *gdict = GDICT_WINDOW (window);
  static GtkWidget *print_dialog=NULL;
  int result;

  gdict->print_job = gnome_print_job_new (NULL);
  
  if (print_dialog == NULL) {
    print_dialog = gnome_print_dialog_new ((GnomePrintJob *) gdict->print_job,
                   _("Print Word Definition"), 0);
    g_signal_connect (G_OBJECT (print_dialog), "response",
          G_CALLBACK (gdict_print_dialog_cb), gdict);
  }
  result = gtk_dialog_run (GTK_DIALOG(print_dialog));

}

static void
close_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  if (gdict->applet)
    gtk_widget_hide (GTK_WIDGET(gdict));
  else {
    g_free (gdict->find_text);
    gtk_main_quit();
  }
}

static void
cut_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  g_signal_emit_by_name (G_OBJECT (gdict->word_entry), "cut_clipboard");
}

static void
copy_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  if (GTK_WIDGET_HAS_FOCUS (gdict->defbox))
    g_signal_emit_by_name (G_OBJECT (gdict->defbox), "copy_clipboard");
  else if (GTK_WIDGET_HAS_FOCUS (gdict->word_entry))
    g_signal_emit_by_name (G_OBJECT (gdict->word_entry), "copy_clipboard");
}

static void
paste_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  g_signal_emit_by_name (G_OBJECT (gdict->word_entry), "paste_clipboard");
}

static void
clear_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  gdict_defbox_clear (gdict->defbox);

  if (gdict->edit_actions)
    gtk_action_group_set_sensitive (gdict->edit_actions, FALSE);
}

static void
select_all_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  GtkTextBuffer* buffer = NULL;
  GtkTextIter start_iter, end_iter;

  gtk_widget_grab_focus (GTK_WIDGET (gdict->defbox));
  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (gdict->defbox));
  gtk_text_buffer_get_bounds (buffer, &start_iter, &end_iter);
  gtk_text_buffer_select_range (buffer, &start_iter, &end_iter);
}

static void
find_word_dialog_cb (GtkDialog *dialog, gint id, gpointer data)
{
  GDictWindow *gdict = data;
  GtkEntry *entry;
  const gchar *str;

  entry = GTK_ENTRY (gdict->find_dialog_entry);

  gtk_statusbar_pop (GTK_STATUSBAR (gdict->statusbar), 0);

  switch (id) {
  case GTK_RESPONSE_APPLY:
    str = gtk_entry_get_text (entry);

    if (str == NULL)
      return;
      
    if (gdict_defbox_find (gdict->defbox, str, gdict->search_from_beginning))
    {
      gdict->search_from_beginning = FALSE;     
      g_free (gdict->find_text);           
      gdict->find_text = g_strdup (str);
      gdict_menu_item_set_sensitive (gdict, "/MenuBar/EditMenu/FindNext", TRUE);
    }
    else {
      gdict->search_from_beginning = TRUE;
      gdict_menu_item_set_sensitive (gdict, "/MenuBar/EditMenu/FindNext", FALSE);
    }
    
    break;

  default:
    gtk_widget_hide (GTK_WIDGET (dialog));
    break;
  }
}

static void
gdict_find_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);

  if (gdict->find_dialog == NULL) {
    GtkWidget *dialog;
    GtkWidget *hbox;
    GtkWidget *label;
    GtkWidget *entry;
    dialog = gtk_dialog_new_with_buttons (_("Find"), GTK_WINDOW(gdict),
        GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
        GTK_STOCK_FIND, GTK_RESPONSE_APPLY,
        NULL);
    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
    gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_APPLY);

    hbox = gtk_hbox_new (FALSE, 8);
    label = gtk_label_new_with_mnemonic (_("_Search for:"));
    entry = gtk_entry_new ();
    gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
    gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
    gdict->find_dialog_entry = entry;

    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
    
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
    
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox);
        
    g_signal_connect (G_OBJECT (dialog), "response", 
        G_CALLBACK (find_word_dialog_cb), gdict);
    g_signal_connect(G_OBJECT (dialog), "destroy",
        G_CALLBACK (gtk_widget_hide), &dialog);

    gtk_widget_show_all (GTK_WIDGET (dialog));

    gdict->search_from_beginning = TRUE;
    gdict->find_dialog = dialog;
    gdict->find_dialog_entry = entry;
  }
  else
    gtk_window_present (GTK_WINDOW (gdict->find_dialog));

  if (gdict->find_text != NULL)
    gtk_entry_set_text (GTK_ENTRY (gdict->find_dialog_entry), gdict->find_text);

  gtk_entry_select_region (GTK_ENTRY (gdict->find_dialog_entry), 0, -1);
  gtk_widget_grab_focus (gdict->find_dialog_entry);
}

static void
find_next_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  if (gdict->find_text)
    gdict->search_from_beginning = !gdict_defbox_find (gdict->defbox, 
                                                       gdict->find_text, 
                                                       gdict->search_from_beginning);
  else
    gdict_find_cb (NULL, NULL);
}

static void
gdict_preferences_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  if (!gdict->pref_dialog)
    gdict->pref_dialog = gdict_pref_dialog_new (gdict);

  gtk_window_present (GTK_WINDOW (gdict->pref_dialog));
}

static void
gdict_help_contents_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  GError *error = NULL;

  gnome_help_display ("gnome-dictionary", NULL, &error);
  if (error) {
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new_with_markup (GTK_WINDOW(gdict),
             GTK_DIALOG_DESTROY_WITH_PARENT,
             GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
             "<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
             _("Could not display help"), error->message); 

    gtk_dialog_run (GTK_DIALOG (dialog));
    gtk_widget_destroy (dialog);
    g_error_free (error);
  }
}

static void
gdict_web_search_cb (GtkAction *action, GtkWidget *widget, gpointer data)
{
  GDictWindow *gdict = GDICT_WINDOW (data);
  GtkWidget *menu_item;
  static int width = GDICT_DEFAULT_WIDTH, height = GDICT_DEFAULT_HEIGHT;
  menu_item = gtk_ui_manager_get_widget (gdict->ui,
              "/SearchTypeMenuBar/SearchTypeMenu/RegularSearch");
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menu_item))) {
    gdict->search_type = SEARCH_TYPE_REGULAR;
    gtk_widget_show (gdict->paned);
    gtk_widget_show (gdict->statusbar);
    gtk_window_set_resizable (GTK_WINDOW (gdict), TRUE);
    gtk_window_resize (GTK_WINDOW (gdict), width, height);
  } else {
    gdict->search_type = SEARCH_TYPE_WEB;
    gtk_widget_hide (gdict->paned);
    gtk_widget_hide (gdict->statusbar);
    gtk_window_get_size (GTK_WINDOW (gdict), &width, &height);
    gtk_window_set_resizable (GTK_WINDOW (gdict), FALSE);
  }
}

static void
gdict_about_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  gdict_about (GTK_WINDOW(gdict));
}

static void
gdict_view_spellings_cb (GtkAction *action, GtkWidget *widget)
{
  GDictWindow *gdict = GDICT_WINDOW (widget);
  
  if (gdict->speller_visible) {
    gtk_widget_hide (GTK_WIDGET(gdict->speller));
    gdict->speller_visible = FALSE;
  } else {
    gtk_widget_show (GTK_WIDGET(gdict->speller));
    gdict->speller_visible = TRUE;
  }
}

static void
gdict_window_close (GtkWidget *widget, GdkEvent *event, gpointer data)
{
  GDictWindow *gdict = data;
  if (gdict->applet != NULL) 
    gtk_widget_hide (widget);
  else {
    gdict_windows = g_slist_remove (gdict_windows, GTK_WINDOW(gdict));    
    dict_context_destroy (gdict->context); /* FIXME */
    gdict_web_search_remove_all (gdict);
    g_slist_free (gdict->web_search_sites);
    gtk_widget_destroy (GTK_WIDGET(gdict));        
    if (gdict_windows == NULL) 
      gtk_main_quit ();
  }
}

GtkWidget *
gdict_window_new (void)
{
  GDictWindow *gdict;
  GtkWidget *window;
  window = g_object_new (GDICT_TYPE_WINDOW, NULL);
  gdict = GDICT_WINDOW (window);
  gdict_init_context (gdict);
  gdict->applet = NULL;
  gdict->search_type = SEARCH_TYPE_REGULAR;
  gdict->web_search_sites = gdict_web_search_default_sites ();
  gdict_windows = g_slist_prepend (gdict_windows, window);
  return window;
}

/* Menu bar */

static const gchar *ui_info = 
"<ui>"
"  <menubar name='MenuBar'>"
"    <menu action='DictionaryMenu'>"
"      <menuitem action='LookUp'/>"
"      <separator/>"
"      <menuitem action='Print'/>"
"      <separator/>"
"      <menuitem action='Quit'/>"
"    </menu>"
"    <menu action='EditMenu'>"
"      <menuitem action='Cut'/>"
"      <menuitem action='Copy'/>"
"      <menuitem action='Paste'/>"
"      <menuitem action='SelectAll'/>"
"      <separator/>"
"      <menuitem action='Find'/>"
"      <menuitem action='FindNext'/>"
"      <separator/>"
"      <menuitem action='Preferences'/>"
"    </menu>"
"    <menu action='ViewMenu'>"
"      <menuitem action='ViewSpellings'/>"
"    </menu>"
"    <menu action='HelpMenu'>"
"      <menuitem action='HelpContents'/>"
"      <menuitem action='About'/>"
"    </menu>"
"  </menubar>"
"</ui>";

static const gchar *search_type_ui = 
"<ui>"
"<menubar name='SearchTypeMenuBar'>"
" <menu action='SearchTypeMenu'>"
"   <menuitem action='RegularSearch'/>"
"   <menuitem action='WebSearch'/>"
" </menu>"
"</menubar>"
"</ui>";

static GtkActionEntry menu_entries[] = {

  /* Top */
  { "DictionaryMenu", NULL, N_("_Dictionary") },
  { "EditMenu", NULL, "_Edit" },
  { "ViewMenu", NULL, "_View" },
  { "HelpMenu", NULL, "_Help" },

  /* Dictionary menu */
  { "LookUp", GTK_STOCK_FIND, N_("_Look Up Word"), "<control>L",
    N_("Lookup word in dictionary"), G_CALLBACK (gdict_lookup_cb) },
  { "Print", GTK_STOCK_PRINT, N_("_Print"), "<control>P",
    N_("Print the current definition"), G_CALLBACK (gdict_print_cb) },
  { "Quit", GTK_STOCK_CLOSE, N_("_Close"), "<control>W",
    N_("Close the application"), G_CALLBACK (close_cb) },

  /* Edit menu, see below for cut, paste etc. */
  { "Find", GTK_STOCK_FIND, N_("_Find..."), "<control>F",
    N_("Find a word in the text"), G_CALLBACK (gdict_find_cb) },
  { "FindNext", NULL, N_("Find Ne_xt"), "<control>G",
    N_("Find next occurrence of the word"), G_CALLBACK (find_next_cb) },
  { "Preferences", GTK_STOCK_PREFERENCES, N_("P_references"), NULL,
    N_("Configure the application"), G_CALLBACK (gdict_preferences_cb) },

  /* Help menu */
  { "HelpContents", GTK_STOCK_HELP, N_("_Contents"), "F1",
     N_("View help for this application"), G_CALLBACK (gdict_help_contents_cb) },
  { "About", GTK_STOCK_ABOUT, N_("_About"), NULL,
    N_("About this application"), G_CALLBACK (gdict_about_cb) }
};


static GtkRadioActionEntry search_type_entries[] = {
  { "SearchTypeMenu", NULL, "Search Type"},
  { "RegularSearch", NULL, N_("On a Dictionary Server"), NULL,
    NULL, 1},
  { "WebSearch", NULL, N_("On a Web Site"), NULL,
    NULL, 2}
};

/* a separate group for items that change sensitivity */
static GtkActionEntry edit_entries[] = {
  { "Cut", GTK_STOCK_CUT, N_("Cu_t"), "<control>X",
    N_("Cut the selection"), G_CALLBACK (cut_cb) },
  { "Copy", GTK_STOCK_COPY, N_("_Copy"), "<control>C",
    N_("Copy the selection"), G_CALLBACK (copy_cb) },
  { "Paste", GTK_STOCK_PASTE, N_("_Paste"), "<control>V",
    N_("Paste clipboard"), G_CALLBACK (paste_cb) },
  { "SelectAll", NULL, N_("Select _All"), "<control>A",
    N_("Select everything"), G_CALLBACK (select_all_cb) }
};

/* a seperate group for toggle entries */
static GtkToggleActionEntry toggle_entries[] = {
  { "ViewSpellings", GTK_STOCK_SPELL_CHECK, N_("_Spellings"), "<control>S",
    N_("View alternate spellings"), G_CALLBACK (gdict_view_spellings_cb) }
};

GtkWidget *
gdict_window_init (GDictWindow *gdict)
{
  GtkWidget *menubar;
  GtkWidget *container;
  GtkWidget *hbox, *paned;
  GtkWidget *button;
  GtkWidget *word_label;
  GtkWidget *scrolled;
  GDictSpeller *speller;
  GtkWidget *window, *main_area;
  GtkTooltips *tooltips;
  GtkToolItem *menu_button;
  GtkUIManager *ui;
  GtkWidget *menu;
  GtkWidget *gnome_word_entry, *word_entry;
  GError *error = NULL;
  static GtkTargetEntry drop_targets [] = {
    { "UTF8_STRING", 0, 0 },
    { "COMPOUND_TEXT", 0, 0 },
    { "TEXT", 0, 0 },
    { "text/plain", 0, 0 },
    { "STRING",     0, 0 }
  };

  tooltips = gtk_tooltips_new ();

  window = GTK_WIDGET (gdict);

  gtk_window_set_default_size (GTK_WINDOW (window), GDICT_DEFAULT_WIDTH, GDICT_DEFAULT_HEIGHT);

  gdict->menu_actions = gtk_action_group_new ("MenuActions");
  gtk_action_group_set_translation_domain (gdict->menu_actions, NULL);
  gtk_action_group_add_actions (gdict->menu_actions, menu_entries,
              G_N_ELEMENTS (menu_entries), gdict);
  gtk_action_group_add_toggle_actions (gdict->menu_actions, toggle_entries,
              G_N_ELEMENTS (toggle_entries), gdict);

  /* a separate group to update sensitivity */
  gdict->edit_actions = gtk_action_group_new ("EditActions");
  gtk_action_group_set_translation_domain (gdict->edit_actions, NULL);
  gtk_action_group_add_actions (gdict->edit_actions, edit_entries,
              G_N_ELEMENTS (edit_entries), gdict);
  gtk_action_group_set_sensitive (gdict->edit_actions, FALSE);

  gdict->search_type_actions = gtk_action_group_new ("SearchTypeActions");
  gtk_action_group_add_radio_actions (gdict->search_type_actions, search_type_entries,
              G_N_ELEMENTS (search_type_entries), 1, 
              G_CALLBACK(gdict_web_search_cb), gdict);

  ui = gtk_ui_manager_new ();
  gtk_ui_manager_insert_action_group (ui, gdict->menu_actions, 0);
  gtk_ui_manager_insert_action_group (ui, gdict->edit_actions, 0);
  gtk_ui_manager_insert_action_group (ui, gdict->search_type_actions, 0);
  gtk_window_add_accel_group (GTK_WINDOW (window), 
            gtk_ui_manager_get_accel_group (ui));

  if (!gtk_ui_manager_add_ui_from_string (ui, ui_info, -1, &error)) {
    g_message ("building menus failed: %s", error->message);
    g_error_free (error);
    return NULL;
  }
  gtk_ui_manager_add_ui_from_string (ui, search_type_ui, -1, &error);

  container = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), container);

  menubar = gtk_ui_manager_get_widget (ui, "/MenuBar");
  gtk_box_pack_start (GTK_BOX (container), menubar, FALSE, FALSE, 0);

  main_area = gtk_vbox_new (FALSE, 0);
  gtk_box_set_spacing (GTK_BOX(main_area), 6);
  gtk_container_set_border_width (GTK_CONTAINER (main_area), 6);
  gtk_box_pack_start (GTK_BOX (container), main_area, TRUE, TRUE, 0);

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_set_spacing (GTK_BOX (hbox), 6);
  gtk_box_pack_start (GTK_BOX (main_area), hbox, FALSE, FALSE, 0);

  word_label = gtk_label_new_with_mnemonic (_("_Word:"));
  gtk_box_pack_start (GTK_BOX (hbox), word_label, FALSE, FALSE, 0);

  button = gdict_label_new_with_stock_image (_("_Look Up Word"), GTK_STOCK_FIND, NULL);
  menu_button = gtk_menu_tool_button_new (button, NULL);
  menu = gtk_widget_get_ancestor (gtk_ui_manager_get_widget (ui, "/SearchTypeMenuBar/SearchTypeMenu/RegularSearch"),
          GTK_TYPE_MENU);
  gtk_menu_tool_button_set_menu (GTK_MENU_TOOL_BUTTON(menu_button), GTK_WIDGET(menu));
  gtk_box_pack_end (GTK_BOX (hbox), GTK_WIDGET(menu_button), FALSE, FALSE, 0);
  gtk_drag_dest_set (GTK_WIDGET(menu_button), GTK_DEST_DEFAULT_ALL, drop_targets, 
         G_N_ELEMENTS (drop_targets), GDK_ACTION_COPY | GDK_ACTION_MOVE);
  gtk_tool_item_set_is_important (GTK_TOOL_ITEM (menu_button), TRUE);
  g_signal_connect (G_OBJECT (menu_button), "clicked",
        G_CALLBACK (gdict_lookup_cb), gdict);
  g_signal_connect (G_OBJECT (button), "drag_data_received",
        G_CALLBACK (gdict_lookup_button_drag_cb), gdict);

  gnome_word_entry = gnome_entry_new ("Wordentry");
  gnome_entry_set_max_saved (GNOME_ENTRY (gnome_word_entry), 10);
  gtk_box_pack_start (GTK_BOX (hbox), gnome_word_entry, TRUE, TRUE, 0);
  word_entry = gnome_entry_gtk_entry(GNOME_ENTRY (gnome_word_entry));
  g_signal_connect (G_OBJECT(word_entry), "activate",
        G_CALLBACK (gdict_lookup_cb), gdict);
  gtk_label_set_mnemonic_widget (GTK_LABEL (word_label), word_entry);
  GTK_WIDGET_SET_FLAGS (word_entry, GTK_CAN_FOCUS);
  gtk_widget_grab_focus (word_entry);
  if (GTK_IS_ACCESSIBLE (gtk_widget_get_accessible (gnome_word_entry))) {
    gail_loaded = TRUE;
    gdict_add_atk_namedesc (GTK_WIDGET(word_label), _("Word"), _("Word"));
    gdict_add_atk_namedesc (gnome_word_entry, _("Word Entry"),
          _("Enter a Word or select one from the list below"));
    gdict_add_atk_namedesc (word_entry, _("Word Entry"), NULL);
    gdict_add_atk_relation (gnome_word_entry, GTK_WIDGET(word_label),
          ATK_RELATION_LABELLED_BY);
    gdict_add_atk_namedesc (GTK_WIDGET(menu_button), NULL, _("Look Up for a Word"));
  }

  paned = gtk_hpaned_new ();
  gtk_box_pack_start (GTK_BOX (main_area), paned, TRUE, TRUE, 0);

  speller = GDICT_SPELLER (gdict_speller_new (gdict));
  gtk_paned_pack1 (GTK_PANED (paned), GTK_WIDGET(speller), FALSE, FALSE);

  scrolled = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
          GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled),
               GTK_SHADOW_ETCHED_IN);
  gdict->defbox = GDICT_DEFBOX (gdict_defbox_new (gdict));
  gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET (gdict->defbox));
  gtk_paned_pack2 (GTK_PANED (paned), scrolled, TRUE, TRUE);

  gdict->statusbar = gtk_statusbar_new ();
  gtk_box_pack_start (GTK_BOX (container), gdict->statusbar, FALSE, FALSE, 0);

  gdict->pref = gdict_pref_new (gdict);
  speller->strat = gdict->pref->dfl_strat;

  g_signal_connect (G_OBJECT (window), "delete_event",
        G_CALLBACK (gdict_window_close), gdict);

  gdict->speller = speller;
  gdict->paned = paned;
  gdict->ui = ui;
  gdict->word_entry = word_entry;
  gdict->gnome_word_entry = gnome_word_entry;
  gdict->find_text = NULL;
  gdict->find_dialog = NULL;
  gdict->pref_dialog = NULL;
  gdict->search_from_beginning = TRUE;

  gdict_menus_set_sensitive (gdict, FALSE);
  gdict_menu_item_set_sensitive (gdict, "/MenuBar/EditMenu/FindNext", FALSE);

  gtk_widget_show_all (container);
  gtk_widget_hide (GTK_WIDGET(speller));
  return GTK_WIDGET(gdict);
}

static void
gdict_window_finalize (GObject *object)
{
  GDictWindow *window = (GDictWindow *) object;
  parent_class->finalize (object);
}

static void
gdict_window_class_init (GDictWindowClass *klass)
{
  GObjectClass *object_class = (GObjectClass *) klass;

  object_class->finalize = gdict_window_finalize;
  parent_class = g_type_class_peek_parent (klass);
}

GType
gdict_window_get_type (void)
{
  static GType object_type = 0;
  
  if (!object_type) {
    static const GTypeInfo object_info = {
      sizeof (GDictWindowClass),
      NULL,   /* base_init */
      NULL,   /* base_finalize */
      (GClassInitFunc) gdict_window_class_init,
      NULL,   /* class_finalize */
      NULL,   /* class_data */
      sizeof (GDictWindow),
      0,              /* n_preallocs */
      (GInstanceInitFunc) gdict_window_init
    };

    object_type = g_type_register_static (GTK_TYPE_WINDOW, "GDictWindow", &object_info, 0);
  }

  return object_type;
}
