#include <bluetooth/bluetooth.h>
#include <bluetooth/rfcomm.h>
#include <openobex/obex.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <assert.h>
#include <errno.h>

#ifndef OBEX_CMD_ACTION
# define OBEX_CMD_ACTION      0x06
#endif /* OBEX_CMD_ACTION */


static void get_server(obex_t *handle, obex_object_t *object)
{
	obex_headerdata_t hv;
	uint8_t hi;
	uint32_t hlen;

	char *name = NULL;

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen))	{
		if(hi == OBEX_HDR_NAME && hlen > 0) {
			name = malloc(hlen);
			assert (name);
			OBEX_UnicodeToChar((uint8_t *) name, hv.bs, hlen);
			break;
		}
	}

	uint8_t *buf;
	int size;
	if (name && strcmp(name, "cde") == 0) {
		size = 10000;
		buf  = malloc(size);
		memset(buf, 0, size);
	} else {
		buf = (uint8_t *) "abc";
		size = sizeof(buf);
	}

	OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);

	hv.bq4 = size;
	OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, hv, sizeof(uint32_t), 0);

	char *tstr;
	int   tstr_len;
	if (name && strcmp(name, "bcd") == 0) {
		tstr = "20060620T154800Z";
		tstr_len = strlen(tstr) + 1;
	} else if (name && strcmp(name, "xyz") == 0) {
		tstr = "abc";
		tstr_len = strlen(tstr);
	} else {
		tstr = "20060620T154800Z";
		tstr_len = strlen(tstr);
	}
        hv.bs = (unsigned char *)tstr;
	OBEX_ObjectAddHeader(handle, object, OBEX_HDR_TIME, hv, tstr_len, 0);

	hv.bs = buf;
	OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, hv, size, 0);

	free(name);
	return;
}

static void connect_server(obex_t *handle, obex_object_t *object)
{
	obex_headerdata_t hv;
	uint8_t hi;
	uint32_t hlen;

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen))
		;

	//add a Conection ID
	hv.bq4 = 123;
	OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION, hv, sizeof(uint32_t), 0);

	OBEX_ObjectSetRsp(object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
}

static void put_server(obex_t *handle, obex_object_t *object)
{
	obex_headerdata_t hv;
	uint8_t hi;
	uint32_t hlen;

	const uint8_t *body = NULL;
	int body_len = 0;
	char *name = NULL;
	char *namebuf = NULL;

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen))	{
		switch(hi)	{
		case OBEX_HDR_BODY:
			body = hv.bs;
			body_len = hlen;
			break;
		case OBEX_HDR_NAME:
			if( (namebuf = malloc(hlen / 2)))	{
				OBEX_UnicodeToChar((uint8_t *) namebuf, hv.bs, hlen);
				name = namebuf;
			}
			break;
		default:
			;
		}
	}
	if(!body)	{
		printf("Got a PUT without a body (delete)\n");
		OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
		goto out;
	}
	if(!name)	{
		name = "OBEX_PUT_Unknown_object";
		printf("Got a PUT without a name. Setting name to %s\n", name);

	}
	if (strcmp(name, "bde") == 0) {
		printf("Forbidden\n");
		OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
	}
	else {
		printf("Success\n");
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
	}

 out:
	free(namebuf);
}

static void server_request(obex_t *handle, obex_object_t *object, int event, int cmd)
{
	switch(cmd) {
	case OBEX_CMD_CONNECT:
		connect_server(handle, object);
		break;
	case OBEX_CMD_DISCONNECT:
		OBEX_ObjectSetRsp(object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
		break;
	case OBEX_CMD_GET:
		get_server(handle, object);
		break;
	case OBEX_CMD_PUT:
		put_server(handle, object);
		break;
	case OBEX_CMD_SETPATH:
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
		break;
	case OBEX_CMD_ACTION:
		OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
		break;
	default:
		fprintf(stderr, "unknown command: %d\n", cmd);
		assert(0);
		break;
	}
	return;
}


static void server_done(obex_t *handle, obex_object_t *object, int obex_cmd, int obex_rsp)
{
	if (obex_cmd == OBEX_CMD_DISCONNECT)
		OBEX_TransportDisconnect(handle);
}

static void obex_event(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	switch (event)	{
	case OBEX_EV_REQDONE:
		printf("Request Done\n");
		assert (mode != OBEX_CLIENT);
		server_done(handle, object, obex_cmd, obex_rsp);
		break;
	case OBEX_EV_REQHINT:
		printf("Request Hint\n");
		OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
		break;

	case OBEX_EV_REQ:
		printf("Request\n");
		server_request(handle, object, event, obex_cmd);
		break;

	case OBEX_EV_PROGRESS:
		printf("Made some progress...\n");
		break;

	case OBEX_EV_REQCHECK:
		printf("Request Check\n");
		break;

	case OBEX_EV_LINKERR:
		fprintf(stderr, "link error!\n");
		assert(0);
		break;

	default:
		fprintf(stderr, "Unknown event %02x!\n", event);
		assert(0);
		break;
	}
}

static void server_do(obex_t *handle)
{
        int ret;

	for(;;) {
	        ret = OBEX_HandleInput(handle, 1);
		if(ret < 0)
			break;
	}
}

static void wait_port(int channel) {

	int r;
	int s;
	int err;
	do {
		s = socket(AF_BLUETOOTH, SOCK_STREAM, BTPROTO_RFCOMM);
		assert(s != -1);

		struct sockaddr_rc loc_addr;
		memset(&loc_addr, 0, sizeof(loc_addr));
		loc_addr.rc_family = AF_BLUETOOTH;
		loc_addr.rc_bdaddr = *BDADDR_ANY;
		loc_addr.rc_channel = channel;

		r = bind(s, (struct sockaddr *)&loc_addr, sizeof(loc_addr));
		err = errno;
		if (r != 0) {
			perror("bind: ");
			fprintf(stderr, "errno = %d\n", err);
			int a = close(s);
			assert(a == 0);
			usleep(10);
		}
	} while(r != 0 && err == EADDRINUSE);

	assert(r == 0 && "bind failed");
	r = close(s);
	assert(r == 0);
}

int main(void) {
	int channel = 4;

	wait_port(channel);

	obex_t *handle = OBEX_Init(OBEX_TRANS_BLUETOOTH, obex_event, 0);
	assert (handle != NULL);

	int r = BtOBEX_ServerRegister(handle, BDADDR_ANY, channel);
	if (r < 0) {
	  perror("ServerRegister: ");
	}
	assert (r >= 0);

	printf("===SYNC===\n");
	fflush(stdout);

	server_do(handle);

	OBEX_Cleanup(handle);
	return 0;
}
