/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gnome-vfs-uri.c - URI handling for the GNOME Virtual File System.

   Copyright (C) 1999 Free Software Foundation

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: Ettore Perazzoli <ettore@gnu.org>

   `split_toplevel_uri()' derived from Midnight Commander code by Norbert
   Warmuth, Miguel de Icaza, Janne Kukonlehto, Dugan Porter, Jakub Jelinek.  */

/* TODO: %xx syntax for providing any character in the URI.  */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#ifdef HAVE_ALLOCA_H
#include <alloca.h>
#endif

#include "gnome-vfs.h"
#include "gnome-vfs-private.h"


#define ALLOCA_SUBSTRING(dest, src, len)		\
        do {						\
	      (dest) = alloca ((len) + 1);		\
	      if ((len) > 0)				\
	              memcpy ((dest), (src), (len));	\
	      dest[(len)] = 0;				\
        } while (0)


/* Extract the hostname and username from the path of length `path_len' pointed
   by `path'.  The path is in the form: [user@]hostname:port/remote-dir, e.g.:

       sunsite.unc.edu/pub/linux
       miguel@sphinx.nuclecu.unam.mx/c/nc
       tsx-11.mit.edu:8192/
       joe@foo.edu:11321/private
       joe:password@foo.se

*/

static gchar *
split_toplevel_uri (const gchar *path, guint path_len,
		    gchar **host_return, gchar **user_return,
		    guint *port_return, gchar **password_return)
{
	const gchar *dir, *colon, *at, *rest;
	const gchar *path_end;
	gchar *retval;

	*host_return = NULL;
	*port_return = 0;
	*user_return = NULL;
	*password_return = NULL;
	retval = NULL;

	if (path_len == 0)
		return retval;

	path_end = path + path_len;
    
	/* Locate path component.  */
	dir = memchr (path, GNOME_VFS_URI_PATH_CHR, path_len);
	if (dir != NULL) {
		retval = g_strndup (dir, path_len - (dir - path));
		if (dir != path)
			at = memchr (path, '@', dir - path);
		else
			at = NULL;
	} else {
		retval = g_strdup (GNOME_VFS_URI_PATH_STR);
		at = strchr (path, '@');
	}

	/* Check for username/password.  */
	if (at != NULL && at != path) {
		const gchar *p;

		p = memchr (path, ':', at - path );
		if (p != NULL && at - p > 1) {
			*password_return = g_strndup (p + 1, at - p - 1);
			if (p != path)
				*user_return = g_strndup (path, p - path);
		} else {
			*user_return = g_strndup (path, at - path);
		}

		if (path_end == at + 1)
			rest = at;
		else
			rest = at + 1;
	} else {
		rest = path;
	}

	/* Check if the host comes with a port spec, if so, chop it.  */
	colon = memchr (rest, ':', dir - rest);
	if (colon != NULL && colon != dir - 1) {
		*host_return = g_strndup (rest, colon - rest);

		if (sscanf (colon + 1, "%d", port_return) == 1) {
			if (*port_return > 0xffff)
				*port_return = 0;
		} else {
			while (1) {
				colon++;
				switch(*colon) {
				case 'C':
					*port_return = 1;
					break;
				case 'r':
					*port_return = 2;
					break;
				case 0:
					goto done;
				}
			}
		}
	} else {
		*host_return = g_strndup (rest, dir - rest);
	}

 done:
	return retval;
}


static void
set_uri_element (GnomeVFSURI *uri,
		 const gchar *text,
		 guint len)
{
	if (text == NULL || len == 0) {
		uri->text = NULL;
		return;
	}

	if (uri->parent == NULL && text[0] == '/' && text[1] == '/') {
		GnomeVFSToplevelURI *toplevel;

		toplevel = (GnomeVFSToplevelURI *) uri;
		uri->text = split_toplevel_uri (text + 2, len - 2,
						&toplevel->host_name,
						&toplevel->user_name,
						&toplevel->host_port,
						&toplevel->password);
	} else {
#if 0
/* This causes problems for URI's that may not be relative, but may still not have a leading / */
		if (text[0] != '/') {
			uri->text = g_malloc (len + 2);
			uri->text[0] = '/';
			memcpy (uri->text + 1, text, len);
			uri->text[len + 1] = 0;
		} else {
#endif
			uri->text = g_strndup (text, len);
#if 0
		}
#endif
	}

	gnome_vfs_canonicalize_pathname (uri->text);
}

/**
 * gnome_vfs_uri_new:
 * @text_uri: A string representing a URI.
 * 
 * Create a new URI from @text_uri.
 * 
 * Return value: The new URI.
 **/
GnomeVFSURI *
gnome_vfs_uri_new (const gchar *text_uri)
{
	GnomeVFSMethod *method;
	GnomeVFSToplevelURI *toplevel;
	GnomeVFSURI *uri, *new_uri;
	const gchar *p;
	gchar *method_string;

	g_return_val_if_fail (text_uri != NULL, NULL);

	if (text_uri[0] == 0)
		return NULL;

	/* FIXME: Correct to look for alpha only?  */
	for (p = text_uri; isalpha (*p); p++)
		;

	if (*p == ':') {
		/* Found toplevel method specification.  */
		ALLOCA_SUBSTRING (method_string, text_uri, p - text_uri);
		method = gnome_vfs_method_get (method_string);
		if (method == NULL)
			return NULL;
		p++;
	} else {
		/* No toplevel method specification.  Use "file" as
                   the default.  */
		method_string = "file";
		method = gnome_vfs_method_get (method_string);
		if (method == NULL)
			return NULL;
		p = text_uri;
	}

	/* The toplevel URI element is special, as it also contains host/user
           information.  */
	toplevel = g_new (GnomeVFSToplevelURI, 1);
	toplevel->host_name = NULL;
	toplevel->host_port = 0;
	toplevel->user_name = NULL;
	toplevel->password = NULL;

	new_uri = (GnomeVFSURI *) toplevel;
	new_uri->parent = NULL;

	while (1) {
		GnomeVFSMethod *new_method;
		const gchar *p1, *p2;
		gchar *new_method_string;

		new_uri->method = method;
		new_uri->method_string = g_strdup (method_string);
		new_uri->ref_count = 1;

		p1 = strchr (p, GNOME_VFS_URI_MAGIC_CHR);
		if (p1 == NULL) {
			set_uri_element (new_uri, p, strlen (p));
			uri = new_uri;
			break;
		}

		set_uri_element (new_uri, p, p1 - p);

		p2 = p1 + 1;
		if (*p2 == 0) {
			gnome_vfs_uri_unref (new_uri);
			return NULL;
		}

		while (*p2 != 0 && *p2 != GNOME_VFS_URI_PATH_CHR && *p2 != GNOME_VFS_URI_MAGIC_CHR)
			p2++;

		ALLOCA_SUBSTRING (new_method_string, p1 + 1, p2 - p1 - 1);
		new_method = gnome_vfs_method_get (new_method_string);

		if (new_method == NULL) {
			/* FIXME: Better error handling/reporting?  */
			gnome_vfs_uri_unref (new_uri);
			return NULL;
		}

		p = p2;

		method = new_method;
		method_string = new_method_string;
		uri = new_uri;

		new_uri = g_new (GnomeVFSURI, 1);
		new_uri->parent = uri;
	}

	return uri;
}


/* Destroy an URI element, but not its parent.  */
static void
destroy_element (GnomeVFSURI *uri)
{
	g_free (uri->text);
	g_free (uri->method_string);

	if (uri->parent == NULL) {
		GnomeVFSToplevelURI *toplevel;

		toplevel = (GnomeVFSToplevelURI *) uri;
		g_free (toplevel->host_name);
		g_free (toplevel->user_name);
		g_free (toplevel->password);
	}

	g_free (uri);
}

/**
 * gnome_vfs_uri_ref:
 * @uri: A GnomeVFSURI.
 * 
 * Increment @uri's reference count.
 * 
 * Return value: @uri.
 **/
GnomeVFSURI *
gnome_vfs_uri_ref (GnomeVFSURI *uri)
{
	GnomeVFSURI *p;

	g_return_val_if_fail (uri != NULL, NULL);

	for (p = uri; p != NULL; p = p->parent)
		p->ref_count++;

	return uri;
}

/**
 * gnome_vfs_uri_unref:
 * @uri: A GnomeVFSURI.
 * 
 * Decrement @uri's reference count.  If the reference count reaches zero,
 * @uri is destroyed.
 **/
void
gnome_vfs_uri_unref (GnomeVFSURI *uri)
{
	GnomeVFSURI *p, *parent;

	g_return_if_fail (uri != NULL);
	g_return_if_fail (uri->ref_count > 0);

	for (p = uri; p != NULL; p = parent) {
		parent = p->parent;
		g_assert (p->ref_count > 0);
		p->ref_count--;
		if (p->ref_count == 0)
			destroy_element (p);
	}
}


/**
 * gnome_vfs_uri_dup:
 * @uri: A GnomeVFSURI.
 * 
 * Duplicate @uri.
 * 
 * Return value: A pointer to a new URI that is exactly the same as @uri.
 **/
GnomeVFSURI *
gnome_vfs_uri_dup (const GnomeVFSURI *uri)
{
	const GnomeVFSURI *p;
	GnomeVFSURI *new, *child;

	if (uri == NULL)
		return NULL;

	new = NULL;
	child = NULL;
	for (p = uri; p != NULL; p = p->parent) {
		GnomeVFSURI *new_element;

		if (p->parent == NULL) {
			GnomeVFSToplevelURI *toplevel;
			GnomeVFSToplevelURI *new_toplevel;

			toplevel = (GnomeVFSToplevelURI *) p;
			new_toplevel = g_new (GnomeVFSToplevelURI, 1);

			new_toplevel->host_name = g_strdup (toplevel->host_name);
			new_toplevel->host_port = toplevel->host_port;
			new_toplevel->user_name = g_strdup (toplevel->user_name);
			new_toplevel->password = g_strdup (toplevel->password);

			new_element = (GnomeVFSURI *) new_toplevel;
		} else {
			new_element = g_new (GnomeVFSURI, 1);
		}

		new_element->ref_count = 1;
		new_element->text = g_strdup (p->text);
		new_element->method_string = g_strdup (p->method_string);
		new_element->method = p->method;
		new_element->parent = NULL;

		if (child != NULL)
			child->parent = new_element;
		else
			new = new_element;
			
		child = new_element;
	}

	return new;
}


/**
 * gnome_vfs_uri_append_path:
 * @uri: A GnomeVFSURI.
 * @path: A path component.
 * 
 * Create a new URI obtained by appending @path to @uri.  This will take care
 * of adding an appropriate directory separator between the end of @uri and
 * the start of @path if necessary.
 * 
 * Return value: The new URI obtained by combining @uri and @path.
 **/
/* FIXME this must be implemented in a much smarter way.  The most important
   issue is that a `#' in `path' will break things rather badly.  */
GnomeVFSURI *
gnome_vfs_uri_append_path (const GnomeVFSURI *uri,
			   const gchar *path)
{
	gchar *uri_string;
	GnomeVFSURI *new;
	gchar *new_string;
	guint len;

	g_return_val_if_fail (uri != NULL, NULL);
	g_return_val_if_fail (path != NULL, NULL);

	/* FIXME this is just a reminder.  */
	if (strchr (path, '#') != NULL)
		g_warning ("gnome_vfs_uri_append_path() is broken with names containing `#'.");

	uri_string = gnome_vfs_uri_to_string (uri, GNOME_VFS_URI_HIDE_NONE);
	len = strlen (uri_string);
	if (len == 0) {
		g_free (uri_string);
		return gnome_vfs_uri_new (path);
	}

	len--;
	while (uri_string[len] == GNOME_VFS_URI_PATH_CHR && len > 0)
		len--;
	uri_string[len + 1] = '\0';

	while (*path == GNOME_VFS_URI_PATH_CHR)
		path++;

	new_string = g_strconcat (uri_string, GNOME_VFS_URI_PATH_STR, path, NULL);
	new = gnome_vfs_uri_new (new_string);

	g_free (new_string);
	g_free (uri_string);

	return new;
}


/**
 * gnome_vfs_uri_to_string:
 * @uri: A GnomeVFSURI.
 * @hide_options: Bitmask specifying what URI elements (e.g. password,
 * user name etc.) should not be represented in the returned string.
 * 
 * Translate @uri into a printable string.  The string will not contain the
 * URI elements specified by @hide_options.
 * 
 * Return value: A malloced printable string representing @uri.
 **/
gchar *
gnome_vfs_uri_to_string (const GnomeVFSURI *uri,
			 GnomeVFSURIHideOptions hide_options)
{
	gchar *toplevel_info;
	const GnomeVFSURI *u;
	const GnomeVFSToplevelURI *toplevel;
	guint size, len;
	gchar *s, *p;

	/* FIXME/TODO: Special characters such as `#' in the URI components
           should be replaced.  */

	g_return_val_if_fail (uri != NULL, NULL);

	size = 0;
	u = uri;
	while (1) {
		if (u->text != NULL)
			size += strlen (u->text);

		if (u->parent != NULL
		    || ! (hide_options
			  & GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD)) {
			if (u->method_string != NULL)
				size += strlen (u->method_string);
			size += 1;	/* '#' or ':' */
		}

		if (u->parent == NULL)
			break;
		u = u->parent;
	}

	toplevel = (GnomeVFSToplevelURI *) u;
	if (toplevel->host_name != NULL) {
		gchar *concat_list[9];
		gchar *port_string;
		guint count;

		count = 0;
		if(!(hide_options & GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD))
			concat_list[count++] = "//";

		if (toplevel->user_name != NULL
		    && ! (hide_options & GNOME_VFS_URI_HIDE_USER_NAME)) {
			concat_list[count++] = toplevel->user_name;
			if (toplevel->password != NULL
			    && ! (hide_options & GNOME_VFS_URI_HIDE_PASSWORD)) {
				concat_list[count++] = ":";
				concat_list[count++] = toplevel->password;
			}
			concat_list[count++] = "@";
		}

		if(!(hide_options & GNOME_VFS_URI_HIDE_HOST_PORT))
			concat_list[count++] = toplevel->host_name;

		if (toplevel->host_port != 0 && !(hide_options & GNOME_VFS_URI_HIDE_HOST_PORT)) {
			port_string = g_strdup_printf ("%d",
						       toplevel->host_port);
			concat_list[count++] = ":";
			concat_list[count++] = port_string;
		} else {
			port_string = NULL;
		}

		concat_list[count] = NULL;

		toplevel_info = g_strjoinv (NULL, concat_list);
		size += strlen (toplevel_info);

		g_free (port_string);
	} else {
		toplevel_info = NULL;
	}

	s = g_malloc (size + 1);
	p = s + size;
	*p = 0;

	u = uri;
	do {
		if (u->text != NULL) {
			len = strlen (u->text);
			p -= len;
			memcpy (p, u->text, len);
		}

		if (u->parent == NULL && toplevel_info != NULL) {
			len = strlen (toplevel_info);
			p -= len;
			memcpy (p, toplevel_info, len);
		}

		if (u->method_string != NULL
		    && (u->parent != NULL
			|| ! (hide_options
			      & GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD))) {
			if (u->parent == NULL)
				*(--p) = ':';
			len = strlen (u->method_string);
			p -= len;
			memcpy (p, u->method_string, len);
			if (u->parent != NULL)
				*(--p) = '#';
		}

		u = u->parent;
	} while (u != NULL);

	g_free (toplevel_info);

	return s;
}


/**
 * gnome_vfs_uri_is_local:
 * @uri: A GnomeVFSURI.
 * 
 * Check if @uri is a local (native) file system.
 * 
 * Return value: %FALSE if @uri is not a local file system, %TRUE otherwise.
 **/
gboolean
gnome_vfs_uri_is_local (const GnomeVFSURI *uri)
{
	g_return_val_if_fail (uri != NULL, FALSE);

	return uri->method->is_local (uri->method, uri);
}

/* FIXME Maybe we have to make more definite decisions on what URLs have to
   look like here.  */

/**
 * gnome_vfs_uri_has_parent:
 * @uri: A GnomeVFSURI.
 * 
 * Check if URI has a parent or not.
 * 
 * Return value: %TRUE if @uri has a parent, %FALSE otherwise.
 **/
gboolean
gnome_vfs_uri_has_parent (const GnomeVFSURI *uri)
{
	g_return_val_if_fail (uri != NULL, FALSE);

	if (uri->parent != NULL)
		return TRUE;

	if (uri->text == NULL)
		return FALSE;

	if (strcmp (uri->text, GNOME_VFS_URI_PATH_STR) == 0)
		return FALSE;

	return TRUE;
}

/**
 * gnome_vfs_uri_get_parent:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve @uri's parent URI.
 * 
 * Return value: A pointer to @uri's parent URI.
 **/
GnomeVFSURI *
gnome_vfs_uri_get_parent (const GnomeVFSURI *uri)
{
	g_return_val_if_fail (uri != NULL, NULL);

	if (uri->text != NULL && uri->text[0] != 0) {
		gchar *p;
		guint len;

		len = strlen (uri->text);
		p = uri->text + len - 1;

		/* Skip trailing slash.  */
		if (*p == GNOME_VFS_URI_PATH_CHR && p != uri->text)
			p--;

		/* Search backwards for next slash.  URIs are normalized, so
                   the next slash we find cannot be a trailing one.  */
		while (p != uri->text && *p != GNOME_VFS_URI_PATH_CHR)
			p--;

		if (p[1] != '\0') {
			GnomeVFSURI *new;

			/* The following is not extremely efficient, but quite
			   safe in case we change the internals.  */

			new = gnome_vfs_uri_dup (uri);
			g_free (new->text);

			if (p == uri->text) {
				new->text = g_strdup (GNOME_VFS_URI_PATH_STR);
			} else {
				new->text = g_malloc (p - uri->text + 2);
				memcpy (new->text, uri->text, p - uri->text);
				new->text[p - uri->text] = GNOME_VFS_URI_PATH_CHR;
				new->text[p - uri->text + 1] = '\0';
			}

			return new;
		}
	}

	return gnome_vfs_uri_dup (uri->parent);
}


/**
 * gnome_vfs_uri_get_toplevel:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve the toplevel URI in @uri.
 * 
 * Return value: A pointer to the toplevel URI object.
 **/
GnomeVFSToplevelURI *
gnome_vfs_uri_get_toplevel (const GnomeVFSURI *uri)
{
	const GnomeVFSURI *p;

	g_return_val_if_fail (uri != NULL, NULL);

	for (p = uri; p->parent != NULL; p = p->parent)
		;

	return (GnomeVFSToplevelURI *) p;
}


/**
 * gnome_vfs_uri_get_host_name:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve the host name for @uri.
 * 
 * Return value: A string representing the host name.
 **/
const gchar *
gnome_vfs_uri_get_host_name (const GnomeVFSURI *uri)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_val_if_fail (uri != NULL, NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);
	return toplevel->host_name;
}

/**
 * gnome_vfs_uri_get_host_port:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve the host port number in @uri.
 * 
 * Return value: The host port number used by @uri.  If the value is zero, the
 * default port value for the specified toplevel access method is used.
 **/
guint
gnome_vfs_uri_get_host_port (const GnomeVFSURI *uri)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_val_if_fail (uri != NULL, 0);

	toplevel = gnome_vfs_uri_get_toplevel (uri);
	return toplevel->host_port;
}

/**
 * gnome_vfs_uri_get_user_name:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve the user name in @uri.
 * 
 * Return value: A string representing the user name in @uri.
 **/
const gchar *
gnome_vfs_uri_get_user_name (const GnomeVFSURI *uri)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_val_if_fail (uri != NULL, NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);
	return toplevel->user_name;
}

/**
 * gnome_vfs_uri_get_password:
 * @uri: A GnomeVFSURI.
 * 
 * Retrieve the password for @uri.
 * 
 * Return value: The password for @uri.
 **/
const gchar *
gnome_vfs_uri_get_password (const GnomeVFSURI *uri)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_val_if_fail (uri != NULL, NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);
	return toplevel->password;
}


/**
 * gnome_vfs_uri_set_host_name:
 * @uri: A GnomeVFSURI.
 * @host_name: A string representing a host name.
 * 
 * Set @host_name as the host name accessed by @uri.
 **/
void
gnome_vfs_uri_set_host_name (GnomeVFSURI *uri,
			     const gchar *host_name)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_if_fail (uri != NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);

	if (toplevel->host_name != NULL)
		g_free (toplevel->host_name);
	toplevel->host_name = g_strdup (host_name);
}

/**
 * gnome_vfs_uri_set_host_port:
 * @uri: A GnomeVFSURI.
 * @host_port: A TCP/IP port number.
 * 
 * Set the host port number in @uri.  If @host_port is zero, the default port
 * for @uri's toplevel access method is used.
 **/
void
gnome_vfs_uri_set_host_port (GnomeVFSURI *uri,
			     guint host_port)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_if_fail (uri != NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);

	toplevel->host_port = host_port;
}

/**
 * gnome_vfs_uri_set_user_name:
 * @uri: A GnomeVFSURI.
 * @user_name: A string representing a user name on the host accessed by @uri.
 * 
 * Set @user_name as the user name for @uri.
 **/
void
gnome_vfs_uri_set_user_name (GnomeVFSURI *uri,
			     const gchar *user_name)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_if_fail (uri != NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);

	if (toplevel->user_name != NULL)
		g_free (toplevel->user_name);
	toplevel->host_name = g_strdup (user_name);
}

/**
 * gnome_vfs_uri_set_password:
 * @uri: A GnomeVFSURI.
 * @password: A password string.
 * 
 * Set @password as the password for @uri.
 **/
void
gnome_vfs_uri_set_password (GnomeVFSURI *uri,
			    const gchar *password)
{
	GnomeVFSToplevelURI *toplevel;

	g_return_if_fail (uri != NULL);

	toplevel = gnome_vfs_uri_get_toplevel (uri);

	if (toplevel->password != NULL)
		g_free (toplevel->password);
	toplevel->host_name = g_strdup (password);
}


static gint
my_streq (const gchar *a,
	  const gchar *b)
{
	if (a == NULL) {
		if (b == NULL)
			return TRUE;
		else
			return FALSE;
	} else if (b == NULL) {
		return FALSE;
	}

	return strcmp (a, b) == 0;
}

static gboolean
compare_elements (const GnomeVFSURI *a,
		  const GnomeVFSURI *b)
{
	if (! my_streq (a->text, b->text)
	    || ! my_streq (a->method_string, b->method_string))
		return FALSE;

	/* The following should not happen, but we make sure anyway.  */
	if (a->method != b->method)
		return FALSE;

	return TRUE;
}

/**
 * gnome_vfs_uri_equal:
 * @a: A GnomeVFSURI.
 * @b: A GnomeVFSURI.
 * 
 * Compare @a and @b.
 * 
 * Return value: %TRUE if @a and @b are equal, %FALSE otherwise.
 **/
gboolean
gnome_vfs_uri_equal (const GnomeVFSURI *a,
		     const GnomeVFSURI *b)
{
	const GnomeVFSToplevelURI *toplevel_a;
	const GnomeVFSToplevelURI *toplevel_b;

	g_return_val_if_fail (a != NULL, FALSE);
	g_return_val_if_fail (b != NULL, FALSE);

	/* First check non-toplevel elements.  */
	while (a->parent != NULL && b->parent != NULL)
		if (! compare_elements (a, b))
			return FALSE;

	/* Now we should be at toplevel for both.  */
	if (a->parent != NULL || b->parent != NULL)
		return FALSE;

	if (! compare_elements (a, b))
		return FALSE;

	toplevel_a = (GnomeVFSToplevelURI *) a;
	toplevel_b = (GnomeVFSToplevelURI *) b;

	/* Finally, compare the extra toplevel members.  */

	if (toplevel_a->host_port != toplevel_b->host_port
	    || ! my_streq (toplevel_a->host_name, toplevel_b->host_name)
	    || ! my_streq (toplevel_a->user_name, toplevel_b->user_name)
	    || ! my_streq (toplevel_a->password, toplevel_b->password))
		return FALSE;

	return TRUE;
}


/**
 * gnome_vfs_uri_get_basename:
 * @uri: A GnomeVFSURI
 * 
 * Retrieve base file name for @uri.
 * 
 * Return value: A pointer to the base file name in @uri.  Notice that the
 * pointer points to the name store in @uri, so the name returned must not
 * be modified nor freed.
 **/
const gchar *
gnome_vfs_uri_get_basename (const GnomeVFSURI *uri)
{
	gchar *p;

	g_return_val_if_fail (uri != NULL, NULL);

	p = strrchr (uri->text, GNOME_VFS_URI_PATH_CHR);
	if (p == NULL)
		return NULL;

	p++;
	if (*p == '\0')
		return NULL;

	return p;
}

/**
 * gnome_vfs_uri_extract_dirname:
 * @uri: A GnomeVFSURI
 * 
 * Extract the name of the directory in which the file pointed to by @uri is
 * stored as a newly allocated string.  The string will end with a
 * GNOME_VFS_URI_PATH_CHR.
 * 
 * Return value: A pointer to the newly allocated string representing the
 * parent directory.
 **/
gchar *
gnome_vfs_uri_extract_dirname (const GnomeVFSURI *uri)
{
	const gchar *base;

	g_return_val_if_fail (uri != NULL, NULL);

	base = gnome_vfs_uri_get_basename (uri);
	if (base == NULL || base == uri->text)
		return g_strdup (GNOME_VFS_URI_PATH_STR);

	return g_strndup (uri->text, base - uri->text);
}

/**
 * gnome_vfs_uri_extract_short_name:
 * @uri: A GnomeVFSURI
 * 
 * Retrieve base file name for @uri, ignoring any trailing path separators.
 * This matches the XPG definition of basename, but not g_basename. This is
 * often useful when you want the name of something that's pointed to by a
 * uri, and don't care whether the uri has a directory or file form.
 * If @uri points to the root, returns GNOME_VFS_URI_PATH_STR.
 * 
 * Return value: A pointer to the newly allocated string representing the
 * short form of the name.
 **/
gchar *
gnome_vfs_uri_extract_short_name (const GnomeVFSURI *uri)
{
	const gchar *p, *short_name_start, *short_name_end;

	g_return_val_if_fail (uri != NULL, NULL);
	g_return_val_if_fail (uri->text != NULL, NULL);
	g_return_val_if_fail (uri->text[0] != '\0', NULL);

	/* Search for the last run of non-'/' characters. */
	p = uri->text;
	short_name_start = NULL;
	short_name_end = p;
	do {
		if (*p == '\0' || *p == GNOME_VFS_URI_PATH_CHR) {
			/* While we are in a run of non-separators, short_name_end is NULL. */
			if (short_name_end == NULL)
				short_name_end = p;
		} else {
			/* While we are in a run of separators, short_name_end is not NULL. */
			if (short_name_end != NULL) {
				short_name_start = p;
				short_name_end = NULL;
			}
		}
	} while (*p++ != '\0');
	g_assert (short_name_end != NULL);
	
	/* If we never found a short name, that means that the string is all
	   directory separators. Since it can't be an empty string, that means
	   it points to the root, so "/" is a good result.
	*/
	if (short_name_start == NULL)
		return g_strdup (GNOME_VFS_URI_PATH_STR);

	/* Return a copy of the short name. */
	return g_strndup (short_name_start, short_name_end - short_name_start);
}


/* The following functions are useful for creating URI hash tables.  */

gint
gnome_vfs_uri_hequal (gconstpointer a,
		      gconstpointer b)
{
	return gnome_vfs_uri_equal (a, b);
}

guint
gnome_vfs_uri_hash (gconstpointer p)
{
	const GnomeVFSURI *uri;
	const GnomeVFSURI *uri_p;
	guint hash_value;

#define HASH_STRING(value, string)		\
	if ((string) != NULL)			\
		(value) ^= g_str_hash (string);

#define HASH_NUMBER(value, number)		\
	(value) ^= number;

	uri = (const GnomeVFSURI *) p;
	hash_value = 0;

	for (uri_p = uri; uri_p != NULL; uri_p = uri_p->parent) {
		HASH_STRING (hash_value, uri_p->text);
		HASH_STRING (hash_value, uri_p->method_string);

		if (uri_p->parent != NULL) {
			const GnomeVFSToplevelURI *toplevel;

			toplevel = (const GnomeVFSToplevelURI *) uri_p;

			HASH_STRING (hash_value, toplevel->host_name);
			HASH_NUMBER (hash_value, toplevel->host_port);
			HASH_STRING (hash_value, toplevel->user_name);
			HASH_STRING (hash_value, toplevel->password);
		}
	}

	return hash_value;

#undef HASH_STRING
#undef HASH_NUMBER
}
