/* GnomeICU
 * Copyright (C) 1998-2002 Jeremy Wise
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * Functions for SNAC family 0x13, server side contacts list management
 * first created by Patrick Sung (2002)
 */


#include "common.h"
#include "auth.h"
#include "cl_migrate.h"
#include "gnomecfg.h"
#include "gnomeicu.h"
#include "groups.h"
#include "response.h"
#include "showlist.h"
#include "util.h"
#include "v7recv.h"
#include "v7send.h"
#include "v7snac13.h"

#define S13_REQID(id) ( ((++s13_reqcnt) << 16) | (id) )
#define SNAC13_SEND(d,l,type,reqid) \
  snac_send (mainconnection, (d), (l), FAMILY_13_SAVED_LIST, \
             (type), NULL, (reqid))

/*** enum and structs ***/

typedef enum {
  SNAC13_NONE = 0,
  SNAC13_ADD_USER_NO_AUTH = 1 << 0,
  SNAC13_ADD_USER_AUTH = 1 << 1,
  SNAC13_REMOVE_USER = 1 << 2,
  SNAC13_AUTH_REQUEST = 1 << 3,
} Snac13State;

/* structure holding infos of the current state (for snac family 0x13 only)
 * used for add/remove contacts which the server sends back result of request
 *
 * uins are added without asking for auth for the first time
 * if fails gnomeicu will try to add again but with auth req
 * This struct holds the data that is needed in case the first try fails.
 */
struct _snac13_state {
  Snac13State state;

  gboolean valid; /* validity of this cookie, when done using, set it to false */
  UIN_T uin;
  gchar nick[20]; /* FIXME */

  guint gid, uid;
  gboolean with_grant; /* whether send with a auth grant when adding new user */
};

struct _invalid_record {
  gint len;
  gchar *tag;
  guint gid;
  guint uid;
  gint tlvlen;
  gchar *tlvval;
};

/*** variables ***/

gboolean contacts_resync = FALSE; /* whether resync contacts list when startup */

/* snac13 packet request id management */
static guint s13_reqid = 0;
static guint s13_reqcnt = 0;

static struct _snac13_state snac13_state = {
  state: SNAC13_NONE
};

static struct _invalid_record invalid_record;

/*** local functions ***/

static void remove_tlv11_record (void);
static void snac13_construct_nickname (TLVstack *t, UIN_T uin, guint16 gid,
				       guint16 uid, const gchar *nick,
				       gboolean needauth);
static void snac13_construct_id_ingroup (TLVstack *t, WORD gid);
static void snac13_construct_visible (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_invisible (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_ignore (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_status (TLVstack *t, gboolean status);
static void snac13_construct_authreq (TLVstack *t, UIN_T uin, const gchar *msg);

static void snac13_send_add_to_list (const TLVstack *tlvs);
static void snac13_send_update_list (const TLVstack *tlvs);
static void snac13_send_remove_from_list (const TLVstack *tlvs);
static void snac13_check_contacts_list_sanity (void);

/*
 * sends 0x13,0x02 and 0x13,0x04/0x13,0x05
 * request server side contacts list
 */
void
v7_request_contacts_list ()
{
  guchar stamp[6];

#ifdef TRACE_FUNCTION
  g_print("v7_requestcontactslist\n");
#endif

  /* request rights (0x13, 0x02) */
  SNAC13_SEND (NULL, 0, F13_CLIENT_REQ_RIGHTS, F13_CLIENT_REQ_RIGHTS);

  if (contacts_resync) {
    list_time_stamp = 0;
    record_cnt = 0;
  }

  printf ("list_time_stamp = %d\n", list_time_stamp);
  printf ("record_cnt = %d\n", record_cnt);
  /* check pref setting, if server list pref not set, check from server*/
  if (!srvlist_exist) {
    s13_reqid = S13_REQID (F13_CLIENT_QUERY_SAVED_LIST);
    SNAC13_SEND (NULL, 0, F13_CLIENT_QUERY_SAVED_LIST, s13_reqid);
  } else {
    /* send request */
    DW_2_CharsBE (stamp, list_time_stamp);
    Word_2_CharsBE (&stamp[4], record_cnt);

    s13_reqid = S13_REQID (F13_CLIENT_REQ_SAVED_LIST);
    SNAC13_SEND (stamp, 6, F13_CLIENT_REQ_SAVED_LIST, s13_reqid);
  }
} /* end v7_request_contacts_list() */


/* read snac 0x13, 0x06 content
 * replace local contact list with the received one */
void
v7_read_contacts_list (Snac *snac, V7Connection *conn)
{
  static gint total_record = 0; /* total record count in user CL */
  static gboolean first_time = TRUE;
  gchar *here = snac->data;
  gchar *rec_str = NULL;
  gint rec_cnt = 0; /* number of CL record in current snac */
  gint i;
  gint len_str, gid, uid;
  TLV *tlv;
  guint32 timestamp;
  GSList *contact;

  gboolean force_migration = FALSE;


#ifdef TRACE_FUNCTION
  g_print("v7_read_contact_list\n");
#endif

  Current_Status = STATUS_ONLINE;

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  here += 2;

  rec_cnt = *here;
  here++;

  total_record += rec_cnt;

  /* reset all because we are reading them anyway */
  contact = Contacts;
  while (contact && first_time) {
    kontakt->vis_list = FALSE;
    kontakt->invis_list = FALSE;
    kontakt->ignore_list = FALSE;
    contact = contact->next;
  }

  first_time = FALSE;

  for (i=0; i < rec_cnt; i++) {
    /* read the BWS string on each CL record */
    len_str = CharsBE_2_Word (here);
    here += 2;
    if (len_str != 0) {
      rec_str = g_strndup (here, len_str);
      here += len_str;
    }

    /* read group-id and uin-id */
    gid = CharsBE_2_Word (here);
    here += 2;
    uid = CharsBE_2_Word (here);
    here += 2;


    /* read the TLV, find out the type */
    tlv = new_tlv (here);

    switch (tlv->type) {
      TLV *tlv2;
      gint tlv_read; /* bytes read inside current tlv */

    case 0: /* contains nickname, for user */
      /* create a user (if not exists) without specifying nickname */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }

      tlv_read = 0;
      while (tlv_read < tlv->len) {
        tlv2 = new_tlv (tlv->value + tlv_read);

        switch (tlv2->type) {
        case 0x131: /* nickname */
          strncpy (kontakt->nick, tlv2->value,
                   ((tlv2->len > 19) ? 19 : tlv2->len));
          kontakt->nick[(tlv2->len > 19) ? 19 : tlv2->len] = 0;
          break;
        case 0x66: /* with auth */
          kontakt->wait_auth = TRUE;
          break;
        default:
          g_warning ("Unknown tlv type (%d) inside type 0", tlv2->type);
          break;
        }

        tlv_read += TLV_HEADER + tlv2->len;
        delete_tlv (tlv2);
      }

      kontakt->uid = uid;
      kontakt->gid = gid;
      kontakt->last_status = STATUS_OFFLINE;
      break;
    case 1: /* contains id's (group or uin) */
      /* group id is not zero, contains uin id */
      if (gid) {
	groups_add (rec_str, gid);
      }

      /* no uin-id in this group, skip */
      if (tlv->len == 0)
	break;

      tlv2 = new_tlv (tlv->value);
      if (tlv2->type != 0xC8) {
	g_warning ("expecting TLV(C8): got TLV(0x%x).",tlv2->type);
      }
      delete_tlv (tlv2);
      break;
    case 2: /* zero length; vis to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->vislist_uid = uid;
      kontakt->vis_list = TRUE;
      break;
    case 3: /* zero length; inv to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->invlist_uid = uid;
      kontakt->invis_list = TRUE;
      break;
    case 4: /* last online status (invis or not invis) */
      tlv2 = new_tlv (tlv->value);
      if (tlv2->type != 0xCA) {
        g_warning ("expecting TLV(CA) inside TLV(4): got TLV(0x%x).",tlv2->type);
      } else {
        status_uid = uid;
        if (*(tlv2->value) == 0x03) {
          if (preset_status != STATUS_INVISIBLE)
            Current_Status = STATUS_INVISIBLE;
        } else
          Current_Status = preset_status;
      }
      delete_tlv (tlv2);
      break;
    case 9: /* unknown meaning, ignore */
      break;
    case 0xE: /* zero length; ignore to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->ignorelist_uid = uid;
      kontakt->ignore_list = TRUE;
      break;
    case 0x11: /* found when user used lite.icq.com before server list exists */
      if ((snac->reqid & 0xFF) == F13_CLIENT_QUERY_SAVED_LIST) {
        invalid_record.len = len_str;
        invalid_record.tag = g_strndup(rec_str, len_str);
        invalid_record.gid = gid;
        invalid_record.uid = uid;
        invalid_record.tlvlen = tlv->len;
        invalid_record.tlvval = g_strndup(tlv->value, tlv->len);
        force_migration = TRUE;
      }
      break;
    case 0x13: /* CL first import time, not really useful to us, ignore */
      break;
    default:
      g_warning ("Unknow TLV from server contact list: type 0x%x.\n\n"
                 "Please send the following packet dump to the mailing-list "
                 "for debugging, thanks.\n\n",tlv->type);
      packet_print_v7_snac (snac);
      break;
    }

    here += TLV_HEADER + tlv->len;

    g_free (rec_str);
    rec_str = NULL;
    delete_tlv (tlv);
  } /* end for(rec_cnt) */

  /* done reading, time stamp remains, read it */
  timestamp = CharsBE_2_DW (here);

  /* no record found, and previous request is 'query contacts list'
     therefore we migrate */
  if (rec_cnt == 0 && (snac->reqid & 0xFF) == F13_CLIENT_QUERY_SAVED_LIST) {
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    /* normal migration */
    g_idle_add (migration_dialog, (gpointer)FALSE);
    return;
  }

  /* done reading all contacts, need force migrate? */
  if (timestamp != 0 && force_migration) {
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    remove_tlv11_record ();
    /* start the migration but use the existing "General" group in the list */
    g_idle_add (migration_dialog, (gpointer)TRUE);
    return;
  }

  /* cannot detect a case to start the migration, so do the default actions */

  if (timestamp != 0) {
    /* finished all reading */
    /* signal ready to use the contacts list */
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    srvlist_exist = TRUE;
    list_time_stamp = timestamp;
    record_cnt = total_record;
    save_preferences ();
    Save_RC ();
  }

  /* update GUI */
  Show_Quick_Status_lower (UPDATE_RECONSTRUCT, NULL);

  /* check sanity of contacts list after reading it */
  snac13_check_contacts_list_sanity ();
}



/*
 * sned snac 0x13, 0x11
 * begin contacts list change
 */
void
v7_begin_CL_edit (gboolean migrate)
{
  gchar data[4] = "\x00\x01\x00\x00";

  if (migrate)
    SNAC13_SEND (data, 4, F13_CLIENT_START_MOD_LIST, F13_CLIENT_START_MOD_LIST);
  else
    SNAC13_SEND (NULL, 0, F13_CLIENT_START_MOD_LIST, F13_CLIENT_START_MOD_LIST);
}

/*
 * send snac 0x13, 0x12
 * end contacts list change
 */
void
v7_end_CL_edit ()
{
  SNAC13_SEND (NULL, 0, F13_CLIENT_END_MOD_LIST, F13_CLIENT_END_MOD_LIST);
}


/* send the contact list (0x13, 0x08)
 *  force - force sending contact, instead of wait until enough records
 * return:
 *  TRUE - if snac is sent
 *  FALSE - snac is not sent
 */
gboolean
v7_send_contacts_list (TLVstack *t, gboolean force)
{
  static gint rec_cnt = 0;

#ifdef TRACE_FUNCTION
  g_print("v7_send_contacts_list\n");
#endif

  rec_cnt++;

  if (force || rec_cnt >= 0x50) {
    rec_cnt = 0;

    if (t->len > 0)
      /* send the tlv stack */
      snac13_send_add_to_list (t);
    return TRUE;
  }

  return FALSE;
}


/* construct the contacts list */
void
v7_migrate_contacts_list ()
{
  GSList *contact;
  guint id, main_gid = 0;
  gint ignore_cnt = 0;
  gint vis_cnt = 0;
  gint inv_cnt = 0;
  TLVstack *tlvs = NULL;
  gboolean grp_exists;

#ifdef TRACE_FUNCTION
  g_print("v7_migrate_contacts_list\n");
#endif

  main_gid = groups_find_gid_by_name ("General");

  if (!main_gid) {
    main_gid = groups_gen_gid ();
    groups_add ("General", main_gid);
    grp_exists = FALSE;
  } else
    grp_exists = TRUE;

  tlvs = new_tlvstack (NULL, 0);

  contact = Contacts;
  /* add contacts first */
  while (contact) {
    /* skip ignore list */
    if (kontakt->ignore_list) {
      contact = contact->next;
      ignore_cnt++;
      continue;
    }

    /* count visible and invisible user */
    if (kontakt->vis_list) {
      vis_cnt++;
    } else if (kontakt->invis_list) {
      inv_cnt++;
    }

    id = contact_gen_uid ();

    snac13_construct_nickname (tlvs, kontakt->uin, main_gid, id, kontakt->nick,
			       FALSE);

    kontakt->gid = main_gid;
    kontakt->uid = id;

    contact = contact->next;

    if (v7_send_contacts_list (tlvs, FALSE)) {
      free_tlvstack (tlvs);
      tlvs = new_tlvstack (NULL, 0);
    }

  }

  /* send special users */
  contact = Contacts;
  while (contact && (ignore_cnt != 0 || vis_cnt != 0 || inv_cnt != 0)) {
    /* only ignore, vis, invis user */
    if (kontakt->ignore_list) {
      snac13_construct_ignore (tlvs, kontakt->uin, id = contact_gen_uid ());
      kontakt->ignorelist_uid = id;
      ignore_cnt--;
    } else if (kontakt->vis_list) {
      snac13_construct_visible (tlvs, kontakt->uin, id = contact_gen_uid ());
      kontakt->vislist_uid = id;
      vis_cnt--;
    } else if (kontakt->invis_list) {
      snac13_construct_invisible (tlvs, kontakt->uin, id =contact_gen_uid ());
      kontakt->invlist_uid = id;
      inv_cnt--;
    }

    contact = contact->next;

    if (v7_send_contacts_list (tlvs, FALSE)) {
      free_tlvstack (tlvs);
      tlvs = new_tlvstack (NULL, 0);
    }
  }

  v7_send_contacts_list (tlvs, TRUE);  /* force a send */
  free_tlvstack (tlvs);
  tlvs = new_tlvstack (NULL, 0);


  /* send the default group - "General" */
  snac13_construct_id_ingroup (tlvs, main_gid);
  v7_send_contacts_list (tlvs, TRUE);
  free_tlvstack (tlvs);
  tlvs = new_tlvstack (NULL, 0);

  if (!grp_exists) {
    /* send master group */
    snac13_construct_id_ingroup (tlvs, 0);
    v7_send_contacts_list (tlvs, TRUE);
    free_tlvstack (tlvs);
    tlvs = new_tlvstack (NULL, 0);

    /* send current status */
    snac13_construct_status (tlvs, TRUE);
    v7_send_contacts_list (tlvs, TRUE);
    free_tlvstack (tlvs);
    tlvs = new_tlvstack (NULL, 0);
  }

  free_tlvstack (tlvs);

} /* end v7_migrate_contacts_list () */


/* send snac 0x13,0x08
 * add _one_ uin to the server contacts list
 *  - authtlv: TRUE add request authorization TLV(0x66)
 *             FALSE don't add the TLV(0x66)
 * snac13_state.state needs to be in these state:
 *   SNAC13_ADD_USER_NO_AUTH
 *   SNAC13_ADD_USER_AUTH
 * snac13_state.uid will be updated with the new uid
 */
void
v7_addto_contacts_list (UIN_T uin, const gchar *nick,
			guint gid, gboolean needauth)
{
  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print("v7_addto_contacts_list\n");
#endif

  if (snac13_state.state != SNAC13_ADD_USER_NO_AUTH &&
      snac13_state.state != SNAC13_ADD_USER_AUTH)
    return;

  v7_begin_CL_edit (FALSE);

  /* generate a new uid for this new user */
  snac13_state.uid = contact_gen_uid ();

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_nickname (tlvs, uin, gid, snac13_state.uid, nick, needauth);
  snac13_send_add_to_list (tlvs);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x14
 */
void
v7_send_grant_auth (UIN_T uin)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);
  snac13_construct_authreq (tlvs, uin, NULL);

  SNAC13_SEND (tlvs->beg, tlvs->len, F13_CLIENT_AUTH_REQUEST_1,
               F13_CLIENT_AUTH_REQUEST_1);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x18
 */
void
v7_send_auth_message (UIN_T uin, const gchar *authmsg)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_authreq (tlvs, uin, authmsg);
  
  SNAC13_SEND (tlvs->beg, tlvs->len, F13_CLIENT_AUTH_REQUEST_2,
               F13_CLIENT_AUTH_REQUEST_2);

  free_tlvstack (tlvs);
}

/*
 * read snac 0x13,0x0E
 * server ack for serveral snac 0x13 request
 */
void
v7_snac13_check_srv_ack (Snac *snac)
{
  guint reqid;
  gchar *here = snac->data;
  gchar state = 0;

  Contact_Member *contactdata;

#ifdef TRACE_FUNCTION
  g_print("v7_snac13_check_srv_ack\n");
#endif

  reqid = snac->reqid;

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  /* read the last byte */
  state = *(here + 1);

  if ((state & 0xFF) != 0x0E) { /* no error */
    switch (snac13_state.state) {
    case SNAC13_ADD_USER_NO_AUTH:
    case SNAC13_ADD_USER_AUTH:
      /* add the user */
      contactdata = g_new0 (Contact_Member, 1);
      contactdata->uin = snac13_state.uin;
      strncpy (contactdata->nick, snac13_state.nick, 20);
      contactdata->status = STATUS_NA;
      contactdata->last_status = STATUS_OFFLINE;
      contactdata->show_again = TRUE;
      contactdata->inlist = TRUE;
      contactdata->tcp_seq = -1;
      contactdata->confirmed = TRUE;
      contactdata->gid = snac13_state.gid;
      contactdata->uid = snac13_state.uid;
      Contacts = g_slist_append (Contacts, contactdata);

      if (snac13_state.state == SNAC13_ADD_USER_AUTH)
        contactdata->wait_auth = TRUE;

      /* update group list before finishing edit */
      v7_update_group_info (snac13_state.gid);
      v7_end_CL_edit ();

      /* refresh the gui */
      Show_Quick_Status_lower (UPDATE_OFFLINE|UPDATE_ONLINE, NULL);

      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_REMOVE_USER:
      /* user already removed in the local copy */
      /* user removal successful, update group */
      v7_update_group_info (snac13_state.gid);
      v7_end_CL_edit ();

      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_AUTH_REQUEST:
      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_NONE:
      break;
    }
  } else { /* error */
    switch (snac13_state.state) {
    case SNAC13_ADD_USER_NO_AUTH:
      /* add user with no auth request unsuccessful, need to send auth req */
      v7_end_CL_edit ();
      auth_request_msg_box (snac13_state.nick, snac13_state.uin);
      /* change to auth request state */
      snac13_state.state = SNAC13_AUTH_REQUEST;
      break;
    case SNAC13_ADD_USER_AUTH:
      g_warning ("User adding failed: uin(%d) ", snac13_state.uin);
      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_REMOVE_USER:
      g_warning ("User removal failed: uin(%d)", snac13_state.uin);
      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_AUTH_REQUEST:
      snac13_state.state = SNAC13_NONE;
      break;
    case SNAC13_NONE:
      break;
    }
  }
} /* end void v7_snac13_check_srv_ack (Snac *snac) */


/*
 * send user status
 *
 * update stauts, TRUE = visible, FALSE = invisible
 */
void
v7_send_status_server (gboolean visible)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_status (tlvs, visible);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x09
 */
void
v7_update_nickname (guint gid, guint uid, UIN_T uin,
                    const gchar *nick, gint nick_len, gboolean wait_auth)
{
  TLVstack *tlvs;
  gchar *mynick;

  /* make sure the nick passed is null terminated */
  mynick = g_strndup (nick, nick_len);

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_nickname (tlvs, uin, gid, uid, mynick, wait_auth);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
  g_free (mynick);
}

/*
 * send snac 0x13,0x09
 */
void
v7_update_group_info (guint gid)
{
  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print("v7_update_group_info\n");
#endif

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_id_ingroup (tlvs, gid);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
}

/* helper functions to add uin */

/* try to add user without using authorization */
void
v7_try_add_uin (UIN_T uin, const gchar *nick, guint gid, gboolean grant)
{
  /* we can also send out authorization grant while adding the target user */
  if (grant) {
    v7_send_grant_auth (uin);
  }

  snac13_state.state = SNAC13_ADD_USER_NO_AUTH;

  snac13_state.uin = uin;

  strncpy (snac13_state.nick, nick, 19); /* FIXME */
  snac13_state.nick[19] = 0; /* null terminate the string */

  snac13_state.gid = gid;
  snac13_state.with_grant = grant;

  /* add new user with auth grant (i.e. no TLV(0x66) set) */
  v7_addto_contacts_list (uin, nick, gid, FALSE);
}

/* send an auth request to the user, according to the data in snac13_state
 * This function should work in conjunction with v7_try_add_uin(), will get
 * called if the adding was not sucessful (see v7_snac13_check_srv_ack(),
 * and auth.c
 */
void
v7_ask_uin_for_auth (gchar *authmsg)
{
  UIN_T uin;

  if (snac13_state.state != SNAC13_AUTH_REQUEST)
    /* something very wrong, we should be at that state */
    return;

  uin = snac13_state.uin;

  /* send the auth message first */
  v7_send_auth_message (uin, authmsg);

  /* send the auth grant again, if the user has intended */
  if (snac13_state.with_grant)
    v7_send_grant_auth (uin);

  snac13_state.state = SNAC13_ADD_USER_AUTH;
  /* now add again, but with the awaiting auth bit set (TLV(0x66) */
  v7_addto_contacts_list (uin, snac13_state.nick, snac13_state.gid, TRUE);
}

/*
 * send snac 0x13,0x0A
 */
void
v7_remove_contact (const UIN_T uin)
{
  TLVstack *tlvs;
  GSList *contact;

#ifdef TRACE_FUNCTION
  g_print("v7_remove_contact\n");
#endif

  if ( !(contact = Find_User (uin)) )
    return;

  v7_begin_CL_edit (FALSE);

  snac13_state.state = SNAC13_REMOVE_USER;

  snac13_state.gid = kontakt->gid;
  snac13_state.uid = kontakt->uid;
  strncpy (snac13_state.nick, kontakt->nick, 19);
  snac13_state.nick[19] = 0;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_nickname (tlvs, kontakt->uin, kontakt->gid, kontakt->uid,
                             kontakt->nick, kontakt->wait_auth);
  snac13_send_remove_from_list (tlvs);

  free_tlvstack (tlvs);

  Save_RC ();
}

/*
 * read snac 0x13,0x19
 */
void
v7_recv_auth_request (Snac *snac)
{
  gchar *here = snac->data;
  UIN_T uin = 0;
  gint str_len;
  gchar *str;

#ifdef TRACE_FUNCTION
  g_print("v7_recv_auth_request\n");
#endif

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  /* read requester uin */
  uin = strtol (here+1, NULL, 10);
  here = here + here[0] + 1;

  str_len = CharsBE_2_Word (here);
  here += 2;

  /* read the request msg */
  str = g_strndup (here, str_len);

  /* got all the info, now call the client auth handler */
  auth_receive_request (uin, str);

  g_free (str);
}

/*
 * send snac 0x13,0x1A
 */
void
v7_grant_auth_request (UIN_T uin)
{
  gint len;
  gchar *strtmp;
  TLVstack *t = NULL;

  t = new_tlvstack (NULL, 0);

  strtmp = g_strdup_printf ("?%d", uin);
  len = strlen (strtmp) - 1;
  strtmp[0] = (gchar)len;
  add_nontlv (t, strtmp, len+1);
  g_free (strtmp);

  add_nontlv (t, "\x1\0\0\0\0", 5);

  SNAC13_SEND (t->beg, t->len, F13_CLIENT_SEND_GRANT_AUTH,
               F13_CLIENT_SEND_GRANT_AUTH);

  free_tlvstack (t);
}

/*
 * user grant your auth request to add to your list (0x1B)
 */
void
v7_user_grant_auth (Snac *snac)
{
  gchar *here = snac->data;
  GSList *contact;
  UIN_T uin;

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  /* read sender uin */
  uin = strtol (here+1, NULL, 10);
  here = here + here[0] + 1;

  contact = Find_User (uin);

  if (contact)
    kontakt->wait_auth = FALSE;

  /* TODO: need to add some GUI call after changing status */
}

/*
 * visible list, add or remove a uin
 */
void
v7_visible (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_visible (tlvs, uin, luid);

  if (add) {
    snac13_send_add_to_list (tlvs);
  } else {
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/*
 * invisible list, add or remove a uin
 */
void
v7_invisible (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_invisible (tlvs, uin, luid);

  if (add) {
    snac13_send_add_to_list (tlvs);
  } else {
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/*
 * ignore list, add or remove a uin
 */
void
v7_ignore (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_ignore (tlvs, uin, luid);

  if (add) {
    snac13_send_add_to_list (tlvs);
  } else {
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/************************* private functions ************************/

/* remove the TLV(0x11) record */
void
remove_tlv11_record ()
{
  TLVstack *tlvs = NULL;

  tlvs = new_tlvstack (NULL, 0);

  add_nontlv_bws (tlvs, invalid_record.tag);

  add_nontlv_w_be (tlvs, invalid_record.gid);
  add_nontlv_w_be (tlvs, invalid_record.uid);

  add_tlv (tlvs, 0x11, invalid_record.tlvval, invalid_record.tlvlen);
  snac13_send_remove_from_list (tlvs);

  g_free (invalid_record.tag);
  g_free (invalid_record.tlvval);

  free_tlvstack (tlvs);
}

/* helpers for composing the snac packet */

/* construct TLV(0) (type 0) packet */
/*   needauth - append 0x66 at the end, flagging the user in queustion is
 *              waiting for authorization
 */
void
snac13_construct_nickname (TLVstack *t, UIN_T uin, guint16 gid, guint16 uid,
                           const gchar *nick, gboolean needauth)
{
  gchar *strtmp;
  TLVstack *tmptlv = NULL;

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);
  
  /* group id, uin id */
  add_nontlv_w_be (t, gid);
  add_nontlv_w_be (t, uid);
  
  /* nickname */
  tmptlv = new_tlvstack (NULL, 0);

  add_tlv (tmptlv, 0x131, nick, strlen (nick));
  if (needauth) {
    add_tlv (tmptlv, 0x66, NULL, 0);
  }

  add_tlv (t, 0, tmptlv->beg, tmptlv->len);

  g_free (tmptlv);
}

void
snac13_construct_id_ingroup (TLVstack *t, WORD gid)
{
  TLVstack *idstlv, *ids;
  gchar *grpname;
  GSList *contact;
  GSList *ginfo;

  grpname = groups_name_by_gid(gid);
  
  /* group name */
  add_nontlv_bws (t, grpname);

  /* group id, zero uin id */
  add_nontlv_w_be (t, gid);
  add_nontlv_w_be (t, 0);

  /* add uinid's */

  ids = new_tlvstack (NULL, 0);

  if (gid == 0) { /* master group */
    for (ginfo = Groups; ginfo != NULL; ginfo = ginfo->next)
      add_nontlv_w_be (ids, ((GroupInfo *)(ginfo->data))->gid);
  } else {
    /* Look up the contact list for members of the group and add them */
    for (contact = Contacts; contact != NULL; contact = contact->next) 
      if (kontakt->gid == gid)
        add_nontlv_w_be (ids, kontakt->uid);
  }

  idstlv = new_tlvstack (NULL, 0);

  add_tlv (idstlv, 0xC8, ids->beg, ids->len);

  free_tlvstack(ids);

  add_tlv (t, 0x1, idstlv->beg, idstlv->len);

  free_tlvstack (idstlv);
}

void
snac13_construct_visible (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x02, NULL, 0);
}

void
snac13_construct_invisible (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x03, NULL, 0);
}

void
snac13_construct_ignore (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x0E, NULL, 0);
}

/*
 * status =
 *   TRUE: visible
 *   FALSE: invisible
 */
void
snac13_construct_status (TLVstack *t, gboolean status)
{
  TLVstack *sts;
  gchar buf;

  status_uid = status_uid ? status_uid : contact_gen_uid ();

  add_nontlv_bws (t, NULL);

  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, status_uid);

  sts = new_tlvstack (NULL, 0);

  buf = status ? '\x4' : '\x3';

  add_tlv (sts, 0xCA, &buf, 1);
  add_tlv (t, 0x4, sts->beg, sts->len);

  free_tlvstack (sts);
}


void
snac13_construct_authreq (TLVstack *t, UIN_T uin, const gchar *msg)
{
  gint len;
  gchar *strtmp;

  /* uin */
  strtmp = g_strdup_printf ("?%d", uin);
  len = strlen (strtmp) - 1; /* length of the string uin */
  strtmp[0] = (guint8)len;
  add_nontlv (t, strtmp, len+1);
  g_free (strtmp);

  add_nontlv_bws (t, msg);

  add_nontlv (t, "\0\0", 2);
}

void
snac13_send_add_to_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_ADD_TO_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_ADD_TO_LIST, s13_reqid);
}

void
snac13_send_update_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_UPDATE_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_UPDATE_LIST, s13_reqid);
}

void
snac13_send_remove_from_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_REMOVE_FROM_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_REMOVE_FROM_LIST, s13_reqid);
}

void
snac13_check_contacts_list_sanity ()
{
  /* match group id from user records */
  /* not done */

  /* check status_uid record, if 0, that means we need to create one */
  if (status_uid == 0) {
    TLVstack *tlvs;

    tlvs = new_tlvstack (NULL, 0);
    snac13_construct_status (tlvs, TRUE);
    snac13_send_add_to_list (tlvs);
    free_tlvstack (tlvs);
  }

  sane_cl = TRUE;
  save_preferences ();
}
