// $Id: mainwin.cc,v 1.29 2001/02/07 17:31:12 cactus Exp $
/*
  Gnomoku Copyright (C) 1998-1999 NAGY Andrs <nagya@telnet.hu>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2
  (included in the RadioActive distribution in doc/GPL) as published by
  the Free Software Foundation.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "config.h"

#include "mainwin.h"
#include <stdarg.h>
#include <errno.h>
#define _GNU_SOURCE
#include <stdio.h>
#include <string.h>
#include <list>

using namespace Gtk;
using namespace Gnome;

#define COPYRIGHT "(C) NAGY Andrs, RDI Gerg"

// GNOMOKU
Gnomoku::Gnomoku(int rows_, int cols_):
    App("main", "Gnomoku"),
    status(true, true, GNOME_PREFERENCES_USER),
    rows(rows_),
    cols(cols_),
    
    opponent(0),
    optype(OP_AI),
    server("localhost"),
    port("23776"),
    beep(false)
{
    // Window
    set_title("Gnomoku");
    set_policy(false, false, false);
    destroy.connect(Gtk::Main::instance()->quit.slot());

    // Status bar
    string greetmsg((string)"Gnomoku" + " " + VERSION + " -- " + COPYRIGHT);
    status.set_default(greetmsg);
    set_statusbar(status);

    // Menu
#if 0
    install_appbar_menu_hints(status, *menubar);
#endif

    list<UI::Info> menubar;

    // Game menu
    list<UI::Info> m_game;
    m_game.push_back (MenuItems::NewGame(
	slot (this, &Gnomoku::start_game)));
    m_game.push_back (UI::Separator ());
    m_game.push_back (MenuItems::Exit (
	slot (this, &Gnomoku::quit)));    
    menubar.push_back (Menus::Game (m_game));

    // Settings menu
    list<UI::Info> m_settings;
    m_settings.push_back (MenuItems::Preferences (
	slot (this, &Gnomoku::options)));
    menubar.push_back (Menus::Settings (m_settings));

    // Help menu
    list<UI::Info> m_help;
    m_help.push_back (MenuItems::About (
	slot (this, &Gnomoku::about)));
    menubar.push_back (Menus::Help (m_help));
    
    create_menus (menubar);
    install_menu_hints ();
    
    // Table
    Table* table = new Table(rows, cols, true);
    Point *p;
    tbl = new (Point**)[rows];
    for (int y=0; y<rows; y++) {
	tbl[y] = new (Point*)[cols];
	for (int x=0; x<cols; x++) {
	    p = new Point(y,x);
	    tbl[y][x] = p;
	    p->clicked.connect(
		bind(slot(this, &Gnomoku::point_pressed), p));
	    table->attach(*p, x, x+1, y, y+1, GTK_EXPAND, GTK_EXPAND);
	}
    }
    table->set_border_width(GNOME_PAD);
    table->show_all();
    
    set_contents(*manage(table));

    // tmout
    Gtk::Main::timeout.connect(slot(this, &Gnomoku::tmout), 100);
}

Gnomoku::~Gnomoku()
{
    if (opponent) {
	delete opponent;
	opponent = 0;
    }
    
    for (int y=0; y<rows; y++) {
	for (int x=0; x<cols; x++) {
	    delete tbl[y][x];
	}
	delete tbl[y];
    }
    delete tbl;
}

gint Gnomoku::status_timeout()
{
    static gfloat i = status.get_progress()->get_adjustment()->get_lower();
    if (i++ == status.get_progress()->get_adjustment()->get_upper())
	i = status.get_progress()->get_adjustment()->get_lower();
    status.get_progress()->set_value(i);
    return true;
}

void Gnomoku::reset()
{
    if (opponent) {
	delete opponent;
	opponent = 0;
	cleanup();
	message(_("Game reset"));
    }
}

#define MAX_MSG_LEN 256
void Gnomoku::message(const char *fmt, ...)
{
    va_list ap;
    char text[MAX_MSG_LEN];
    
    va_start(ap, fmt);
    vsnprintf(text, MAX_MSG_LEN, fmt, ap);
    
    status.pop();
    status.push(text);
    
    va_end(ap);
}

void Gnomoku::cleanup()
{
    for (int y=0; y<rows; y++)
	for (int x=0; x<cols; x++) {
	    tbl[y][x]->set(0);
	    tbl[y][x]->set_blink(false);
	}
    my_turn = true;
}

void Gnomoku::start_game()
{
    msg_t msg;

    cleanup();
    
    if (opponent) {
	if (opponent->ok()) {
	    msg.type = MSG_START;
	    opponent->put_msg(msg);
	    message(_("Game restarted"));
	}
    } else {
	if (optype==OP_AI) {
	    opponent = new AI(rows, cols);
	} else {
	    message(_("Estabilishing connection..."));
	    status.get_progress()->set_activity_mode(true);
	    if (!status_conn)
		status_conn = Gtk::Main::timeout.connect(slot(this, &Gnomoku::status_timeout), 50);
	    opponent = new User(optype, server, port);
	}
    }
}

gint Gnomoku::tmout()
{
    if (opponent) {
	if (int err=opponent->err()) {
	    message(_("ERROR: %s"), strerror(err));
	    status_conn.disconnect();
	    status.get_progress()->set_activity_mode(false);
	    status.get_progress()->set_value(0);
	    delete opponent;
	    opponent = 0;
	} else {
	    if ((opponent->ok()) &&
		(opponent->ready())) get_msg();
	}
    }
    return true;
}

void Gnomoku::get_msg()
{
    msg_t msg;

    opponent->get_msg(msg);
    switch (msg.type) {
    case MSG_START:
	message(_("Your opponent restarted the game"));
	cleanup();
	break;
    case MSG_PUT:
	if ((msg.y>=0) && (msg.y<rows) &&
	    (msg.x>=0) && (msg.x<cols) &&
	    !tbl[msg.y][msg.x]->get())
	{
	    tbl[msg.y][msg.x]->set(2);
	    tbl[msg.y][msg.x]->grab_focus();
	    if (!won(msg.y, msg.x)) {
		my_turn = true;
		message(_("It's your turn"));
		if (beep) gdk_beep();
	    }
	}
	break;
    case MSG_CLOSE:
	message(_("Your party closed the connection"));
	delete opponent;
	opponent = 0;
	break;
    case MSG_GREET:
	message(_("Your opponent is %s"), msg.ident.c_str());
	status_conn.disconnect();
	status.get_progress()->set_activity_mode(false);
	status.get_progress()->set_value(0);
	break;
    }
}

void Gnomoku::point_pressed(Point *p)
{
    msg_t msg;
    
    if (opponent && opponent->ok() && my_turn && !p->get()) {
	msg.type = MSG_PUT;
	msg.y = p->gety();
	msg.x = p->getx();
	opponent->put_msg(msg);
	p->set(1);
	if (!won(msg.y, msg.x)) {
	    my_turn = false;
	    message(_("Waiting for your party to respond"));
	}
    }
}

bool Gnomoku::won(int y, int x)
{
    int who=tbl[y][x]->get();
    bool wins=false;
    int pieces[4];
    Point *points[4][5];
    int seq=0;

    for (int i=0; i<4; i++) pieces[i] = 0;

    for (int ay=-1; ay<=1; ay++)
	for (int ax=-1; ax<=1; ax++)
	    if (ay || ax) {
		int ny=y, nx=x;
		while (((ny+=ay)>=0) && ((nx+=ax)>=0) &&
		       (ny<rows) && (nx<cols) &&
		       tbl[ny][nx]->get()==who)
		{
		    int s = (seq<4 ? seq : 7-seq);
		    points[s][++pieces[s]] = tbl[ny][nx];
		}
		seq++;
	    }

    for (int i=0; i<4; i++)
	if (pieces[i]>=4) {
	    wins=true;
	    points[i][0] = tbl[y][x];
	    for (int p=0; p<5; p++)
		points[i][p]->set_blink(true);
	    opponent->won();
	    
	    if (who==1)
		message(_("You won the game"));
	    else
		message(_("Your opponent won the game"));
	    my_turn=false;
	}

    return wins;
}

void Gnomoku::quit()
{
    Gnome::Dialog* dialog = Dialogs::question_modal(
	_("Do you really want to quit the game?"), 0);
    if (!dialog->run())
	destroy();
}

void Gnomoku::options()
{
    Options* optwin = new Options(optype, server, port, beep);
    optwin->set_transient_for(*this);
    optwin->changed.connect(slot(this, &Gnomoku::reset));
}

void Gnomoku::about()
{
    vector<string> authors(2);
    authors[0] = "NAGY Andrs <nagya@telnet.hu>";
    authors[1] = "RDI Gerg <cactus@cactus.rulez.org>";
    About* about = new About("Gnomoku", VERSION, COPYRIGHT, authors,
			     _("Gomoku game for GNOME"));
    about->run();
}
