/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2002-2003 Imendio AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtkwidget.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkmain.h>
#include <glib/gi18n.h>

#include <libgnome/gnome-program.h>
#include <libgnomeui/gnome-ui-init.h>

#include <libgossip/gossip-account-manager.h>
#include <libgossip/gossip-account.h>

#include "gossip-preferences.h"
#include "gossip-stock.h"
#include "gossip-app.h"

static void
setup_default_window_icon (void)
{
	GdkPixbuf *pixbuf;

	pixbuf = gdk_pixbuf_new_from_file (DATADIR "/pixmaps/gossip.png", NULL);
	gtk_window_set_default_icon (pixbuf);
	g_object_unref (pixbuf);
}

int
main (int argc, char *argv[])
{
	GnomeProgram          *program;
	GossipAccountManager  *account_manager;
	gboolean               no_connect = FALSE;
	gboolean               list_accounts = FALSE;
	poptContext            popt_context;
	gchar                 *account_name = NULL;
	const gchar          **args;
	GList                 *accounts;

	struct poptOption   options[] = {
		{ "no-connect",
			'n',
			POPT_ARG_NONE,
			&no_connect,
			0,
			N_("Don't connect on startup"),
		  NULL },
		{ "account",
			'a',
			POPT_ARG_STRING,
			&account_name,
			0,
			N_("Which account to connect to on startup"),
		  N_("ACCOUNT-NAME") },
		{ "list-accounts",
			'l',
			POPT_ARG_NONE,
			&list_accounts,
			0,
			N_("List the available accounts"),
		  NULL },

		{ NULL, '\0', 0, NULL, 0, NULL, NULL }
	};

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
        bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	program = gnome_program_init ("gossip", PACKAGE_VERSION,
				      LIBGNOMEUI_MODULE,
                                      argc, argv,
                                      GNOME_PROGRAM_STANDARD_PROPERTIES,
				      GNOME_PARAM_POPT_TABLE, options,
				      GNOME_PARAM_HUMAN_READABLE_NAME, PACKAGE_NAME,
				      NULL);

	g_set_application_name (PACKAGE_NAME);

	g_object_get (program,
		      GNOME_PARAM_POPT_CONTEXT,
		      &popt_context,
		      NULL);

	args = poptGetArgs (popt_context);

	/* get all accounts */
 	account_manager = gossip_account_manager_new (NULL);

	accounts = gossip_account_manager_get_accounts (account_manager);

	if (list_accounts) {
		GList         *l;
		GossipAccount *def = NULL;

		if (g_list_length ((GList*)accounts) < 1) {
			g_print (_("No accounts available."));
			g_print ("\n");
		} else {
			def = gossip_account_manager_get_default (account_manager);
			
			g_print (_("Available accounts:"));
			g_print ("\n");
		}
		
		for (l = accounts; l; l = l->next) {
			GossipAccount *account = l->data;
			
			g_print (" %s", gossip_account_get_name (account));
			if (gossip_account_equal (account, def)) {
				g_print (" ");
				g_print (_("[default]"));
			}

			g_print ("\n");
		}

		g_list_free (accounts);

		return EXIT_SUCCESS;
	}
	
	if (account_name) {
		GossipAccount *account;

		account = gossip_account_manager_find (account_manager,
						       account_name);
		if (!account) {
			g_print (_("There is no account with the name '%s'."),
				 account_name);
			g_print ("\n");

			g_list_free (accounts);

			return EXIT_FAILURE;
		}

		/* use the specified account as default account. */
		gossip_account_manager_set_overridden_default (account_manager,
							       account_name);
	}

	g_list_free (accounts);

	gossip_stock_init ();
	setup_default_window_icon ();
	gossip_app_create (account_manager);
	
	if (!no_connect) {
		gossip_app_connect (TRUE);
	}
	
	gtk_main ();

	g_object_unref (account_manager);

	return EXIT_SUCCESS;
}
