/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <stdio.h>
#include <string.h>
#include "ccd-border.h"

#define SET_COMMON_WIDTH(spec_, border_, val_)				       \
if (CCD_PROPERTY_SPEC_SET == spec_ && 					       \
    !(CCD_BORDER_FLAGS_SPECIFIC_WIDTH & border_.flags)) {		       \
	border_.width = val_;						       \
	border_.flags |= CCD_BORDER_FLAGS_COMMON_WIDTH;			       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.width_spec = spec_;					       \
}

#define SET_SPECIFIC_WIDTH(spec_, border_, val_)			       \
if (CCD_PROPERTY_SPEC_SET == spec_) {					       \
	border_.width = val_;						       \
	border_.flags |= CCD_BORDER_FLAGS_SPECIFIC_WIDTH;		       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.width_spec = spec_;					       \
}

#define SET_COMMON_STYLE(spec_, border_, val_)				       \
if (CCD_PROPERTY_SPEC_SET == spec_ && 					       \
    !(CCD_BORDER_FLAGS_SPECIFIC_STYLE & border_.flags)) {		       \
	border_.style = val_;						       \
	border_.flags |= CCD_BORDER_FLAGS_COMMON_STYLE;			       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.style_spec = spec_;					       \
}

#define SET_SPECIFIC_STYLE(spec_, border_, val_)			       \
if (CCD_PROPERTY_SPEC_SET == spec_) {					       \
	border_.style = val_;						       \
	border_.flags |= CCD_BORDER_FLAGS_SPECIFIC_STYLE;		       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.style_spec = spec_;					       \
}

#define SET_COMMON_COLOR(spec_, border_, val_)				       \
if (CCD_PROPERTY_SPEC_SET == spec_ && 					       \
    !(CCD_BORDER_FLAGS_SPECIFIC_COLOR & border_.flags)) { 		       \
	border_.color.red = val_.red;					       \
	border_.color.green = val_.green;				       \
	border_.color.blue = val_.blue;					       \
	border_.flags |= CCD_BORDER_FLAGS_COMMON_COLOR;			       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.color_spec = spec_;					       \
}

#define SET_SPECIFIC_COLOR(spec_, border_, val_)			       \
if (CCD_PROPERTY_SPEC_SET == spec_) { 					       \
	border_.color.red = val_.red;					       \
	border_.color.green = val_.green;				       \
	border_.color.blue = val_.blue;					       \
	border_.flags |= CCD_BORDER_FLAGS_SPECIFIC_COLOR;		       \
}									       \
if (spec_ != CCD_PROPERTY_SPEC_UNSET) {					       \
	border_.color_spec = spec_;					       \
}

/*!
 * Map between border style css string and internal value.
 */
static const struct {
	ccd_border_style_t border_style;
	char const *css;
} _border_style_map[] = {
	{ CCD_BORDER_STYLE_HIDDEN,	"hidden" },
	{ CCD_BORDER_STYLE_DOTTED,	"dotted" },
	{ CCD_BORDER_STYLE_DASHED,	"dashed" },
	{ CCD_BORDER_STYLE_SOLID,	"solid"  },
	{ CCD_BORDER_STYLE_DOUBLE,	"double" },
	{ CCD_BORDER_STYLE_GROOVE,	"groove" },
	{ CCD_BORDER_STYLE_RIDGE,	"ridge"  },
	{ CCD_BORDER_STYLE_INSET,	"inset"  },
	{ CCD_BORDER_STYLE_OUTSET,	"outset" }
};

static bool
match_style (char const		*css_border_style,
	     ccd_border_style_t	*style)
{
	g_return_val_if_fail (css_border_style && *css_border_style, false);

	for (unsigned int i = 0; i < G_N_ELEMENTS (_border_style_map); i++) {
		if (0 == strcmp (_border_style_map[i].css, css_border_style)) {
			*style = _border_style_map[i].border_style;
			return true;
		}
	}

	return false;
}

static char const *
lookup_name (ccd_border_style_t border_style)
{
	for (unsigned int i = 0; i < G_N_ELEMENTS (_border_style_map); i++) {
		if (_border_style_map[i].border_style == border_style) {
			return _border_style_map[i].css;
		}
	}	

	return NULL;
}

ccd_border_t * 
ccd_border_new (void)
{
	return g_new0 (ccd_border_t, 1);
}

void
ccd_border_free (ccd_border_t *self)
{
	g_free (self);
}

/*
 * Parse properties of the form
 * - border: ;		# only to prevent errors
 * - border: 1px;
 * - border: solid;
 * - border: 1px solid;
 * - border: red;
 * - border: 1px red;
 * - border: 1px solid red;
 */
bool
ccd_border_parse (ccd_border_t	*self,
		  char const	*property,
		  CRTerm const	*values)
{
	CRTerm const		*iter;
	double			 width;
	ccd_border_style_t	 style;
	ccd_color_t		 color;
	ccd_property_spec_t	 width_spec;
	ccd_property_spec_t	 color_spec;
	ccd_property_spec_t	 style_spec;

	iter = values;

	width = 0;	/* prevent warning */
	width_spec = ccd_property_parse_spec (&iter);
	if (CCD_PROPERTY_SPEC_SET == width_spec &&
	    iter && 
	    TERM_NUMBER == iter->type) {
		/* width */
		width = iter->content.num->val;
		iter = iter->next;
		width_spec = CCD_PROPERTY_SPEC_SET;
	}

	style_spec = ccd_property_parse_spec (&iter);
	style = CCD_BORDER_STYLE_SOLID;	/* prevent warning */
	if (CCD_PROPERTY_SPEC_SET == style_spec &&
	    iter && 
	    TERM_IDENT == iter->type &&
	    match_style (cr_string_peek_raw_str (iter->content.str), &style)) {
		/* style */
		iter = iter->next;
		style_spec = CCD_PROPERTY_SPEC_SET;
	}

	color_spec = ccd_property_parse_spec (&iter);
	color.red = color.green = color.blue = 0; /* prevent warning */
	if (CCD_PROPERTY_SPEC_SET == color_spec &&
	    iter &&
	    ccd_color_parse (&color, &iter)) {
		/* color */
		/* iter is advanced by ccd_color_parse() */
		color_spec = CCD_PROPERTY_SPEC_SET;
	}

	if (0 == strcmp ("border", property)) {

		SET_COMMON_WIDTH (width_spec, self->left, width);
		SET_COMMON_WIDTH (width_spec, self->top, width);
		SET_COMMON_WIDTH (width_spec, self->right, width);
		SET_COMMON_WIDTH (width_spec, self->bottom, width);

		SET_COMMON_STYLE (style_spec, self->left, style);
		SET_COMMON_STYLE (style_spec, self->top, style);
		SET_COMMON_STYLE (style_spec, self->right, style);
		SET_COMMON_STYLE (style_spec, self->bottom, style);
		
		SET_COMMON_COLOR (color_spec, self->left, color);
		SET_COMMON_COLOR (color_spec, self->top, color);
		SET_COMMON_COLOR (color_spec, self->right, color);
		SET_COMMON_COLOR (color_spec, self->bottom, color);

	} else if (0 == strcmp ("border-left", property)) {

		SET_SPECIFIC_WIDTH (width_spec, self->left, width);
		SET_SPECIFIC_STYLE (style_spec, self->left, style);
		SET_SPECIFIC_COLOR (color_spec, self->left, color);

	} else if (0 == strcmp ("border-top", property)) {

		SET_SPECIFIC_WIDTH (width_spec, self->top, width);
		SET_SPECIFIC_STYLE (style_spec, self->top, style);
		SET_SPECIFIC_COLOR (color_spec, self->top, color);

	} else if (0 == strcmp ("border-right", property)) {

		SET_SPECIFIC_WIDTH (width_spec, self->right, width);
		SET_SPECIFIC_STYLE (style_spec, self->right, style);
		SET_SPECIFIC_COLOR (color_spec, self->right, color);

	} else if (0 == strcmp ("border-bottom", property)) {

		SET_SPECIFIC_WIDTH (width_spec, self->bottom, width);
		SET_SPECIFIC_STYLE (style_spec, self->bottom, style);
		SET_SPECIFIC_COLOR (color_spec, self->bottom, color);

	} else {
		return false;
	}

	return true;
}

static void
draw_none (ccd_border_stroke_t const	*stroke,
	   cairo_t			*cr,
	   double			 x1,
	   double			 y1,
	   double			 x2,
	   double			 y2)
{}

static void
draw_dotted (ccd_border_stroke_t const	*stroke,
	     cairo_t			*cr,
	     double			 x1,
	     double			 y1,
	     double			 x2,
	     double			 y2)
{
	double dash_len;

	dash_len = stroke->width;

	cairo_save (cr);

	cairo_move_to (cr, x1, y1);
	cairo_line_to (cr, x2, y2);
	cairo_set_dash (cr, &dash_len, 1, 0);
	cairo_set_line_width (cr, stroke->width);
	cairo_set_source_rgb (cr, stroke->color.red, stroke->color.green, 
				stroke->color.blue);
	cairo_stroke (cr);

	cairo_restore (cr);
}

static void
draw_dashed (ccd_border_stroke_t const	*stroke,
	     cairo_t			*cr,
	     double			 x1,
	     double			 y1,
	     double			 x2,
	     double			 y2)
{
	double dashes[2];

	dashes[0] = stroke->width * 3;
	dashes[1] = stroke->width * 3;

	cairo_save (cr);

	cairo_move_to (cr, x1, y1);
	cairo_line_to (cr, x2, y2);
	cairo_set_dash (cr, dashes, G_N_ELEMENTS (dashes), 0);
	cairo_set_line_width (cr, stroke->width);
	cairo_set_source_rgb (cr, stroke->color.red, stroke->color.green, 
				stroke->color.blue);
	cairo_stroke (cr);

	cairo_restore (cr);
}

static void
draw_solid (ccd_border_stroke_t const	*stroke,
	    cairo_t			*cr,
	    double			 x1,
	    double			 y1,
	    double			 x2,
	    double			 y2)
{
	cairo_save (cr);

	cairo_move_to (cr, x1, y1);
	cairo_line_to (cr, x2, y2);
	cairo_set_line_width (cr, stroke->width);
	cairo_set_source_rgb (cr, stroke->color.red, stroke->color.green, 
				stroke->color.blue);
	cairo_stroke (cr);

	cairo_restore (cr);
}

typedef void (*draw_f) (ccd_border_stroke_t const *, cairo_t *, 
			double, double, double, double);

static draw_f
get_draw_func (ccd_border_stroke_t const *stroke)
{
	if (CCD_PROPERTY_SPEC_NONE == stroke->style_spec) {
		return draw_none;
	}

	switch (stroke->style) {
	case CCD_BORDER_STYLE_HIDDEN: 
		g_warning ("CCD_BORDER_STYLE_HIDDEN not implemented");
		return draw_none;
	case CCD_BORDER_STYLE_DOTTED:
		return draw_dotted;
	case CCD_BORDER_STYLE_DASHED:
		return draw_dashed;
	case CCD_BORDER_STYLE_SOLID:
		return draw_solid;
	case CCD_BORDER_STYLE_DOUBLE:
		g_warning ("CCD_BORDER_STYLE_DOUBLE not implemented");
	case CCD_BORDER_STYLE_GROOVE:
		g_warning ("CCD_BORDER_STYLE_GROOVE not implemented");
	case CCD_BORDER_STYLE_RIDGE:
		g_warning ("CCD_BORDER_STYLE_RIDGE not implemented");
	case CCD_BORDER_STYLE_INSET:
		g_warning ("CCD_BORDER_STYLE_INSET not implemented");
	case CCD_BORDER_STYLE_OUTSET:
		g_warning ("CCD_BORDER_STYLE_OUTSET not implemented");
	default:
		g_assert_not_reached ();
		return draw_solid;
	}
}

void
ccd_border_draw (ccd_border_stroke_t const	*left,
		 ccd_border_stroke_t const	*top,
		 ccd_border_stroke_t const	*right,
		 ccd_border_stroke_t const	*bottom,
		 cairo_t			*cr,
		 double				 x,
		 double				 y,
		 double				 width,
		 double				 height)
{
	draw_f	draw_func;
	double	off;

	if (left && left->flags) {
		draw_func = get_draw_func (left);
		off = left->width / 2.;
		draw_func (left, cr,
			   x + off, y, x + off, y + height);
	}

	if (top && top->flags) {
		draw_func = get_draw_func (top);
		off = top->width / 2.;
		draw_func (top, cr,
			   x, y + off, x + width, y + off);
	}

	if (right && right->flags) {
		draw_func = get_draw_func (right);
		off = right->width / 2.;
		draw_func (right, cr,
			   x + width - off, y, x + width - off, y + height);
	}

	if (bottom && bottom->flags) {
		draw_func = get_draw_func (bottom);
		off = bottom->width / 2.;
		draw_func (bottom, cr,
			   x + width, y + height - off, x, y + height - off);
	}
}

void
ccd_border_stroke_dump (ccd_border_stroke_t const *self)
{
	if (self->flags & CCD_BORDER_FLAGS_WIDTH_MASK) {
		printf ("%.1f ", self->width);
	}

	if (self->flags & CCD_BORDER_FLAGS_STYLE_MASK) {
		printf ("%s ", lookup_name (self->style));
	}

	if (self->flags & CCD_BORDER_FLAGS_COLOR_MASK) {
		ccd_color_dump (&self->color);
	}

	printf (";\n");
}

void
ccd_border_dump (ccd_border_t const *self)
{
	if (self->left.flags) {
		printf (CCD_PROPERTY_DUMP_PREFIX "border-left: ");
		ccd_border_stroke_dump (&self->left);
	}

	if (self->top.flags) {
		printf (CCD_PROPERTY_DUMP_PREFIX "border-top: ");
		ccd_border_stroke_dump (&self->top);
	}

	if (self->right.flags) {
		printf (CCD_PROPERTY_DUMP_PREFIX "border-right: ");
		ccd_border_stroke_dump (&self->right);
	}

	if (self->bottom.flags) {
		printf (CCD_PROPERTY_DUMP_PREFIX "border-bottom: ");
		ccd_border_stroke_dump (&self->bottom);
	}
}

