/* The Cairo CSS Drawing Library.
 * Copyright (C) 2008 Robert Staudinger
 *
 * This  library is free  software; you can  redistribute it and/or
 * modify it  under  the terms  of the  GNU Lesser  General  Public
 * License  as published  by the Free  Software  Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed  in the hope that it will be useful,
 * but  WITHOUT ANY WARRANTY; without even  the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License  along  with  this library;  if not,  write to  the Free
 * Software Foundation, Inc., 51  Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#ifndef CCD_NODE_H
#define CCD_NODE_H

#include <stdbool.h>
#include <glib.h>
#include <ccd/ccd-features.h>

G_BEGIN_DECLS

/**
 * ccd_node_t:
 * 
 * Stack-allocatable struct representing a document node. Used for querying the 
 * #ccd_stylesheet_t.
 * 
 * <emphasis>Memory management:</emphasis> Unless specified otherwise, objects 
 * of this kind are under the responsibility of the libccd consumer.
 **/
typedef void * ccd_node_t;

/** 
 * ccd_node_is_a_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query whether a #ccd_node_t is of a certain type.
 *
 * Returns: %TRUE if matches.
 **/
typedef bool (*ccd_node_is_a_f) (ccd_node_t const *self, 
				 char const *type_name);

/** 
 * ccd_node_get_container_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the container of a #ccd_node_t.
 *
 * Returns: newly allocated container node or %NULL.
 **/
typedef ccd_node_t * (*ccd_node_get_container_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_base_style_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the name of the style a #ccd_node_t derives from.
 *
 * Returns: base style or %NULL. The returned value must be valid until
 * it is released.
 **/
typedef ccd_node_t * (*ccd_node_get_base_style_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_type_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the type name of a #ccd_node_t.
 *
 * Returns: node type name or %NULL. The returned value must be valid until
 * the current stylesheet query returns.
 **/
typedef const char * (*ccd_node_get_type_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_id_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the ID of a #ccd_node_t.
 *
 * Returns: node ID or %NULL. The returned value must be valid until
 * the current stylesheet query returns.
 **/
typedef const char * (*ccd_node_get_id_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_class_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the class name of a #ccd_node_t.
 *
 * Returns: node class name or %NULL. The returned value must be valid until
 * the current stylesheet query returns.
 **/
typedef const char * (*ccd_node_get_class_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_pseudo_class_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to query the pseudo-class name of a #ccd_node_t.
 *
 * Returns: node pseudo-class name or %NULL. The returned value must be valid until
 * the current stylesheet query returns.
 **/
typedef const char * (*ccd_node_get_pseudo_class_f) (ccd_node_t const *self);

/** 
 * ccd_node_get_attribute_f:
 * @self: a #ccd_node_t.
 * @name: attribute name.
 *
 * Hook function to query a #ccd_node_t's attributes.
 *
 * Returns: attribute value or %NULL. The returned value must be valid until
 * the current stylesheet query returns.
 **/
typedef char * (*ccd_node_get_attribute_f) (ccd_node_t const	*self,
					    char const		*name);

/** 
 * ccd_node_release_f:
 * @self: a #ccd_node_t.
 *
 * Hook function to deallocate a #ccd_node_t instance.
 **/
typedef void (*ccd_node_release_f) (ccd_node_t *self);

/**
 * ccd_node_class_t:
 * @get_container:	a #ccd_node_get_container_f.
 * @get_base_style:	a #ccd_node_get_base_style_f.
 * @get_id:		a #ccd_node_get_id_f.
 * @get_type:		a #ccd_node_get_type_f.
 * @get_class:		a #ccd_node_get_class_f.
 * @get_pseudo_class:	a #ccd_node_get_pseudo_class_f.
 * @get_attribute:	a #ccd_node_get_attribute_f.
 * @release:		a #ccd_node_release_f.
 *
 * Dispatch table a CCD consumer has to fill so the selection engine can 
 * retrieve information about the document the document.
 *
 * The implemented dispatch table needs to be passed to #ccd_init.
 **/
typedef struct {
	ccd_node_is_a_f			is_a;
	ccd_node_get_container_f	get_container;
	ccd_node_get_base_style_f	get_base_style;
	ccd_node_get_id_f		get_id;
	ccd_node_get_type_f		get_type;
	ccd_node_get_class_f		get_class;
	ccd_node_get_pseudo_class_f	get_pseudo_class;
	ccd_node_get_attribute_f	get_attribute;
	ccd_node_release_f		release;
} ccd_node_class_t;

ccd_node_class_t const *	ccd_node_get_class	(void);
void				ccd_node_set_class	(ccd_node_class_t const *node_class);
void				ccd_node_reset_class	(void);

G_END_DECLS

#endif /* CCD_NODE_H */

