/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* This file is part of the GtkHTML library

   Copyright (C) 2000 Helix Code, Inc.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include "gtkhtmldebug.h"
#include "htmlclue.h"
#include "htmlcolor.h"
#include "htmlcolorset.h"
#include "htmltextmaster.h"
#include "htmlselection.h"
#include "htmlengine-edit.h"
#include "htmlengine-edit-movement.h"
#include "htmlengine-edit-selection-updater.h"
#include "htmlsettings.h"

#include "htmlengine-edit-fontstyle.h"

/* #define PARANOID_DEBUG */


static GtkHTMLFontStyle
get_font_style_from_selection (HTMLEngine *engine)
{
	GtkHTMLFontStyle style;
	GtkHTMLFontStyle conflicts;
	gboolean backwards;
	gboolean first;
	HTMLObject *p;

	g_return_val_if_fail (engine->clue != NULL, GTK_HTML_FONT_STYLE_DEFAULT);
	g_assert (engine->mark != NULL);

	/* printf ("%s mark %p,%d cursor %p,%d\n",
		__FUNCTION__,
		engine->mark, engine->mark->position,
		engine->cursor, engine->cursor->position); */

	if (engine->mark->position < engine->cursor->position)
		backwards = TRUE;
	else
		backwards = FALSE;

	style = GTK_HTML_FONT_STYLE_DEFAULT;
	conflicts = GTK_HTML_FONT_STYLE_DEFAULT;
	first = TRUE;

	p = engine->cursor->object;

	while (1) {
		if (html_object_is_text (p) && p->selected) {
			if (first) {
				style = HTML_TEXT (p)->font_style;
				first = FALSE;
			} else {
				conflicts |= HTML_TEXT (p)->font_style ^ style;
			}
		}

		if (p == engine->mark->object)
			break;

		if (backwards)
			p = html_object_prev_leaf (p);
		else
			p = html_object_next_leaf (p);

		g_assert (p != NULL);
	}

	return style & ~conflicts;
}

static HTMLColor *
get_color_from_selection (HTMLEngine *engine)
{
	HTMLColor *color = NULL;
	gboolean backwards;
	HTMLObject *p;

	g_return_val_if_fail (engine->clue != NULL, NULL);
	g_assert (engine->mark != NULL);

	backwards = (engine->mark->position < engine->cursor->position) ? TRUE : FALSE;

	p = engine->cursor->object;
	while (1) {
		if (html_object_is_text (p) && p->selected) {
			color = HTML_TEXT (p)->color;
			break;
		}

		if (p == engine->mark->object)
			break;

		if (backwards)
			p = html_object_prev_leaf (p);
		else
			p = html_object_next_leaf (p);

		g_assert (p != NULL);
	}

	return color;
}

GtkHTMLFontStyle
html_engine_get_document_font_style (HTMLEngine *engine)
{
	HTMLObject *curr;

	g_return_val_if_fail (engine != NULL, GTK_HTML_FONT_STYLE_DEFAULT);
	g_return_val_if_fail (HTML_IS_ENGINE (engine), GTK_HTML_FONT_STYLE_DEFAULT);
	g_return_val_if_fail (engine->editable, GTK_HTML_FONT_STYLE_DEFAULT);

	curr = engine->cursor->object;

	if (curr == NULL)
		return GTK_HTML_FONT_STYLE_DEFAULT;
	else if (! html_object_is_text (curr))
		return GTK_HTML_FONT_STYLE_DEFAULT;
	else if (! html_engine_is_selection_active (engine))
		return HTML_TEXT (curr)->font_style;
	else
		return get_font_style_from_selection (engine);
}

HTMLColor *
html_engine_get_document_color (HTMLEngine *engine)
{
	HTMLObject *curr;

	g_return_val_if_fail (engine != NULL, NULL);
	g_return_val_if_fail (HTML_IS_ENGINE (engine), NULL);
	g_return_val_if_fail (engine->editable, NULL);

	curr = engine->cursor->object;

	if (curr == NULL)
		return NULL;
	else if (! html_object_is_text (curr))
		return NULL;
	else if (! html_engine_is_selection_active (engine))
		return HTML_TEXT (curr)->color;
	else
		return get_color_from_selection (engine);
}

GtkHTMLFontStyle
html_engine_get_font_style (HTMLEngine *engine)
{
	return (engine->insertion_font_style == GTK_HTML_FONT_STYLE_DEFAULT)
		? html_engine_get_document_font_style (engine)
		: engine->insertion_font_style;
}

HTMLColor *
html_engine_get_color (HTMLEngine *engine)
{
	return engine->insertion_color;
}

/**
 * html_engine_update_insertion_font_style:
 * @engine: An HTMLEngine
 * 
 * Update @engine's current insertion font style according to the
 * current selection and cursor position.
 * 
 * Return value: 
 **/
gboolean
html_engine_update_insertion_font_style (HTMLEngine *engine)
{
	GtkHTMLFontStyle new_style;

	new_style = html_engine_get_document_font_style (engine);

	if (new_style != engine->insertion_font_style) {
		engine->insertion_font_style = new_style;
		return TRUE;
	}

	return FALSE;
}

/**
 * html_engine_update_insertion_style:
 * @engine: An HTMLEngine
 * 
 * Update @engine's current insertion font style/color according to the
 * current selection and cursor position.
 * 
 * Return value: 
 **/
gboolean
html_engine_update_insertion_color (HTMLEngine *engine)
{
	HTMLColor *new_color;

	new_color = html_engine_get_document_color (engine);

	if (new_color && !html_color_equal (new_color, engine->insertion_color)) {
		html_color_unref (engine->insertion_color);
		engine->insertion_color = new_color;
		html_color_ref (engine->insertion_color);
		return TRUE;
	}

	return FALSE;
}

/**
 * html_engine_set_font_style:
 * @engine: An HTMLEngine
 * @style: An HTMLFontStyle
 * 
 * Set the current font style for `engine'.  This has the same semantics as the
 * bold, italics etc. buttons in a word processor, i.e.:
 *
 * - If there is a selection, the selection gets the specified style.
 *
 * - If there is no selection, the style gets "attached" to the cursor.  So
 *   inserting text after this will cause text to have this style.
 *
 * Instead of specifying an "absolute" style, we specify it as a "difference"
 * from the current one, through an AND mask and an OR mask.
 *
 **/
struct tmp_font {
	GtkHTMLFontStyle and_mask;
	GtkHTMLFontStyle or_mask;
};

static void
object_set_font_style (HTMLObject *o, struct tmp_font *tf)
{
	if (html_object_is_text (o)) {
		HTML_TEXT (o)->font_style |= tf->or_mask;
		HTML_TEXT (o)->font_style &= tf->and_mask;
	}
}

gboolean
html_engine_set_font_style (HTMLEngine *e,
			    GtkHTMLFontStyle and_mask,
			    GtkHTMLFontStyle or_mask)
{
	g_return_val_if_fail (e != NULL, FALSE);
	g_return_val_if_fail (HTML_IS_ENGINE (e), FALSE);
	g_return_val_if_fail (e->editable, FALSE);

	if (html_engine_is_selection_active (e)) {
		struct tmp_font *tf = g_new (struct tmp_font, 1);
		tf->and_mask = and_mask;
		tf->or_mask  = or_mask;
		html_engine_cut_and_paste (e, "Set font style", (GFunc) object_set_font_style, tf);
		g_free (tf);
		return FALSE;
	} else {
		GtkHTMLFontStyle old = e->insertion_font_style;

		e->insertion_font_style |= or_mask;
		e->insertion_font_style &= and_mask;

		return (old == e->insertion_font_style) ? FALSE : TRUE;
	}
}

gboolean
html_engine_toggle_font_style (HTMLEngine *engine, GtkHTMLFontStyle style)
{
	GtkHTMLFontStyle cur_style;

	cur_style = html_engine_get_font_style (engine);

	if (cur_style & style)
		return html_engine_set_font_style (engine, GTK_HTML_FONT_STYLE_MAX & ~style, 0);
	else
		return html_engine_set_font_style (engine, GTK_HTML_FONT_STYLE_MAX, style);
}

static GtkHTMLFontStyle
inc_dec_size (GtkHTMLFontStyle style, gboolean inc)
{
	GtkHTMLFontStyle size;

	if (style == GTK_HTML_FONT_STYLE_DEFAULT)
		style = GTK_HTML_FONT_STYLE_SIZE_3;

	size = style & GTK_HTML_FONT_STYLE_SIZE_MASK;
	if (inc && size < GTK_HTML_FONT_STYLE_SIZE_7)
		size++;
	else if (!inc && size > GTK_HTML_FONT_STYLE_SIZE_1)
		size--;

	style &= ~GTK_HTML_FONT_STYLE_SIZE_MASK;
	style |= size;

	return style;
}

static void
inc_dec_size_cb (HTMLObject *obj, gpointer data)
{
	if (html_object_is_text (obj))
		HTML_TEXT (obj)->font_style = inc_dec_size (HTML_TEXT (obj)->font_style, GPOINTER_TO_INT (data));
}

void
html_engine_font_size_inc_dec (HTMLEngine *e, gboolean inc)
{
	if (html_engine_is_selection_active (e))
		html_engine_cut_and_paste (e, (inc) ? "Increase font size" : "Decrease font size",
					   (GFunc) inc_dec_size_cb, GINT_TO_POINTER (inc));
	else
		e->insertion_font_style = inc_dec_size (e->insertion_font_style, inc);
}

static void
set_color (HTMLObject *o, HTMLColor *color)
{
	if (html_object_is_text (o))
		html_text_set_color (HTML_TEXT (o), NULL, color);
}

void
html_engine_set_color (HTMLEngine *e, HTMLColor *color)
{
	if (!color)
		color = html_colorset_get_color (e->settings->color_set, HTMLTextColor);

	if (html_engine_is_selection_active (e)) {
		html_engine_cut_and_paste_begin (e, "Set color");
		g_list_foreach (e->cut_buffer, (GFunc) set_color, color);
		html_engine_cut_and_paste_end (e);
	} else {
		html_color_unref (e->insertion_color);
		e->insertion_color = color;
		html_color_ref (e->insertion_color);
	}
}

/* URL/Target

   get actual url/target
*/

const gchar *
html_engine_get_url (HTMLEngine *e)
{
	return e->insertion_url;
}

const gchar *
html_engine_get_target (HTMLEngine *e)
{
	return e->insertion_target;
}

void
html_engine_set_url (HTMLEngine *e, const gchar *url)
{
	if (e->insertion_url)
		g_free (e->insertion_url);
	e->insertion_url = g_strdup (url);
}

void
html_engine_set_target (HTMLEngine *e, const gchar *target)
{
	if (e->insertion_target)
		g_free (e->insertion_target);
	e->insertion_target = g_strdup (target);
}

/* get url/target from document */

static const gchar *
get_url_or_target_from_selection (HTMLEngine *e, gboolean get_url)
{
	const gchar *str = NULL;
	gboolean backwards;
	HTMLObject *p;

	g_return_val_if_fail (e->clue != NULL, NULL);
	g_assert (html_engine_is_selection_active (e));
	g_assert (e->mark != NULL);

	backwards = (e->mark->position < e->cursor->position) ? TRUE : FALSE;

	p = e->cursor->object;
	while (1) {
		str = (get_url) ? html_object_get_url (p) : html_object_get_target (p);
		if (str) break;
		if (p == e->mark->object) break;
		p = (backwards) ? html_object_prev_leaf (p) : html_object_next_leaf (p);
		g_assert (p != NULL);
	}

	return str;
}

const gchar *
html_engine_get_document_url (HTMLEngine *e)
{
	if (html_engine_is_selection_active (e))
		return get_url_or_target_from_selection (e, TRUE);
	else
		return html_object_get_url (e->cursor->object);
}

const gchar *
html_engine_get_document_target (HTMLEngine *e)
{
	if (html_engine_is_selection_active (e))
		return get_url_or_target_from_selection (e, FALSE);
	else
		return html_object_get_target (e->cursor->object);
}

gboolean
html_engine_update_insertion_url_and_target (HTMLEngine *engine)
{
	const gchar *url, *target;
	gboolean retval = FALSE;

	url    = html_engine_get_document_url    (engine);
	target = html_engine_get_document_target (engine);

	if (url != engine->insertion_url) {
		html_engine_set_url (engine, url);
		retval = TRUE;
	}

	if (target != engine->insertion_target) {
		html_engine_set_target (engine, target);
		retval = TRUE;
	}

	return retval;
}
