// -*- c++ -*-
/* $Id: fileutils.cc,v 1.1 2002/03/04 18:34:04 daniel Exp $ */

/* fileutils.cc
 *
 * Copyright (C) 2002 The gtkmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glib/gdir.h>
#include <glib/gstrfuncs.h>
#include <glibmm/fileutils.h>
#include <glibmm/utility.h>


namespace Glib
{

/**** Glib::FileError ******************************************************/

FileError::FileError(GError* gobject)
:
  Error (gobject)
{}

GFileError FileError::code() const
{
  return static_cast<GFileError>(Error::code());
}


/**** Glib::Dir ************************************************************/

Dir::Dir(const std::string& path, unsigned int flags)
{
  GError* error = 0;
  gobject_ = g_dir_open(path.c_str(), flags, &error);

  if(error)
    Glib::Error::throw_exception(error);
}

Dir::Dir(GDir* gobject)
:
  gobject_ (gobject)
{}

Dir::~Dir()
{
  if(gobject_)
    g_dir_close(gobject_);
}

std::string Dir::read_name()
{
  const char *const name = g_dir_read_name(gobject_);
  return (name) ? std::string(name) : std::string();
}

void Dir::rewind()
{
  g_dir_rewind(gobject_);
}

void Dir::close()
{
  if(gobject_)
  {
    g_dir_close(gobject_);
    gobject_ = 0;
  }
}

DirIterator Dir::begin()
{
  g_dir_rewind(gobject_);
  return DirIterator(gobject_, g_dir_read_name(gobject_));
}

DirIterator Dir::end()
{
  return DirIterator(gobject_, 0);
}


/**** Glib::DirIterator ****************************************************/

DirIterator::DirIterator()
:
  gobject_ (0),
  current_ (0)
{}

DirIterator::DirIterator(GDir* gobject, const char* current)
:
  gobject_ (gobject),
  current_ (current)
{}

std::string DirIterator::operator*() const
{
  return (current_) ? std::string(current_) : std::string();
}

DirIterator& DirIterator::operator++()
{
  current_ = g_dir_read_name(gobject_);
  return *this;
}

void DirIterator::operator++(int)
{
  current_ = g_dir_read_name(gobject_);
}

bool DirIterator::operator==(const DirIterator& rhs) const
{
  return (current_ == rhs.current_);
}

bool DirIterator::operator!=(const DirIterator& rhs) const
{
  return (current_ != rhs.current_);
}


bool file_test(const std::string& filename, GFileTest test)
{
  return g_file_test(filename.c_str(), test);
}

int mkstemp(std::string& filename_template)
{
  char *const buf = g_strndup(filename_template.data(), filename_template.size());
  const int fileno = g_mkstemp(buf);

  filename_template = ScopedPtr<char>(buf).get();
  return fileno;
}

} // namespace Glib

