// -*- c++ -*-
/* $Id: clipboard.cc,v 1.2 2002/04/22 00:05:38 murrayc Exp $ */

/* stockid.cc
 *
 * Copyright (C) 1998-1999 The Gtk-- Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtkmm/clipboard.h>

namespace Gtk
{

//SignalProxy_GetClear:

//This Signal Proxy allows the C++ coder to specify a SigC::Slot instead of a static function.
class SignalProxy_GetClear
{
public:
  SignalProxy_GetClear(const Clipboard::SlotGet& slot_get, const Clipboard::SlotClear& slot_clear);
  ~SignalProxy_GetClear();

  static void gtk_callback_get(GtkClipboard* clipboard, GtkSelectionData* selection_data, guint info, gpointer data);
  static void gtk_callback_clear(GtkClipboard* clipboard, gpointer data);

protected:
  Clipboard::SlotGet slot_get_;
  Clipboard::SlotClear slot_clear_;
};

SignalProxy_GetClear::SignalProxy_GetClear(const Clipboard::SlotGet& slot_get, const Clipboard::SlotClear& slot_clear)
{
  slot_get_ = slot_get;
  slot_clear_ = slot_clear;
}

SignalProxy_GetClear::~SignalProxy_GetClear()
{
}

void SignalProxy_GetClear::gtk_callback_get(GtkClipboard* clipboard, GtkSelectionData* selection_data, guint info, gpointer data)
{
  SignalProxy_GetClear* pSignalProxy = static_cast<SignalProxy_GetClear*>(data);
  if(pSignalProxy)
  {
    //Use Slot::operator():
    (pSignalProxy->slot_get_)(selection_data, info);
    pSignalProxy->slot_get_ = Clipboard::SlotGet(); //The callback is only used once.
  }
}

void SignalProxy_GetClear::gtk_callback_clear(GtkClipboard* clipboard, gpointer data)
{
  SignalProxy_GetClear* pSignalProxy = static_cast<SignalProxy_GetClear*>(data);
  if(pSignalProxy)
  {
    //Use Slot::operator():
    (pSignalProxy->slot_clear_)();
    delete pSignalProxy; //The callbacks are only used once, and clear is called last.
  }
}

//SignalProxy_Received:

//This Signal Proxy allows the C++ coder to specify a SigC::Slot instead of a static function.
class SignalProxy_Received
{
public:
  typedef Clipboard::SlotReceived SlotType;
  SignalProxy_Received(const SlotType& slot);
  ~SignalProxy_Received();

  static void gtk_callback(GtkClipboard* clipboard, GtkSelectionData* selection_data, gpointer data);

protected:
  SlotType slot_;
};

SignalProxy_Received::SignalProxy_Received(const SlotType& slot)
{
  slot_ = slot;
}

SignalProxy_Received::~SignalProxy_Received()
{
}

void SignalProxy_Received::gtk_callback(GtkClipboard* clipboard, GtkSelectionData* selection_data, gpointer data)
{
  SignalProxy_Received* pSignalProxy = static_cast<SignalProxy_Received*>(data);
  if(pSignalProxy)
  {
    //Use Slot::operator():
    (pSignalProxy->slot_)(selection_data);
    delete pSignalProxy;
  }
}


//SignalProxy_TextReceived:

//This Signal Proxy allows the C++ coder to specify a SigC::Slot instead of a static function.
class SignalProxy_TextReceived
{
public:
  typedef Clipboard::SlotTextReceived SlotType;
  SignalProxy_TextReceived(const SlotType& slot);
  ~SignalProxy_TextReceived();

  static void gtk_callback(GtkClipboard* clipboard, const gchar* text, gpointer data);

protected:
  SlotType slot_;
};

SignalProxy_TextReceived::SignalProxy_TextReceived(const SlotType& slot)
{
  slot_ = slot;
}

SignalProxy_TextReceived::~SignalProxy_TextReceived()
{
}

void SignalProxy_TextReceived::gtk_callback(GtkClipboard* clipboard, const gchar* text, gpointer data)
{
  SignalProxy_TextReceived* pSignalProxy = static_cast<SignalProxy_TextReceived*>(data);
  if(pSignalProxy)
  {
    Glib::ustring strText;
    if(text)
      strText = text;

    //Use Slot::operator():
    (pSignalProxy->slot_)(strText);
    delete pSignalProxy;//The callback is only used once.
  }
}



Clipboard::Clipboard(GdkAtom selection)
: gobject_(gtk_clipboard_get(selection))
{}

Clipboard::Clipboard(GtkClipboard* gobject)
: gobject_(gobject)
{}

Clipboard::Clipboard(const Clipboard& src)
: gobject_(src.gobject_)
{}

Clipboard::~Clipboard()
{
}

Clipboard& Clipboard::operator=(const Clipboard& src)
{
  gobject_ = src.gobject_;
  return *this;
}

bool Clipboard::operator==(const Clipboard& src) const
{
  return gobject_ == src.gobject_;
}

GtkClipboard* Clipboard::gobj()
{
  return gobject_;
}

const GtkClipboard* Clipboard::gobj() const
{
  return gobject_;
}

bool Clipboard::set(const GtkTargetEntry* targets, guint n_targets, SlotGet slot_get, SlotClear slot_clear)
{
  //Create a signal proxy. A pointer to this will be passed through the callback's data parameter.
  SignalProxy_GetClear* pSignalProxy = new SignalProxy_GetClear(slot_get, slot_clear);

  return gtk_clipboard_set_with_data(gobj(), targets, n_targets,
    &SignalProxy_GetClear::gtk_callback_get, &SignalProxy_GetClear::gtk_callback_clear, (gpointer)pSignalProxy);
}

Glib::RefPtr<Glib::Object> Clipboard::get_owner()
{
  return Glib::wrap(gtk_clipboard_get_owner(gobj()), true); //true = take_copy.
}

void Clipboard::clear()
{
  gtk_clipboard_clear(gobj());
}

void Clipboard::set_text(const Glib::ustring& text)
{
  gtk_clipboard_set_text(gobj(), text.c_str(), text.size());
}

void Clipboard::request_contents(GdkAtom target, SlotReceived slot)
{
  //Create a signal proxy. A pointer to this will be passed through the callback's data parameter.
  SignalProxy_Received* pSignalProxy = new SignalProxy_Received(slot);

  gtk_clipboard_request_contents(gobj(), target,
    &SignalProxy_Received::gtk_callback, (gpointer)pSignalProxy);
}

void Clipboard::request_text(SlotTextReceived slot)
{
  //Create a signal proxy. A pointer to this will be passed through the callback's data parameter.
  SignalProxy_TextReceived* pSignalProxy = new SignalProxy_TextReceived(slot);

  gtk_clipboard_request_text(gobj(),
    &SignalProxy_TextReceived::gtk_callback, (gpointer)pSignalProxy);
}

GtkSelectionData* Clipboard::wait_for_contents(GdkAtom target)
{
  return gtk_clipboard_wait_for_contents(gobj(), target);
}

Glib::ustring Clipboard::wait_for_text()
{
  return Glib::convert_return_gchar_ptr_to_ustring(gtk_clipboard_wait_for_text(gobj()));
}

bool Clipboard::wait_is_text_available()
{
   return (gboolean)gtk_clipboard_wait_is_text_available(gobj());
}

} /* namespace Gtk */

