// -*- c++ -*-
#include <glibmm/object.h>
#include <glibmm/private/object_p.h>
// -*- c++ -*-
/* $Id: object.cc,v 1.30 2002/04/02 13:24:00 daniel Exp $ */

/* 
 *
 * Copyright 1998-1999 The Gtk-- Development Team
 * Copyright 2001      Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdarg.h>
#include <glib-object.h>
//#include <gtkmm/main.h>

//Thoughts:
//Q. Should Object be a shared-reference smart pointer, so that we can do this: Object object2 = object1; ?
//A. Probably not. It would be more natural to do Object* pObject2 = pObject1 and do your own appropriate memory management.
//   You could still use a generic shared-reference smart pointer if you like, but that should be orthogonal.
//   e.g.  shared_ptr<Object> refObject1(...);  shared_ptr<Object> refObject2 = refObject1;
//   We could recommend the use of boost::shared_ptr<> for this.

//Thoughts:
//Q. Can we allow a C++ Glib::Object to be destroyed?
//A. Apparently not. The C++ wrapper needs to live as long as the underlying C instance, because the underlying instance uses our signal handlers.
//   And we can't just allow premature destruction, because GObject has no destroy signal so users of the objects can not respond to their destruction intelligently.

//Weak references:
//I'm not sure what the point of these are apart from being a hacky way out of circular references,
//but maybe we could make it easier to use them by making a Java Reference Object -style class like so:
// Glib::WeakRef<SomeDerivedObject> weakrefSomeObject(object1);
// ...
// if(weakrefSomeObject->isStillAlive())
// {
//   weakrefSomeObject->some_method();
// }
// else
// {
//   //Deal with it, maybe recreating the object.
// }
//
// Without this, the coder has to define his own signal handler which sets his own isStillAlive boolean.
// weakrefSomeObject<> could still have its own signal_destroyed signal so that coders can choose to deal
// with the destruction as soon as it happens instead of just checking later before they try to use it.

//TODO: Properties.

namespace Glib
{

/*
void Object::glibmm_sigsegv(const char *pos) {
    char c[10240];
    sprintf(c,"glibmm_sigsegv(%s): assertion failed inside glibmm. Please make a stack dump with gdb and send it to glibmm-main@sourceforge.net\n", pos);
    g_warning(c);
    char *f=0;
    *f=0; // sigsegv.
  };
*/


/* Object_Class */

GType Object_Class::get_type()
{
  if (!gtype_) //Create the GType if necessary.
  {
    //Make sure that the parent type has been created:
    //CppClassParent::CppObjectType::get_type();

    GTypeInfo info =
      {
        sizeof(BaseClassType),
        (GBaseInitFunc) 0,
        (GBaseFinalizeFunc) 0,
        (GClassInitFunc) class_init_function,
        0,           /* class_finalize */
        0,           /* class_data */
        sizeof (BaseObjectType),
        0,              /* n_preallocs */
        (GInstanceInitFunc) object_init_function,
        0 /* value_table */
      };

    gtype_ = g_type_register_static(G_TYPE_OBJECT, "glibmm__Object", &info, GTypeFlags(0));
  }

  return gtype_;
}

void Object_Class::class_init_function(GObjectClass* klass)
{}

void Object_Class::object_init_function(GObject* gobject)
{}

Object* Object_Class::wrap_new(GObject* object)
{
  return new Object(object);
}


/* Object */

Object::CppClassType Object::object_class;

Object::Object()
{
  //This constructor is ONLY for derived classes that are NOT wrappers of derived C objects.
  //For instance, Gtk::Object should NOT use this constructor.
  g_warning("Object::Object(): Did you really mean to call this?");
  //TODO: Remove the g_warning()

  //Set gobject_, because we can't call an InstanceBase non-default constructor, because it's a virtual base class.
  //Store a pointer to this wrapper in the underlying instance,
  //so that we never create a second wrapper for the same underlying instance.
  //Also, specify a callback that will tell us when it's time to delete this C++ wrapper instance:
  ObjectBase::initialize(static_cast<GObject*>(g_object_new(G_TYPE_OBJECT, 0)));
}

Object::Object(GObject *castitem)
//: ObjectBase(castitem)  //This doesn't work with virtual base classes - see the comment in ObjectBase().
{
  //Set gobject_, because we can't call an InstanceBase non-default constructor, because it's a virtual base class.
  //Store a pointer to this wrapper in the underlying instance,
  //so that we never create a second wrapper for the same underlying instance.
  //Also, specify a callback that will tell us when it's time to delete this C++ wrapper instance:
  ObjectBase::initialize(castitem);
}

Object::~Object()
{
  cpp_destruction_in_progress_ = true;

  // remove our hook.
  GObject* object = gobject_;
  gobject_ = 0;
  if (object)
  {
    g_object_steal_qdata(object, quark_); //Remove the pointer to the wrapper from the underlying instance. See the constructor.
    gobject_ = 0; //We can't use it anymore. The RefPtr will do the unref-ing.
  }
}

RefPtr<Object> Object::create()
{
  return RefPtr<Object>(); //Derived classes will actually return RefPtr<>s that contain useful instances.
}

GType Object::get_type()
{
  return object_class.get_type();
}

GType Object::get_base_type()
{
  return G_TYPE_OBJECT;
}

// Data services
void* Object::get_data(const QueryQuark& id)
{
  return g_object_get_qdata(gobj(),id);
}

void Object::set_data(const Quark& id, void* data)
{
  g_object_set_qdata(gobj(),id,data);
}

void Object::set_data(const Quark& id, void* data, DestroyNotify destroy)
{
  g_object_set_qdata_full(gobj(), id, data, destroy);
}

void Object::remove_data(const QueryQuark& id)
{
  // missing in glib??
  g_return_if_fail(id.id() > 0);
  g_datalist_id_remove_data(&gobj()->qdata,id);
}

void* Object::steal_data(const QueryQuark& id)
{
  return g_object_steal_qdata(gobj(), id);
}

/*
void Object::set_manage()
{
  //Glib::Object can not be dynamic because it lacks a float state.
}
*/

/*
GObject* Object::gobj_copy()
{
  reference();
  return gobj();
}
*/

Glib::RefPtr<Object> Object::wrap_specific_type(GObject* gobject, bool take_copy /* = false */) //static
{
  Glib::RefPtr<Object> refPtr;
  //Check for an existing wrapper:
  Object* pCppObject = dynamic_cast<Object*>(_get_current_wrapper(gobject));
  if(pCppObject)
  {
    //Return the existing wrapper:
    refPtr = Glib::RefPtr<Object>(pCppObject);
  }
  else
  {
    //Create a new wrapper:
    refPtr = Glib::RefPtr<Object>( new Object(gobject) );
  }

  if(take_copy && refPtr)
    refPtr->reference();

  return refPtr;
}



} /* namespace Glib */


