// -*- c++ -*-
/* $Id: dispatcher.cc,v 1.3 2002/06/16 22:08:39 daniel Exp $ */

/* Copyright 2002 Free Software Foundation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <glibmm/dispatcher.h>
#include <glibmm/exceptionhandler.h>
#include <glibmm/fileutils.h>
#include <glibmm/main.h>
#include <glibmm/thread.h>
#include <sigc++/class_slot.h>

#include <cerrno>
#include <glib.h>
#include <fcntl.h>

#ifndef G_OS_WIN32
#include <unistd.h>
#else
/* This is what glib/gspawn-win32.c includes, I've no idea whether it works :) */
#include <windows.h>
#include <io.h>
#include <direct.h>
#endif /* G_OS_WIN32 */


namespace
{

struct DispatchNotifyData
{
  unsigned long           tag;
  Glib::Dispatcher*       dispatcher;
  Glib::DispatchNotifier* notifier;
};

void warn_failed_pipe_io(const char* what, int err_no)
{
  g_warning("Error in inter-thread communication: %s() failed: %s", what, g_strerror(err_no));
}

/* Try to set the close-on-exec flag of the file descriptor,
 * so that it won't be leaked if a new process is spawned.
 */
void fd_set_close_on_exec(int fd)
{
  const int flags = fcntl(fd, F_GETFD, 0);
  g_return_if_fail(flags >= 0);

  fcntl(fd, F_SETFD, flags | FD_CLOEXEC);
}

/* One word: paranoia.
 */
void fd_close_and_invalidate(int& fd)
{
  if(fd >= 0)
  {
    int result;

    do { result = close(fd); }
    while(result < 0 && errno == EINTR);

    if(result < 0)
      warn_failed_pipe_io("close", errno);

    fd = -1;
  }
}

} // anonymous namespace


namespace Glib
{

class DispatchNotifier
{
public:
  ~DispatchNotifier();

  static DispatchNotifier* reference_instance();
  static void unreference_instance(DispatchNotifier* notifier);

  void send_notification(Dispatcher* dispatcher);
  bool pipe_io_handler(Glib::IOCondition condition);

protected:
  // Only used by reference_instance().  Should be private, but that triggers
  // a silly gcc warning even though DispatchNotifier has static methods.
  DispatchNotifier();

private:
  static Glib::StaticPrivate<DispatchNotifier> thread_specific_instance_;

  int               ref_count_;
  int               fd_receiver_;
  int               fd_sender_;
  SigC::Connection  conn_io_handler_;

  void create_pipe();

  // noncopyable
  DispatchNotifier(const DispatchNotifier&);
  DispatchNotifier& operator=(const DispatchNotifier&);
};


/**** Glib::DispatchNotifier ***********************************************/

Glib::StaticPrivate<DispatchNotifier>
DispatchNotifier::thread_specific_instance_ = GLIBMM_STATIC_PRIVATE_INIT;

DispatchNotifier::DispatchNotifier()
:
  ref_count_    (0),
  fd_receiver_  (-1),
  fd_sender_    (-1)
{
  create_pipe();

  conn_io_handler_ = Glib::signal_io().connect(
      SigC::slot_class(*this, &DispatchNotifier::pipe_io_handler),
      fd_receiver_, Glib::IO_IN);
}

DispatchNotifier::~DispatchNotifier()
{
  conn_io_handler_.disconnect();

  fd_close_and_invalidate(fd_sender_);
  fd_close_and_invalidate(fd_receiver_);
}

void DispatchNotifier::create_pipe()
{
  int filedes[2] = { -1, -1 };

  if(pipe(filedes) < 0)
  {
    GError *const error = g_error_new(
        G_FILE_ERROR, g_file_error_from_errno(errno),
        "Failed to create pipe for inter-thread communication: %s", g_strerror(errno));

    throw Glib::FileError(error);
  }

  fd_receiver_ = filedes[0];
  fd_sender_   = filedes[1];

  fd_set_close_on_exec(fd_receiver_);
  fd_set_close_on_exec(fd_sender_);
}

// static
DispatchNotifier* DispatchNotifier::reference_instance()
{
  DispatchNotifier* instance = thread_specific_instance_.get();

  if(!instance)
  {
    instance = new DispatchNotifier();
    thread_specific_instance_.set(instance);
  }

  ++instance->ref_count_; // initially 0

  return instance;
}

// static
void DispatchNotifier::unreference_instance(DispatchNotifier* notifier)
{
  DispatchNotifier *const instance = thread_specific_instance_.get();

  // Yes, the notifier argument is only used to check for sanity.
  g_return_if_fail(instance == notifier);

  if(--instance->ref_count_ <= 0)
  {
    g_return_if_fail(instance->ref_count_ == 0); // could be < 0 if messed up

    // This will cause deletion of the notifier object.
    thread_specific_instance_.set(0);
  }
}

void DispatchNotifier::send_notification(Dispatcher* dispatcher)
{
  DispatchNotifyData data =
  {
    0xdeadbeef,
    dispatcher,
    this
  };

  gsize n_written = 0;

  // Look at this code -- we're prepared for the worst.  Hah!
  do
  {
    void * const buffer = reinterpret_cast<guint8*>(&data) + n_written;
    const gssize result = write(fd_sender_, buffer, sizeof(data) - n_written);

    if(result < 0)
    {
      if(errno == EINTR)
        continue;

      warn_failed_pipe_io("write", errno);
      return;
    }

    n_written += result;
  }
  while(n_written < sizeof(data));
}

bool DispatchNotifier::pipe_io_handler(Glib::IOCondition)
{
  DispatchNotifyData data = { 0, };
  gsize n_read = 0;

  // Look at this code -- we're prepared for the worst.  Hah!
  do
  {
    void * const buffer = reinterpret_cast<guint8*>(&data) + n_read;
    const gssize result = read(fd_receiver_, buffer, sizeof(data) - n_read);

    if(result < 0)
    {
      if(errno == EINTR)
        continue;

      warn_failed_pipe_io("read", errno);
      return true;
    }

    n_read += result;
  }
  while(n_read < sizeof(data));

  g_return_val_if_fail(data.tag == 0xdeadbeef, true);
  g_return_val_if_fail(data.notifier == this, true);

  // Actually, we wouldn't need the try/catch block because the Glib::Source
  // C callback already does it for us.  However, we do it anyway because the
  // default return value is 'false', which is not what we want.
  try
  {
    data.dispatcher->signal_();
  }
  catch(...)
  {
    Glib::exception_handlers_invoke();
  }

  return true;
}


/**** Glib::Dispatcher *****************************************************/

Dispatcher::Dispatcher()
:
  signal_   (),
  notifier_ (DispatchNotifier::reference_instance())
{}

Dispatcher::~Dispatcher()
{
  DispatchNotifier::unreference_instance(notifier_);
}

void Dispatcher::emit()
{
  notifier_->send_notification(this);
}

SigC::Connection Dispatcher::connect(const SigC::Slot0<void>& slot)
{
  return signal_.connect(slot);
}

} // namespace Glib

