/*
 *  Copyright (C) 1999-2001 Bruno Pires Marinho
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include "gtm.h"
#include "file-data.h"
#include "file-list.h"
#include "wget-log.h"
#include "download-info.h"
#include "log.h"

/* Function to convert the wget notation of file size (55,449,600) */ 
static guint32 
convert_wget_size (char *size)
{
   char *p = size;

   while (*p != ' ') {
      if (*p == ',') {
	 while (*p != ' ')
	    *p++ = *(p+1);
	 p = size;
      } else
	 p++;
   }
   return atoi (size);
}

static void
show_error (FileData *file_data, gchar *error_msg)
{
    gchar *message;

    /* Before showing the error message disable auto-download. If we don't do
     * this auto download retries to download the file again and with an error.
     */
    file_data_set_use_auto_dl (file_data, FALSE);
    message = g_strconcat (error_msg, "\t", file_data->url, NULL);
    log_insert_entry (message);
    g_free (message);
}


static int
wget_log_process_line (FileData *file_data)
{
    gchar *p;
    struct stat file_stat;

    switch (file_data->state) {
	case DL_NOT_CONNECTED:
	    /* First check to see if connected to the host correctly */
	    if (strstr (file_data->line, "connected!") != NULL) {
		file_data_set_state (file_data, DL_CONNECTED, TRUE);
		break;
	    }

	    /* We are not connected to the host so we must find the problem */
	    if (strncmp (file_data->line, "--", 2) == 0 ||
		strncmp (file_data->line, "  ", 2) == 0 ||
		strncmp (file_data->line, "Connecting to ", 14) == 0)
		break;
	    else if (strncmp (file_data->line, "socket: ", 8) == 0)
		show_error (file_data,
                            _ ("Socket error when downloading URL:\n"));
	    else if (strncmp (file_data->line, "Connection to ", 14) == 0)
		show_error (file_data,
                            _ ("Connection refused when downloading URL:\n"));
	    else if (strstr (file_data->line, "No route to host") != NULL)
		show_error (file_data,
                            _ ("No route to host:\n"));
	    else if (strncmp (file_data->line, "connect: ", 9) == 0)
		show_error (file_data,
                            _ ("Connection refused when downloading URL:\n"));
	    else if (strstr (file_data->line, "Host not found.") != NULL)
		show_error (file_data,
                            _ ("Host not found when downloading URL:\n"));
	    else
		show_error (file_data,
                            _ ("Unknown error when downloading URL:\n"));
	    kill (file_data->wget_pid, SIGKILL);
	    return 1;
	    break;

	case DL_CONNECTED:
	    /* File not found for FTP */
	    if (strncmp (file_data->line, "No such file ", 13) == 0) {
		show_error (file_data,
                            _ ("File not found when downloading URL:\n"));
		break;
	    }

	    /* File not found for HTTP or Proxy */
	    if (strstr (file_data->line, "ERROR") != NULL) {
		show_error (file_data, 
                            _ ("File not found when downloading URL:\n"));
		break;
	    }

	    /* Get the leght of file being downloaded */
	    p = strstr (file_data->line, "Length: ");
	    if (p != NULL) {
		p += 8;
                /* Only set the total size of we don't have one yet. */
                if (file_data->total_size == 0)
                    file_data_set_total_size (file_data,
                                              convert_wget_size (p), TRUE);
		file_data_set_state (file_data, DL_RETRIEVING, TRUE);
		/* Get session start time and session file start size */
		if (stat (file_data->local_filename, &file_stat) != -1) {
		    file_data->session_start_time = file_stat.st_ctime;
		    file_data->session_start_size = file_stat.st_size;
		} else {
		    file_data->session_start_time = 0;
		    file_data->session_start_size = 0;
		}
		file_data->session_elapsed = 0;
	    } else {
                /* We didn't get a length so, probably it's unspecified size
                   so get the start of download by trying to catch the
                   string "K ->" */
                p = strstr (file_data->line, "K -> ");
                if (p != NULL) {
                    /* Unspecified size, so set total_size to 0 */
                    file_data_set_total_size (file_data, 0, TRUE);
                    file_data_set_state (file_data, DL_RETRIEVING, TRUE);
                    if (stat (file_data->local_filename, &file_stat) != -1) {
                        file_data->session_start_time = file_stat.st_ctime;
                        file_data->session_start_size = file_stat.st_size;
                    } else {
                        file_data->session_start_time = 0;
                        file_data->session_start_size = 0;
                    }
                }
                file_data->session_elapsed = 0;
            }
	    break;

	default:
	    break;
    }
    return 0;
}

void
wget_log_process (FileData *file_data)
{
    gchar buffer[MAX_WGET_LINE_SIZE * 2 + 1];
    gint total_read, pos, res;

    do {
	total_read = read (file_data->log_fd, buffer, MAX_WGET_LINE_SIZE * 2);
	pos = 0;
	while (pos < total_read) {
	    if (buffer[pos] != '\n') {
		if (file_data->line_pos < MAX_WGET_LINE_SIZE)
		    file_data->line[file_data->line_pos++] = buffer[pos++];
		else
		    pos++;
	    } else {
		file_data->line[file_data->line_pos] = '\0';
		res = wget_log_process_line (file_data);
		if (res == 1)
                    break;
		file_data->line_pos = 0;
		pos++;
	    }
	}
	if (file_data->state == DL_RETRIEVING)
	    file_data_update_statistics (file_data, TRUE);
    } while (total_read == MAX_WGET_LINE_SIZE * 2);
}
