/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Johan Dahlin <jdahlin@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <glade/glade.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtktreeviewcolumn.h>
#include <gtk/gtktreeselection.h>
#include <libgnome/gnome-i18n.h>
#include "song-list-view.h"
#include "id3-tag.h"
#include "eggtreemultidnd.h"
#include "utils.h"

static void         song_list_view_class_init  (SongListViewClass *klass);
static void         song_list_view_init        (SongListView      *view);

static void         song_title_func            (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_rating_func           (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_year_func             (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_time_func             (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_artist_func           (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_album_func            (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_genre_func            (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_play_count_func       (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_last_played_func      (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_track_number_func     (GtkTreeViewColumn *tree_column,
						GtkCellRenderer   *cell,
						GtkTreeModel      *tree_model,
						GtkTreeIter       *iter,
						gpointer           data);
static void         song_column_clicked_cb     (GtkTreeViewColumn *column,
						SongListView      *view);


GType
song_list_view_get_type (void)
{
  static GType type = 0;
	
  if (!type)
    {
      static const GTypeInfo info =
	{
	  sizeof (SongListViewClass),
	  NULL,           /* base_init */
	  NULL,           /* base_finalize */
	  (GClassInitFunc) song_list_view_class_init,
	  NULL,           /* class_finalize */
	  NULL,           /* class_data */
	  sizeof (SongListView),
	  0,
	  (GInstanceInitFunc) song_list_view_init,
	};

      type = g_type_register_static (GTK_TYPE_TREE_VIEW, "SongListView",
				     &info, 0);
    }

  return type;
}

static void
song_list_view_class_init (SongListViewClass *klass)
{
}

static GtkTreeViewColumn *
setup_column (SongListView        *song_view,
	      const char          *title,
	      GtkTreeCellDataFunc  func,
	      int                  sort_id,
	      gboolean             pack_start,
	      gboolean             visible)
{
  GtkTreeView       *view;
  GtkCellRenderer   *cell;
  GtkTreeViewColumn *column;

  view = GTK_TREE_VIEW (song_view);
  
  cell = gtk_cell_renderer_text_new ();
  gtk_cell_renderer_text_set_fixed_height_from_font (GTK_CELL_RENDERER_TEXT (cell), 1);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, title);

  if (pack_start)
    gtk_tree_view_column_pack_start (column, cell, TRUE);
  else
    gtk_tree_view_column_pack_end (column, cell, TRUE);

  gtk_tree_view_column_set_cell_data_func (column,
                                           cell,
					   func,
					   NULL,
					   NULL);
  gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
  gtk_tree_view_column_set_resizable (column, TRUE);

  gtk_tree_view_column_set_clickable (column, TRUE);
  g_object_set_data (G_OBJECT (column), "sort_id", GINT_TO_POINTER (sort_id));
  g_signal_connect (column,
		    "clicked",
		    G_CALLBACK (song_column_clicked_cb),
		    view);
  
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), column);

  gtk_tree_view_column_set_visible (column, visible);

  return column;
}

static gboolean
drag_drop_cb (GtkWidget      *widget,
	      GdkDragContext *drag_context,
	      int             x,
	      int             y,
	      guint           time,
	      gpointer        user_data)
{
  g_signal_stop_emission_by_name (widget, "drag_drop");

  return TRUE;
}

static const GtkTargetEntry drag_types[] = {
  {  "x-special/jamboree-song-list", 0, 0 }
};

static void
song_list_view_init (SongListView *view)
{
  GtkTreeModel *model;
  GtkTreeSelection *selection;
  
  model = song_list_model_new ();

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (view), TRUE);
  gtk_tree_view_set_model (GTK_TREE_VIEW (view), model);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (view), TRUE);
  egg_tree_multi_drag_add_drag_support (GTK_TREE_VIEW (view));

  gtk_tree_view_enable_model_drag_source (GTK_TREE_VIEW (view),
					  GDK_BUTTON1_MASK,
					  drag_types,
					  G_N_ELEMENTS (drag_types),
					  GDK_ACTION_COPY);

  g_signal_connect (view,
		    "drag_drop",
		    G_CALLBACK (drag_drop_cb),
		    view);
  
  setup_column (view, _("#"), song_track_number_func, SONG_SORT_ID_TRACK_NUMBER, TRUE, TRUE);
  view->play_column = setup_column (view, _("Title"), song_title_func, SONG_SORT_ID_TITLE, FALSE, TRUE);
  setup_column (view, _("Artist"), song_artist_func, SONG_SORT_ID_ARTIST, TRUE, TRUE);
  setup_column (view, _("Album"), song_album_func, SONG_SORT_ID_ALBUM, TRUE, TRUE);
  setup_column (view, _("Genre"), song_genre_func, SONG_SORT_ID_GENRE, TRUE, FALSE);
  setup_column (view, _("Length"), song_time_func, SONG_SORT_ID_TIME, TRUE, TRUE);
  setup_column (view, _("Year"), song_year_func, SONG_SORT_ID_YEAR, TRUE, FALSE);
  setup_column (view, _("Rating"), song_rating_func, SONG_SORT_ID_RATING, TRUE, FALSE);
  setup_column (view, _("Play Count"), song_play_count_func, SONG_SORT_ID_PLAY_COUNT, TRUE, FALSE);
  setup_column (view, _("Last Played"), song_last_played_func, SONG_SORT_ID_LAST_PLAYED, TRUE, FALSE);

  view->play_cell = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (view->play_column, view->play_cell, FALSE);
  
  song_list_model_set_sorting (SONG_LIST_MODEL (model),
			       SONG_SORT_ID_DEFAULT,
			       GTK_SORT_ASCENDING);  
}

static void
song_title_func (GtkTreeViewColumn *tree_column,
		 GtkCellRenderer   *cell,
		 GtkTreeModel      *tree_model,
		 GtkTreeIter       *iter,
		 gpointer           data)
{
  Song *song;
  
  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
	
  g_object_set (cell, "text", song->title, NULL);
}

static void
song_artist_func (GtkTreeViewColumn *tree_column,
		  GtkCellRenderer   *cell,
		  GtkTreeModel      *tree_model,
		  GtkTreeIter       *iter,
		  gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
	
  g_object_set (cell, "text", song->artist, NULL);
}

static void
song_album_func (GtkTreeViewColumn *tree_column,
		 GtkCellRenderer   *cell,
		 GtkTreeModel      *tree_model,
		 GtkTreeIter       *iter,
		 gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
	
  g_object_set (cell, "text", song->album, NULL);
}

static void
song_genre_func (GtkTreeViewColumn *tree_column,
		 GtkCellRenderer   *cell,
		 GtkTreeModel      *tree_model,
		 GtkTreeIter       *iter,
		 gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
	
  g_object_set (cell, "text", id3_genres[song->genre], NULL);
}

static void
song_year_func (GtkTreeViewColumn *tree_column,
		GtkCellRenderer   *cell,
		GtkTreeModel      *tree_model,
		GtkTreeIter       *iter,
		gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);

  if (song->year > 0)
    {
      char buf[8];
      snprintf (buf, 7, "%d", song->year);
      g_object_set (cell, "text", buf, NULL);
    }
  else
    g_object_set (cell, "text", NULL, NULL);
}

static void
song_time_func (GtkTreeViewColumn *tree_column,
		GtkCellRenderer   *cell,
		GtkTreeModel      *tree_model,
		GtkTreeIter       *iter,
		gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);

  if (song->length > -1)
    {
      char *buf = format_time_for_display (song->length);
      g_object_set (cell, "text", buf, NULL);
      g_free (buf);
    }
  else
    g_object_set (cell, "text", NULL, NULL);
}

static void
song_rating_func (GtkTreeViewColumn *tree_column,
		  GtkCellRenderer   *cell,
		  GtkTreeModel      *tree_model,
		  GtkTreeIter       *iter,
		  gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);

  /* Doesn't do anything yet. */
  
  g_object_set (cell, "text", "", NULL);
}

static void
song_play_count_func (GtkTreeViewColumn *tree_column,
		      GtkCellRenderer   *cell,
		      GtkTreeModel      *tree_model,
		      GtkTreeIter       *iter,
		      gpointer           data)
{
  Song *song;
  char buf[8];
  
  song = song_list_model_get_song ((SongListModel *) tree_model, iter);

  snprintf (buf, 7, "%d", song->play_count);
  
  g_object_set (cell, "text", buf, NULL);
}

static void
song_last_played_func (GtkTreeViewColumn *tree_column,
		       GtkCellRenderer   *cell,
		       GtkTreeModel      *tree_model,
		       GtkTreeIter       *iter,
		       gpointer           data)
{
  Song *song;
  
  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
  
  /* Doesn't do anything yet. */
  
  g_object_set (cell, "text", "", NULL);
}

static void
song_track_number_func (GtkTreeViewColumn *tree_column,
			GtkCellRenderer   *cell,
			GtkTreeModel      *tree_model,
			GtkTreeIter       *iter,
			gpointer           data)
{
  Song *song;

  song = song_list_model_get_song ((SongListModel *) tree_model, iter);
  
  if (song->track_number != -1)
    {
      char buf[4];
      snprintf (buf, 4, "% 2d", song->track_number);
      g_object_set (cell, "text", buf, NULL);
    }
  else
    g_object_set (cell, "text", NULL, NULL);
}

static void
song_column_clicked_cb (GtkTreeViewColumn *column,
			SongListView      *view)
{
  GtkTreeModel *model;
  GtkSortType order;
  SongSortId id;
  gboolean sorted;
  GList *columns, *l;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (view));

  sorted = gtk_tree_view_column_get_sort_indicator (column);

  columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (view));
  for (l = columns; l; l = l->next)
      gtk_tree_view_column_set_sort_indicator (l->data, FALSE); 
  
  id = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "sort_id"));

  if (sorted)
    {
      order = gtk_tree_view_column_get_sort_order (column);
      if (order == GTK_SORT_ASCENDING)
	order = GTK_SORT_DESCENDING;
      else
	order = -1;
    }
  else
    order = GTK_SORT_ASCENDING;

  if (order != -1)
    {
      gtk_tree_view_column_set_sort_indicator (column, TRUE);
      gtk_tree_view_column_set_sort_order (column, order);
  
      song_list_model_set_sorting (SONG_LIST_MODEL (model), id, order);
    }
  else
    song_list_model_set_sorting (SONG_LIST_MODEL (model),
				 SONG_SORT_ID_DEFAULT,
				 GTK_SORT_ASCENDING);
}

GtkWidget *
song_list_view_new (void)
{
  return g_object_new (TYPE_SONG_LIST_VIEW, NULL);
}

#if 0
static SongSortId
song_sort_id_from_string (const char * string)
{
  if (!strcmp (string, "title"))
    return SONG_SORT_ID_TITLE;
  else if (!strcmp (string, "artist"))
    return SONG_SORT_ID_ARTIST;
  else if (!strcmp (string, "time"))
    return SONG_SORT_ID_TIME;
  else if (!strcmp (string, "album"))
    return SONG_SORT_ID_ALBUM;
  else if (!strcmp (string, "genre"))
    return SONG_SORT_ID_GENRE;
  else if (!strcmp (string, "year"))
    return SONG_SORT_ID_YEAR;
  else if (!strcmp (string, "rating"))
    return SONG_SORT_ID_RATING;
  else if (!strcmp (string, "track_number"))
    return SONG_SORT_ID_TRACK_NUMBER;
  else if (!strcmp (string, "last_played"))
    return SONG_SORT_ID_LAST_PLAYED;
  else if (!strcmp (string, "play_count"))
    return SONG_SORT_ID_PLAY_COUNT;
  else
    {
      g_assert_not_reached ();
      return 0;
    }
}
#endif

const char *
song_sort_id_to_string (SongSortId sort_id)
{
  switch (sort_id)
    {
    case SONG_SORT_ID_TITLE:
      return "title";
    case SONG_SORT_ID_ARTIST:
      return "artist";
    case SONG_SORT_ID_TIME:
      return "time";
    case SONG_SORT_ID_ALBUM:
      return "album";
    case SONG_SORT_ID_GENRE:
      return "genre";
    case SONG_SORT_ID_YEAR:
      return "year";
    case SONG_SORT_ID_RATING:
      return "rating";
    case SONG_SORT_ID_TRACK_NUMBER:
      return "track_number";
    case SONG_SORT_ID_LAST_PLAYED:
      return "last_played";
    case SONG_SORT_ID_PLAY_COUNT:
      return "play_count";
      
    default:
      g_assert_not_reached();
      break;
    }
  
  return NULL;
}

static GtkWidget *
get_column_by_sort_id (SongListView *view, SongSortId  sort_id)
{
  GList *columns, *l;
  GtkWidget *column;
  gpointer data;
  
  columns = gtk_tree_view_get_columns (GTK_TREE_VIEW (view));

  for (l = columns; l; l = l->next)
    {
      column = l->data;
      data = g_object_get_data (G_OBJECT (column), "sort_id");
      if (GPOINTER_TO_INT (data) == sort_id)
	return column;
    }

  g_assert_not_reached ();
  
  return NULL;
}

static void
toggle_column_cb (GtkToggleButton   *button,
		  GtkTreeViewColumn *column)
{
  gboolean state;
  SongSortId sort_id;
  const char *sort_key;
  char *key;
  
  state = gtk_toggle_button_get_active (button);
  gtk_tree_view_column_set_visible (column, state);

  sort_id = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (column), "sort_id"));
  sort_key = song_sort_id_to_string (sort_id);

  key = g_strdup_printf ("/apps/jamboree/columns/%s", sort_key);
  gconf_client_set_bool (gconf_client, key, state, NULL);
  g_free (key);
}

static void
hookup_column (SongListView      *view,
	       GtkTreeViewColumn *column,
	       GladeXML          *glade,
	       const char        *name)
{
  GtkWidget *w;

  w = glade_xml_get_widget (glade, name);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w), gtk_tree_view_column_get_visible (column));
  
  g_signal_connect (w,
		    "toggled",
		    G_CALLBACK (toggle_column_cb),
		    column);
}

void
song_list_view_setup_columns (SongListView *view)
{
  int sort_id;
  char *key;
  GError *error = NULL;
  gboolean state;
  GtkWidget *column;

  for (sort_id = SONG_SORT_ID_DEFAULT + 1; sort_id < SONG_SORT_ID_NO_COLUMNS; sort_id++)
    {
      key = g_strdup_printf ("/apps/jamboree/columns/%s",
			     song_sort_id_to_string (sort_id));
      
      state = gconf_client_get_bool (gconf_client, key, &error);
      g_free (key);
      if (error)
	{
	  g_error_free (error);
	  continue;
	}
      
      column = get_column_by_sort_id (view, sort_id);

      gtk_tree_view_column_set_visible (GTK_TREE_VIEW_COLUMN (column), state);
    }
}

void
song_list_view_column_chooser (SongListView *view,
			       GtkWindow    *parent)
{
  GladeXML *glade;
  static GtkWidget *window = NULL;
  GList *l;
  
  if (window)
    {
      if (parent)
	gtk_window_set_transient_for (GTK_WINDOW (window), parent);
      gtk_window_present (GTK_WINDOW (window));
      return;
    }
  
  glade = glade_xml_new (DATADIR "/jamboree/column-chooser.glade", NULL, NULL);

  window = glade_xml_get_widget (glade, "column_chooser");
  if (parent)
    gtk_window_set_transient_for (GTK_WINDOW (window), parent);
      
  g_object_add_weak_pointer (G_OBJECT (window), (gpointer) &window);

  l = gtk_tree_view_get_columns (GTK_TREE_VIEW (view));
  /*hookup_column (view, l->data, glade, "playing_checkbutton");*/

  hookup_column (view, l->data, glade, "track_number_checkbutton");

  l = l->next;
  /*hookup_column (view, l->data, glade, "title_checkbutton");*/

  l = l->next;
  hookup_column (view, l->data, glade, "artist_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "album_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "genre_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "length_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "year_checkbutton");
  
  l = l->next;
  hookup_column (view, l->data, glade, "rating_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "play_count_checkbutton");

  l = l->next;
  hookup_column (view, l->data, glade, "last_played_checkbutton");

  gtk_widget_show (window);
}

