/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 * Copyright (C) 2003 Johan Dahlin <jdahlin@gnome.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>
#include <gdk/gdkx.h>
#include <gtk/gtkbbox.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>
#include <gtk/gtkvbox.h>
#include <libgnome/gnome-i18n.h>
#include "utils.h"

#define KILOBYTE_FACTOR 1024.0
#define MEGABYTE_FACTOR 1048576.0
#define GIGABYTE_FACTOR 1073741824.0

#define MINUTE_FACTOR   (1000.0 * 60.0)
#define HOUR_FACTOR     (1000.0 * 60.0 * 60.0)
#define DAY_FACTOR      (1000.0 * 60.0 * 60.0 * 24.0)
#define WEEK_FACTOR     (1000.0 * 60.0 * 60.0 * 24.0 * 7.0) 

static GdkFilterReturn event_filter (GdkXEvent *xevent,
				     GdkEvent  *event,
				     gpointer   user_data);

static KeyCode         grab_key     (const char *keystring);

typedef struct {
  char *name;
  XF86Key key;
  KeyCode code;
} KeyEntry;

static KeyEntry keys[] = {
  { "XF86AudioNext", XF86AUDIO_NEXT, 0 },
  { "XF86AudioPrev", XF86AUDIO_PREV, 0 },
  { "XF86AudioPlay", XF86AUDIO_PLAY, 0 },
  { "XF86AudioStop", XF86AUDIO_STOP, 0 },
  { "XF86AudioPause", XF86AUDIO_PAUSE, 0 },
  { NULL, 0, 0 }
};

typedef struct
{
  KeyGrabFunction function;
  gpointer user_data;
} Helper;

GtkWidget *
hig_dialog_new (GtkWindow      *parent,
		GtkDialogFlags  flags,
		GtkMessageType  type,
		GtkButtonsType  buttons,
		const char      *header,
		const char      *messagefmt,
		...)
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *dialog_action_area;
  GtkWidget *hbox;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *image;
  char *title;
  va_list args;
  char *msg;
  char *hdr;

  if (messagefmt && messagefmt[0])
    {
      va_start (args, messagefmt);
      msg = g_strdup_vprintf (messagefmt, args);
      va_end (args);
    }
  else 
    msg = NULL;

  if (header) 
    hdr = g_markup_escape_text (header, -1);
  else 
    hdr = NULL;
	
  dialog = gtk_dialog_new ();
	
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
  gtk_window_set_title (GTK_WINDOW (dialog), "");
  
  dialog_vbox = GTK_DIALOG (dialog)->vbox;
  gtk_box_set_spacing (GTK_BOX (dialog_vbox), 12);

  hbox = gtk_hbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX (dialog_vbox), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width (GTK_CONTAINER (hbox), 5);
  gtk_widget_show (hbox);

  switch (type)
    {
    case GTK_MESSAGE_ERROR:
      image = gtk_image_new_from_stock ("gtk-dialog-error", GTK_ICON_SIZE_DIALOG);
      break;
      
    case GTK_MESSAGE_QUESTION:
      image = gtk_image_new_from_stock ("gtk-dialog-question", GTK_ICON_SIZE_DIALOG);
      break;
      
    case GTK_MESSAGE_INFO:
      image = gtk_image_new_from_stock ("gtk-dialog-info", GTK_ICON_SIZE_DIALOG);
      break;
      
    case GTK_MESSAGE_WARNING:
      image = gtk_image_new_from_stock ("gtk-dialog-warning", GTK_ICON_SIZE_DIALOG);
      break;
      
    default:
      image = NULL;
      g_assert_not_reached ();
    }
  
  gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
  gtk_widget_show (image);

  vbox = gtk_vbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
  gtk_widget_show (vbox);

  if (hdr)
    {
      title = g_strconcat ("<b>", hdr, "</b>", NULL);
      label = gtk_label_new (title);  
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
      gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
      gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
      gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
      gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
      gtk_widget_show (label);
      g_free (title);
    }
  
  if (msg && msg[0])
    {
      label = gtk_label_new (msg);
      gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
      gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
      gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
      gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
      gtk_widget_show (label);
    }
  
  dialog_action_area = GTK_DIALOG (dialog)->action_area;
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_END);
  
  switch (buttons)
    {		
    case GTK_BUTTONS_NONE:
      break;
      
    case GTK_BUTTONS_OK:
      button = gtk_button_new_from_stock (GTK_STOCK_OK);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_OK);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_show (button);
      
      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
      break;
      
    case GTK_BUTTONS_CLOSE:
      button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CLOSE);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_show (button);
		
      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
      break;
		
    case GTK_BUTTONS_CANCEL:
      button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CANCEL);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_show (button);
		
      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
      break;
		
    case GTK_BUTTONS_YES_NO:
      button = gtk_button_new_from_stock (GTK_STOCK_NO);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_NO);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_show (button);

      button = gtk_button_new_from_stock (GTK_STOCK_YES);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_YES);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
      gtk_widget_show (button);

      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_YES);
      break;
		
		
    case GTK_BUTTONS_OK_CANCEL:
      button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
      gtk_widget_show (button);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_CANCEL);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

      button = gtk_button_new_from_stock (GTK_STOCK_OK);
      gtk_widget_show (button);
      gtk_dialog_add_action_widget (GTK_DIALOG (dialog), button, GTK_RESPONSE_OK);
      GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);

      gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
      break;

    default:
      g_assert_not_reached ();
      break;
    }

  if (parent != NULL) 
    gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (parent));

  if (flags & GTK_DIALOG_MODAL)
    gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);

  if (flags & GTK_DIALOG_DESTROY_WITH_PARENT)
    gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
  
  g_free (msg);
  g_free (hdr);
  
  return dialog;
}

static GdkFilterReturn
event_filter (GdkXEvent *xevent,
	      GdkEvent  *event,
	      gpointer   user_data)
{
  XKeyEvent *keyevent = (XKeyEvent *) xevent;
  int i;
  Helper *helper = (Helper*) user_data;
  
  if (((XEvent *) xevent)->type != KeyPress)
    return GDK_FILTER_CONTINUE;
  
  for (i = 0; i < XF86AUDIO_MAX; i++)
    if (keys[i].code == keyevent->keycode)
      break;

  switch (keys[i].key) {
  case XF86AUDIO_PLAY:
  case XF86AUDIO_STOP:
  case XF86AUDIO_PREV:
  case XF86AUDIO_NEXT:
  case XF86AUDIO_PAUSE:
    helper->function (keys[i].key, helper->user_data);
    break;

  default:
    return GDK_FILTER_CONTINUE;
  }

  return GDK_FILTER_REMOVE;
}

static KeyCode
grab_key (const char *keystring)
{
  KeySym sym;
  KeyCode code;
  
  sym = XStringToKeysym (keystring);
  if (sym == NoSymbol)
    return 0;
  
  code = XKeysymToKeycode (GDK_DISPLAY (), sym);
  if (code == 0)
    return 0;
  
  gdk_error_trap_push ();

  XGrabKey (GDK_DISPLAY (), code,
	    0,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod2Mask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod5Mask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    LockMask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod2Mask | LockMask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod5Mask | LockMask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod2Mask | Mod5Mask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);
  XGrabKey (GDK_DISPLAY (), code,
	    Mod2Mask | Mod5Mask | LockMask,
	    GDK_ROOT_WINDOW (), True,
	    GrabModeAsync, GrabModeAsync);

  gdk_flush ();

  if (gdk_error_trap_pop ())
    {
      /* Silently fail... */
      return 0;
    }

  return code;
}

void
grab_keys (KeyGrabFunction function,
	   gpointer        user_data)
{
  int i;
  KeyCode code;
  Helper *helper;

  for (i = 0; keys[i].name; i++)
    {
      code = grab_key (keys[i].name);
      if (code)
	keys[i].code = code;
    }

  helper = g_new (Helper, 1);
  helper->function = function;
  helper->user_data = user_data;
  
  gdk_window_add_filter (gdk_get_default_root_window (),
			 event_filter,
			 helper);
}

char*
format_file_size_for_display (long long size)
{
  gdouble displayed_size;
  
  if (size < (long long) KILOBYTE_FACTOR) {
    if (size == 1)
      return g_strdup (_("1 byte"));
    else
      return g_strdup_printf (_("%u bytes"), (guint) size);
  } else {    
    if (size < (long long) MEGABYTE_FACTOR) {
      displayed_size = (gdouble) size / KILOBYTE_FACTOR;
      return g_strdup_printf (_("%.1f K"), displayed_size);
    } else if (size < (long long) GIGABYTE_FACTOR) {
      displayed_size = (gdouble) size / MEGABYTE_FACTOR;
      return g_strdup_printf (_("%.1f MB"), displayed_size);
    } else {
      displayed_size = (gdouble) size / GIGABYTE_FACTOR;
      return g_strdup_printf (_("%.1f GB"), displayed_size);
    }
  }
}

char *
format_time_nicely_for_display (guint time)
{
  time /= 1000;
  
  if (time > 3600)
    return g_strdup_printf (_("%d h %d min and %d sec"),
			    time / 3600,
			    (time % 3600) / 60,
			    (time % 3600) % 60);
  else if (time > 60)
    return g_strdup_printf (_("%d min %d sec"), time / 60, time % 60);
  else if (time > 1)
    return g_strdup_printf (_("%d sec"), time);
  else
    return g_strdup (_("1 sec"));
}

/* How to display times longer than 1 hour? 120:00 or 2:00:00? */
char *
format_time_for_display (guint time)
{
  time /= 1000;
  
  if (time > 3600)
    return g_strdup_printf ("%d:%02d:%02d",
			    time / 3600,
			    (time % 3600) / 60,
			    (time % 3600) % 60);
  else
    return g_strdup_printf ("%d:%02d", time / 60, time % 60);    
}

char *
format_length_for_display (guint time)
{
  if (time < MINUTE_FACTOR)
    if (time == 1)
      return g_strdup (_("1 second"));
    else
      return g_strdup_printf (_("%u seconds"), time / 1000);
  else
    {
      double displayed_len;
      
      if (time < HOUR_FACTOR)
	{
	  displayed_len = time / MINUTE_FACTOR;

	  return g_strdup_printf (_("%.1f minutes"), displayed_len);
	}
      else if (time < DAY_FACTOR)
	{
	  displayed_len = (double) time / HOUR_FACTOR;

	  return g_strdup_printf (_("%.1f hours"), displayed_len);
	}
      else if (time < WEEK_FACTOR)
	{
	  displayed_len = (double) time / DAY_FACTOR;

	  return g_strdup_printf (_("%.1f days"), displayed_len);
	}
      else
	{
	  displayed_len = (double) time / WEEK_FACTOR;

	  return g_strdup_printf (_("%.1f weeks"), displayed_len);
	}
    }
}

/* Mostly stolen from eel/eel-stock-dialogs.c */
#define RESPONSE_DETAILS 1000

static GtkDialog *
create_message_dialog (const char     *message,
                       const char     *dialog_title,
                       GtkMessageType  type,
                       GtkButtonsType  buttons_type,
                       GtkWindow      *parent)
{
  GtkWidget *box;
                                                                                                         
  g_assert (dialog_title != NULL);
  
  box = gtk_message_dialog_new (parent, 0, type, buttons_type, "%s", message);
  gtk_window_set_title (GTK_WINDOW (box), dialog_title);
  gtk_window_set_wmclass (GTK_WINDOW (box), "stock_dialog", "Eel");
  
  return GTK_DIALOG (box);
}
                                                                                                         
static GtkDialog *
show_message_dialog (const char     *message,
                     const char     *dialog_title,
                     GtkMessageType  type,
                     GtkButtonsType  buttons_type,
                     const char     *additional_button_label,
                     int             additional_button_response_id,
                     GtkWindow      *parent)
{
  GtkDialog *dialog;
  GtkWidget *additional_button;
  GtkWidget *box;
  
  dialog = create_message_dialog (message, dialog_title, type,
				  buttons_type, parent);
  
  if (additional_button_label != NULL)
    {
      additional_button = gtk_dialog_add_button (dialog,
						 additional_button_label,
						 additional_button_response_id);
      box = gtk_widget_get_parent (additional_button);
      if (GTK_IS_BOX (box))
	gtk_box_reorder_child (GTK_BOX (box), additional_button, 0);
    }
  
  gtk_widget_show (GTK_WIDGET (dialog));
  
  return dialog;
}


static void
details_dialog_response_callback (GtkDialog  *dialog,
                                  int         response_id,
                                  const char *detailed_message)
{
  if (response_id == RESPONSE_DETAILS)
    {
      gtk_label_set_text (GTK_LABEL (GTK_MESSAGE_DIALOG (dialog)->label),
			  detailed_message);
      gtk_dialog_set_response_sensitive (dialog, RESPONSE_DETAILS, FALSE);
    }
  else
    gtk_object_destroy (GTK_OBJECT (dialog));

}

static GtkDialog *
show_ok_dialog (const char     *message,
                const char     *dialog_title,
                GtkMessageType  type,
                GtkWindow      *parent)
{
  GtkDialog *dialog;
 
  dialog = show_message_dialog (message, dialog_title, type,
				GTK_BUTTONS_OK, NULL, 0, parent);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
  
  g_signal_connect (dialog, "response", G_CALLBACK (gtk_object_destroy), NULL);
 
  return dialog;
}

GtkDialog *
eel_show_error_dialog (const char *error,
                       const char *dialog_title,
                       GtkWindow  *parent)
{
  return show_ok_dialog (error,
			 dialog_title == NULL ? _("Error") : dialog_title,
			 GTK_MESSAGE_ERROR, parent);
}

GtkDialog *
eel_show_error_dialog_with_details (const char *error_message,
                                    const char *dialog_title,
                                    const char *detailed_error_message,
                                    GtkWindow  *parent)
{
  GtkDialog *dialog;
  GClosure* closure;
 
  g_return_val_if_fail (error_message != NULL, NULL);
  g_return_val_if_fail (parent == NULL || GTK_IS_WINDOW (parent), NULL);
  
  if (detailed_error_message == NULL
      || strcmp (error_message, detailed_error_message) == 0)
    return eel_show_error_dialog (error_message, dialog_title, parent);
  
  dialog = show_message_dialog (error_message,
				dialog_title == NULL ? _("Error") : dialog_title,
				GTK_MESSAGE_ERROR,
				GTK_BUTTONS_OK, _("Details"), RESPONSE_DETAILS,
				parent);
         
  closure = g_cclosure_new (G_CALLBACK (details_dialog_response_callback),
			    g_strdup (detailed_error_message),
			    (GClosureNotify) g_free);
    
  /* Show the details when you click on the details button. */
  g_signal_connect_closure (G_OBJECT (dialog), "response", closure, FALSE);
  return dialog;
}
