/* -*- mode: C; c-file-style: "gnu" -*- */
/*
 * Copyright (C) 2003 Richard Hult <richard@imendio.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* Small wrapper around the id3 tag library of the month, so we can swap it
 * easily.
 */

#include <config.h>
#include <stdlib.h>
#include <id3tag.h>
#include "id3-tag.h"

static char *genres[] = {
  "Blues", "Classic Rock", "Country", "Dance", "Disco", "Funk", "Grunge",
  "Hip-Hop", "Jazz", "Metal", "New Age", "Oldies", "Other", "Pop", "R&B",
  "Rap", "Reggae", "Rock", "Techno", "Industrial", "Alternative", "Ska",
  "Death Metal", "Pranks", "Soundtrack", "Euro-Techno", "Ambient", "Trip-Hop",
  "Vocal", "Jazz+Funk", "Fusion", "Trance", "Classical", "Instrumental",
  "Acid", "House", "Game", "Sound Clip", "Gospel", "Noise", "Alternative Rock",
  "Bass", "Soul", "Punk", "Space", "Meditative", "Instrumental Pop",
  "Instrumental Rock", "Ethnic", "Gothic", "Darkwave", "Techno-Industrial",
  "Electronic", "Pop-Folk", "Eurodance", "Dream", "Southern Rock", "Comedy",
  "Cult", "Gangsta", "Top 40", "Christian Rap", "Pop/Funk", "Jungle",
  "Native American", "Cabaret", "New Wave", "Psychadelic", "Rave", "Showtunes",
  "Trailer", "Lo-Fi", "Tribal", "Acid Punk", "Acid Jazz", "Polka", "Retro",
  "Musical", "Rock & Roll", "Hard Rock", "Folk", "Folk/Rock", "National Folk",
  "Swing", "Fast-Fusion", "Bebob", "Latin", "Revival", "Celtic", "Bluegrass",
  "Avantgarde", "Gothic Rock", "Progressive Rock", "Psychedelic Rock",
  "Symphonic Rock", "Slow Rock", "Big Band", "Chorus", "Easy Listening",
  "Acoustic", "Humour", "Speech", "Chanson", "Opera", "Chamber Music",
  "Sonata", "Symphony", "Booty Bass", "Primus", "Porn Groove", "Satire",
  "Slow Jam", "Club", "Tango", "Samba", "Folklore", "Ballad", "Power Ballad",
  "Rhytmic Soul", "Freestyle", "Duet", "Punk Rock", "Drum Solo", "Acapella",
  "Euro-House", "Dance Hall", "Goa", "Drum & Bass", "Club-House", "Hardcore",
  "Terror", "Indie", "BritPop", "Negerpunk", "Polsk Punk", "Beat",
  "Christian Gangsta Rap", "Heavy Metal", "Black Metal", "Crossover",
  "Contemporary Christian", "Christian Rock"
};

char **id3_genres = genres;

static char *
tag_get_string (struct id3_tag *tag, const char *field_name)
{
  unsigned int nstrings, j; 
  const struct id3_frame *frame;
  const union id3_field *field;
  const id3_ucs4_t *ucs4;
  id3_utf8_t *utf8 = NULL;
  char *ret = NULL;
  
  frame = id3_tag_findframe (tag, field_name, 0);
  if (frame == 0)
    return NULL;
  
  field = &frame->fields[1];
  nstrings = id3_field_getnstrings (field);
  for (j = 0; j < nstrings; j++)
    {
      ucs4 = id3_field_getstrings (field, j);
      if (ucs4)
	{
	  utf8 = id3_ucs4_utf8duplicate (ucs4);
	  break;
	}
    }

  if (utf8)
    {
      ret = g_strdup (g_strstrip (utf8));
      free (utf8);
    }

  return ret;
}

static int
tag_get_int (Id3Tag *tag, const char *field)
{
  char *str;
  char *endptr = NULL;
  int ret;
  
  str = tag_get_string (tag, field);
  if (!str)
    return -1;
  
  ret = strtol ((char *) str, &endptr, 10);
  if (endptr == (char *) str)
    ret = -1;
  
  g_free (str);
  
  return ret;
}

char *
id3_tag_get_title (Id3Tag *tag)
{
  return tag_get_string (tag, ID3_FRAME_TITLE);
}

char *
id3_tag_get_artist (Id3Tag *tag)
{
  return tag_get_string (tag, ID3_FRAME_ARTIST);
}

char *
id3_tag_get_album (Id3Tag *tag)
{
  return tag_get_string (tag, ID3_FRAME_ALBUM);
}

int
id3_tag_get_genre (Id3Tag *tag)
{
  return tag_get_int (tag, ID3_FRAME_GENRE);
}

int
id3_tag_get_year (Id3Tag *tag)
{
  return tag_get_int (tag, ID3_FRAME_YEAR);
}

int
id3_tag_get_number (Id3Tag *tag)
{
  return tag_get_int (tag, ID3_FRAME_TRACK);
}
