package org.gnu.libebook;

import org.gnu.glib.EventMap;
import org.gnu.glib.EventType;
import org.gnu.glib.GObject;
import org.gnu.glib.Handle;
import org.gnu.libedataserver.EDSObject;

import java.util.ArrayList;

/**
 * EBookView - Represents a search against an {@link EBook}.
 */
public class EBookView extends EDSObject
{
    private static EventMap _eventMap = new EventMap();
    private ArrayList _eBookViewListeners;

    static {
        addEvents(_eventMap);
    }

    /**
     * Construct a new EBookView using the handle returned by a call to the native layer.
     *
     * @param handle The handle (which is retrieved from a native layer) which represents an EBookView
     */
    public EBookView(Handle handle)
    {
        super(handle);
    }

    protected static EBookView getEBookView(Handle handle)
    {
        if (handle == null)
            return null;

        EBookView obj = (EBookView) GObject.getGObjectFromHandle(handle);
        return obj == null ? new EBookView(handle) : obj;
    }

    /**
     * This adds a listener to the EBookView, it is required to, at the very least, gather the contacts returned from a search against an EBook
     *
     * @param listener An {@link EBookViewListener}
     */
    public void addListener(EBookViewListener listener)
    {
        if (_eBookViewListeners == null)
            initialiseListenerList();

        if (findListener(listener) == -1)
            _eBookViewListeners.add(listener);
    }

    private void initialiseListenerList()
    {
        _eventMap.initialize(this, EBookViewEvent.Type.ADDED);
        _eventMap.initialize(this, EBookViewEvent.Type.CHANGED);
        _eventMap.initialize(this, EBookViewEvent.Type.COMPLETE);
        _eventMap.initialize(this, EBookViewEvent.Type.REMOVED);
        _eventMap.initialize(this, EBookViewEvent.Type.STATUS_MESSAGE);
        _eBookViewListeners = new ArrayList();
    }

    protected int findListener(Object listener)
    {
        return _eBookViewListeners == null || listener == null ? -1 : _eBookViewListeners.indexOf(listener);
    }

    public Class getEventListenerClass(String signal)
    {
        Class className = _eventMap.getEventListenerClass(signal);
        if (className == null)
            className = super.getEventListenerClass(signal);
        return className;
    }

    public EventType getEventType(String signal)
    {
        EventType eventType = _eventMap.getEventType(signal);
        if (eventType == null)
            eventType = super.getEventType(signal);
        return eventType;
    }

    /**
     * Start the EBookView search. This should be eventually stopped.
     */
    public void start()
    {
        e_book_view_start(getHandle());
    }

    /**
     * Stops the EBookView search.
     */
    public void stop()
    {
        e_book_view_stop(getHandle());
    }

    //SC: handle is a glist of econtacts
    private void handleAdded(Handle handle)
    {
        EContact[] contacts = getEContactsFromHandle(handle);
        if (_eBookViewListeners != null)
            for (int i1 = 0; i1 < _eBookViewListeners.size(); i1++)
                ((EBookViewListener) _eBookViewListeners.get(i1)).onAdd(contacts);
    }

    private EContact[] getEContactsFromHandle(Handle handle)
    {
        Handle[] contactHandles = e_book_view_get_contacts_from_glist(handle);
        EContact[] contacts = new EContact[contactHandles.length];
        for (int i = 0; i < contactHandles.length; i++)
            contacts[i] = EContact.getEContact(contactHandles[i]);
        return contacts;
    }

    //SC: handle is a glist of econtacts
    private void handleChanged(Handle handle)
    {
        EContact[] contacts = getEContactsFromHandle(handle);
        if (_eBookViewListeners != null)
            for (int i1 = 0; i1 < _eBookViewListeners.size(); i1++)
                ((EBookViewListener) _eBookViewListeners.get(i1)).onChanged(contacts);
    }

    //SC: handle is a glist of ids
    private void handleRemoved(Handle handle)
    {
        if (_eBookViewListeners != null)
            for (int i = 0; i < _eBookViewListeners.size(); i++)
                ((EBookViewListener) _eBookViewListeners.get(i)).onRemove();
    }

    private void handleComplete(int something)
    {
        if (_eBookViewListeners != null)
            for (int i = 0; i < _eBookViewListeners.size(); i++)
                ((EBookViewListener) _eBookViewListeners.get(i)).onComplete();
    }

    private void handleStatusMessage(String message)
    {
        if (_eBookViewListeners != null)
            for (int i = 0; i < _eBookViewListeners.size(); i++)
                ((EBookViewListener) _eBookViewListeners.get(i)).onStatusMessage(message);
    }

    private static void addEvents(EventMap anEvtMap)
    {
        anEvtMap.addEvent("contacts-added", "handleAdded", EBookViewEvent.Type.ADDED, EBookViewListener.class);
        anEvtMap.addEvent("contacts-changed", "handleChanged", EBookViewEvent.Type.CHANGED, EBookViewListener.class);
        anEvtMap.addEvent("contacts-removed", "handleRemoved", EBookViewEvent.Type.REMOVED, EBookViewListener.class);
        anEvtMap.addEvent("sequence-complete", "handleComplete", EBookViewEvent.Type.COMPLETE, EBookViewListener.class);
        anEvtMap.addEvent("status-message", "handleStatusMessage", EBookViewEvent.Type.STATUS_MESSAGE, EBookViewListener.class);
    }

    native static private void e_book_view_start(Handle handle);

    native static private void e_book_view_stop(Handle handle);

    native static private Handle[] e_book_view_get_contacts_from_glist(Handle handle);
}