/* -*- Mode: C; c-set-style: gnu indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/* gnome-url.c
 * Copyright (C) 1998, James Henstridge <james@daa.com.au>
 * Copyright (C) 1999, 2000 Red Hat, Inc.
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,  59 Temple Place - Suite 330, Cambridge, MA 02139, USA.
 */
/*
  @NOTATION@
 */

#include <config.h>
#include <string.h>
#include <glib.h>
#include <limits.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>

#include "gnome-i18nP.h"

#include <gconf/gconf-client.h>

#include "gnome-exec.h"
#include "gnome-util.h"
#include "gnome-init.h"
#include "gnome-i18n.h"
#include "gnome-gconfP.h"

#include "gnome-url.h"

#include <popt.h>

#define DEFAULT_HANDLER "mozilla"
#define HELP_URIS_HANDLER "gnome-help"

static gchar *
gnome_url_default_handler (void)
{
	static gchar *default_handler = NULL;
	
	if (!default_handler) {
		gchar *str, *app;
		GConfClient *client;

		/* init our gconf stuff if necessary */
		gnome_gconf_lazy_init ();

		client = gconf_client_get_default ();
		
		str = gconf_client_get_string (client, "/desktop/gnome/url-handlers/default-show", NULL);

		if (str) {
			default_handler = str;
		} else {
			/* It's the first time gnome_url_show is run, so up some useful defaults */

			app = g_find_program_in_path (HELP_URIS_HANDLER);
			if (app) {
				g_free (app);
				app = HELP_URIS_HANDLER " \"%s\"";
			} else
				app = DEFAULT_HANDLER " \"%s\"";

			default_handler = DEFAULT_HANDLER " \"%s\"";
			gconf_client_set_string (client, "/desktop/gnome/url-handlers/default-show",
						 default_handler, NULL);

			if (gconf_client_dir_exists (client, "/desktop/gnome/url-handlers/info-show", NULL) == FALSE) {
				gconf_client_set_string (client, "/desktop/gnome/url-handlers/info-show", app, NULL);
			}

			if (gconf_client_dir_exists (client, "/desktop/gnome/url-handlers/man-show", NULL) == FALSE) {
				gconf_client_set_string (client, "/desktop/gnome/url-handlers/man-show", app, NULL);
			}

			if (gconf_client_dir_exists (client, "/desktop/gnome/url-handlers/ghelp-show", NULL) == FALSE) {
				gconf_client_set_string (client, "/desktop/gnome/url-handlers/ghelp-show", app, NULL);
			}

			gconf_client_suggest_sync (client, NULL);
		}

		g_object_unref (G_OBJECT (client));
	}
	
	return g_strdup (default_handler);
}

/**
 * gnome_url_show
 * @url: The url to display. Should begin with the protocol to use (e.g.
 * "http:", "ghelp:", etc)
 * @error: Used to store any errors that result from trying to display the @url.
 *
 * Displays the given URL in an appropriate viewer. The appropriate viewer is
 * user definable. It is determined by extracting the protocol from the @url,
 * then seeing if the /desktop/gnome/url-handlers/<protocol>-show key exists in
 * the configuration database. It it does, this entry is used as the template
 * for the command. 
 *
 * If no protocol specific handler exists, the
 * /desktop/gnome/url-handlers/default-show key is used to determine the
 * viewer and if that doesn't exist a compiled in default is used (Nautilus,
 * or failing that gnome-help-browser).
 *
 * Once a viewer is determined, it is called with the @url as a parameter. If
 * any errors occur, they are returned in the @error parameter. These errors
 * will either be in the %GNOME_URL_ERROR or the %G_SPAWN_ERROR domains.
 *
 * Returns: %TRUE if everything went fine, %FALSE otherwise (in which case
 * @error will contain the actual error).
 */
gboolean
gnome_url_show (const gchar *url, GError **error)
{
	gint i;
	gchar *pos, *template;
	int argc;
	const char **argv;
	char **newargv;
	gboolean ret;
	gchar *tmp;
	
	g_return_val_if_fail (url != NULL, FALSE);

	pos = strchr (url, ':');

	if (pos != NULL) {
		gchar *protocol, *path;
		GConfClient *client;
		
		g_return_val_if_fail (pos >= url, FALSE);

		protocol = g_new (gchar, pos - url + 1);
		strncpy (protocol, url, pos - url);
		protocol[pos - url] = '\0';
		g_ascii_strdown (protocol, -1);

		/* init our gconf stuff if necessary */
		gnome_gconf_lazy_init ();

		path = g_strconcat ("/desktop/gnome/url-handlers/", protocol, "-show", NULL);
		client = gconf_client_get_default ();

		template = gconf_client_get_string (client, path, NULL);

		if (template == NULL) {
			gchar* template_temp;
			
			template_temp = gnome_url_default_handler ();
						
			/* Retry to get the right url handler */
			template = gconf_client_get_string (client, path, NULL);

			if (template == NULL) 
				template = template_temp;
			else
				g_free (template_temp);

		}
		
		g_free (path);
		g_free (protocol);

		g_object_unref (G_OBJECT (client));
	} else {
		/* no ':' ? this shouldn't happen. Use default handler */
		template = gnome_url_default_handler ();
	}

	/* We use a popt function as it does exactly what we want to do and
	   gnome already uses popt */
	if (poptParseArgvString (template, &argc,
				 /* casting is evil, but necessary */&argv) != 0) {
		/* can't parse */
		g_set_error (error,
			     GNOME_URL_ERROR,
			     GNOME_URL_ERROR_PARSE,
			     _("Cannot parse: %s"),
			     template);
		g_free (template);
		return FALSE;
	}

	newargv = g_new0 (char *, argc + 1);
	for (i = 0; i < argc; i++) {
		if (strcmp (argv[i], "%s") == 0)
			newargv[i] = g_strdup (url);
		else
			newargv[i] = g_strdup (argv[i]);
	}
	newargv[i] = NULL;
	
	tmp = g_strjoinv (" ", newargv);
	g_strfreev (newargv);
	newargv = g_new0 (char *, 4);
	newargv[0] = gnome_util_user_shell ();
	newargv[1] = g_strdup ("-c");
	newargv[2] = tmp;
	newargv[3] = NULL;

	/* the way the poptParseArgvString works is that the entire thing
	 * is allocated as one buffer, so just free will suffice, also
	 * it must be free and not g_free */
	free (argv);
	
	/* This can return some errors */
	ret = g_spawn_async (NULL /* working directory */,
			     newargv,
			     NULL /* envp */,
			     0 /* flags */,
			     NULL /* child_setup */,
			     NULL /* data */,
			     NULL /* child_pid */,
			     error);

	g_strfreev (newargv);
	g_free (template);

	return ret;
}

/**
 * gnome_url_error_quark
 *
 * Returns: A quark representing gnome-url module errors.
 */
GQuark
gnome_url_error_quark (void)
{
	static GQuark error_quark = 0;

	if (error_quark == 0)
		error_quark =
			g_quark_from_static_string ("gnome-url-error-quark");

	return error_quark;
}
